#include <dlfcn.h>
#include <stdio.h>
#include <string.h>
#include <stdint.h>

#include "ql_tee_service.h"
#include "mbtk_log.h"
#include <tee_client_api.h>


struct test_ctx {
	TEEC_Context ctx;
	TEEC_Session sess;
};


struct test_ctx ctx;

const void *obj_id;
uint32_t obj_size;

#define lib_secure_path "/lib/libsecure_storage.so"
static void *dlHandle_secure;




int (*prepare_tee_session)(struct test_ctx *ctx);
void (*terminate_tee_session)(struct test_ctx *ctx);
TEEC_Result (*read_secure_object)(struct test_ctx *ctx, const void *id, uint32_t id_size, char *data, size_t data_len);
TEEC_Result (*write_secure_object)(struct test_ctx *ctx, const void *id, uint32_t id_size, char *data, size_t data_len);
TEEC_Result (*delete_secure_object)(struct test_ctx *ctx, const void *id, uint32_t id_size);


static int tee_api_import(void)
{
    dlHandle_secure = dlopen(lib_secure_path, RTLD_NOW);
    if (dlHandle_secure == NULL) 
    {
        return -1;
    }

    prepare_tee_session = (int (*)(struct test_ctx *ctx))dlsym(dlHandle_secure, "prepare_tee_session");
    if (prepare_tee_session == NULL) 
    {
        LOGE("prepare_tee_session dlsym fail\n");
        return -1;
    }

    terminate_tee_session = (void (*)(struct test_ctx *ctx))dlsym(dlHandle_secure, "terminate_tee_session");
    if (terminate_tee_session == NULL) 
    {
        LOGE("terminate_tee_session dlsym fail\n");
        return -1;
    }

    read_secure_object = (TEEC_Result (*)(struct test_ctx *ctx, const void *id, uint32_t id_size,char *data, size_t data_len))dlsym(dlHandle_secure, "read_secure_object");
    if (read_secure_object == NULL) 
    {
        LOGE("read_secure_object dlsym fail\n");
        return -1;
    }

    write_secure_object = (TEEC_Result (*)(struct test_ctx *ctx, const void *id, uint32_t id_size, char *data, size_t data_len))dlsym(dlHandle_secure, "write_secure_object");
    if (write_secure_object == NULL) 
    {
        LOGE("write_secure_object dlsym fail\n");
        return -1;
    }

    delete_secure_object = (TEEC_Result (*)(struct test_ctx *ctx, const void *id, uint32_t id_size))dlsym(dlHandle_secure, "delete_secure_object");
    if (delete_secure_object == NULL) 
    {
        LOGE("delete_secure_object dlsym fail\n");
        return -1;
    }

    return 0;
}

/**
* @brief init tee sdk
* @param  [in] None
* @param  [out] None
* @retval GSW_HAL_SUCCESS is success\other is fail
*/
ql_tee_error_t ql_ss_initialize(void)
{
    int32_t ret = 0;
    ret = tee_api_import();
    if(ret)
    {
	LOGE("tee_api_import fail\n");
        return ret;
    }
    ret = prepare_tee_session(&ctx);

    return ret;
}

void ql_ss_deinitialize(void)
{
    
     terminate_tee_session(&ctx);

}
ql_tee_error_t ql_ss_open(const void *id, uint32_t id_size, uint32_t *object)
{

    obj_id = id;
    obj_size = id_size;
    return 0;
}

ql_tee_error_t ql_ss_close(uint32_t object)
{
   
    obj_id = NULL;
    obj_size = 0;
    return 0;

}

/**
* @brief read sensitive data from tee
* @param  [in] char* in_obj_name :Sensitive data name
* @param  [in] unsigned int* p_out_buf_len：The size of sensitive data output cache
* @param  [out] char* out_buf：Cache of sensitive data output
* @param  [out] unsigned int* p_out_buf_len：Sensitive data length
* @retval GSW_HAL_SUCCESS is success\other is fail
*/

ql_tee_error_t ql_ss_read(uint32_t object, void *data, uint32_t data_size, uint32_t *count)
{
    int32_t ret = 0;
    TEEC_Result res;

    res = read_secure_object(&ctx, obj_id, obj_size, data, data_size);
    if (res != TEEC_SUCCESS)
    {
	LOGE("Failed to read an object from the secure storage");
        ret = -1;
    }

    *count = strlen(data);
    return ret;
}


/**
* @brief write sensitive data to tee
* @param  [in] char* in_obj_name :Sensitive data name
* @param  [in] char* in_buf：A cache for writing sensitive data
* @param  [out] unsigned int in_buf_len：Sensitive data length
* @retval GSW_HAL_SUCCESS is success\other is fail
*/
ql_tee_error_t ql_ss_write(uint32_t object, void *data, uint32_t data_size)
{
    int32_t ret = 0;
    TEEC_Result res;
    res = write_secure_object(&ctx, obj_id, obj_size, data, data_size);
    if (res != TEEC_SUCCESS)
    {
	LOGE("Failed to write an object from the secure storage");
        ret = -1;
    }

    return ret;
}


/**
* @brief delete sensitive data from tee
* @param  [in] char* in_obj_name :Sensitive data name
* @retval GSW_HAL_SUCCESS is success\other is fail
*/
ql_tee_error_t ql_ss_unlink(uint32_t object)
{
    int32_t ret = 0;
    TEEC_Result res;
    res = delete_secure_object(&ctx, obj_id, obj_size);
    if (res != TEEC_SUCCESS)
    {
	LOGE("Failed to delete the object: 0x%x", res);
        ret = -1;
    }


    return ret;

}




