/*-----------------------------------------------------------------------------------------------*/
/**
  @file main_apn.c
  @brief Example how to set and get APN configuration
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2018 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  20190624   tyler.kuang  Created .
-------------------------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <stdint.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "ql_type.h"
#include "ql_data_call.h"

const char *g_options_short = "a:n:v:f:u:p:sg";

struct option g_options_long[] = {
    {"set",        no_argument,          0,     's'},
    {"get",        no_argument,          0,     'g'},
    {"apn_id",     required_argument,    0,     'a'},
    {"apn_name",   required_argument,    0,     'n'},
    {"ip_ver",     required_argument,    0,     'v'},
    {"auth_pref",  required_argument,    0,     'f'},
    {"username",   required_argument,    0,     'u'},
    {"password",   required_argument,    0,     'p'},
    {NULL,         0,                    0,      0 }
};

#define USAGE_STRING "exec --[set|s]|[get|g], set or get apn configuration\r\n"\
    "     --apn_id|a number, apn id, range:1-16\r\n"\
    "     [--ip_ver|v number], ip vresion:4-IPV4,6-IPV6,10-IPV4V6\r\n"\
    "     [--apn_name|n string], apn name\r\n"\
    "     [--auth_pref|f number], auth pref:0-none,1-pap,2-chap,3-pan&chap\r\n"\
    "     [--username|u string], \r\n"\
    "     [--password|p string], auth pref:0-none,1-pap,2-chap,3-pan&chap\r\n"\
    "exp : exec -g -a 1\r\n " \
    "      exec -s -a 4 -v 4 -n test_apn -f 0 -u test_username -p test_password\r\n "


void usage(void)
{
    printf("%s", USAGE_STRING);
}


int main(int argc, char *argv[])
{
    int c;
    int dat;
    int ret;
    int opt_idx = 0;
    int retry_cnt = 0;

    int opt_type = -1; /*1-get 2-set*/
    int apn_id = -1;
    int ip_ver = QL_NET_IP_VER_V4;
    int auth_pref = QL_NET_AUTH_PREF_PAP_CHAP_NOT_ALLOWED;
    char apn_name[128] = {0};
    char username[128] = {0};
    char password[128] = {0};

    ql_data_call_apn_config_t cfg;

    while(1)
    {
        c = getopt_long(argc, argv, g_options_short, g_options_long, &opt_idx);
        if(c==-1)
        {
            break;
        }

        switch(c) {
            case 's':
                opt_type = 2;
                break;
            case 'g':
                opt_type = 1;
                break;

            case 'a':
                apn_id = atoi(optarg);
                if(apn_id<=0 || apn_id>QL_NET_MAX_APN_ID)
                {
                    printf("Invalid apn_id : %d\n", apn_id);
                    return -1;
                }

                break;
            case 'f':
                dat = atoi(optarg);
                switch(dat)
                {
                    case 0:
                        auth_pref = QL_NET_AUTH_PREF_PAP_CHAP_NOT_ALLOWED;
                        break;
                    case 1:
                        auth_pref = QL_NET_AUTH_PREF_PAP_ONLY_ALLOWED;
                        break;
                    case 2:
                        auth_pref = QL_NET_AUTH_PREF_CHAP_ONLY_ALLOWED;
                        break;
                    case 3:
                        auth_pref = QL_NET_AUTH_PREF_PAP_CHAP_BOTH_ALLOWED;
                        break;
                    default:
                        printf("Invalid auth pref : %d\n", dat);
                        return -1;
                }
                break;
            case 'v':
                dat = atoi(optarg);
                if(4 == dat)
                {
                    ip_ver = QL_NET_IP_VER_V4;
                }
                else if(6 == dat)
                {
                    ip_ver = QL_NET_IP_VER_V6;
                }
                else if(10 == dat)
                {
                    ip_ver = QL_NET_IP_VER_V4V6;
                }
                else
                {
                    printf("Invalid ip_ver : %d\n", dat);
                    return -1;
                }
                break;

            case 'n':
                strncpy(apn_name, optarg, sizeof(apn_name)-1);
                break;

            case 'u':
                strncpy(username, optarg, sizeof(username)-1);
                break;

            case 'p':
                strncpy(password, optarg, sizeof(password)-1);
                break;

            default:
                usage();
                return 0;
        }
    }

    if(opt_type!=1 && opt_type!=2)
    {
        usage();
        return 0;
    }

    if(apn_id<=0 || apn_id>QL_NET_MAX_APN_ID)
    {
        usage();
        return 0;
    }

    retry_cnt = 20*1000/100; /** timeout : 20S */
    while(retry_cnt>0)
    {
        ret = ql_data_call_init();
        if(ret == QL_ERR_SERVICE_NOT_READY)
        {
            retry_cnt--;
            usleep(100*1000); /** sleep 100ms */
        }
        break;
    }

    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_init, ret=%d", ret);
        return -1;
    }


    /** get apn info */
    if(opt_type == 1)
    {
        memset(&cfg, 0 ,sizeof(cfg));
        ret = ql_data_call_get_apn_config(apn_id, &cfg);
        if(ret != QL_ERR_OK)
        {
            printf("Failed to get apn configuration, ret=%d\n", ret);
            return -1;
        }
        printf("Succeed to get apn configuration\n");
        printf("apn_id     : %d\n", apn_id);
        printf("ip_ver     : %d\n", cfg.ip_ver);
        printf("auth_pref  : %d\n", cfg.auth_pref);
        printf("apn_name   : %s\n", cfg.apn_name);
        printf("username   : %s\n", cfg.username);
        printf("password   : %s\n", cfg.password);
    }
    else
    {
        memset(&cfg, 0, sizeof(cfg));
        cfg.ip_ver = ip_ver;
        cfg.auth_pref = auth_pref;
        strncpy(cfg.apn_name, apn_name, sizeof(cfg.apn_name)-1);
        strncpy(cfg.username, username, sizeof(cfg.username)-1);
        strncpy(cfg.password, password, sizeof(cfg.password)-1);

        printf("Start to set apn configuration\n");
        printf("apn_id     : %d\n", apn_id);
        printf("ip_ver     : %d\n", cfg.ip_ver);
        printf("auth_pref  : %d\n", cfg.auth_pref);
        printf("apn_name   : %s\n", cfg.apn_name);
        printf("username   : %s\n", cfg.username);
        printf("password   : %s\n", cfg.password);

        ret = ql_data_call_set_apn_config(apn_id, &cfg);
        if(ret != QL_ERR_OK)
        {
            printf("Failed to set apn configuration, ret=%d\n", ret);
            return -1;
        }
        printf("Succeed to set apn configuration\n");
    }


    return 0;
}

