/*-----------------------------------------------------------------------------------------------*/
/**
  @file main_data_call.c
  @brief Example how to data call API
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2018 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  20190624   tyler.kuang  Created .
-------------------------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <stdint.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "ql_type.h"
#include "ql_data_call.h"

const char *g_options_short = "i:n:a:v:r:dhd";

struct option g_options_long[] = {
    {"call_id",    required_argument,    0,     'i'},
    {"call_name",  required_argument,    0,     'n'},
    {"apn_id",     required_argument,    0,     'a'},
    {"ip_ver",     required_argument,    0,     'v'},
    {"reconnect",  required_argument,    0,     'r'},
    {"default",    no_argument,          0,     'd'},
    {"help",       no_argument,          0,     'h'},
    {NULL,         0,                    0,      0 }
};

#define USAGE_STRING "exec --call_id|i,data call ID\r\n"\
                     "     [--call_name|n], data call name\r\n"\
                     "     --apn_id|a, APN ID, range:1-16\r\n"\
                     "     [--ip_ver|v], IP vresion:4-IPV4,6-IPV6\r\n"\
                     "     [--reconnect|r] reconnect interval in SEC\r\n"\
                     "     [--default|d], set to system default network\r\n"\
                     "exp : exec -a 1 -n test_network -i 4 -d\r\n "

int g_is_default_network = 0;
int g_call_id = -1;

void usage(void)
{
    printf("%s", USAGE_STRING);
}

void data_call_status_ind_cb(int call_id,
        QL_NET_DATA_CALL_STATUS_E pre_call_status,
        ql_data_call_status_t *p_msg)
{
    FILE *fp;
    char cmd_buf[256];

    /** not for this data call */
    if(call_id != g_call_id)
    {
        return;
    }

    printf("data call status change, form %d to %d\n", pre_call_status, p_msg->call_status);
    if(p_msg->call_status == QL_NET_DATA_CALL_STATUS_CONNECTED)
    {
        printf("call_id     : %d\n", p_msg->call_id);
        printf("call_name   : %s\n", p_msg->call_name);
        printf("device_name : %s\n", p_msg->device);
        if(p_msg->has_addr)
        {
            printf("IPV4 addr    : %s\n", p_msg->addr.addr);
            printf("IPV4 gateway : %s\n", p_msg->addr.gateway);
            printf("IPV4 netmask : %s\n", p_msg->addr.netmask);
            printf("IPV4 dnsp    : %s\n", p_msg->addr.dnsp);
            printf("IPV4 dnss    : %s\n", p_msg->addr.dnss);

            if(g_is_default_network)
            {
                /** set system default router */
                snprintf(cmd_buf, sizeof(cmd_buf), "ip ro add default via %s dev %s",
                        p_msg->addr.gateway, p_msg->device);
                system(cmd_buf);

                system("iptables -t filter -F");

                snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t nat -A POSTROUTING -o %s -j MASQUERADE", p_msg->device);
                system(cmd_buf);

                /** set system dns configuration*/
                fp = fopen("/tmp/resolv_v4.conf", "w");
                if(!fp)
                {
                    printf("Failed to write resolv file, err=%s\n", strerror(errno));
                    return;
                }

                if(p_msg->addr.dnsp[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr.dnsp);
                }

                if(p_msg->addr.dnss[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr.dnss);
                }
                fclose(fp);

                system("echo \"\" > /etc/resolv.conf");
                if(access("/tmp/resolv_v4.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v4.conf >> /etc/resolv.conf");
                }
                if(access("/tmp/resolv_v6.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v6.conf >> /etc/resolv.conf");
                }
            }
        }

        if(p_msg->has_addr6)
        {
            printf("IPV6 addr    : %s\n", p_msg->addr6.addr);
            printf("IPV6 gateway : %s\n", p_msg->addr6.gateway);
            printf("IPV6 netmask : %s\n", p_msg->addr6.prefix);
            printf("IPV6 dnsp    : %s\n", p_msg->addr6.dnsp);
            printf("IPV6 dnss    : %s\n", p_msg->addr6.dnss);

            if(g_is_default_network)
            {
                /** set system default router */
                snprintf(cmd_buf, sizeof(cmd_buf), "ip -6 ro add default via %s dev %s",
                        p_msg->addr6.gateway, p_msg->device);
                system(cmd_buf);

                /** set system dns configuration*/
                fp = fopen("/tmp/resolv_v6.conf", "w");
                if(!fp)
                {
                    printf("Failed to write resolv file, err=%s\n", strerror(errno));
                    return;
                }

                if(p_msg->addr6.dnsp[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr6.dnsp);
                }

                if(p_msg->addr6.dnss[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr6.dnss);
                }
                fclose(fp);

                system("echo \"\" > /etc/resolv.conf");
                if(access("/tmp/resolv_v4.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v4.conf >> /etc/resolv.conf");
                }
                if(access("/tmp/resolv_v6.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v6.conf >> /etc/resolv.conf");
                }
            }
        }
    }
}


int main(int argc, char *argv[])
{
    int c;
    int dat;
    int ret;
    int opt_idx = 0;
    int retry_cnt = 0;

    ql_data_call_param_t *p_cfg;
    int reconnect_interval = 0;
    int apn_id = -1;
    int ip_ver = QL_NET_IP_VER_V4;
    char call_name[128] = "test_network";

    while(1)
    {
        c = getopt_long(argc, argv, g_options_short, g_options_long, &opt_idx);
        if(c==-1)
        {
            break;
        }

        switch(c) {
            case 'i':
                g_call_id = atoi(optarg);
                if(g_call_id < 0)
                {
                    printf("Invalid call id : %d", g_call_id);
                    return -1;
                }
                break;

            case 'r':
                reconnect_interval = atoi(optarg);
                if(reconnect_interval<=0)
                {
                    printf("Invalid reconnect interval : %d\n", reconnect_interval);
                    return -1;
                }
                break;

            case 'n':
                strncpy(call_name, optarg, sizeof(call_name)-1);
                break;

            case 'a':
                apn_id = atoi(optarg);
                if(apn_id<=0 || apn_id>QL_NET_MAX_APN_ID)
                {
                    printf("Invalid apn_id : %d\n", apn_id);
                    return -1;
                }

                break;
            case 'v':
                dat = atoi(optarg);
                if(4 == dat)
                {
                    ip_ver = QL_NET_IP_VER_V4;
                }
                else if(6 == dat)
                {
                    ip_ver = QL_NET_IP_VER_V6;
                }
                else
                {
                    printf("Invalid ip_ver : %d\n", dat);
                    return -1;
                }
                break;

            case 'd':
                g_is_default_network = 1;
                break;
            default:
                usage();
                return 0;
        }
    }

    if(g_call_id < 0)
    {
        usage();
        return 0;
    }

    retry_cnt = 20*1000/100; /** timeout : 20S */
    while(retry_cnt>0)
    {
        ret = ql_data_call_init();
        if(ret == QL_ERR_SERVICE_NOT_READY)
        {
            retry_cnt--;
            usleep(100*1000); /** sleep 100ms */
	    continue;
        }
        break;
    }

    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_init, ret=%d", ret);
        return -1;
    }

    ql_data_call_set_status_ind_cb(data_call_status_ind_cb);

    ret = ql_data_call_create(g_call_id, call_name, 0);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to create data call, ret=%d\n", ret);
        return -1;
    }

    p_cfg = ql_data_call_param_alloc();
    ql_data_call_param_set_apn_id(p_cfg, apn_id);
    ql_data_call_param_set_ip_version(p_cfg, ip_ver);

    if(reconnect_interval > 0)
    {
        int time_list[2] = {0};
        time_list[0] = reconnect_interval;
        ql_data_call_param_set_reconnect_mode(p_cfg, QL_NET_DATA_CALL_RECONNECT_NORMAL);
        ql_data_call_param_set_reconnect_interval(p_cfg, time_list, sizeof(time_list)/sizeof(time_list[0]));
    }

    ret = ql_data_call_config(g_call_id, p_cfg);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to config data call, ret=%d", ret);
        return -1;
    }

    ret = ql_data_call_start(g_call_id);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to start data call, ret=%d", ret);
        return -1;
    }

    while(1)
    {
        sleep(10);
    }

    return 0;
}
