/*-----------------------------------------------------------------------------------------------*/
/**
  @file main_apn.c
  @brief Sample code for multiple data call
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2018 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  20191127   tyler.kuang  Created .
-------------------------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <stdint.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "ql_type.h"
#include "ql_data_call.h"

#define APN_NAME_PUBLIC    "apnpublic"
#define APN_NAME_PRIVATE   "apnprivate"

#define DATA_CALL_APN_PUBLIC  3
#define DATA_CALL_APN_PRIVATE 4

#define DATA_CALL_ID_PUBLIC   1
#define DATA_CALL_ID_PRIVATE  2

void data_call_status_ind_cb(int call_id,
        QL_NET_DATA_CALL_STATUS_E pre_call_status,
        ql_data_call_status_t *p_msg)
{
    FILE *fp;
    char cmd_buf[256];

    /** not for this data call */
    if(call_id!=DATA_CALL_ID_PUBLIC && call_id!=DATA_CALL_ID_PRIVATE)
    {
        return;
    }

    printf("data call %d:%s status change, form %d to %d\n",
            call_id, p_msg->call_name, pre_call_status, p_msg->call_status);
    if(p_msg->call_status == QL_NET_DATA_CALL_STATUS_CONNECTED)
    {
        printf("call_id     : %d\n", p_msg->call_id);
        printf("call_name   : %s\n", p_msg->call_name);
        printf("device_name : %s\n", p_msg->device);
        if(p_msg->has_addr)
        {
            printf("IPV4 addr    : %s\n", p_msg->addr.addr);
            printf("IPV4 gateway : %s\n", p_msg->addr.gateway);
            printf("IPV4 netmask : %s\n", p_msg->addr.netmask);
            printf("IPV4 dnsp    : %s\n", p_msg->addr.dnsp);
            printf("IPV4 dnss    : %s\n", p_msg->addr.dnss);

            if(call_id == DATA_CALL_ID_PUBLIC)
            {
                /** set system default router */
                snprintf(cmd_buf, sizeof(cmd_buf), "ip ro add default via %s dev %s",
                        p_msg->addr.gateway, p_msg->device);
                system(cmd_buf);

                system("iptables -t filter -F");

                snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t nat -A POSTROUTING -o %s -j MASQUERADE", p_msg->device);
                system(cmd_buf);

                /** set system dns configuration*/
                fp = fopen("/tmp/resolv_v4.conf", "w");
                if(!fp)
                {
                    printf("Failed to write resolv file, err=%s\n", strerror(errno));
                    return;
                }

                if(p_msg->addr.dnsp[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr.dnsp);
                }

                if(p_msg->addr.dnss[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr.dnss);
                }
                fclose(fp);

                system("echo \"\" > /etc/resolv.conf");
                if(access("/tmp/resolv_v4.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v4.conf >> /etc/resolv.conf");
                }
                if(access("/tmp/resolv_v6.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v6.conf >> /etc/resolv.conf");
                }
            }
        }

        if(p_msg->has_addr6)
        {
            printf("IPV6 addr    : %s\n", p_msg->addr6.addr);
            printf("IPV6 gateway : %s\n", p_msg->addr6.gateway);
            printf("IPV6 netmask : %s\n", p_msg->addr6.prefix);
            printf("IPV6 dnsp    : %s\n", p_msg->addr6.dnsp);
            printf("IPV6 dnss    : %s\n", p_msg->addr6.dnss);

            if(call_id == DATA_CALL_ID_PUBLIC)
            {
                /** set system default router */
                snprintf(cmd_buf, sizeof(cmd_buf), "ip -6 ro add default via %s dev %s",
                        p_msg->addr6.gateway, p_msg->device);
                system(cmd_buf);

                /** set system dns configuration*/
                fp = fopen("/tmp/resolv_v6.conf", "w");
                if(!fp)
                {
                    printf("Failed to write resolv file, err=%s\n", strerror(errno));
                    return;
                }

                if(p_msg->addr6.dnsp[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr6.dnsp);
                }

                if(p_msg->addr6.dnss[0])
                {
                    fprintf(fp, "nameserver %s\n", p_msg->addr6.dnss);
                }
                fclose(fp);

                system("echo \"\" > /etc/resolv.conf");
                if(access("/tmp/resolv_v4.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v4.conf >> /etc/resolv.conf");
                }
                if(access("/tmp/resolv_v6.conf", F_OK) == 0)
                {
                    system("cat /tmp/resolv_v6.conf >> /etc/resolv.conf");
                }
            }
        }
    }
}

static void data_call_service_error_cb(int error)
{
    if(error == QL_ERR_ABORTED) {
        printf("RIL service exit!!!\n");
    }
}

int main(int argc, char *argv[])
{
    int ret = 0;
    ql_data_call_apn_config_t apn_cfg;
    ql_data_call_param_t *p_param = NULL;
    int time_interval_list[20] = {0};
    int retry_cnt = 20;

    while(retry_cnt > 0)
    {
        ret = ql_data_call_init();

        if(ret == QL_ERR_SERVICE_NOT_READY)
        {
            sleep(1);
            retry_cnt --;
            continue;
        }
        break;
    }

    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_init, ret=%d\n", ret);
        return -1;
    }

    ql_data_call_set_status_ind_cb(data_call_status_ind_cb);

    ql_data_call_set_service_error_cb(data_call_service_error_cb);

    /**
     * STEP 1: Set LTE default attach APN
     */
    memset(&apn_cfg, 0, sizeof(apn_cfg));
    strncpy(apn_cfg.apn_name, APN_NAME_PRIVATE, sizeof(apn_cfg.apn_name));
    apn_cfg.ip_ver = QL_NET_IP_VER_V4;

    ret = ql_data_call_set_apn_config(1, &apn_cfg);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_set_apn_config, APN1, ret=%d\n", ret);
        return -1;
    }

    /**
     *  STEP 2: Set APN used by Data call
     *  APN6 for public network
     *  APN7 For private network
     */
    memset(&apn_cfg, 0, sizeof(apn_cfg));
    strncpy(apn_cfg.apn_name, APN_NAME_PUBLIC, sizeof(apn_cfg.apn_name));
    apn_cfg.ip_ver = QL_NET_IP_VER_V4;

    ret = ql_data_call_set_apn_config(DATA_CALL_APN_PUBLIC, &apn_cfg);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_set_apn_config, APN_ID=%d, ret=%d\n", DATA_CALL_APN_PUBLIC, ret);
        return -1;
    }

    memset(&apn_cfg, 0, sizeof(apn_cfg));
    strncpy(apn_cfg.apn_name, APN_NAME_PRIVATE, sizeof(apn_cfg.apn_name));
    apn_cfg.ip_ver = QL_NET_IP_VER_V4;

    ret = ql_data_call_set_apn_config(DATA_CALL_APN_PRIVATE, &apn_cfg);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_set_apn_config, APN_ID=%d, ret=%d\n", DATA_CALL_APN_PRIVATE, ret);
        return -1;
    }

    p_param = ql_data_call_param_alloc();
    if(p_param == NULL)
    {
        printf("Failed to ql_data_call_param_alloc, memory is not enough\n");
        return -1;
    }

    /**
     * STEP 3: create and start data call
     * Public network data call: call_id=1, call_name=public, APN6
     * Private network data_call: call_id=2, call_name=private, APN7
     */
    ret = ql_data_call_create(DATA_CALL_ID_PUBLIC, "public", 0);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_create, call_id=%d, ret=%d\n", DATA_CALL_ID_PUBLIC, ret);
        return -1;
    }

    ql_data_call_param_init(p_param);
    ql_data_call_param_set_apn_id(p_param, DATA_CALL_APN_PUBLIC);
    ql_data_call_param_set_ip_version(p_param, QL_NET_IP_VER_V4);
    ql_data_call_param_set_reconnect_mode(p_param, QL_NET_DATA_CALL_RECONNECT_NORMAL);
    time_interval_list[0] = 20;
    ql_data_call_param_set_reconnect_interval(p_param, time_interval_list, 1);

    ret = ql_data_call_config(DATA_CALL_ID_PUBLIC, p_param);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_config, call_id=%d, ret=%d\n", DATA_CALL_ID_PUBLIC, ret);
        return -1;
    }

    printf("Start data call : public\n");
    ret = ql_data_call_start(DATA_CALL_ID_PUBLIC);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_start, call_id=%d, ret=%d\n", DATA_CALL_ID_PUBLIC, ret);
        return -1;
    }

    ret = ql_data_call_create(DATA_CALL_ID_PRIVATE, "private", 0);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_create, call_id=%d, ret=%d\n", DATA_CALL_ID_PRIVATE, ret);
        return -1;
    }

    ql_data_call_param_init(p_param);
    ql_data_call_param_set_apn_id(p_param, DATA_CALL_APN_PRIVATE);
    ql_data_call_param_set_ip_version(p_param, QL_NET_IP_VER_V4);
    ql_data_call_param_set_reconnect_mode(p_param, QL_NET_DATA_CALL_RECONNECT_NORMAL);
    time_interval_list[0] = 20;
    ql_data_call_param_set_reconnect_interval(p_param, time_interval_list, 1);

    ret = ql_data_call_config(DATA_CALL_ID_PRIVATE, p_param);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_config, call_id=%d, ret=%d\n", DATA_CALL_ID_PRIVATE, ret);
    }

    printf("Start data call : private\n");
    ret = ql_data_call_start(DATA_CALL_ID_PRIVATE);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_data_call_start, call_id=%d, ret=%d\n", DATA_CALL_ID_PRIVATE, ret);
        return -1;
    }

    while(1)
    {
        sleep(1);
    }

    return 0;
}
