/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_gnss_test.c
  @brief GNSS service API 
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2024 mobiletek Wireless Solution, Co., Ltd. All Rights Reserved.
  mobiletek Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---------    -----------------------------------------------------------------
  20241022    yq.wang      Created .
-------------------------------------------------------------------------------------------------*/

#include <stdio.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>

#include "ql_test_utils.h"
#include "ql_type.h"
#include "ql_gnss.h"

#define IS_DIGIT(x)    ( ((x-'0')>=0) && ((x-'0')<=9) )


static void gnss_ind_cb(void *msg)
{
    uint8_t *msg_id = NULL; 
    msg_id = msg;
    if(*msg_id == QL_GNSS_NMEA_MSG)
    {
        nmea_srv_ind_msg *data = (nmea_srv_ind_msg *)msg;
        printf("time=%d %s",data->time,data->nmea_sentence);
    }
    else if (*msg_id == QL_GNSS_STATUS_MSG)
    {
        gnss_status_ind_msg *data = (gnss_status_ind_msg *)msg;
        switch(data->g_status)
        {
            case QL_GNSS_STATUS_FOTA_START:
                printf("gnss engine fota req\n");
                break;    
            case QL_GNSS_STATUS_FOTA_BOOT:
                printf("gnss engine fota bootloader\n");
                break;    
            case  QL_GNSS_STATUS_FOTA_FIRM:
                printf("gnss engine fota firmware\n");
                break;    
            case  QL_GNSS_STATUS_WORKING:
                printf("gnss engine working\n");
                break;    
            default:
                break;
        };
    }
    else
    {
      printf("Invalue msg !!!");
    }
}

#if 0
static void gnss_service_error_cb(int error)
{
    printf("===== GNSS Service Abort =====\n");
}
#endif

static void item_ql_gnss_init(void)
{
    int err = QL_ERR_OK;
    printf("Start to ql_gnss_init\n");
    err = ql_gnss_init();
    if(err == QL_ERR_OK)
    {
        printf("Successful\n");
    }
    else
    {
        printf("Failed to ql_gnss_init, err=%d\n", err);
    }
}


static void item_ql_gnss_set_ind_cb(void)
{
    int err = QL_ERR_OK;

    printf("Start to ql_gnss_set_ind_cb\n");
    err = ql_gnss_set_ind_cb(gnss_ind_cb);
    if(err != QL_ERR_OK)
    {
        printf("Failed to ql_gnss_set_ind_cb, err=%d\n", err);
    }
    else
    {
        printf("Successful\n");
    }
}

#if 0
static void item_ql_gnss_set_service_error_cb(void)
{
    int err = QL_ERR_OK;
    printf("Start to item_ql_gnss_set_service_error_cb\n");
    err = ql_gnss_set_service_error_cb(gnss_service_error_cb);
    if(err == QL_ERR_OK)
    {
        printf("Successful\n");
    }
    else
    {
        printf("Failed to item_ql_gnss_set_service_error_cb, err=%d\n", err);
    }

}
#endif

static void item_ql_gnss_start(void)
{
    int err = QL_ERR_OK;

    printf("Start to ql_gnss_start\n");
    err = ql_gnss_start();
    if(err != QL_ERR_OK)
    {
        printf("Failed to start gnss , err=%d\n", err);
    }
    else
    {
        printf("Successful\n");
    }
}


static void item_ql_gnss_stop(void)
{
    int err = QL_ERR_OK;
    
    printf("Start to ql_gnss_stop\n");
    err = ql_gnss_stop();
    if(err != QL_ERR_OK)
    {
        printf("Failed to stop gnss, err=%d\n", err);
    }
    else
    {
        printf("Successful\n");
    }
}

#if 0
static void item_ql_gnss_get_engine_state(void)
{
    int err = QL_ERR_OK;
    QL_GNSS_ENGINE_STATE_E state = 0;

    printf("Start ql_gnss_get_engine_state : \n");
    err = ql_gnss_get_engine_state(&state);
    if(err != QL_ERR_OK)
    {
        printf("Get engine state  faild %d\n",err);
    }
    else
    {
        if(state)
        {
            printf("Engine state is on\n");
        }
        else
        {
            printf("Engine state is off\n");
        }
    }
}
#endif

#if 0
static void ql_gnss_get_constellation_show(char *str_p, QL_GNSS_CONSTELLATION_MASK_E mask)
{
    switch (mask)
    {
    case GPS_ONLY:
        snprintf(str_p, 100, "%s", "GPS");
        break;
    case BDS_ONLY:
        snprintf(str_p, 100, "%s", "BDS_ONLY");
        break;
    case GPS_BDS:
        snprintf(str_p, 100, "%s", "GPS+BDS");
        break;
    case GLONASS_ONLY:
        snprintf(str_p, 100, "%s", "GLONASS_ONLY");
        break;
    case GPS_GLONASS:
        snprintf(str_p, 100, "%s", "GPS+GLONASS");
        break; 
    case BDS_GLONASS:
        snprintf(str_p, 100, "%s", "BDS+GLONASS");
        break;
    case GPS_BDS_GLONASS:
        snprintf(str_p, 100, "%s", "GPS+BDS+GLONASS");
        break;
    case GPS_SBAS_QZSS:
        snprintf(str_p, 100, "%s", "GPS+SBAS+QZSS");
        break;
    case GPS_BDS_GALILEO_SBAS_QZSS:
        snprintf(str_p, 100, "%s", "GPS+BDS+GALILEO+SBAS+QZSS");
        break;
    case GPS_GLONASS_GALILEO_SBAS_QZSS:
        snprintf(str_p, 100, "%s", "GPS+GLONASS+GALILEO+SBAS+QZSS");
        break;
    default:
        snprintf(str_p, 100, "%s", "error");
        break;
    };
}

static void item_ql_gnss_get_constellation(void)
{
    int err = QL_ERR_OK;
    QL_GNSS_CONSTELLATION_MASK_E mask = 0;
    printf("Start ql_gnss_get_constellation : \n");
    err = ql_gnss_get_constellation(&mask);
    if (err != QL_ERR_OK)
    {
        printf("Set Constellation  faild %d\n", err);
    }
    else
    {
        switch (mask)
        {
        case GPS_SBAS_QZSS:
            printf("GPS+SBAS+QZSS\n");
            break;
        case BDS_ONLY:
            printf("BDS_ONLY\n");
            break;
        case GPS_BDS_GALILEO_SBAS_QZSS:
            printf("GPS+BDS+GALILEO+SBAS+QZSS\n");
            break;
        case GPS_GLONASS_GALILEO_SBAS_QZSS:
            printf("GPS+GLONASS+GALILEO+SBAS+QZSS\n");
            break;
        case GPS_ONLY:
            printf("GPS_ONLY\n");
            break;
        case GPS_BDS:
            printf("GPS_BDS\n");
            break;
        case GLONASS_ONLY:
            printf("GLONASS_ONLY\n");
            break;
        case GPS_GLONASS:
            printf("GPS+GLONASS\n");
            break;
        case BDS_GLONASS:
            printf("BDS+GLONASS\n");
            break;
        case GPS_BDS_GLONASS:
            printf("GPS+BDS+GLONASS\n");
            break;
        default:
            break;
        };
    }
}

/*
EC200 Supported constellations
(GPS+SBAS+QZSS--0x08|BDS_ONLY--0x10|GPS+BDS+GALILEO+SBAS+QZSS--0x11|GPS+GLONASS+GALILEO+SBAS+QZSS--0x101)");

AG35CET  Supported constellations
(GPS_ONLY--0x01|BDS_ONLY--0x02|GPS_BDS--0x03|GLONASS_ONLY--0x04)|GPS_GLONASS--0x05|BDS_GLONASS--0x06|GPS_BDS_GLONASS--0x07")

AG35EUT  Supported constellations
(GPS_ONLY--0x01|BDS_ONLY--0x02|GPS_BDS--0x03|GLONASS_ONLY--0x04)|GPS_GLONASS--0x05|BDS_GLONASS--0x06|GPS_BDS_GLONASS--0x07|
GALILEO_ONLY--0x08|GPS_GALILEO--0x09|BD_GALILEO--0x10|GPS_BDS_GLONASS_GALILEO--0x11")
*/
static void item_ql_gnss_set_constellation(void)
{
    int err = QL_ERR_OK;
    uint32_t input = 0;
    QL_GNSS_CONSTELLATION_MASK_E constellation = 0;
    printf("Please input gnss constellation:\n");
    scanf("%x",&input);
    printf("input=%x\n",input);
    constellation = (QL_GNSS_CONSTELLATION_MASK_E)input;
    err = ql_gnss_set_constellation(constellation);
    if(err != QL_ERR_OK)
    {
        printf("Failed to set gnss constellation mask:%x err=%d\n",constellation,err);
    }
    else
    {
        printf("Successful\n");
    }
}

static void item_ql_gnss_set_nmea_type(void)
{
    int err = QL_ERR_OK;
    uint32_t input = 0;

    printf("Please input nmea type bitmask:\n");
    printf("(GST|ZDA|VTG|RMC|GSV|GSA|GLL|GGA)(HEX Base, i.e.0xff)\n");
    scanf("%x",&input);
    printf("input=%x\n",input);
    err = ql_gnss_set_nmea_type(input);
    if(err != QL_ERR_OK)
    {
        printf("Failed to set nmea type with type:%x: err=%d\n",input,err);
    }
    else
    {
        printf("Successful\n");
    }
}

static void item_ql_gnss_get_nmea_version(void)
{
    int err = QL_ERR_OK;
    QL_GNSS_NMEA_VERSION_ID_E version = 0;

    printf("Start to get nmea_version \n");
    err = ql_gnss_get_nmea_version(&version);
    if(err != QL_ERR_OK)
    {
        printf("Failed to get nmea_version with version:%d, err=%d\n",version, err);
    }
    else
    {
        printf("Successful\n");
        if(QL_GNSS_NMEA_VERSION_V30 == version)
        {
            printf("Nmea_version is 3.0\n");
        }
        else if(QL_GNSS_NMEA_VERSION_V41 == version)
        {
            printf("Nmea_version is 4.0\n");
        }
    }
    
}


static void item_ql_gnss_set_nmea_version(void)
{
    int err = QL_ERR_OK;
    uint32_t input=0;
    QL_GNSS_NMEA_VERSION_ID_E version;
    
    printf("Please input nmea verson : 0--nema v3.0|1--nema v4.1\n");
    scanf("%d",&input);
    printf("Start to set nmea_version with version:%d\n",input);
    version = (QL_GNSS_NMEA_VERSION_ID_E)input;
    err = ql_gnss_set_nmea_version(version);
    if(err != QL_ERR_OK)
    {
        printf("Failed to set nmea_version with version:%d, err=%d\n",version, err);
    }
    else
    {
        printf("Successful\n");
    }
}
#endif

static void item_ql_gnss_set_start_mode(void)
{
    int err = QL_ERR_OK;
    uint32_t input=0;
    QL_GNSS_START_MODE_E mode;

    printf("Please input mode : 0--cold start|1--warm start|2--hot start\n");
    scanf("%d",&input);
    printf("Start to ql_set_start_mode with modem:%d\n",input);
    mode = (QL_GNSS_START_MODE_E)input;
    err = ql_gnss_set_start_mode(mode);
    if(err != QL_ERR_OK)
    {
        printf("Failed to Start to ql_set_start_mode with modem:%d, err=%d\n",mode, err);
    }
    else
    {
        printf("Successful\n");
    }
}


static void item_ql_gnss_set_agnss_mode(void)
{
    int err = QL_ERR_OK;
    uint32_t input=0;

    printf("Please input mode : 0x01--GPS| 0x02--BD| 0x04--GLONASS| 0x08--GALILEO| 0x10--QZSS \n");
    scanf("%x",&input);
    printf("Start to ql_set_agnss_mode with modem:%x\n",input);
    err = ql_gnss_set_agnss_mode(input);
    if(err != QL_ERR_OK)
    {
        printf("Failed to Start to ql_set_agnss_mode with modem:%d, err=%d\n",input, err);
    }
    else
    {
        printf("Successful\n");
    }

}

static void item_ql_gnss_inject_agnss_data(void)
{
    int err = QL_ERR_OK;

    err = ql_gnss_inject_agnss_data();
    if (err != QL_ERR_OK)
    {
        printf("Failed to inject agnss  err=%d\n", err);
    }
    else
    {
        printf("Successful\n");
    }
}

#if 0
static int convertUtcTimeMsec(char *utc_str,uint64_t *msec)
{
    int i =0;
    int temp =0;
    int date[6] = {0};
    char ch;
    unsigned char field = 0;
    struct tm t;
    while(1)
    {
        ch = utc_str[i++];

        if(IS_DIGIT(ch))
        {
            temp *= 10;
            temp += ch - '0';
        }
        else if(ch == ' ')
        {
            continue;
        }
        else if(ch == '-' || ch == ' ' || ch == ':' || ch == '\0')
        {
            date[field] = temp;
            field++;
            temp = 0;

            if(ch == '\0')
            {
                printf("parse char success\n");
                break;
            }
        }
        else
        {
            printf("ch=%c,ch=%d,invalid char\n",ch,ch);
            return -1;
        }
    }

    if(date[0] < 1900
        || date[1] < 1 || date[1] > 12
        || date[2] < 1 || date[2] > 31
        || date[3] < 0 || date[3] > 23
        || date[4] < 0 || date[4] > 59
        || date[5] < 0 || date[5] > 59
         ) 
     {
         return -1;
     }

     t.tm_sec = date[5];
     t.tm_min = date[4];
     t.tm_hour = date[3];
     t.tm_mday = date[2];
     t.tm_mon = date[1] -1;
     t.tm_year = date[0] - 1900;
     t.tm_isdst = 0;

     *msec = (uint64_t)1000 * (uint64_t)mktime(&t);
     return 0;

}

static void item_ql_gnss_inject_utc_time(void)
{
    int err = QL_ERR_OK;
    int input = 0;
    uint64_t utc_time;
    char buff[128]={0};
    printf("Start ql_inject_utc_time : \n");
    printf("Please Select Time:\n1-Current System time\n2-Input Time,Format:YYYY-MM-DD hh:mm:ss\n:");
    scanf("%d",&input);
    if(err !=0)
    {
        printf("Input error\n");
        return ;
    }
    switch(input)
    {
        case 1:
            utc_time=(uint64_t)time(NULL)*1000;
            system("date");
            break;
        case 2:
            err = t_get_string(buff, sizeof(buff));
            if(err !=0)
            {
                printf("Input error\n");
                return ;
            }
            convertUtcTimeMsec(buff,&utc_time);
            break;
            default:
                printf("Not select!!!\n");
                return ;
    }
    err = ql_gnss_inject_utc_time(utc_time);
    if(err != QL_ERR_OK)
    {
        printf("Failed to Inject Utc Time  %d\n",err);
    }
    else
    {
        printf("Successful!\n");
    }
}




static void item_ql_gnss_suspend(void)
{
    int err = QL_ERR_OK;

    printf("Start to ql_gnss_suspend\n");
    err = ql_gnss_suspend();
    if(err != QL_ERR_OK)
    {
        printf("Failed to set ql_gnss_suspend\n");
    }
    else
    {
        printf("Successful\n");
    }
}


static void item_ql_gnss_resume(void)
{
    int err = QL_ERR_OK;

    printf("Start to ql_gnss_resume\n");
    err = ql_gnss_resume();
    if(err != QL_ERR_OK)
    {
        printf("Failed to set ql_gnss_resume\n");
    }
    else
    {
        printf("Successful\n");
    }
}
#endif

static void item_ql_gnss_deinit(void)
{
    int err = QL_ERR_OK;

    printf("Start to ql_gnss_deinit\n");
    err = ql_gnss_deinit();
    if(err == QL_ERR_OK)
    {
        printf("Successful\n");
    }
    else
    {
        printf("Failed to ql_gnss_deinit, err=%d\n", err);
    }
}

static t_item_t ql_gnss_items[] = 
{
    {"ql_gnss_init", item_ql_gnss_init},
    {"ql_gnss_nmea_ind_cb", item_ql_gnss_set_ind_cb},
//    {"ql_gnss_get_engine_state",item_ql_gnss_get_engine_state},
    {"ql_gnss_start",item_ql_gnss_start},
    {"ql_gnss_stop", item_ql_gnss_stop},
//    {"ql_gnss_get_nmea_version",item_ql_gnss_get_nmea_version},
//    {"ql_gnss_set_nmea_version",item_ql_gnss_set_nmea_version},
//    {"ql_gnss_set_nmea_type",item_ql_gnss_set_nmea_type},
//    {"ql_gnss_get_constellation",item_ql_gnss_get_constellation},
//    {"ql_gnss_set_constellation",item_ql_gnss_set_constellation},  
    {"ql_gnss_set_start_mode",item_ql_gnss_set_start_mode},
    {"ql_gnss_set_agnss_mode",item_ql_gnss_set_agnss_mode},
    {"ql_gnss_inject_agnss_data",item_ql_gnss_inject_agnss_data},
//    {"ql_gnss_inject_utc_time",item_ql_gnss_inject_utc_time},
//    {"ql_gnss_suspend", item_ql_gnss_suspend},
//    {"ql_gnss_resume", item_ql_gnss_resume},
//    {"ql_gnss_set_service_error_cb",item_ql_gnss_set_service_error_cb},
    {"ql_gnss_deinit", item_ql_gnss_deinit},
};

t_module_t ql_gnss_module =
{
    "gnss",
    T_ARRAY_SIZE(ql_gnss_items),
    ql_gnss_items
};

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief Read a int value from stdin 
  @param[out] val, Return read data
  @return 
  0 - successful
  1 - read an enter 
  -1 - invalid input
  */
/*-----------------------------------------------------------------------------------------------*/
int t_get_int(int *val)
{
    int dat;
    char *ptr_end = NULL;
    char buf[256] = {0};

    if(NULL == fgets(buf, sizeof(buf)-1, stdin))
    {
        return -1;
    }
#if 0    
    if(0 == buf[0])
    {
        return -1;
    }
#endif
    if(buf[0] == '\n')
    {
        return 1;
    }

    dat = strtol(buf, &ptr_end, 10);
    if(ptr_end!=NULL && ptr_end[0]!='\n')
    {
        return -1;
    }

    if(val)
    {
        val[0] = dat;
    }

    return 0;
}

void dump_items()
{
    int i;

    printf("\n");
    printf("The current module is: \n");

    for(i=0; i< ql_gnss_module.item_len; i++)
    {
        printf("%d\t%s\n", i, ql_gnss_module.item_list[i].name);
    }
    printf("-1\texit\n");
}

int main(int argc, char *argv[])
{
    int ret;
    int idx;

    dump_items();

    while(1)
    {
        printf("Please enter your choice: ");
        ret = t_get_int(&idx);
        printf("\n");
        if(ret < 0)
        {
            printf("Invalid input\n");
            continue;
        }
        else if(ret == 1)
        {
            dump_items();
            continue;
        }

        if(idx == -1)
        {
            break;
        }

        if(idx<0 || idx>=ql_gnss_module.item_len)
        {
            printf("Not support idx: %d\n", idx);
            continue;
        }

        printf("->Item : %s\n", ql_gnss_module.item_list[idx].name);
        ql_gnss_module.item_list[idx].handle();
    }

    return 0;
}

