/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_nw_test.c 
  @brief nw test module 
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2024 mobiletek Wireless Solution, Co., Ltd. All Rights Reserved.
  mobiletek Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---------    -----------------------------------------------------------------
  20241022    yq.wang      Created .
-------------------------------------------------------------------------------------------------*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "ql_test_utils.h"
#include "ql_type.h"
#include "ql_nw.h"


#if 0
static int internal_nw_get_nas_rat(int radio, char* buf, int buf_len);
#endif
static int internal_nw_get_net_status(QL_NW_NETWORK_STATUS_TYPE_E net, char* buf, int buf_len);
static int internal_nw_get_radio_tech(QL_NW_RADIO_TECH_TYPE_E radio, char* buf, int buf_len);
static int internal_nw_get_tech_domain(QL_NW_TECH_DOMAIN_TYPE_E domain, char* buf, int buf_len);
static int internal_nw_get_signal_strength_level(QL_NW_SIGNAL_STRENGTH_LEVEL_E level, char* buf, int buf_len);
static void internal_nw_get_mcc_mnc_value(char *plmn, int plmn_len, unsigned short *mcc, unsigned short *mnc);
static void internal_nw_get_service_option(unsigned short so_mask, char *buf, int buf_len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief Read a uint32 value from stdin 
  @param[out] val, Return read data
  @return 
  0 - successful
  1 - read an enter 
  -1 - invalid input
  */
/*-----------------------------------------------------------------------------------------------*/
int t_get_hex(uint32_t *val)
{
    int dat;
    char *ptr_end = NULL;
    char buf[256] = {0};

    if(fgets(buf, sizeof(buf)-1, stdin) == NULL)
    {
        return -1;
    }
    
    if(0 == buf[0])
    {
        return -1;
    }

    if(buf[0] == '\n')
    {
        return 1;
    }

    dat = strtol(buf, &ptr_end, 16);
    if(ptr_end!=NULL && ptr_end[0]!='\n')
    {
        return -1;
    }

    if(val)
    {
        val[0] = dat;
    }

    return 0;
}

void nw_voice_reg_event_ind_cb(ql_nw_reg_status_info_t *ind)
{
    char domain_info[16] = {0};
    char radio_info[16] = {0};
    char svc_opt[128] = {0};

    if(ind == NULL)
    {
        printf("recv voice reg ind event, param is null\n");
        return;
    }
    printf("recv voice reg ind event, detail info:\n");

    if(internal_nw_get_tech_domain(ind->tech_domain, domain_info, sizeof(domain_info)) == 0)
    {
        printf("\ttech_domain is unrecognized:%d, ", ind->tech_domain);
    }
    else
    {
        printf("\ttech_domain = %s, ", domain_info);
    }

    if(internal_nw_get_radio_tech(ind->radio_tech, radio_info, sizeof(radio_info)) == 0)
    {
        printf("radio_tech is unrecognized:%d, ", ind->radio_tech);
    }
    else
    {
        printf("radio_tech = %s, ", radio_info);
    }

    printf("roaming = %d, reg_status = %d, deny_reason = %d\n", 
            ind->roaming, ind->reg_state, ind->deny_reason);

    if(QL_NW_RADIO_TECH_NR5G != ind->radio_tech)
    {
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%X, lac=%d, psc=%d, tac=%d, endc_available=%d\n",
                ind->mcc,
                ind->mnc,
                ind->forbidden,
                ind->cid,
                ind->lac,
                ind->psc,
                ind->tac,
                ind->endc_available);

        printf("\tinPRL=%d, css=%d, sid=%d, nid=%d, bsid=%d\n",
                ind->inPRL,
                ind->css,
                ind->sid,
                ind->nid,
                ind->bsid);
    }
    else
    { 
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%llX, pci=%d, tac=%d\n",
                ind->mcc,
                ind->mnc,
                ind->forbidden,
                ind->nr5g_cid,
                ind->nr5g_pci,
                ind->nr5g_tac);

        internal_nw_get_service_option(ind->nr5g_svc_opt, svc_opt, sizeof(svc_opt));
        printf("\tsvc_opt(%d)=%s\n", ind->nr5g_svc_opt, svc_opt);
    } 
}

#if 0
void nw_data_reg_event_ind_cb(ql_nw_reg_status_info_t *ind)
{
    char domain_info[16] = {0};
    char radio_info[16] = {0};
    char svc_opt[128] = {0};
    
    if(ind == NULL)
    {
        printf("recv data reg ind event, param is null\n");
        return;
    }
    printf("recv data reg ind event, detail info:\n");
    
    if(internal_nw_get_tech_domain(ind->tech_domain, domain_info, sizeof(domain_info)) == 0)
    {
        printf("\ttech_domain is unrecognized:%d, ", ind->tech_domain);
    }
    else
    {
        printf("\ttech_domain = %s, ", domain_info);
    }

    if(internal_nw_get_radio_tech(ind->radio_tech, radio_info, sizeof(radio_info)) == 0)
    {
        printf("radio_tech is unrecognized:%d, ", ind->radio_tech);
    }
    else
    {
        printf("radio_tech = %s, ", radio_info);
    }

    printf("roaming = %d, reg_status = %d, deny_reason = %d\n", 
            ind->roaming, ind->reg_state, ind->deny_reason);

    if(QL_NW_RADIO_TECH_NR5G != ind->radio_tech)
    {
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%X, lac=%d, psc=%d, tac=%d, endc_available=%d\n",
                ind->mcc,
                ind->mnc,
                ind->forbidden,
                ind->cid,
                ind->lac,
                ind->psc,
                ind->tac,
                ind->endc_available);

        printf("\tinPRL=%d, css=%d, sid=%d, nid=%d, bsid=%d\n",
                ind->inPRL,
                ind->css,
                ind->sid,
                ind->nid,
                ind->bsid);
    }
    else
    { 
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%llX, pci=%d, tac=%d\n",
                ind->mcc,
                ind->mnc,
                ind->forbidden,
                ind->nr5g_cid,
                ind->nr5g_pci,
                ind->nr5g_tac);

        internal_nw_get_service_option(ind->nr5g_svc_opt, svc_opt, sizeof(svc_opt));
        printf("\tsvc_opt(%d)=%s\n", ind->nr5g_svc_opt, svc_opt);
    }
}
#endif

void nw_signal_strength_event_ind_cb(ql_nw_signal_strength_info_t *ind, QL_NW_SIGNAL_STRENGTH_LEVEL_E level)
{
    char level_info[16] = {0};

    if(ind == NULL)
    {
        printf("recv signal strength ind event, param is null\n");
        return;
    }

    printf("recv event indication : signal strength event\n");

    if(ind->has_gsm)
    {
        printf("gsm_sig_info: rssi=%d\n", ind->gsm.rssi);
    } 

    if(ind->has_wcdma)
    {
        printf("wcdma_sig_info: rssi=%d, ecio=%d\n", 
                ind->wcdma.rssi, 
                ind->wcdma.ecio); 
    } 
    if(ind->has_tdscdma)
    {
        printf("tdscdma_sig_info: rssi=%d, rscp=%d, ecio=%d, sinr=%d\n", 
                ind->tdscdma.rssi, 
                ind->tdscdma.rscp,
                ind->tdscdma.ecio,
                ind->tdscdma.sinr);
    } 
    if(ind->has_lte)
    {
        printf("lte_sig_info: rssi=%d, rsrq=%d, rsrp=%d, snr=%d\n", 
                ind->lte.rssi, 
                ind->lte.rsrq,
                ind->lte.rsrp,
                ind->lte.snr);
    } 
    if(ind->has_nr5g)
    {
        printf("nr5g_sig_info: rsrp=%hd, rsrq=%hd, snr=%hd\n",
                ind->nr5g.rsrp,
                ind->nr5g.rsrq,
                ind->nr5g.snr);
    }
    if(ind->has_cdma)
    {
        printf("cdma_sig_info: rssi=%d, ecio=%d\n", 
                ind->cdma.rssi, 
                ind->cdma.ecio);
    } 
    if(ind->has_hdr)
    {
        printf("hdr_sig_info: rssi=%d, ecio=%d, sinr=%d, io=%d\n", 
                ind->hdr.rssi, 
                ind->hdr.ecio,
                ind->hdr.sinr,
                ind->hdr.io);
    }
    
    if(internal_nw_get_signal_strength_level(level, level_info, sizeof(level_info)) == 0)
    {
        printf("signal strength level is %d, unrecognized\n", level);
    }
    else
    {
        printf("signal strength level is %s\n", level_info);
    }
}

#if 0
void nw_cell_access_status_event_ind_cb(QL_NW_CELL_ACCESS_STATE_TYPE_E status)
{
    printf("recv event indication : cell access status event\n");
    printf("status = %d\n", status);
}

void nw_nitz_time_update_event_ind_cb(ql_nw_nitz_time_info_t *ind)
{
    if(ind == NULL)
    {
        printf("recv nitz time update ind event, param is null\n");
        return;
    }
    printf("recv event indication : nitz time update event\n");
    printf("nitz_time=%s, abs_time=%lld, leap_sec=%d\n",
            ind->nitz_time, ind->abs_time, ind->leap_sec);
}

void nw_wea_alert_event_ind_cb(ql_nw_wea_alert_info_t *ind)
{
    if(ind == NULL)
    {
        printf("recv wea alert ind event, param is null\n");
        return;
    }
    
    printf("recv event indication : wea alert event, wea_alert_info_valid:[%d]\n", 
            ind->wea_alert_info_valid);
    if(ind->wea_alert_info_valid)
    {
        printf("wea alert info:[%s] \n",ind->wea_alert_info);
    }
}

void nw_etws_alert_event_ind_cb(ql_nw_etws_alert_info_t *ind)
{
    if(ind == NULL)
    {
        printf("recv etws alert ind event, param is null\n");
        return;
    }
    printf("recv event indication : etws alert event, etws_primary_info_valid:[%d] etws_secondary_info_valid:[%d]\n", 
            ind->etws_primary_info_valid, ind->etws_secondary_info_valid);
    if(ind->etws_primary_info_valid)
    {
        printf("etws primary alert info:[%s] \n", ind->etws_primary_info);
    }
    if(ind->etws_secondary_info_valid)
    {
        printf("etws secondary alert info:[%s] \n",
            ind->etws_secondary_info);
    }
}

void nw_network_scan_async_cb(int async_id, ql_nw_scan_result_list_info_t *info)
{
    int i = 0;
    char net_info[16] = {0};
    char radio_info[16] = {0};

    printf("network scan async callback, async id is %d, list_len=%d, detail info:\n", async_id, info->entry_len);
    for(i = 0; i < info->entry_len; i++)
    {
        memset(net_info, 0, sizeof(net_info));
        memset(radio_info, 0, sizeof(radio_info));
        printf("\t[%d]: long_eons=%s, short_eons=%s, mcc=%s, mnc=%s, ",
               i,
               info->entry[i].operator_name.long_eons,
               info->entry[i].operator_name.short_eons,
               info->entry[i].operator_name.mcc,
               info->entry[i].operator_name.mnc);


        if(internal_nw_get_net_status(info->entry[i].network_status, net_info, sizeof(net_info)) == 0)
        {
            printf("unrecognized network_status:%d, ", info->entry[i].network_status);
        }
        else
        {
            printf("network_status=%s, ", net_info);
        }

        if(internal_nw_get_radio_tech(info->entry[i].rat, radio_info, sizeof(radio_info)) == 0)
        {
            printf("unrecognized rat:%d\n ", info->entry[i].rat);
        }
        else
        {
            printf("radio_tech=%s\n", radio_info);
        }
    }
}
#endif

void nw_service_error_cb(int error)
{
    printf("===== NW Service Abort =====\n");
}

void item_ql_nw_init(void)
{
    int ret = 0;
    
    printf("Start to ql_nw_init: ");
    ret = ql_nw_init();
    if(ret == QL_ERR_OK)
    {
        printf("nw init ok\n");
    }
    else
    {
        printf("failed, ret=%d\n", ret);
    }
}

void item_ql_nw_deinit(void)
{
    int ret = 0;
    
    printf("Start to ql_nw_deinit: ");
    ret = ql_nw_deinit();
    if(ret == QL_ERR_OK)
    {
        printf("nw deinit ok\n");
    }
    else
    {
        printf("failed, ret=%d\n", ret);
    }
}

#if 0
void item_ql_nw_network_scan(void)
{
    int ret;
    int async_id = 0;

    ret = ql_nw_network_scan(&async_id, nw_network_scan_async_cb);
    if(ret == QL_ERR_OK)
    {
        printf("async network scan succeed, token id is %d\n", async_id);
    }
    else
    {
        printf("async network scan failed, token id is %d, ret=%d", async_id, ret);
    }
}
#endif

void item_ql_nw_set_power_mode(void)
{
    int ret;
    uint32_t mode = 0;

    printf("please input power mode mask hex(VOICE | SMS | SIM | NETWORK | NORMAL): ");
    ret = t_get_hex(&mode);
    if(ret != 0)
    {
        printf("Invalid input\n");
        return;
    }
    ret = ql_nw_set_power_mode(mode);
    printf("ql_nw_set_lower_power_mode ret = %d\n", ret);
}

#if 0
void item_ql_nw_set_pref_nwmode_roaming(void)
{
    int ret;
    uint32_t mask = 0;
    ql_nw_pref_nwmode_roaming_info_t t_info;

    memset(&t_info, 0, sizeof(t_info));
    printf("please input config mask hex(TDSCDMA | LTE | EVDO | CDMA | WCDMA | GSM): ");
    ret = t_get_hex(&mask);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }
    t_info.preferred_nw_mode = mask;

#if 0
    printf("please input roaming pref(0:off 1:on): ");
    ret = t_get_int((int *)&mask);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }

    t_info.preferred_roaming = mask;
#endif
    ret = ql_nw_set_pref_nwmode_roaming(&t_info);
    printf("ql_nw_set_config ret = %d\n", ret);
}

void item_ql_nw_get_pref_nwmode_roaming(void)
{
    int ret;
    ql_nw_pref_nwmode_roaming_info_t t_info;

    memset(&t_info, 0, sizeof(t_info));
    ret = ql_nw_get_pref_nwmode_roaming(&t_info);
#if 0    
    printf("ql_nw_get_config ret = %d\npreferred_nw_mode=%#llx, preferred_roaming=%d\n", 
            ret, t_info.preferred_nw_mode, t_info.preferred_roaming);
#else
    printf("ql_nw_get_config ret = %d\npreferred_nw_mode=%#llx\n", 
            ret, t_info.preferred_nw_mode);
#endif            
}

void item_ql_nw_get_mobile_operator_name(void)
{
    int ret;
    ql_nw_mobile_operator_name_info_t t_info;

    memset(&t_info, 0, sizeof(t_info));
    ret = ql_nw_get_mobile_operator_name(&t_info);
    printf("ql_nw_get_operator_name ret = %d, long_eons=%s, short_eons=%s, mcc=%s, mnc=%s\n", 
        ret,
        t_info.long_eons, 
        t_info.short_eons, 
        t_info.mcc, 
        t_info.mnc);
}

void item_ql_nw_get_cell_info(void)
{
    int i;
    int ret;
    unsigned short mcc;
    unsigned short mnc;
    ql_nw_cell_info_t t_info;

    memset(&t_info, 0, sizeof(t_info));
    ret = ql_nw_get_cell_info(&t_info);
    printf("ql_nw_get_cell_info ret = %d, detail info:\n", ret);
    
    if(t_info.gsm_info_valid)
    {
        printf("gsm cell information:\n");
        for(i = 0; i < t_info.gsm_info_len; i++)
        {
            printf("\tcid=%d,plmn=0x%02x 0x%02x 0x%02x,lac=%d,arfcn=%d,bsic=%d,rssi=%d,", 
                    t_info.gsm_info[i].cid, 
                    t_info.gsm_info[i].plmn[0], 
                    t_info.gsm_info[i].plmn[1], 
                    t_info.gsm_info[i].plmn[2], 
                    t_info.gsm_info[i].lac, 
                    t_info.gsm_info[i].arfcn, 
                    t_info.gsm_info[i].bsic, 
                    t_info.gsm_info[i].rssi);

            internal_nw_get_mcc_mnc_value(t_info.gsm_info[i].plmn, 3, &mcc, &mnc);
            printf("convert plmn to mcc=%d,mnc=%02d\n", mcc, mnc);
        }
    }

    if(t_info.umts_info_valid)
    {
        printf("umts cell information:\n");
        for(i = 0; i < t_info.umts_info_len; i++)
        {
            printf("\tcid=%d,lcid=%d,plmn=0x%02x 0x%02x 0x%02x,lac=%d,uarfcn=%d,psc=%d,rssi=%d,", 
                    t_info.umts_info[i].cid,
                    t_info.umts_info[i].lcid, 
                    t_info.umts_info[i].plmn[0], 
                    t_info.umts_info[i].plmn[1], 
                    t_info.umts_info[i].plmn[2], 
                    t_info.umts_info[i].lac, 
                    t_info.umts_info[i].uarfcn, 
                    t_info.umts_info[i].psc, 
                    t_info.umts_info[i].rssi);
            internal_nw_get_mcc_mnc_value(t_info.umts_info[i].plmn, 3, &mcc, &mnc);
            printf("convert plmn to mcc=%d,mnc=%02d\n", mcc, mnc);
        }
    }

    if(t_info.lte_info_valid)
    {
        printf("lte cell information:\n");
        for(i = 0; i < t_info.lte_info_len; i++)
        {
            printf("\tcid=%d,plmn=0x%02x 0x%02x 0x%02x,tac=%d,pci=%d,earfcn=%d,rssi=%d,", 
                    t_info.lte_info[i].cid, 
                    t_info.lte_info[i].plmn[0], 
                    t_info.lte_info[i].plmn[1], 
                    t_info.lte_info[i].plmn[2], 
                    t_info.lte_info[i].tac, 
                    t_info.lte_info[i].pci, 
                    t_info.lte_info[i].earfcn, 
                    t_info.lte_info[i].rssi);
            internal_nw_get_mcc_mnc_value(t_info.lte_info[i].plmn, 3, &mcc, &mnc);
            printf("convert plmn to mcc=%d,mnc=%02d\n", mcc, mnc);
        }
    }
  
    if(t_info.nr5g_info_valid)
    {
        printf("nr5g cell information:\n");
        printf("\tcid=%lld,plmn=0x%02x 0x%02x 0x%02x,tac=%d,pci=%d,arfcn=%d,rsrp=%d,rsrq=%d,snr=%d,", 
                t_info.nr5g_info.cid, 
                t_info.nr5g_info.plmn[0], 
                t_info.nr5g_info.plmn[1], 
                t_info.nr5g_info.plmn[2], 
                t_info.nr5g_info.tac, 
                t_info.nr5g_info.pci, 
                t_info.nr5g_info.arfcn, 
                t_info.nr5g_info.rsrp,
                t_info.nr5g_info.rsrq,
                t_info.nr5g_info.snr);
        internal_nw_get_mcc_mnc_value(t_info.nr5g_info.plmn, 3, &mcc, &mnc);
        printf("convert plmn to mcc=%d,mnc=%02d\n", mcc, mnc);
    }

    if(t_info.cdma_info_valid)
    {
        printf("cdma cell information:\n");
        printf("\tsid=%d,nid=%d,bid=%d,refpn=%d,base_lat=%d,base_long=%d,rssi=%d\n", 
                t_info.cdma_info.sid,
                t_info.cdma_info.nid,
                t_info.cdma_info.bid,
                t_info.cdma_info.refpn,
                t_info.cdma_info.base_lat,
                t_info.cdma_info.base_long,
                t_info.cdma_info.rssi);  
    }
}
#endif

void item_ql_nw_get_voice_reg_status(void)
{
    int ret;
    char domain_info[16] = {0};
    char radio_info[16] = {0};
    char svc_opt[128] = {0};
    ql_nw_reg_status_info_t t_info;

    memset(&t_info, 0, sizeof(t_info));
    ret = ql_nw_get_voice_reg_status(&t_info);

    printf("ql_nw_get_voice_reg_status ret = %d, detail info:\n", ret);

    if(internal_nw_get_tech_domain(t_info.tech_domain, domain_info, sizeof(domain_info)) == 0)
    {
        printf("\ttech_domain is unrecognized:%d, ", t_info.tech_domain);
    }
    else
    {
        printf("\ttech_domain = %s, ", domain_info);
    }

    if(internal_nw_get_radio_tech(t_info.radio_tech, radio_info, sizeof(radio_info)) == 0)
    {
        printf("radio_tech is unrecognized:%d, ", t_info.radio_tech);
    }
    else
    {
        printf("radio_tech = %s, ", radio_info);
    }

    printf("roaming = %d, reg_status = %d, deny_reason = %d\n", 
            t_info.roaming, t_info.reg_state, t_info.deny_reason);

    if(QL_NW_RADIO_TECH_NR5G != t_info.radio_tech)
    {
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%X, lac=%d, psc=%d, tac=%d, endc_available=%d\n",
                t_info.mcc,
                t_info.mnc,
                t_info.forbidden,
                t_info.cid,
                t_info.lac,
                t_info.psc,
                t_info.tac,
                t_info.endc_available);

        printf("\tinPRL=%d, css=%d, sid=%d, nid=%d, bsid=%d\n",
                t_info.inPRL,
                t_info.css,
                t_info.sid,
                t_info.nid,
                t_info.bsid);
    }
    else
    { 
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%llX, pci=%d, tac=%d\n",
                t_info.mcc,
                t_info.mnc,
                t_info.forbidden,
                t_info.nr5g_cid,
                t_info.nr5g_pci,
                t_info.nr5g_tac);

        internal_nw_get_service_option(t_info.nr5g_svc_opt, svc_opt, sizeof(svc_opt));
        printf("\tsvc_opt(%d)=%s\n", t_info.nr5g_svc_opt, svc_opt);
    }
}

void item_ql_nw_get_data_reg_status(void)
{
    int ret;
    char domain_info[16] = {0};
    char radio_info[16] = {0};
    char svc_opt[128] = {0};
    ql_nw_reg_status_info_t t_info;

    memset(&t_info, 0, sizeof(t_info));
    ret = ql_nw_get_data_reg_status(&t_info);
    
    printf("ql_nw_get_data_reg_status ret = %d, detail info:\n", ret);
    
    if(internal_nw_get_tech_domain(t_info.tech_domain, domain_info, sizeof(domain_info)) == 0)
    {
        printf("\ttech_domain is unrecognized:%d, ", t_info.tech_domain);
    }
    else
    {
        printf("\ttech_domain = %s, ", domain_info);
    }

    if(internal_nw_get_radio_tech(t_info.radio_tech, radio_info, sizeof(radio_info)) == 0)
    {
        printf("radio_tech is unrecognized:%d, ", t_info.radio_tech);
    }
    else
    {
        printf("radio_tech = %s, ", radio_info);
    }

    printf("roaming = %d, reg_status = %d, deny_reason = %d\n", 
            t_info.roaming, t_info.reg_state, t_info.deny_reason);

    if(QL_NW_RADIO_TECH_NR5G != t_info.radio_tech)
    {
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%X, lac=%d, psc=%d, endc_available=%d\n",
                t_info.mcc,
                t_info.mnc,
                t_info.forbidden,
                t_info.cid,
                t_info.lac,
                t_info.psc,
                //t_info.tac,  // tac not valid  in item_ql_nw_get_data_reg_status ,use item_ql_nw_get_cell_info
                t_info.endc_available);

        printf("\tinPRL=%d, css=%d, sid=%d, nid=%d, bsid=%d\n",
                t_info.inPRL,
                t_info.css,
                t_info.sid,
                t_info.nid,
                t_info.bsid);
    }
    else
    { 
        printf("\tmcc=%s, mnc=%s, forbidden=%d, cid=0x%llX, pci=%d, tac=%d\n",
                t_info.mcc,
                t_info.mnc,
                t_info.forbidden,
                t_info.nr5g_cid,
                t_info.nr5g_pci,
                t_info.nr5g_tac);

        internal_nw_get_service_option(t_info.nr5g_svc_opt, svc_opt, sizeof(svc_opt));
        printf("\tsvc_opt(%d)=%s\n", t_info.nr5g_svc_opt, svc_opt);
    }

}

void item_ql_nw_get_signal_strength(void)
{
    int ret;
    char level_info[16] = {0};
    ql_nw_signal_strength_info_t info;
    QL_NW_SIGNAL_STRENGTH_LEVEL_E level = QL_NW_SIGNAL_STRENGTH_LEVEL_NONE;

    memset(&info, 0, sizeof(info));
    ret = ql_nw_get_signal_strength(&info, &level);
    if (QL_ERR_OK != ret)
    {
        printf("failed, ret = %d\n", ret);
        return;
    }

    if(info.has_gsm)
    {
        printf("gsm_sig_info: rssi=%hhd\n", info.gsm.rssi);
    } 

    if(info.has_wcdma)
    {
        printf("wcdma_sig_info: rssi=%hhd, ecio=%hd\n",
                info.wcdma.rssi,
                info.wcdma.ecio);
    } 

    if(info.has_tdscdma)
    {
        printf("tdscdma_sig_info: rssi=%hhd, rscp=%hhd, ecio=%hd, sinr=%hhd\n",
                info.tdscdma.rssi,
                info.tdscdma.rscp,
                info.tdscdma.ecio,
                info.tdscdma.sinr);
    } 

    if(info.has_lte)
    {
        printf("lte_sig_info: rssi=%hhd, rsrq=%hhd, rsrp=%hd, snr=%hd\n",
                info.lte.rssi,
                info.lte.rsrq,
                info.lte.rsrp,
                info.lte.snr);
    } 
    
    if(info.has_nr5g)
    {
        printf("nr5g_sig_info: rsrp=%hd, rsrq=%hd, snr=%hd\n",
                info.nr5g.rsrp,
                info.nr5g.rsrq,
                info.nr5g.snr);
    }

    if(info.has_cdma)
    {
        printf("cdma_sig_info: rssi=%hhd, ecio=%hd\n",
                info.cdma.rssi,
                info.cdma.ecio);
    } 

    if(info.has_hdr)
    {
        printf("hdr_sig_info: rssi=%hhd, ecio=%hd, sinr=%hd, io=%d\n",
                info.hdr.rssi,
                info.hdr.ecio,
                info.hdr.sinr,
                info.hdr.io);
    }

    if(internal_nw_get_signal_strength_level(level, level_info, sizeof(level_info)) == 0)
    {
        printf("signal strength level is %d, unrecognized\n", level);
    }
    else
    {
        printf("signal strength level is %s\n", level_info);
    }
}

#if 0
void item_ql_nw_get_cell_access_status(void)
{
    int ret;
    QL_NW_CELL_ACCESS_STATE_TYPE_E e_state;

    ret = ql_nw_get_cell_access_status(&e_state);
    printf("ql_nw_get_cell_access_state ret = %d, e_state=%d\n", ret, e_state);
}

void item_ql_nw_get_nitz_time_info(void)
{
    int ret;
    ql_nw_nitz_time_info_t t_info;

    memset(&t_info, 0, sizeof(t_info));
    ret = ql_nw_get_nitz_time_info(&t_info);
    printf("ql_nw_get_nitz_time_info ret = %d\n nitz_time=%s, abs_time=%lld, leap_sec=%hhd\n", 
        ret,
        t_info.nitz_time, 
        t_info.abs_time, 
        t_info.leap_sec);
}
#endif

void item_ql_nw_set_voice_reg_ind_cb(void)
{
    int ret = 0;
    int reg_flag = 0;
    
    printf("please input voice reg option: (0: unreg, other: reg): ");
    ret = t_get_int(&reg_flag);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }
    if(reg_flag)
    {
        ret = ql_nw_set_voice_reg_ind_cb(nw_voice_reg_event_ind_cb);
    }
    else
    {
        ret = ql_nw_set_voice_reg_ind_cb(NULL);
    }
    printf("ql_nw_reg_voice_reg_event ret = %d\n", ret);
}

#if 0
void item_ql_nw_set_data_reg_ind_cb(void)
{
    int ret = 0;
    int reg_flag = 0;
    
    printf("please input data reg option: (0: unreg, other: reg): ");
    ret = t_get_int(&reg_flag);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }
    if(reg_flag)
    {
        ret = ql_nw_set_data_reg_ind_cb(nw_data_reg_event_ind_cb);
    }
    else
    {
        ret = ql_nw_set_data_reg_ind_cb(NULL);
    }
    printf("ql_nw_reg_data_reg_event ret = %d\n", ret);
}
#endif

void item_ql_nw_set_signal_strength_chg_ind_cb(void)
{
    int ret = 0;
    int reg_flag = 0;
    
    printf("please input signal strength change reg option: (0: unreg, other: reg): ");
    ret = t_get_int(&reg_flag);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }
    
    if(reg_flag)
    {
        ret = ql_nw_set_signal_strength_ind_cb(nw_signal_strength_event_ind_cb);
    }
    else
    {
        ret = ql_nw_set_signal_strength_ind_cb(NULL);
    }
    printf("ql_nw_reg_signal_strength_chg_event ret = %d\n", ret);
}

#if 0
void item_ql_nw_set_cell_access_status_chg_ind_cb(void)
{
    int ret = 0;
    int reg_flag = 0;
    
    printf("please input cell access status change reg option: (0: unreg, other: reg): ");
    ret = t_get_int(&reg_flag);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }
    
    if(reg_flag)
    {
        ret = ql_nw_set_cell_access_status_ind_cb(nw_cell_access_status_event_ind_cb);
    }
    else
    {
        ret = ql_nw_set_cell_access_status_ind_cb(NULL);
    }
    printf("ql_nw_reg_cell_access_status_event ret = %d\n", ret);
}

void item_ql_nw_set_nitz_time_update_ind_cb(void)
{
    int ret = 0;
    int reg_flag = 0;
    
    printf("please input nitz time update reg option: (0: unreg, other: reg): ");
    ret = t_get_int(&reg_flag);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }

    if(reg_flag)
    {
        ret = ql_nw_set_nitz_time_update_ind_cb(nw_nitz_time_update_event_ind_cb);
    }
    else
    {
        ret = ql_nw_set_nitz_time_update_ind_cb(NULL);
    }
    printf("ql_nw_reg_nitz_time_update_event ret = %d\n", ret);
}

void item_ql_nw_set_wea_alert_ind_cb(void)
{
    int ret = 0;
    int reg_flag = 0;
    
    printf("please input wea alert reg option: (0: unreg, other: reg): ");
    ret = t_get_int(&reg_flag);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }

    if(reg_flag)
    {
        ret = ql_nw_set_wea_alert_ind_cb(nw_wea_alert_event_ind_cb);
    }
    else
    {
        ret = ql_nw_set_wea_alert_ind_cb(NULL);
    }
    printf("ql_nw_set_wea_alert_ind_cb ret = %d\n", ret);
}

void item_ql_nw_set_etws_alert_ind_cb(void)
{
    int ret = 0;
    int reg_flag = 0;
    
    printf("please input etws alert reg option: (0: unreg, other: reg): ");
    ret = t_get_int(&reg_flag);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }

    if(reg_flag)
    {
        ret = ql_nw_set_etws_alert_ind_cb(nw_etws_alert_event_ind_cb);
    }
    else
    {
        ret = ql_nw_set_etws_alert_ind_cb(NULL);
    }
    printf("ql_nw_set_etws_alert_ind_cb ret = %d\n", ret);
}
#endif

static void internal_nw_get_mcc_mnc_value(char *plmn, int plmn_len, unsigned short *mcc, unsigned short *mnc)
{
    int mcc1 = 0, mcc2 = 0, mcc3 = 0;
    int mnc1 = 0, mnc2 = 0, mnc3 = 0;
    
    if(plmn_len < 3 || plmn == NULL) {
        printf("get mcc mnc from plmn fail, param is invalid\n");
        return;
    }

    mcc1 = plmn[0] & 0X0F;
    mcc2 = plmn[0] >> 4;
    mcc3 = plmn[1] & 0x0F;
    
    mnc3 = plmn[1] >> 4;
    mnc2 = plmn[2] >> 4;
    mnc1 = plmn[2] & 0x0F;
    
    *mcc = mcc1 * 100 + mcc2 * 10 + mcc3;
    
    if(0X0F == mnc3) {
        *mnc = mnc1 * 10 + mnc2;
    }
    else {
        *mnc = mnc1 * 100 + mnc2 * 10 + mnc3;
    }
}

static int internal_nw_get_net_status(QL_NW_NETWORK_STATUS_TYPE_E net, char* buf, int buf_len)
{
    int ret_val = 1;
    
    if(buf == NULL || buf_len < 2)
    {
        printf("param is valid\n");
        return 0;
    }

    memset(buf, 0, buf_len);
    
    switch(net)
    {
        case QL_NW_NETWORK_STATUS_NONE:
            strncpy(buf, "NONE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_NETWORK_STATUS_CURRENT_SERVING:
            strncpy(buf, "CURRENT_SERVING", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_NETWORK_STATUS_PREFERRED:
            strncpy(buf, "PREFERED", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_NETWORK_STATUS_NOT_PREFERRED:
            strncpy(buf, "NOT_PREFERRED", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_NETWORK_STATUS_AVAILABLE:
            strncpy(buf, "AVAILABLE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_NETWORK_STATUS_FORBIDDEN:
            strncpy(buf, "FORBIDDEN", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        default:
            ret_val = 0;
            break;
    }
    return ret_val;
}

#if 0
static int internal_nw_get_nas_rat(int radio, char* buf, int buf_len)
{
    int ret_val = 1;
    
    if(buf == NULL || buf_len < 2)
    {
        printf("param is valid\n");
        return 0;
    }

    memset(buf, 0, buf_len);
    
    switch(radio)
    {
        case 0:
            strncpy(buf, "UNKNOWN", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case 1:
            strncpy(buf, "GSM", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case 2:
            strncpy(buf, "WCDMA", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case 3:
            strncpy(buf, "TDSCDMA", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case 4:
            strncpy(buf, "LTE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case 5:
            strncpy(buf, "NR5G", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case 6:
            strncpy(buf, "CDMA", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case 7:
            strncpy(buf, "HDR", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        default:
            ret_val = 0;
            break;
    }
    return ret_val;
}
#endif

static int internal_nw_get_radio_tech(QL_NW_RADIO_TECH_TYPE_E radio, char* buf, int buf_len)
{
    int ret_val = 1;
    
    if(buf == NULL || buf_len < 2)
    {
        printf("param is valid\n");
        return 0;
    }

    memset(buf, 0, buf_len);
    
    switch(radio)
    {
        case QL_NW_RADIO_TECH_TD_SCDMA:
            strncpy(buf, "TD_SCDMA", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_GSM:
            strncpy(buf, "GSM", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_HSPAP:
            strncpy(buf, "HSPAP", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_LTE:
            strncpy(buf, "LTE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_EHRPD:
            strncpy(buf, "EHRPD", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_EVDO_B:
            strncpy(buf, "EVDO_B", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_HSPA:
            strncpy(buf, "HSPA", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_HSUPA:
            strncpy(buf, "HSUPA", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_HSDPA:
            strncpy(buf, "HSDPA", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_EVDO_A:
            strncpy(buf, "EVDO_A", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_EVDO_0:
            strncpy(buf, "EVDO_0", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_1xRTT:
            strncpy(buf, "1xRTT", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_IS95B:
            strncpy(buf, "IS95B", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_IS95A:
            strncpy(buf, "IS95A", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_UMTS:
            strncpy(buf, "UMTS", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_EDGE:
            strncpy(buf, "EDGE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_GPRS:
            strncpy(buf, "GPRS", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_NR5G:
            strncpy(buf, "NR5G", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_RADIO_TECH_NONE:
            strncpy(buf, "NONE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        default:
            ret_val = 0;
            break;
    }
    return ret_val;
}

static int internal_nw_get_tech_domain(QL_NW_TECH_DOMAIN_TYPE_E domain, char* buf, int buf_len)
{
    int ret_val = 1;
    
    if(buf == NULL || buf_len < 2)
    {
        printf("param is valid\n");
        return 0;
    }

    memset(buf, 0, buf_len);
    
    switch(domain)
    {
        case QL_NW_TECH_DOMAIN_NONE:
            strncpy(buf, "NONE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_TECH_DOMAIN_3GPP:
            strncpy(buf, "3GPP", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_TECH_DOMAIN_3GPP2:
            strncpy(buf, "3GPP2", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        default:
            ret_val = 0;
            break;
    }
    return ret_val;
}

static int internal_nw_get_signal_strength_level(QL_NW_SIGNAL_STRENGTH_LEVEL_E level, char* buf, int buf_len)
{
    int ret_val = 1;
    
    if(buf == NULL || buf_len < 2)
    {
        printf("param is valid\n");
        return 0;
    }

    memset(buf, 0, buf_len);
    
    switch(level)
    {
        case QL_NW_SIGNAL_STRENGTH_LEVEL_NONE:
            strncpy(buf, "UNKNOWN", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_SIGNAL_STRENGTH_LEVEL_POOR:
            strncpy(buf, "POOR", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_SIGNAL_STRENGTH_LEVEL_MODERATE:
            strncpy(buf, "MODERATE", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_SIGNAL_STRENGTH_LEVEL_GOOD:
            strncpy(buf, "GOOD", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        case QL_NW_SIGNAL_STRENGTH_LEVEL_GREAT:
            strncpy(buf, "GREAT", buf_len - 1);
            buf[buf_len - 1] = '\0';
            break;
        default:
            ret_val = 0;
            break;
    }
    return ret_val;
}

static void internal_nw_get_service_option(unsigned short so_mask, char *buf, int buf_len)
{
    int remain_len = buf_len;

    if(NULL == buf || buf_len < 2)
    {
        printf("param is valid\n");
        return;
    }
    memset(buf, 0, buf_len);

    if(so_mask & QL_NW_NR5G_SO_TDD)
    {
        if(remain_len > strlen("NR5G_TDD|") + 1)
        {
            strcat(buf + (buf_len - remain_len), "NR5G_TDD|");
            remain_len = buf_len - strlen("NR5G_TDD|");
        }
    }

    if(so_mask & QL_NW_NR5G_SO_SUB6)
    {
        if(remain_len > strlen("NR5G_SUB6|") + 1)
        {
            strcat(buf + (buf_len - remain_len), "NR5G_SUB6|");
            remain_len = buf_len - strlen("NR5G_SUB6|");
        }
    }

    if(so_mask & QL_NW_NR5G_SO_MMWAVE)
    {
        if(remain_len > strlen("NR5G_MMWAVE|") + 1)
        {
            strcat(buf + (buf_len - remain_len), "NR5G_MMWAVE|");
            remain_len = buf_len - strlen("NR5G_MMWAVE|");
        }
    }
    
    if(so_mask & QL_NW_NR5G_SO_NSA)
    {
        if(remain_len > strlen("NR5G_NSA|") + 1)
        {
            strcat(buf + (buf_len - remain_len), "NR5G_NSA|");
            remain_len = buf_len - strlen("NR5G_NSA|");
        }
    }
    
    if(so_mask & QL_NW_NR5G_SO_SA)
    {
        if(remain_len > strlen("NR5G_SA|") + 1)
        {
            strcat(buf + (buf_len - remain_len), "NR5G_SA|");
            remain_len = buf_len - strlen("NR5G_SA|");
        }
    }

    if(strlen(buf) > 0)
    {
        buf[strlen(buf) - 1] = '\0';
    }
    return;
}



#if 0
void item_ql_nw_wea_set_config(void)
{
    int ret = 0;
    int choice = 0;
    int item = 0;
    ql_nw_wea_config_t config = {0};

    printf("test ql_nw_wea_set_config: \n");
    printf(" 1)Presidential_alert\n");
    printf(" 2)Extreme_alert\n");
    printf(" 3)Severe_alert\n");
    printf(" 4)Amber_alert\n");
    printf(" 5)PublicSafety_alert\n");
    printf(" 6)StateLocalTest_alert\n");
    printf(" 7)RMT_alert\n");
    printf(" 8)Exercise_alert\n");
    printf(" 9)CMSPDefined_alert\n");
    printf("10)Spanish_alert\n");
    printf("please enter item to config: ");
    ret = t_get_int(&choice);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }
    printf("\n");

    switch (choice)
    {
        case 1:
            item |= QL_NW_WEA_CONFIG_PRESIDENTIAL_ALERT;
            printf("Presidential_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.Presidential_alert) != 1)
            {
                printf("Invalid input\n");
                return;
            }
            break;
        case 2:
            item |= QL_NW_WEA_CONFIG_EXTREME_ALERT;
            printf("Extreme_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.Extreme_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        case 3:
            item |= QL_NW_WEA_CONFIG_SEVERE_ALERT;
            printf("Severe_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.Severe_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        case 4:
            item |= QL_NW_WEA_CONFIG_AMBER_ALERT;
            printf("Amber_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.Amber_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }   
            break;
        case 5:
            item |= QL_NW_WEA_CONFIG_PUBLIC_SAFETY_ALERT;
            printf("PublicSafety_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.PublicSafety_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        case 6:
            item |= QL_NW_WEA_CONFIG_STATE_LOCAL_TEST_ALERT;
            printf("StateLocalTest_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.StateLocalTest_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        case 7:
            item |= QL_NW_WEA_CONFIG_RMT_ALERT;
            printf("RMT_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.RMT_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        case 8:
            item |= QL_NW_WEA_CONFIG_EXERCISE_ALERT;
            printf("Exercise_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.Exercise_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        case 9:
            item |= QL_NW_WEA_CONFIG_CMSP_DEFINED_ALERT;
            printf("CMSPDefined_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.CMSPDefined_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        case 10:
            item |= QL_NW_WEA_CONFIG_SPANISH_ALERT;
            printf("Spanish_alert: 0 - disable, 1 - enable:");
            if(scanf("%hhu", &config.Spanish_alert) != 1)
            {    
                printf("Invalid input\n");
                return;
            }
            break;
        default:
            printf("bad choice: %d\n", choice);
            return;
    }
    ret = getchar();

    ret = ql_nw_set_wea_config(item, &config);
    if (ret == QL_ERR_OK)
    {
        printf("ok\n");
    }
    else
    {
        printf("failed, ret = %d\n", ret);
    }
}

void item_ql_nw_wea_get_config(void)
{
    int ret = 0;
    ql_nw_wea_config_t config = {0};

    printf("test ql_nw_get_wea_config: \n");
    ret = ql_nw_get_wea_config(&config);
    if (ret == QL_ERR_OK)
    {
        printf("****** wea Config ******\n");
        printf("Presidential_alert: %s\n", config.Presidential_alert ? "true" : "false");
        printf("Extreme_alert: %s\n", config.Extreme_alert ? "true" : "false");
        printf("Severe_alert: %s\n", config.Severe_alert ? "true" : "false");
        printf("Amber_alert: %s\n", config.Amber_alert ? "true" : "false");
        printf("PublicSafety_alert: %s\n", config.PublicSafety_alert ? "true" : "false");
        printf("StateLocalTest_alert: %s\n", config.StateLocalTest_alert ? "true" : "false");
        printf("RMT_alert: %s\n", config.RMT_alert ? "true" : "false");
        printf("Exercise_alert: %s\n", config.Exercise_alert ? "true" : "false");
        printf("CMSPDefined_alert: %s\n", config.CMSPDefined_alert ? "true" : "false");
        printf("Spanish_alert: %s\n", config.Spanish_alert ? "true" : "false");
    }
    else
    {
        printf("failed, ret = %d\n", ret);
    }
}

void item_ql_nw_etws_set_config(void)
{
    int ret = 0;
    int etws_enable = 0;

    printf("please input etws config (0: disable,  1: enable): ");
    ret = t_get_int(&etws_enable);
    if(ret != 0)
    {    
        printf("Invalid input\n");
        return;
    }

    ret = ql_nw_set_etws_config((unsigned char)etws_enable);
    printf("item_ql_nw_etws_set_config ret = %d\n", ret);
}

void item_ql_nw_etws_get_config(void)
{
    int ret = 0;
    unsigned char etws_enable = 0;

    ret = ql_nw_get_etws_config(&etws_enable);
    printf("ql_nw_get_etws_config ret = %d \t etws_enable=%d\n", 
            ret, etws_enable);
}
#endif

void item_ql_nw_set_service_error_cb(void)
{
    int ret = 0;

    ret = ql_nw_set_service_error_cb(nw_service_error_cb);
    if(ret != QL_ERR_OK)
    {
        printf("Failed to ql_nw_set_service_error_cb, ret=%d\n", ret);
    }
    else
    {
        printf("Sucessful\n");
    }
}
void item_ql_set_nw_slot(void)
{
    int choice = -1;
    int ret = -1;
    printf("1:sim card 1, 2:sim card 2\n");
    while (1)
    {
        if (scanf("%d", &choice) == 1) 
        {  
            if (choice == 1 || choice == 2) 
            {
                break;  
            }
            else
            {
                printf("invlid input \n");
            }
        }
        else
        {
            int c;
            while ((c = getchar()) != '\n' && c != EOF);
            printf("invalid input \n");
        }
                    
    }
                
    if (choice == 1)
    {
        ret = ql_set_sms_slot(QL_SIM_SLOT_1);
        if(ret != QL_ERR_OK)
        {
            printf("ql_set_sms_slot failed %d\n",ret);
        }
        else
        {
            printf("ql_set_sms_slot success\n");
        }
    } 
    else if (choice == 2)
    {
       ret = ql_set_sms_slot(QL_SIM_SLOT_2);
       if(ret != QL_ERR_OK)
       {
            printf("ql_set_sms_slot failed %d\n",ret);
       }
       else
       {
            printf("ql_set_sms_slot success\n");
       }
    }
    
}


static t_item_t ql_nw_items[] = 
{
      
    {"ql_nw_init", item_ql_nw_init},
//    {"ql_nw_network_scan", item_ql_nw_network_scan},
    {"ql_nw_set_power_mode", item_ql_nw_set_power_mode},
//    {"ql_nw_set_pref_nwmode_roaming", item_ql_nw_set_pref_nwmode_roaming},
//    {"ql_nw_get_pref_nwmode_roaming", item_ql_nw_get_pref_nwmode_roaming},
//    {"ql_nw_get_mobile_operator_name", item_ql_nw_get_mobile_operator_name},
//    {"ql_nw_get_cell_info", item_ql_nw_get_cell_info},
    {"ql_nw_get_voice_reg_status", item_ql_nw_get_voice_reg_status},
    {"ql_nw_get_data_reg_status", item_ql_nw_get_data_reg_status},
    {"ql_nw_get_signal_strength", item_ql_nw_get_signal_strength},
//    {"ql_nw_get_cell_access_status", item_ql_nw_get_cell_access_status},
//    {"ql_nw_get_nitz_time_info", item_ql_nw_get_nitz_time_info},
    {"ql_nw_set_voice_reg_ind_cb", item_ql_nw_set_voice_reg_ind_cb},
//    {"ql_nw_set_data_reg_ind_cb", item_ql_nw_set_data_reg_ind_cb},
    {"ql_nw_set_signal_strength_chg_ind_cb", item_ql_nw_set_signal_strength_chg_ind_cb},
//    {"ql_nw_set_cell_access_status_chg_ind_cb", item_ql_nw_set_cell_access_status_chg_ind_cb},
//    {"ql_nw_set_nitz_time_update_ind_cb", item_ql_nw_set_nitz_time_update_ind_cb},
//    {"ql_nw_set_wea_alert_ind_cb", item_ql_nw_set_wea_alert_ind_cb},
//    {"ql_nw_wea_set_config", item_ql_nw_wea_set_config},
//    {"ql_nw_wea_get_config", item_ql_nw_wea_get_config},
//    {"ql_nw_set_etws_alert_ind_cb", item_ql_nw_set_etws_alert_ind_cb},
//    {"ql_nw_etws_set_config", item_ql_nw_etws_set_config},
//    {"ql_nw_etws_get_config", item_ql_nw_etws_get_config},
    {"ql_nw_set_service_error_cb", item_ql_nw_set_service_error_cb},
    {"ql_set_nw_slot", item_ql_set_nw_slot},
    {"ql_nw_deinit", item_ql_nw_deinit},
};

t_module_t ql_nw_module =
{
    "nw",
    T_ARRAY_SIZE(ql_nw_items),
    ql_nw_items
};

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief Read a int value from stdin 
  @param[out] val, Return read data
  @return 
  0 - successful
  1 - read an enter 
  -1 - invalid input
  */
/*-----------------------------------------------------------------------------------------------*/
int t_get_int(int *val)
{
    int dat;
    char *ptr_end = NULL;
    char buf[256] = {0};

    if(NULL == fgets(buf, sizeof(buf)-1, stdin))
    {
        return -1;
    }
#if 0    
    if(0 == buf[0])
    {
        return -1;
    }
#endif
    if(buf[0] == '\n')
    {
        return 1;
    }

    dat = strtol(buf, &ptr_end, 10);
    if(ptr_end!=NULL && ptr_end[0]!='\n')
    {
        return -1;
    }

    if(val)
    {
        val[0] = dat;
    }

    return 0;
}

void dump_items()
{
    int i;

    printf("\n");
    printf("The current module is: \n");

    for(i=0; i< ql_nw_module.item_len; i++)
    {
        printf("%d\t%s\n", i, ql_nw_module.item_list[i].name);
    }
    printf("-1\texit\n");
}

int main(int argc, char *argv[])
{
    int ret;
    int idx;

    dump_items();

    while(1)
    {
        printf("Please enter your choice: ");
        ret = t_get_int(&idx);
        printf("\n");
        if(ret < 0)
        {
            printf("Invalid input\n");
            continue;
        }
        else if(ret == 1)
        {
            dump_items();
            continue;
        }

        if(idx == -1)
        {
            break;
        }

        if(idx<0 || idx>=ql_nw_module.item_len)
        {
            printf("Not support idx: %d\n", idx);
            continue;
        }

        printf("->Item : %s\n", ql_nw_module.item_list[idx].name);
        ql_nw_module.item_list[idx].handle();
    }

    return 0;
}

