/*
 * hostapd / AP table
 * Copyright (c) 2002-2009, Jouni Malinen <j@w1.fi>
 * Copyright (c) 2003-2004, Instant802 Networks, Inc.
 * Copyright (c) 2006, Devicescape Software, Inc.
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

#include "utils/includes.h"

#include "utils/common.h"
#include "utils/eloop.h"
#include "common/ieee802_11_defs.h"
#include "common/ieee802_11_common.h"
#include "hostapd.h"
#include "ap_config.h"
#include "ieee802_11.h"
#include "sta_info.h"
#include "beacon.h"
#include "ap_list.h"


/* AP list is a double linked list with head->prev pointing to the end of the
 * list and tail->next = NULL. Entries are moved to the head of the list
 * whenever a beacon has been received from the AP in question. The tail entry
 * in this link will thus be the least recently used entry. */


static int ap_list_beacon_olbc(struct hostapd_iface *iface, struct ap_info *ap)
{
	int i;

	if (iface->current_mode == NULL ||
	    iface->current_mode->mode != HOSTAPD_MODE_IEEE80211G ||
	    iface->conf->channel != ap->channel)
		return 0;

	if (ap->erp != -1 && (ap->erp & ERP_INFO_NON_ERP_PRESENT))
		return 1;

	for (i = 0; i < WLAN_SUPP_RATES_MAX; i++) {
		int rate = (ap->supported_rates[i] & 0x7f) * 5;
		if (rate == 60 || rate == 90 || rate > 110)
			return 0;
	}

	return 1;
}


static struct ap_info * ap_get_ap(struct hostapd_iface *iface, const u8 *ap)
{
	struct ap_info *s;

	s = iface->ap_hash[STA_HASH(ap)];
	while (s != NULL && os_memcmp(s->addr, ap, ETH_ALEN) != 0)
		s = s->hnext;
	return s;
}


static void ap_ap_list_add(struct hostapd_iface *iface, struct ap_info *ap)
{
	if (iface->ap_list) {
		ap->prev = iface->ap_list->prev;
		iface->ap_list->prev = ap;
	} else
		ap->prev = ap;
	ap->next = iface->ap_list;
	iface->ap_list = ap;
}


static void ap_ap_list_del(struct hostapd_iface *iface, struct ap_info *ap)
{
	if (iface->ap_list == ap)
		iface->ap_list = ap->next;
	else
		ap->prev->next = ap->next;

	if (ap->next)
		ap->next->prev = ap->prev;
	else if (iface->ap_list)
		iface->ap_list->prev = ap->prev;
}


static void ap_ap_hash_add(struct hostapd_iface *iface, struct ap_info *ap)
{
	ap->hnext = iface->ap_hash[STA_HASH(ap->addr)];
	iface->ap_hash[STA_HASH(ap->addr)] = ap;
}


static void ap_ap_hash_del(struct hostapd_iface *iface, struct ap_info *ap)
{
	struct ap_info *s;

	s = iface->ap_hash[STA_HASH(ap->addr)];
	if (s == NULL) return;
	if (os_memcmp(s->addr, ap->addr, ETH_ALEN) == 0) {
		iface->ap_hash[STA_HASH(ap->addr)] = s->hnext;
		return;
	}

	while (s->hnext != NULL &&
	       os_memcmp(s->hnext->addr, ap->addr, ETH_ALEN) != 0)
		s = s->hnext;
	if (s->hnext != NULL)
		s->hnext = s->hnext->hnext;
	else
		wpa_printf(MSG_INFO, "AP: could not remove AP " MACSTR
			   " from hash table",  MAC2STR(ap->addr));
}


static void ap_free_ap(struct hostapd_iface *iface, struct ap_info *ap)
{
	ap_ap_hash_del(iface, ap);
	ap_ap_list_del(iface, ap);

	iface->num_ap--;
	os_free(ap);
}


static void hostapd_free_aps(struct hostapd_iface *iface)
{
	struct ap_info *ap, *prev;

	ap = iface->ap_list;

	while (ap) {
		prev = ap;
		ap = ap->next;
		ap_free_ap(iface, prev);
	}

	iface->ap_list = NULL;
}


static struct ap_info * ap_ap_add(struct hostapd_iface *iface, const u8 *addr)
{
	struct ap_info *ap;

	ap = os_zalloc(sizeof(struct ap_info));
	if (ap == NULL)
		return NULL;

	/* initialize AP info data */
	os_memcpy(ap->addr, addr, ETH_ALEN);
	ap_ap_list_add(iface, ap);
	iface->num_ap++;
	ap_ap_hash_add(iface, ap);

	if (iface->num_ap > iface->conf->ap_table_max_size && ap != ap->prev) {
		wpa_printf(MSG_DEBUG, "Removing the least recently used AP "
			   MACSTR " from AP table", MAC2STR(ap->prev->addr));
		ap_free_ap(iface, ap->prev);
	}

	return ap;
}


#ifdef CONFIG_24G_BW_SWITCH
// To check 2.4G band 20M/40M bandwidth switch. Added by Yewei
static int ap_list_handle_obss(struct hostapd_data *hapd, struct ieee802_11_elems *elems)
{
	int switch_to_20 = 0;
	struct hostapd_iface *iface = hapd->iface;

	// OBSS operation only applicable to 2.4G band.
	if (hapd->iface->current_mode && (hapd->iface->current_mode->mode == HOSTAPD_MODE_IEEE80211A))
	    return 0;
	
	// Only applicable when supporting at least HT 40
	if ((!(iface->conf->ht_capab & HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET)) ||
		(!(iface->drv_flags & WPA_DRIVER_FLAGS_HT_2040_COEX)))
		return 0;
	
	// Only applicable when auto_20m_40m is enabled
	if (!iface->conf->auto_20m_40m)
	{
		return 0;
	}

	// If the Beacon is on the same channel, no need to handle
	if (elems->ds_params && hapd->iface->current_mode &&
	    (hapd->iface->current_mode->mode == HOSTAPD_MODE_IEEE80211G ||
	     hapd->iface->current_mode->mode == HOSTAPD_MODE_IEEE80211B) &&
	    (hapd->iconf->channel == elems->ds_params[0])) {
		return 0;
	}

	// Check if need to switch to 20M bandwidth
	if (elems->ht_capabilities) {
		struct ieee80211_ht_capabilities *ht_cap = (struct ieee80211_ht_capabilities *)elems->ht_capabilities;
		if (0 != iface->conf->secondary_channel) {
			iface->secondary_ch = iface->conf->secondary_channel;
			iface->conf->secondary_channel = 0;
			//ieee802_11_update_beacons(iface);
			switch_to_20 = 1;
			wpa_printf(MSG_INFO, "ap_list_handle_obss, %s, switch_to_20:%d", iface->phy, switch_to_20);
		}
	}
	else if (!elems->ht_capabilities && !elems->vht_capabilities && !elems->he_capabilities) {
		if (iface->conf->secondary_channel) {
			iface->secondary_ch = iface->conf->secondary_channel;
			iface->conf->secondary_channel = 0;
			//ieee802_11_update_beacons(iface);
			switch_to_20 = 1;
			wpa_printf(MSG_INFO, "ap_list_handle_obss, %s, switch_to_20:%d", iface->phy, switch_to_20);			
		}
	} 	

	return switch_to_20;
}
#endif /* CONFIG_24G_BW_SWITCH */


#ifdef CONFIG_5G_BW_SWITCH
// To check 5G band 20M/40M/80M bandwidth switch in AP list timer. Added by Liangyu Chu
static int ap_list_5g_BW_switch(struct hostapd_iface *iface)
{
	// Only applicable to 5G band
	if (iface->current_mode && 
		(iface->current_mode->mode == HOSTAPD_MODE_IEEE80211G ||
	     iface->current_mode->mode == HOSTAPD_MODE_IEEE80211B))
	{
		/*wpa_printf(MSG_INFO, "ap_list_5g_BW_switch: %s, Not at 5G band", iface->phy);*/	
	     return 0;	
	}
	
	// Only applicable when supporting at least HT 40
    if (!iface->conf->ht_capab || !(iface->conf->ht_capab & HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET)) 
	{
		/*wpa_printf(MSG_INFO, "ap_list_5g_BW_switch: %s, Not support HT 40M bandwidth", iface->phy);*/
		return 0;
	}
	
	// Only applicable to VHT or HE mode
	// if (!iface->conf->vht_capab || !iface->conf->he_phy_capab)
	if (!iface->conf->ieee80211ac && !iface->conf->ieee80211ax)
	{
		/*wpa_printf(MSG_INFO, "ap_list_5g_BW_switch: %s, Not support VHT or HE capabilities", iface->phy);*/
		return 0;
	}

	// Only applicable when auto_80M is enabled
	if (!iface->conf->auto_80m)
	{
		/*wpa_printf(MSG_INFO, "ap_list_5g_BW_switch: %s, auto BW switch is not enabled", iface->phy);*/
		return 0;
	}

	int bw_switch = 0;
	int current_bw = 20;
	int target_bw = 80;

	// Compute current bandwith
	if (iface->conf->vht_oper_chwidth 
#ifdef CONFIG_IEEE80211AX
        || iface->conf->he_oper_chwidth
#endif
       )
		current_bw = 80;
	else if (iface->conf->secondary_channel)
		current_bw = 40;

	// Compute target bandwidth from AP list statistics
	if ((iface->num_ap > AP_LIST_MAX_NUM_STA_TO_20M) || 
		(iface->max_rssi && (iface->max_rssi > AP_LIST_MAX_RSSI_TO_20M)))
		target_bw = 20;
	else if ((iface->num_ap > AP_LIST_MAX_NUM_STA_TO_40M) || 
		(iface->max_rssi && (iface->max_rssi > AP_LIST_MAX_RSSI_TO_40M)))
		target_bw = 40;

	// Only applicable when BW changes 
	if (current_bw == target_bw)
	{
		wpa_printf(MSG_INFO, "ap_list_5g_BW_switch: %s, num_ap=%d, max_rssi=%d, target_bw=%d, current_bw=%d, no need to switch BW",
			iface->phy, iface->num_ap, iface->max_rssi, target_bw, current_bw);
		return 0;
	}	
	else		
	{	
		wpa_printf(MSG_INFO, "ap_list_5g_BW_switch: %s, num_ap=%d, max_rssi=%d, target_bw=%d, current_bw=%d, need to switch BW",
			iface->phy, iface->num_ap, iface->max_rssi, target_bw, current_bw);
	}

	// 80M -> 20M switch
	if (current_bw == 80 && target_bw == 20)
	{
		// Record previous settings
		iface->secondary_ch = iface->conf->secondary_channel;
		iface->vht_oper_ch_bw = iface->conf->vht_oper_chwidth;
		iface->vht_center_freq_seg0_80m = iface->conf->vht_oper_centr_freq_seg0_idx;	

		// 20M bandwidth configuration
		iface->conf->secondary_channel = 0;
		iface->conf->vht_oper_chwidth = 0;
		iface->conf->vht_oper_centr_freq_seg0_idx = 0;

		bw_switch = 1;
	}
	// 20M -> 80M switch
	else if (current_bw == 20 && target_bw == 80)
	{
		// 80M bandwidth configuration
		iface->conf->secondary_channel = iface->secondary_ch;
		iface->conf->vht_oper_chwidth = iface->vht_oper_ch_bw;
		iface->conf->vht_oper_centr_freq_seg0_idx = iface->vht_center_freq_seg0_80m;

		bw_switch = 1;
	}

	// 80M -> 40M switch
	else if (current_bw == 80 && target_bw == 40)
	{
		// Record previous settings
		iface->vht_oper_ch_bw = iface->conf->vht_oper_chwidth;
		iface->vht_center_freq_seg0_80m = iface->conf->vht_oper_centr_freq_seg0_idx;	

		// 40M bandwidth configuration
		iface->conf->vht_oper_chwidth = 0;
		iface->conf->vht_oper_centr_freq_seg0_idx = iface->vht_center_freq_seg0_40m;
		
		bw_switch = 1;
	}
	// 40M -> 80M switch
	else if (current_bw == 40 && target_bw == 80)
	{
		// Record previous settings
		iface->vht_center_freq_seg0_40m = iface->conf->vht_oper_centr_freq_seg0_idx;	
	
		// 80M bandwidth configuration
		iface->conf->vht_oper_chwidth = iface->vht_oper_ch_bw;
		iface->conf->vht_oper_centr_freq_seg0_idx = iface->vht_center_freq_seg0_80m;

		bw_switch = 1;
	}
	
	// 40M -> 20M switch
	else if (current_bw == 40 && target_bw == 20)
	{
		// Record previous settings
		iface->secondary_ch = iface->conf->secondary_channel;
		iface->vht_center_freq_seg0_40m = iface->conf->vht_oper_centr_freq_seg0_idx;			

		// 20M bandwidth configuration
		iface->conf->secondary_channel = 0;
		iface->conf->vht_oper_centr_freq_seg0_idx = 0;
		
		bw_switch = 1;
	}
	// 20M -> 40M switch
	else if (current_bw == 20 && target_bw == 40)
	{
		// 40M bandwidth configuration		
		iface->conf->secondary_channel = iface->secondary_ch;
		iface->conf->vht_oper_centr_freq_seg0_idx = iface->vht_center_freq_seg0_40m;

		bw_switch = 1;
	}

	// Always set HE operation IEs the same as VHT operation IEs if applicable
#ifdef CONFIG_IEEE80211AX	
	if (bw_switch && iface->conf->ieee80211ac && iface->conf->ieee80211ax)
	{
		iface->conf->he_oper_chwidth = iface->conf->vht_oper_chwidth;
		iface->conf->he_oper_centr_freq_seg0_idx = iface->conf->vht_oper_centr_freq_seg0_idx;
	}
#endif
	
	return bw_switch;
}
#endif /* CONFIG_5G_BW_SWITCH */


#if defined CONFIG_24G_BW_SWITCH || defined CONFIG_5G_BW_SWITCH
void ap_list_process_beacon(struct hostapd_data *hapd,
#else
void ap_list_process_beacon(struct hostapd_iface *iface,
#endif /* defined CONFIG_24G_BW_SWITCH || defined CONFIG_5G_BW_SWITCH */
			    const struct ieee80211_mgmt *mgmt,
			    struct ieee802_11_elems *elems,
			    struct hostapd_frame_info *fi)
{
	struct ap_info *ap;
	int new_ap = 0;
	int set_beacon = 0;
#if defined CONFIG_24G_BW_SWITCH || defined CONFIG_5G_BW_SWITCH
	struct hostapd_iface *iface = hapd->iface;
#endif /* defined CONFIG_24G_BW_SWITCH || defined CONFIG_5G_BW_SWITCH */
#ifdef CONFIG_5G_BW_SWITCH
	int ssi_signal = fi ? fi->ssi_signal : 0;
	int ch_width = 20; // Default set each AP as 20M bandwidth
#endif /* CONFIG_5G_BW_SWITCH */

	if (iface->conf->ap_table_max_size < 1)
		return;

	ap = ap_get_ap(iface, mgmt->bssid);
	if (!ap) {
		ap = ap_ap_add(iface, mgmt->bssid);
		if (!ap) {
			wpa_printf(MSG_INFO,
				   "Failed to allocate AP information entry");
			return;
		}
		new_ap = 1;
	}

	merge_byte_arrays(ap->supported_rates, WLAN_SUPP_RATES_MAX,
			  elems->supp_rates, elems->supp_rates_len,
			  elems->ext_supp_rates, elems->ext_supp_rates_len);

	// Check ERP related info
	if (elems->erp_info)
		ap->erp = elems->erp_info[0];
	else
		ap->erp = -1;

	// Check channel info
	if (elems->ds_params)
		ap->channel = elems->ds_params[0];
	else if (elems->ht_operation)
		ap->channel = elems->ht_operation[0];
	else if (fi)
		ap->channel = fi->channel;

	// Check HT capablities
	if (elems->ht_capabilities)
		ap->ht_support = 1;
	else
		ap->ht_support = 0;

#ifdef CONFIG_5G_BW_SWITCH
	// Check VHT capabilities
	if (elems->vht_capabilities)
		ap->vht_support = 1;
	else
		ap->vht_support = 0;	

	// Check HE capabilities
	if (elems->he_capabilities)
		ap->he_support = 1;
	else
		ap->he_support = 0;

	// Check operation bandwidth as reported in Beacon frame. Only consider VHT and HE.
	if ((ap->vht_support || ap->he_support) && (elems->vht_operation))
	{
		struct ieee80211_vht_operation *vht_op = (struct ieee80211_vht_operation *)elems->vht_operation;
		int op_bw = vht_op->vht_op_info_chwidth;
		switch (op_bw)
		{
			case 0: ch_width = 20; break;
			case 1: ch_width = 80; break;
			case 2: ch_width = 160; break;
			default: break;
		}
	}
	ap->ch_width = ch_width;

	// Check RSSI of the received Beacon frame
	ap->ssi_signal = ssi_signal;
	if ((iface->max_rssi == 0) || (ssi_signal > iface->max_rssi))
		iface->max_rssi = ssi_signal;

	/*wpa_printf(MSG_INFO, "ap_list_process_beacon: BSSID=" MACSTR ", channel=%d, ch_width=%d, ssi_signal=%d, ht_support=%d, vht_support=%d, he_support=%d",
		MAC2STR(mgmt->bssid), ap->channel, ap->ch_width, ap->ssi_signal, ap->ht_support, ap->vht_support, ap->he_support);*/		
#endif /* CONFIG_5G_BW_SWITCH */

	os_get_reltime(&ap->last_beacon);

	if (!new_ap && ap != iface->ap_list) {
		/* move AP entry into the beginning of the list so that the
		 * oldest entry is always in the end of the list */
		ap_ap_list_del(iface, ap);
		ap_ap_list_add(iface, ap);
	}

	if (!iface->olbc &&
	    ap_list_beacon_olbc(iface, ap)) {
		iface->olbc = 1;
		wpa_printf(MSG_DEBUG, "OLBC AP detected: " MACSTR
			   " (channel %d) - enable protection",
			   MAC2STR(ap->addr), ap->channel);
		set_beacon++;
	}

	if (!iface->olbc_ht && !ap->ht_support &&
	    (ap->channel == 0 ||
	     ap->channel == iface->conf->channel ||
	     ap->channel == iface->conf->channel +
	     iface->conf->secondary_channel * 4)) {
		iface->olbc_ht = 1;
		hostapd_ht_operation_update(iface);
		wpa_printf(MSG_DEBUG, "OLBC HT AP detected: " MACSTR
			   " (channel %d) - enable protection",
			   MAC2STR(ap->addr), ap->channel);
		set_beacon++;
	}

#ifdef CONFIG_24G_BW_SWITCH
	// 2.4G band 20M/40M bandwidth adaptation. Added by Yewei
	if (ap_list_handle_obss(hapd, elems))
		set_beacon++;
#endif /* CONFIG_24G_BW_SWITCH */
#if 0
	// 5G band 20M/40M/80M bandwidth adaptation. Added by Liangyu Chu
	if (ap_list_5g_BW_switch(iface))
		set_beacon++;
#endif

	if (set_beacon)
		ieee802_11_update_beacons(iface);
}


void ap_list_timer(struct hostapd_iface *iface)
{
	struct os_reltime now;
	struct ap_info *ap;
	int set_beacon = 0;

	if (!iface->ap_list)
		return;

	os_get_reltime(&now);

	while (iface->ap_list) {
		ap = iface->ap_list->prev;
		if (!os_reltime_expired(&now, &ap->last_beacon,
					iface->conf->ap_table_expiration_time))
			break;

		ap_free_ap(iface, ap);
	}

#ifdef CONFIG_24G_BW_SWITCH
	/* yewei, if the number of neighbor aps is zero, need swtich to 40Mhz, 2024-01-04 */
	ap = iface->ap_list;
	if ((!ap) && (iface->secondary_ch) && (0 == iface->conf->secondary_channel) && 
	    (iface->conf->ht_capab & HT_CAP_INFO_SUPP_CHANNEL_WIDTH_SET) && 
	    (iface->drv_flags & WPA_DRIVER_FLAGS_HT_2040_COEX) &&
	    (iface->conf->auto_20m_40m)) {    
        	iface->conf->secondary_channel = iface->secondary_ch;
        	set_beacon++;
			wpa_printf(MSG_INFO, "ap_list_timer, %s, switch_to_40", iface->phy);			
    	}
#endif /* CONFIG_24G_BW_SWITCH */

#ifdef CONFIG_5G_BW_SWITCH
	// 5G band 20M/40M/80M bandwidth adaptation. Added by Liangyu Chu
	if (ap_list_5g_BW_switch(iface))
		set_beacon++;

	// Update max RSSI level
	iface->max_rssi = 0;
#endif /* CONFIG_5G_BW_SWITCH */

	if (iface->olbc || iface->olbc_ht) {
		int olbc = 0;
		int olbc_ht = 0;

		ap = iface->ap_list;
		while (ap && (olbc == 0 || olbc_ht == 0)) {
			if (ap_list_beacon_olbc(iface, ap))
				olbc = 1;
			if (!ap->ht_support)
				olbc_ht = 1;
			ap = ap->next;
		}
		if (!olbc && iface->olbc) {
			wpa_printf(MSG_DEBUG, "OLBC not detected anymore");
			iface->olbc = 0;
			set_beacon++;
		}
		if (!olbc_ht && iface->olbc_ht) {
			wpa_printf(MSG_DEBUG, "OLBC HT not detected anymore");
			iface->olbc_ht = 0;
			hostapd_ht_operation_update(iface);
			set_beacon++;
		}
	}

	if (set_beacon)
		ieee802_11_update_beacons(iface);
}


int ap_list_init(struct hostapd_iface *iface)
{
	return 0;
}


void ap_list_deinit(struct hostapd_iface *iface)
{
	hostapd_free_aps(iface);
}
