/**
 *
 *  (C) Copyright 2009-2016 Marvell International Ltd. All Rights Reserved
 *
 *  MARVELL CONFIDENTIAL
 *  The source code contained or described herein and all documents related to
 *  the source code ("Material") are owned by Marvell International Ltd or its
 *  suppliers or licensors. Title to the Material remains with Marvell
 *  International Ltd or its suppliers and licensors. The Material contains
 *  trade secrets and proprietary and confidential information of Marvell or its
 *  suppliers and licensors. The Material is protected by worldwide copyright
 *  and trade secret laws and treaty provisions. No part of the Material may be
 *  used, copied, reproduced, modified, published, uploaded, posted,
 *  transmitted, distributed, or disclosed in any way without Marvell's prior
 *  express written permission.
 *
 *  No license under any patent, copyright, trade secret or other intellectual
 *  property right is granted to or conferred upon you by disclosure or delivery
 *  of the Materials, either expressly, by implication, inducement, estoppel or
 *  otherwise. Any license under such intellectual property rights must be
 *  express and approved by Marvell in writing.
 *
 */


#define pr_fmt(fmt) KBUILD_MODNAME " core :%s:%d: " fmt, __func__, __LINE__

#include "fp_common.h"
#include "fp_core.h"

static const char *fastpath_driver_version = "Fastpath V2.1";
static struct fastpath_driver *fastpath;

#ifdef FASTPATH_WARN_AS_ERR
unsigned int debug_level = (DBG_WARN_AS_ERR | DBG_INFO);
#else
unsigned int debug_level = (DBG_INFO);
#endif

/* fastpath modules - defines modules add/remove order */
static struct fastpath_module_ops *fp_modules_ops[] = {
	&fp_device_ops,
	&fp_database_ops,
	&fp_learner_ops,
	&fp_classifier_ops,
	&fp_forward_ops,
	&fp_netlink_ops,
#ifdef CONFIG_ASR_TOE
	&fp_cm_ops,
#endif

};

struct fastpath_module *fp_module_get_by_name(const char *name)
{
	int i;

	for (i = 0; i < fastpath->num_modules; i++) {
		if (!fastpath->modules[i])
			continue;
		if (!strncmp(fastpath->modules[i]->name, name, MAX_NAME_SZ)) {
			kobject_get(&fastpath->modules[i]->kobj);
			return fastpath->modules[i];
		}
	}

	return NULL;
}

struct fastpath_module *fp_module_get_by_idx(int idx)
{
	if (fastpath->modules[idx]) {
		kobject_get(&fastpath->modules[idx]->kobj);
		return fastpath->modules[idx];
	}
	return NULL;
}

void fp_module_put(struct fastpath_module *m)
{
	kobject_put(&m->kobj);
}

/* debug flags: WDET*/
static ssize_t fastpath_debug_show(struct kobject *kobj,
			       struct kobj_attribute *attr, char *buf)
{
	char flags[] = "WDET";
	int i;

	for (i = 0; i < DBG_LEVEL_END_BIT; i++)
		if (!(debug_level & (1 << i)))
			flags[i] = '-';

	return scnprintf(buf, PAGE_SIZE, "fastpath debug flags: %s\n"
				"\nAvailable Flags:\n"
				"W - warnings as errors\n"
				"D - fpdb entry debug info\n"
				"E - Extended fpdb entry debug info (not used)\n"
				"T - fpdb trace logging\n", flags);
}

static int set_or_clear_flag(char flag, int set)
{
	switch (flag) {
	case 'w':
	case 'W':
		debug_level = (set) ? (debug_level | DBG_WARN_AS_ERR) :
				      (debug_level & ~DBG_WARN_AS_ERR);
		break;
	case 'd':
	case 'D':
		debug_level = (set) ? (debug_level | DBG_INFO) :
				      (debug_level & ~DBG_INFO);
		break;
	case 'e':
	case 'E':
		debug_level = (set) ? (debug_level | DBG_INFO_EXT) :
				      (debug_level & ~DBG_INFO_EXT);
		break;
	case 't':
	case 'T':
		debug_level = (set) ? (debug_level | DBG_TRACE_LOG) :
				      (debug_level & ~DBG_TRACE_LOG);
		break;
	default: return -EINVAL;
	}

	return 0;
}

static ssize_t fastpath_debug_store(struct kobject *kobj,
				      struct kobj_attribute *attr,
				      const char *buf, size_t size)
{
	int count = size;
	char c, flag;
	const char *ptr = buf;
	int ret, op;

	if (ptr[0] == '0') {
		debug_level = 0;
		return size;
	}

	while (count > 1) {
		c = ptr[0];
		flag = ptr[1];

		if (c == ' ') {
			ptr++;
			count--;
			continue;
		}
		switch (c) {
		case '+':
			op = 1;
			break;
		case '-':
			op = 0;
			break;
		default: return -EINVAL;
		}

		ret = set_or_clear_flag(flag, op);
		if (ret < 0)
			return ret;
		ptr += 2;
		count -= 2;
	}
	return size;
}

static struct kobj_attribute fastpath_debug_attribute =
__ATTR(debug, S_IRUGO|S_IWUSR, fastpath_debug_show, fastpath_debug_store);

static const struct attribute *fp_core_attrs[] = {
	&fastpath_debug_attribute.attr,
	NULL, /* need to NULL terminate the list of attributes */
};

#define to_attr(a) container_of(a, struct fp_attr, attr)

static ssize_t fastpath_module_show(struct kobject *kobj,
				     struct attribute *attr, char *buf)
{
	struct fastpath_module *module = to_fpmod(kobj);
	struct fp_attr *mattr = to_attr(attr);

	if (!module || !mattr || !mattr->show)
		return -EINVAL;

	return mattr->show(module, buf);
}

static ssize_t fastpath_module_store(struct kobject *kobj,
				      struct attribute *attr,
				      const char *buf, size_t count)
{
	struct fastpath_module *module = to_fpmod(kobj);
	struct fp_attr *mattr = to_attr(attr);

	if (!module || !mattr || !mattr->store)
		return -EINVAL;

	return mattr->store(module, buf, count);
}

const struct sysfs_ops fp_sysfs_ops = {
	.show = fastpath_module_show,
	.store = fastpath_module_store,
};

static int __init fastpath_init(void)
{
	int ret, i;
	struct fastpath_module *m;
	struct net *net;

	pr_debug("init start\n");

	fastpath = kzalloc(sizeof(*fastpath), GFP_KERNEL);
	if (!fastpath) {
		pr_err("No Memory\n");
		return -ENOMEM;
	}

	fastpath->version = fastpath_driver_version;
	fastpath->num_modules = ARRAY_SIZE(fp_modules_ops);
	fastpath->modules = kzalloc(fastpath->num_modules *
				sizeof(struct fastpath_module *), GFP_KERNEL);
	if (!fastpath->modules) {
		pr_err("No Memory\n");
		ret = -ENOMEM;
		goto free_fastpath;
	}

	fastpath->kobj = kobject_create_and_add("fastpath", kernel_kobj);
	if (!fastpath->kobj) {
		pr_err("kobject_create_and_add failed\n");
		ret = -ENOMEM;
		goto free_modules;
	}

	ret = sysfs_create_files(fastpath->kobj, fp_core_attrs);
	if (ret < 0) {
		pr_err("sysfs_create_files failed (%d)\n", ret);
		goto kobj_put;
	}

	for_each_net(net)
		nf_ct_netns_get(net, NFPROTO_BRIDGE);

	/* load fastpath modules */
	for (i = 0; i < fastpath->num_modules; i++) {
		struct fastpath_module_ops *ops = fp_modules_ops[i];

		m = kzalloc(sizeof(*m), GFP_KERNEL);
		if (!m) {
			pr_err("no memeory\n");
			ret = -ENOMEM;
			goto unload_modules;
		}

		m->ops = ops;
		m->fastpath = fastpath;
		m->idx = i;

		ret = m->ops->probe(m);
		if (ret < 0) {
			pr_err("Error loading [%d]:%s (%d)\n", i, m->name, ret);
			kfree(m);
			goto unload_modules;
		}

		/* module loaded */
		fastpath->modules[i] = m;
	}

	pr_debug("init done\n");
	return 0;

unload_modules:
	for (i = i - 1; i >= 0 ; i--) {
		m = fastpath->modules[i];
		m->ops->remove(m);
	}
	sysfs_remove_files(fastpath->kobj, fp_core_attrs);
kobj_put:
	kobject_put(fastpath->kobj);
free_modules:
	kfree(fastpath->modules);
free_fastpath:
	kfree(fastpath);
	return ret;
}


static void __exit fastpath_exit(void)
{
	struct fastpath_module *m;
	int i;
	struct net *net;

	pr_debug("exit start\n");

	/* unload fastpath modules */
	for (i = 0; i < fastpath->num_modules; i++) {
		m = fastpath->modules[fastpath->num_modules - i - 1];
		m->ops->remove(m);
	}
	sysfs_remove_files(fastpath->kobj, fp_core_attrs);
	kobject_put(fastpath->kobj);
	kfree(fastpath->modules);
	kfree(fastpath);

	for_each_net(net)
		nf_ct_netns_put(net, NFPROTO_BRIDGE);

	pr_debug("exit done\n");
}

module_param(debug_level, uint, 0644);

module_init(fastpath_init)
module_exit(fastpath_exit)

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Tomer Eliyahu tomere@marvell.com");
MODULE_AUTHOR("Ram Marzin ramm@marvell.com");
MODULE_AUTHOR("Yair Weiss yairw@marvell.com");
MODULE_DESCRIPTION("Marvell Fastpath");
MODULE_INFO(Version, __MODULE_STRING(FP_DRV_VERSION));
