#ifndef FP_DATABASE_H
#define FP_DATABASE_H

/*
 *	Fast path data base
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

/**--------------------------------------
 * Enum
 *--------------------------------------*/


static const char *const entry_state_names[] = {
	"initialized",
	"alive",
	"dying",
};

enum entry_state {
	ENTRY_INITIALIZED,
	ENTRY_ALIVE,
	ENTRY_DYING,
};

/**--------------------------------------
 * STRUCTURES
 *--------------------------------------*/

/* Trace buffer per entry (enabled via DBG_LVL) */
struct fpdb_trace {
	struct list_head list;
	int sz;
	unsigned int hit_counter;
	unsigned int timeout;
	struct tcphdr tcph;
	unsigned int tcp_state;
	unsigned long ct_status; /* have we seen direction in both ways? */
};

struct fpdb_debug {
	unsigned int in_route_type;
	unsigned int out_route_type;
	struct fpdb_trace trace;
};

struct fpdb_entry {
	/* Managment fields - DO NOT USE!*/
	struct hlist_node hlist;
	struct rcu_head rcu;
	atomic_t rc;
	spinlock_t lock;
	struct timer_list *guard_timer;
	struct work_struct work;

	/* Entry data*/
	unsigned long tstamp; /* data base aging out use this field */
	struct nf_conn *ct;
	enum ip_conntrack_dir dir;
	unsigned long flags;
	struct hh_cache hh;
	struct nf_conntrack_tuple in_tuple;
	struct nf_conntrack_tuple out_tuple;
	struct fp_net_device *out_dev;	/* destination interface handle */
	struct fp_net_device *in_dev;	/* source interface handle */
	enum entry_state state;
	bool block;

	/* statistics & debug */
	unsigned int bucket;
	unsigned int hit_counter;
	struct fpdb_debug debug;
#ifdef CONFIG_ASR_TOE
	unsigned int nl_flag;
	unsigned long detect_speed_jiffies;
	unsigned int detect_speed_bytes;
	unsigned int speed; /* Kbps */
#endif
};

struct nf_conn_fastpath {
	struct fpdb_entry *fpd_el[IP_CT_DIR_MAX];
};

static inline struct nf_conn_fastpath *nfct_fastpath(const struct nf_conn *ct)
{
	return nf_ct_ext_find(ct, NF_CT_EXT_FASTPATH);
}

struct hist_entry {
	unsigned int buckets;
	unsigned int entries;
};

#define HISTOGRAM_SIZE			(16)

struct fpdb_stats {
	/* HASH statistics */
	u32 size;		/* number of buckets */
	u32 lookups;		/* total lookups */
	u32 iterations;		/* iterations per lookups */
	u32 hits;		/* successfull lookups */
	u32 largest_bucket;	/* Number of entries in largest bucket */
	u32 num_occupied;	/* Number of occupied buckets */
	u32 load_factor;	/* num of hashed entries / num of buckets */

				/* HISTOGRAM */
	struct hist_entry hist[HISTOGRAM_SIZE + 1];

	/* Database statistics */
	u32 avg_lookup_latency;
	u32 max_lookup_latency;
	u32 max_entries;	/* max num of hashed entries */
};

/**--------------------------------------
 * API FUNCTIONS
 *--------------------------------------*/

void fpdb_lock_bh(void);
void fpdb_unlock_bh(void);
struct fpdb_entry *fpdb_alloc(gfp_t flags);
void fpdb_dump_entry(char *msg, struct fpdb_entry *el);
void fpdb_add(struct fpdb_entry *el);
void fpdb_replace(struct fpdb_entry *el, struct fpdb_entry *nel);
void fpdb_del(struct fpdb_entry *entry);
struct fpdb_entry *fpdb_get(struct nf_conntrack_tuple *tuple);
void fpdb_put(struct fpdb_entry *entry);
void fpdb_del_by_dev(struct net_device *dev);
void fpdb_del_by_port(unsigned int);
void fpdb_flush(void);
void fpdb_del_least_used_entry(int max_num);
void fpdb_trace(struct fpdb_entry *entry, struct tcphdr *tcph);
void fpdb_iterate(int (*iter)(struct fpdb_entry *e, void *data), void *data);
void fpdb_free(struct fpdb_entry * el);
int fpdb_del_block_entry_by_dev(struct fpdb_entry *el, void *data);

#ifdef FP_USE_SRAM_POOL_OPT
extern unsigned long sram_pool_alloc(size_t size);
extern void sram_pool_free(unsigned long addr, size_t size);
#endif

#endif /* FP_DATABASE_H */
