#!/bin/bash

# script to setup work enviroment for linux developers on open WRT
#Usage:
#  cd to <open wrt root>
#  source ./scripts/env_tools/env_setup.bash
#
# this script will setup git autocomplete. make sure phabricator client is in the PATH
# and setup PS1 to show GIT status.

function echo_blue()
{
	echo -e '\033[1;34m'"$@"'\033[0m'
}

function echo_red()
{
	echo -e '\033[1;31m'"$@"'\033[0m'
}

function echo_purple()
{
	echo -e '\033[1;35m'"$@"'\033[0m'
}

export OWRT_ROOT=`pwd -P`

function git_set_openwrt_alias()
{
	alias r='cd ${OWRT_ROOT}'
	alias l='cd ${OWRT_ROOT}/marvell/linux'
	alias t='cd ${OWRT_ROOT}/marvell/lte-telephony'
	alias w='cd ${OWRT_ROOT}/marvell/linux/drivers/marvell/sd8897'
	alias f='cd ${OWRT_ROOT}/marvell/fastpath'
	alias i='cd ${OWRT_ROOT}/marvell/ims'
	alias s='cd ${OWRT_ROOT}/marvell/services'
	alias u='cd ${OWRT_ROOT}/marvell/uboot'
	alias obm='cd ${OWRT_ROOT}/marvell/obm'
	alias out='cd ${OWRT_ROOT}/bin/'

	# scripts aliases
	alias mgit='${OWRT_ROOT}/mgit.sh'
	alias sa_lindent='${OWRT_ROOT}/scripts/env_tools/sa_lindent.sh'

}

function set_openwrt_git()
{
	ENV_TOOLS_DIR=$OWRT_ROOT/scripts/env_tools
	
	echo_blue "Environment setup script for NZ2/3 open wrt project"
	
	echo "setting PS1 with Git info"
	source ${ENV_TOOLS_DIR}/git_prompt.bash
	setps1
	echo "adding git auto compleate to shell"
	source ${ENV_TOOLS_DIR}/git_completion.bash
	
	# make sure /usr/local/bin/ to your PATH environment variable
	if ( ! (echo $PATH | grep -c -q ":/usr/local/bin") ); then
		export PATH=$PATH:/usr/local/bin
	fi
	
	echo_blue "Done !"
}

function set_p305_git()
{
	#clean previous device setup
	if [ -n "${DEVICE+set}" ]; then
		unset DEVICE
	fi

	echo_blue "Device setup for p305"
	# insert saar specific configuration here
	export DEVICE=pxa1826p305
	export BOARD=pxa1826p305
	cd ${OWRT_ROOT} && make defconfig_${DEVICE} && cd -
	echo_blue "Done !"
}

function set_spinor_git()
{
	#clean previous device setup
	if [ -n "${DEVICE+set}" ]; then
		unset DEVICE
	fi

	echo_blue "Device setup for spinor"
	# insert spinor specific configuration here
	export DEVICE=pxa1826spinor
	export BOARD=pxa1826
	cd ${OWRT_ROOT} && make defconfig_${DEVICE} && cd -
	echo_blue "Done !"
}

function set_pcie_git()
{
	#clean previous device setup
	if [ -n "${DEVICE+set}" ]; then
		unset DEVICE
	fi

	echo_blue "Device setup for pxa1826pcie"
	# insert pcie specific configuration here
	export DEVICE=pxa1826pcie
	export BOARD=pxa1826pcie
	cd ${OWRT_ROOT} && make defconfig_${DEVICE} && cd -
	echo_blue "Done !"
}

function set_dkb_git()
{
	#clean previous device setup
	if [ -n "${DEVICE+set}" ]; then
		unset DEVICE
	fi

	echo_blue "Device setup for pxa1826DKB"
	# insert DKB specific configuration here
	export DEVICE=pxa1826
	export BOARD=pxa1826
	cd ${OWRT_ROOT} && make defconfig_${DEVICE} && cd -
	echo_blue "Done !"
}

function mk()
{
	BIN_DIR=$OWRT_ROOT/bin
	DEFAULT='-j4 V=s'

	if [ -z "${DEVICE}" ]; then
		export DEVICE=pxa1826
	fi

	if [ -z "${BOARD}" ]; then
		export BOARD=pxa1826
	fi
	
	local rc
	if [ "$1" != "I_am_respawned" ];then
		# respawn myself with output redirection
		mk I_am_respawned $* 2>&1 |  mrvlfilter ${OWRT_ROOT}/buildall.log
		return
	fi
	shift 	# removing "I_am_respawned" parameter

	if [ -z "$1" ] ; then	# no input parameter
		echo_blue "what would you like to build?"
		echo " 1.  build all"
		echo " 2.  build Kernel"
		echo " 3.  build Telephony"
		echo " 4.  build uboot"
		echo " 5.  build Obm"
		echo " 6.  Clean previous builds"
		echo " 7.  build clean - Clean previous builds and build all"
		echo "=========================="
		echo "for debug mode add the flag: -j1"
		read x_ANSWER_ALL; ANSWER_ALL=( $x_ANSWER_ALL );
		ANSWER=${ANSWER_ALL[0]}
		for ((i=1; i<${#ANSWER_ALL[@]}; i++));do
			VAROPTS+="${ANSWER_ALL[$i]} "
		done
	else
		echo $1
		ANSWER=$1
		shift
		VAROPTS="$@"
	fi
	
	date
	BUILD_START_TIME=$(date)
	BUILD_START_TIME_S=$(date +%s)
	
	case $ANSWER in
	1)
		echo_blue "build NZ open WRT $OWRT_ROOT"
		rm -rf $BIN_DIR/*
		echo_blue "build $DEVICE device"
		echo MRVL_START_FILTER_ACTION_START
		cd ${OWRT_ROOT} && make package/network/services/lte-telephony/clean && make $DEFAULT $VAROPTS
		check_build_result $?
		echo MRVL_START_FILTER_ACTION_END
		;;
	2)
		echo MRVL_START_FILTER_ACTION_START
		cd ${OWRT_ROOT} && make target/linux/{compile,install} IB=y $DEFAULT $VAROPTS
		check_build_result $?
		echo MRVL_START_FILTER_ACTION_END
		;;
	3)
		echo MRVL_START_FILTER_ACTION_START
		cd ${OWRT_ROOT} && make package/network/services/lte-telephony/{clean,configure,compile} $DEFAULT $VAROPTS
		[ $? -eq 0 ] && make package/install $DEFAULT $VAROPTS
		[ $? -eq 0 ] && make target/linux/install $DEFAULT $VAROPTS
		check_build_result $?
		echo MRVL_START_FILTER_ACTION_END
		;;
	4)
		echo MRVL_START_FILTER_ACTION_START
		cd ${OWRT_ROOT} && make package/boot/uboot-mmp/{configure,compile} $DEFAULT $VAROPTS
		mv -f ${BIN_DIR}/${BOARD}/*u-boot.bin ${BIN_DIR}/${BOARD}/swd/
		check_build_result $?
		echo MRVL_START_FILTER_ACTION_END
		;;
	5)
		echo MRVL_START_FILTER_ACTION_START
		cd ${OWRT_ROOT} && make package/obm-mmp/compile $DEFAULT $VAROPTS
		mv -f ${BIN_DIR}/${BOARD}/*NonTLoader_NAND_DDR.bin ${BIN_DIR}/${BOARD}/swd/
		check_build_result $?
		echo MRVL_START_FILTER_ACTION_END
		;;
	6)
		echo MRVL_START_FILTER_ACTION_START
		echo_blue "Clean previous builds"
		cd ${OWRT_ROOT} && make distclean V=s
		./scripts/feeds update -a
		./scripts/feeds install -a 
		make defconfig_${DEVICE}
		echo MRVL_START_FILTER_ACTION_END
		;;
	7)
		echo MRVL_START_FILTER_ACTION_START
		echo_blue "Clean previous builds"
		cd ${OWRT_ROOT} && make distclean V=s
		./scripts/feeds update -a
		./scripts/feeds install -a 
		make defconfig_${DEVICE}
		echo_blue "build NZ open WRT $OWRT_ROOT"
		make $DEFAULT $VAROPTS
		check_build_result $?
		echo MRVL_START_FILTER_ACTION_END
		;;
	*)
		echo_red "This option isn't defined"
		;;
	esac		
}

function mrvlfilter()
{
	rm $1 2> /dev/null || true
	awk -v logfile=$1 '
		BEGIN{
			c_pur = "\033[1;35m"; c_red = "\033[1;31m";c_end = "\033[0m";start=0;
		}
		/MRVL_START_FILTER_ACTION_START/{wrns=0;start=1;next;};
		/MRVL_START_FILTER_ACTION_END/{start=0;
		printf("%s@ ==================== @ %s\n",c_red, c_end);
		printf("%s@ WARNINGS_COUNT %d %s\n",c_red, wrns, c_end);
		printf("%s@ ==================== @ %s\n",c_red, c_end);
		next;};

		{print $0 >> logfile;};
		start == 0 {print $0; next;};
		/[Ww]arning[ \t]*:/||/[Ee]rror[ \t]*:/||/ERROR[ \t]*:/||/WARNING[ \t]*:/{
			wrns += 1;
			printf("%s%s%s\n",c_red, $0, c_end); next;};
		/Removing unused section/{gsub(/([^[:space:]])*(linux-x86\/toolchain\/arm-eabi)([^[:space:]])*/,"");
			printf("%s%s%s\n",c_pur, $0, c_end); next;}
		{print $0;};
	'
}

function copy_to_windows()
{
	if [ "${SYNC_PATH}" == "" ]; then
		export SYNC_PATH=/nfs/pt/swdev/areas/projects/gr_platdev/${USER}/owrt_out
	fi
	echo_blue "copy open WRT build output to ${SYNC_PATH}"
	mkdir -p ${SYNC_PATH}
	rsync -L --chmod=Dgo+w,Fgo+w -avq ${OWRT_ROOT}/bin/ ${SYNC_PATH}/
}

function check_build_result() {
	local result=${1:-$?}
	if [ $result -ne 0 ] ; then
		echo_red "build $DEVICE device"
		echo_red "BUILD FAILS.\n"
	else
		echo_purple "build $DEVICE device"
		echo_purple "BUILD SUCCESSFUL"
		copy_to_windows
	fi
	BUILD_END_TIME=$(date)
	BUILD_END_TIME_S=$(date +%s)

	DIFF=$(($BUILD_END_TIME_S - $BUILD_START_TIME_S))
	H1=$((DIFF / 3600))
	M1=$((DIFF / 60 % 60))
	S1=$((DIFF % 60))

	echo ""
	echo_purple "Build Start Time: $BUILD_START_TIME"
	echo_purple "Build End Time:   $BUILD_END_TIME"

	if (($H1 > 0)); then
		printf "Build Duration:   %d seconds = %02dh%02dm%02ds\n" $DIFF $H1 $M1 $S1
	else
		printf "Build Duration:   %d seconds = %02dm%02ds\n" $DIFF $M1 $S1
	fi
	echo_purple "log found in ${OWRT_ROOT}/buildall.log"
}

function arc_land()
{
	echo_blue "arc_land"
	branch="$(git branch | sed -n -e 's/^\* \(.*\)/\1/p')"
	[ "$branch" == "master" ] && echo_red on branch master, abort. && return 1

	# check if master branch exists
	git show-ref --verify --quiet refs/heads/master
	if [ $? -eq 0 ]; then
		echo_red In order to land, need to create master branch. Delete current master branch?
		select yn in "Yes" "No"; do
			case $yn in
				Yes ) git branch -D master; break;;
				No ) echo_red abort.; return 1;;
			esac
		done
	fi
	
	# Create a temporary master branch for arc land command
	git branch --track master origin/master
	[ ! $? -eq 0 ] && echo_red failed to create master branch, abort. && return 1

	if [ "$branch" == "(no branch)" ]; then
		tmp_branch=$(date +tmp_%N)
		git checkout -b $tmp_branch
	fi

	arc land
	result=$?

	repo sync -d .
	git branch -D master
	[ ! -z "$tmp_branch" ] && git branch -D $tmp_branch

	if [ $result -eq 0 ]; then
		echo_blue Done.
	else
		echo_red Failed.
	fi

	return $result
}
