#!/bin/sh
set -e

if [ $# -eq 0 -o $# -eq 1 -a "$1" = help ]; then
	echo "Usage: $0 <add | del | list | help> [arguments...]"
	echo "   run $0 help <command>  to show help for <command>"
	exit 0
fi

HELP=false
if [ "$1" = help ]; then
	HELP=true
	shift
fi

die () {
	if [ $# -gt 0 ]; then
		echo "$@"
	fi
	exit 1
}

row() {
	FMT="%s"
	for c in `seq 2 $#`; do
		FMT="%-15s $FMT"
	done
	printf "$FMT\n" "$@"
}

MMP_DIR=target/linux/mmp

case "$1" in
	add)
		RET=0
		if [ $# -lt 5 ]; then
			echo "Insufficient arguments"
			RET=1
			HELP=true
		fi
		if $HELP; then
			echo "$0 add <template_profile> <chip> <name> <code> [kernel_config_suffix]"
			echo "   Add a new profile named <chip>p<code> based on template config file <template_profile> and specified params"
			echo "   Model name would be <name><code>, BLF_PREFIX and FOTA_PREFIX would be p<code>"
			echo "Sample usage: $0 add asr1803p401 asr1802s NEZAS 201"
			exit $RET
		fi
		TMPL="config/defconfig_$2"
		[ -s "$TMPL" ] || die "Invalid template file $TMPL"
		TMPL_CHIP="$(sed 's/^CONFIG_TARGET_mmp_\([^_]*\)=y$/\1/;t;d' "$TMPL")"
		[ -n "$TMPL_CHIP" ] || die "No chip found in $TMPL"
		TMPL_MODL="$(sed "s/^CONFIG_TARGET_mmp_${TMPL_CHIP}_\\([^_]*\)=y$/\\1/;t;d" "$TMPL")"
		[ -n "$TMPL_MODL" ] || die "No model found in $TMPL"
		TMPL_NAME="$(echo "$TMPL_MODL" | sed 's/[0-9].*$//')"
		TMPL_NAME_L="$(echo "$TMPL_NAME" | tr A-Z a-z)"
		TMPL_CODE="$(echo "$TMPL_MODL" | sed 's/^[^0-9]*//')"
		TMPL_CONF=CONFIG_TARGET_mmp_${TMPL_CHIP}_${TMPL_MODL}
		TMPL_KCFG="$(echo $MMP_DIR/$TMPL_CHIP/config-* | sed 's/ .*//')"
		[ -s "$TMPL_KCFG" ] || die "No kernel config found for $TMPL_CHIP"
		TMPL_OEMF=$MMP_DIR/$TMPL_CHIP/oem_fs
		[ -d "$TMPL_OEMF" ] || die "No oem_fs found for $TMPL_CHIP"
		CHIP=$3
		[ -n "$CHIP" ] || die "Empty chip"
		NAME=$4
		[ -n "$NAME" ] || die "Empty name"
		NAME_L="$(echo "$NAME" | tr A-Z a-z)"
		CODE=$5
		[ -n "$CODE" ] || die "Empty code"
		CONF=CONFIG_TARGET_mmp_${CHIP}_$NAME$CODE
		KCONF=$6
		TGTF="config/defconfig_${CHIP}p$CODE"
		[ ! -e "$TGTF" ] || die "$TGTF already exists"
		KCFG=$MMP_DIR/$CHIP/`basename $TMPL_KCFG`
		OEMF=$MMP_DIR/$CHIP/oem_fs
		TGTM=$MMP_DIR/$CHIP/target.mk
		CPFX="$(echo $CHIP | sed 's/[0-9].*//')"
		CSFX="$(echo $CHIP | sed 's/^[^0-9]*//')"
		[ "$CPFX$CSFX" = "$CHIP" ] || die "Unexpected $CHIP"
		CPFX_U="$(echo $CPFX | tr a-z A-Z)"
		TMPL_CPFX="$(echo $TMPL_CHIP | sed 's/[0-9].*//')"
		TMPL_CSFX="$(echo $TMPL_CHIP | sed 's/^[^0-9]*//')"
		[ "$TMPL_CPFX$TMPL_CSFX" = "$TMPL_CHIP" ] || die "Unexpected $TMPL_CHIP"
		TMPL_CPFX_U="$(echo $TMPL_CPFX | tr a-z A-Z)"
		set -x
		set -e
		mkdir -p $MMP_DIR/$CHIP $MMP_DIR/$CHIP/profiles
		[ -s $KCFG ] || cp $TMPL_KCFG $KCFG
		[ -d $OEMF ] || cp -a $TMPL_OEMF $OEMF
		if [ ! -s $TGTM ] || [ `grep -Fc $CHIP $TGTM` -ne 3 ]; then
			cp $MMP_DIR/$TMPL_CHIP/target.mk $TGTM
			sed -i "s/$TMPL_CHIP/$CHIP/g;s/$TMPL_CPFX_U/$CPFX_U/g;s/$TMPL_CSFX/$CSFX/g" $TGTM
		fi
		if ! grep -Fq 'define Model' $MMP_DIR/$CHIP/profiles/*.mk 2>/dev/null; then
			PROF=$MMP_DIR/$CHIP/profiles/$CPFX.mk
			cp $MMP_DIR/$TMPL_CHIP/profiles/*.mk $PROF
			sed -i '/^define Profile/,/\$(eval \$(call Model,/d' $PROF
		else
			PROF=`grep -Fl 'define Model' $MMP_DIR/$CHIP/profiles/*.mk | head -1`
		fi
		sed -i ":n;/^$/{N;s/^\\n$//;t n;};/^\n*define Profile\/$NAME$CODE$/,/^endef/d;/,$NAME$CODE[,)]/d" $PROF
		cat >> $PROF << EOF

define Profile/$NAME$CODE
  NAME:=$CPFX_U `echo $NAME | awk '{printf("%c%s\n", substr($0,1,1), tolower(substr($0,2)))}'` Profile $CODE
endef
\$(eval \$(call Profile,$NAME$CODE))
\$(eval \$(call Model,$NAME$CODE,${CHIP}p$CODE))
EOF
		[ -d $MMP_DIR/$TMPL_CHIP/$TMPL_MODL ] && cp -a $MMP_DIR/$TMPL_CHIP/$TMPL_MODL $MMP_DIR/$CHIP/$NAME$CODE
		SUBT="$(sed 's/^\s*\<SUBTARGETS\>[^=]*=\s*//;t n;d;:n;s/ \+$//;s/ \+/\n/g' $MMP_DIR/Makefile)"
		if ! echo "$SUBT" | grep -Fx $CHIP; then
			sed -i "/^\\s*\\<SUBTARGETS\\>[^=]*=/{s/ *$/ $CHIP/}" $MMP_DIR/Makefile
		fi
		TMPL_RULE="$(awk "/^ifeq \\(\\\$\\($TMPL_CONF\\),y\\)/{++x;print;next};x{print;if(\$1~/^if/)++x;else if(\$1==\"endif\")--x}" rules.mk)"
		if [ -n "$TMPL_RULE" ]; then
			awk "/^ifeq \\(\\\$\\(CONFIG_TARGET_mmp_/{++x;if(/$CONF\\),y\\)/)y=NR};!y{print};x{if(\$1~/^if/&&!/CONFIG_TARGET_mmp_/)++x;else if(\$1==\"endif\"){--x;if(!x){getline;if(/^ *$/)n=NR;else n=NR-1;if(y){m=n-y+1;y=0;if(n<NR)print}else print}}};END{print n-m >\"rules.mk.nr\"}" rules.mk > rules.mk.tmp
			#diff -u rules.mk rules.mk.tmp || true
			LNUM="$(cat rules.mk.nr)"
			head -$LNUM rules.mk.tmp > rules.mk
			echo "$TMPL_RULE" | sed "s/$TMPL_CHIP/$CHIP/g;s/$TMPL_MODL/$NAME$CODE/g;s/$TMPL_CODE/$CODE/g" >> rules.mk
			echo >> rules.mk
			tail -n +`expr $LNUM + 1` rules.mk.tmp >> rules.mk
			#diff -u rules.mk.tmp rules.mk || true
			rm rules.mk.tmp rules.mk.nr
		else
			echo "Warning: rules.mk not updated"
		fi
		TMPL_IMG_MKFL="$(awk "/^ifeq \\(\\\$\\(CONFIG_TARGET_mmp_$TMPL_CHIP\\),y\\)/{++x;print;next};x{print;if(\$1~/^if/)++x;else if(\$1==\"endif\")--x}" $MMP_DIR/image/Makefile)"
		if [ -n "$TMPL_IMG_MKFL" ]; then
			awk "/^ifeq \\(\\\$\\(CONFIG_TARGET_mmp_/{++x;if(/CONFIG_TARGET_mmp_$CHIP\\),y\\)/)y=NR};!y{print};x{if(\$1~/^if/&&!/CONFIG_TARGET_mmp_/)++x;else if(\$1==\"endif\"){--x;if(!x){getline;if(/^ *$/)n=NR;else n=NR-1;if(y){m=n-y+1;y=0;if(n<NR)print}else print}}};END{print n-m >\"image.mk.nr\"}" $MMP_DIR/image/Makefile > image.mk.tmp
			#diff -u $MMP_DIR/image/Makefile image.mk.tmp || true
			LNUM="$(cat image.mk.nr)"
			head -$LNUM image.mk.tmp > $MMP_DIR/image/Makefile
			echo "$TMPL_IMG_MKFL" | sed "s/$TMPL_CHIP/$CHIP/g;s/$TMPL_MODL/$NAME$CODE/g;s/$TMPL_NAME/$NAME/g;s/$TMPL_CODE/$CODE/g" >> $MMP_DIR/image/Makefile
			echo >> $MMP_DIR/image/Makefile
			tail -n +`expr $LNUM + 1` image.mk.tmp >> $MMP_DIR/image/Makefile
			#diff -u image.mk.tmp $MMP_DIR/image/Makefile || true
			rm image.mk.tmp image.mk.nr
		else
			echo "Warning: image/Makefile not updated"
		fi
		cp $TMPL $TGTF
		if [ $TMPL_CHIP = $CHIP ]; then
			sed -i "s/$NAME$CODE/$TMPL_MODL/g;t;s/$TMPL_MODL/$NAME$CODE/g;s/p$TMPL_CODE/p$CODE/g" $TGTF
		else
			sed -i "s/$CHIP/$TMPL_CHIP/g;t;s/$TMPL_CHIP/$CHIP/g;s/$TMPL_MODL/$NAME$CODE/g;s/$TMPL_NAME_L/$NAME_L/g;s/p$TMPL_CODE/p$CODE/g" $TGTF
		fi
		#diff -u $TMPL $TGTF || true
		make prepare-tmpinfo
		./scripts/config/conf --defconfig=$TGTF -w $TGTF.new Config.in
		#diff -u $TGTF $TGTF.new || true
		mv $TGTF.new $TGTF
		set +x
		sh $0 list | grep -F ${CHIP}p$CODE
		;;
	list)
		if $HELP; then
			echo "$0 list"
			echo "   show the list of defined profiles, as well as the"
			echo "   list of unused target configurations in uboot /obm"
			exit 0
		fi
		[ -d "$MMP_DIR" ] || die "Please run under openwrt directory"
		PROFILES=
		MODELS=
		for c in $MMP_DIR/*/profiles/*.mk; do
			[ -s "$c" ] || die "Invalid file $c"
			CHIP_DIR="$(echo "$c" | sed "s=^$MMP_DIR/==;s=/profiles.*==")"
			echo "$CHIP_DIR" | grep -Fvq / || die "Slash in $CHIP_DIR"
			for d in `sed "s/^.*(call Profile,//;t n;d;:n;s/).*//" "$c"`; do
				MODELS="$MODELS $d"
				p="$(sed "s/^.*(call Model,$d,//;t n;d;:n;s/).*//" "$c")"
				[ -n "$p" ] || die "Missing model $d"
				PROFILES="$PROFILES $p"
				eval "PROFILE_$d=$p"
				eval "MODEL_$p=$d"
				eval "CHIP_$p=$CHIP_DIR"
			done
		done
		UBOOT_CONF_LIST="$(sed -n '/^define uboot\//,/^endef$/{s/^ *DEPS:=@//;t n;d;:n;s/[()]//g;s/||/\n/g;s/ //g;p}' package/boot/uboot-mmp/Makefile | sed 's/^/CONFIG_/')"
		OBM_CONF_LIST="$(sed -n '/^define obm\//,/^endef$/{s/^ *DEPS:=@//;t n;d;:n;s/[()]//g;s/||/\n/g;s/ //g;p}' package/boot/obm-mmp/Makefile | sed 's/^/CONFIG_/')"
		for p in $PROFILES; do
			eval "CONF=CONFIG_TARGET_mmp_\${CHIP_$p}_\${MODEL_$p}"
			CFG="$(grep -Fxl $CONF=y config/* | sed 's=^.*/defconfig_==')"
			if [ -z "$CFG" ]; then
				echo "Warning: profile $p has no config"
			elif [ "$CFG" != $p ]; then
				CFG="$(echo -n "$CFG" | tr '\n' ',')"
				echo "Warning: profile $p has config: $CFG"
				echo "$CFG" | grep -q "^$p," || echo "         without $p"
			fi
			eval "CONFIG_$p=\"$CFG\""
			if [ -z "$CFG" ]; then
				VAL=XXX
			else
				if [ "$CFG" != $p ]; then
					CFG="$(echo "$CFG" | sed 's/,.*//')"
				fi
				if eval grep -Fxq CONFIG_TARGET_mmp_\${CHIP_$p}=y config/defconfig_$CFG; then
					VAL=OO
				else
					VAL=XO
				fi
				if eval grep -Fxq CONFIG_TARGET_ARCH_PACKAGES=\\\"\${CHIP_$p}\\\" config/defconfig_$CFG; then
					VAL=${VAL}O
				else
					VAL=${VAL}X
				fi
			fi
			if true; then
				if grep -Fxq "ifeq (\$($CONF),y)" rules.mk; then
					VAL=${VAL}O
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* ARCH_PROFILE.*= *//;t n;d;:n;p}" rules.mk)"
					if [ "$c" = "$p" ]; then
						VAL=${VAL}O
					else
						VAL=${VAL}X
					fi
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* BLF_PREFIX.*= *//;t n;d;:n;p}" rules.mk)"
					eval "BLF_$p=\"$c\""
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* FOTA_PREFIX.*= *//;t n;d;:n;p}" rules.mk)"
					eval "FOTA_$p=\"$c\""
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* KERNEL_CONFIG_FILE_NAME_APPENDIX.*= *//;t n;d;:n;p}" rules.mk)"
					eval "KCONF_$p=\"$c\""
				else
					VAL=${VAL}XX
				fi
			fi
			if echo "$UBOOT_CONF_LIST" | grep -Fxq "$CONF"; then
				VAL=${VAL}O
			else
				VAL=${VAL}X
			fi
			if echo "$OBM_CONF_LIST" | grep -Fxq "$CONF"; then
				VAL=${VAL}O
			else
				VAL=${VAL}X
			fi
			eval "VALID_$p=$VAL"
		done
		row Profile Chip Model BLF/FOTA/Kconf Config "Validity(O/X)"
		for p in $PROFILES; do
			eval row \$p \${CHIP_$p} \${MODEL_$p} \"\${BLF_$p}/\${FOTA_$p}/\${KCONF_$p}\" \"\${CONFIG_$p}\" \${VALID_$p}
		done
		for c in `echo "$UBOOT_CONF_LIST
$OBM_CONF_LIST" | sort | uniq | sed 's/^\s*CONFIG_TARGET_mmp_//'`; do
			d="$(echo "$c" | sed 's/^[^_]*_//')"
			if [ -n "$(eval echo "\$PROFILE_$d")" ]; then
				continue
			fi
			CHIP_DIR="$(echo "$c" | sed 's/_.*//')"
			CONF="CONFIG_TARGET_mmp_$c"
			CFG="$(grep -Fxl $CONF=y config/* | sed 's=^.*/defconfig_==')"
			if [ -z "$CFG" ]; then
				VAL=XXX
			else
				CFG="$(echo "$CFG" | head -1)"
				if grep -Fxq CONFIG_TARGET_mmp_$CHIP_DIR=y config/defconfig_$CFG; then
					VAL=OO
				else
					VAL=XO
				fi
				if grep -Fxq "CONFIG_TARGET_ARCH_PACKAGES=\"$CHIP_DIR\"" config/defconfig_$CFG; then
					VAL=${VAL}O
				else
					VAL=${VAL}X
				fi
			fi
			if true; then
				if grep -Fxq "ifeq (\$($CONF),y)" rules.mk; then
					VAL=${VAL}O
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* ARCH_PROFILE.*= *//;t n;d;:n;p}" rules.mk)"
					if [ "$c" = "$CFG" ]; then
						VAL=${VAL}O
					else
						VAL=${VAL}X
					fi
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* BLF_PREFIX.*= *//;t n;d;:n;p}" rules.mk)"
					BLF="$c"
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* FOTA_PREFIX.*= *//;t n;d;:n;p}" rules.mk)"
					FOTA="$c"
					c="$(sed -n "/^ifeq (\\\$($CONF),y)$/,/^endif$/{s/.* KERNEL_CONFIG_FILE_NAME_APPENDIX.*= *//;t n;d;:n;p}" rules.mk)"
					KCONF="$c"
				else
					VAL=${VAL}XX
					BLF=
					FOTA=
					KCONF=
				fi
			fi
			if echo "$UBOOT_CONF_LIST" | grep -Fxq "$CONF"; then
				VAL=${VAL}O
			else
				VAL=${VAL}X
			fi
			if echo "$OBM_CONF_LIST" | grep -Fxq "$CONF"; then
				VAL=${VAL}O
			else
				VAL=${VAL}X
			fi
			row "" "$CHIP_DIR" "$d" "$BLF/$FOTA/$KCONF" "$CFG" "$VAL"
		done
		;;

	*)
		die "Unknown command $1"
		;;
esac
