#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <pthread.h>
#include <stdarg.h>
#include <unistd.h>
#include <libubox/ustream.h>
#include <libubus.h>

#include "gnss_info.h"
#include "mbtk_log.h"

//static struct blob_buf b;
static struct blob_buf gps_blob;

const struct blobmsg_policy gnss_init_policy[] ={
	[0] = {
		.name = "gnss_init_param",
		.type = BLOBMSG_TYPE_INT32,
	},
};


const struct blobmsg_policy get_agps_policy[] ={
	[0] = {
		.name = "server_name",
		.type = BLOBMSG_TYPE_STRING,
	},
	[1] = {
		.name = "alam_flag",
		.type = BLOBMSG_TYPE_INT32,
	},
};

const struct blobmsg_policy gnss_sleep_policy[] ={
    [0] = {
        .name = "gnss_sleep_param",
        .type = BLOBMSG_TYPE_INT32,
    },
};

const struct blobmsg_policy gnss_setting_policy[] ={
    [0] = {
        .name = "gnss_setting_param",
        .type = BLOBMSG_TYPE_STRING,
    },
};

static int gps_ubus_send_reply(struct ubus_context *ctx, struct ubus_request_data *req, int ret)
{
    blob_buf_init(&gps_blob, 0);
    blobmsg_add_u32(&gps_blob, "event", ret);
    ubus_send_reply(ctx, req, gps_blob.head);
    return 0;
}

static int gps_ubus_string_reply(struct ubus_context *ctx, struct ubus_request_data *req, char *str)
{
    blob_buf_init(&gps_blob, 0);
    blobmsg_add_string(&gps_blob, "gps_state_resp", str);
    ubus_send_reply(ctx, req, gps_blob.head);
    return 0;
}

static int ubus_gnss_init(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg)
{
    UNUSED(obj);
    UNUSED(method);
    struct blob_attr *tb[1];
    struct blob_attr *cur;
    int init_mode = 0, err = 0;
    int status = 0;
    gnss_err_enum gnss_result = GNSS_ERR_OK;

    /*parsing blob to be accessed easily with tb array - parse "1" argument*/
    err = blobmsg_parse(gnss_init_policy, 1, tb, blob_data(msg), blob_len(msg));
    if (err < 0)
    {
        LOGE("blobmsg_parse fail");
        return -1;
    }
    cur = tb[0];
    if (!cur) {
        LOGE("missing parameter");
        return -2;
    }

    init_mode = blobmsg_get_u32(cur);
    LOGD("init_mode=%d", init_mode);
    if(init_mode == 0) { // Close gnss.
        gnss_result = gnss_deinit();
    } else {
        if(((GNSS_PRINT_PORT_UART1 | GNSS_PRINT_PORT_USB_NMEA | GNSS_PRINT_PORT_USB_AT | GNSS_PRINT_PORT_TTY_AT) & init_mode) == init_mode) {
            gnss_result = gnss_init(init_mode);
        } else { // ARG error, no print nmea.
            gnss_result = GNSS_ERR_UNSUPPORT;
        }
    }

    LOGD("ubus_gnss_init() ret=%d", gnss_result);
    gps_ubus_send_reply(ctx, req, (int)gnss_result);

    return 0;
}

static int ubus_gnss_deinit(struct ubus_context *ctx, struct ubus_object *obj,
            struct ubus_request_data *req, const char *method,
            struct blob_attr *msg)
{
    UNUSED(ctx);
    UNUSED(obj);
    UNUSED(req);
    UNUSED(method);
    UNUSED(msg);

    gnss_err_enum gnss_result = GNSS_ERR_OK;

    gnss_result = gnss_deinit();
    gps_ubus_send_reply(ctx, req, (int)gnss_result);

    return 0;
}

static int ubus_gnss_get_agps(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg)
{
    UNUSED(obj);
    UNUSED(method);
    struct blob_attr *tb[ARRAY_SIZE(get_agps_policy)];
    struct blob_attr *cur;
    char *server_name = NULL;
    int err = 0, alm_flag = 0;
    gnss_err_enum gnss_result = GNSS_ERR_OK;

    err = blobmsg_parse(get_agps_policy, ARRAY_SIZE(get_agps_policy), tb, blob_data(msg), blob_len(msg));
    if (err < 0)
    {
        LOGE("blobmsg_parse table fail");
        return -1;
    }

    cur = tb[0];
    if (cur)
        server_name = blobmsg_get_string(cur);
    else
        LOGE("missing parameter1");

    cur = tb[1];
    if (cur)
        alm_flag = blobmsg_get_u32(cur);
    else
        LOGE("missing parameter2");

    LOGD("server_name=%s, alm_flag=%d", server_name, alm_flag);

    gnss_result = GNSS_ERR_UNSUPPORT;
    gps_ubus_send_reply(ctx, req, (int)gnss_result);

    return 0;
}

static int ubus_gnss_set_agps(struct ubus_context *ctx, struct ubus_object *obj,
            struct ubus_request_data *req, const char *method,
            struct blob_attr *msg)
{
    UNUSED(ctx);
    UNUSED(obj);
    UNUSED(req);
    UNUSED(method);
    UNUSED(msg);

    gnss_err_enum gnss_result = GNSS_ERR_OK;


    gnss_result = GNSS_ERR_UNSUPPORT;
    gps_ubus_send_reply(ctx, req, gnss_result);

    return 0;
}

static int ubus_gnss_sleep(struct ubus_context *ctx, struct ubus_object *obj,
              struct ubus_request_data *req, const char *method,
              struct blob_attr *msg)
{
    UNUSED(ctx);
    UNUSED(obj);
    UNUSED(req);
    UNUSED(method);
    struct blob_attr *tb[1];
    struct blob_attr *cur;
    int workmode = 0, err = 0;
    int status = 0;
    gnss_err_enum gnss_result = GNSS_ERR_OK;

    /*parsing blob to be accessed easily with tb array - parse "1" argument*/
    err = blobmsg_parse(gnss_sleep_policy, 1, tb, blob_data(msg), blob_len(msg));
    if (err < 0)
    {
        LOGE("blobmsg_parse fail");
        return -1;
    }
    cur = tb[0];
    if (!cur) {
        LOGE("missing parameter");
        return -2;
    }

    workmode = blobmsg_get_u32(cur);
    LOGD("workMode=%d", workmode);
    /* Goto Sleeping.....*/

    LOGD("ret=%d", status);




    gnss_result = GNSS_ERR_UNSUPPORT;

    gps_ubus_send_reply(ctx, req, gnss_result);

    return 0;
}

static int ubus_gnss_setting(struct ubus_context *ctx, struct ubus_object *obj,
            struct ubus_request_data *req, const char *method,
            struct blob_attr *msg)
{
    UNUSED(ctx);
    UNUSED(obj);
    UNUSED(req);
    UNUSED(method);
    struct blob_attr *tb[1];
    struct blob_attr *cur;
    int err = 0;
    char *gpsCfg = NULL;
    int status = 0;
    gnss_err_enum gnss_result = GNSS_ERR_OK;

    /*parsing blob to be accessed easily with tb array - parse "1" argument*/
    err = blobmsg_parse(gnss_setting_policy, 1, tb, blob_data(msg), blob_len(msg));
    if (err < 0)
    {
      LOGE("blobmsg_parse fail");
      return -1;
    }
    cur = tb[0];
    if (!cur) {
      LOGE("missing parameter");
      return -2;
    }

    gpsCfg = blobmsg_get_string(cur);
    LOGD("gpsCfg=%s", gpsCfg);

    char rsp[1024];
    gnss_result = gnss_set(gpsCfg, strlen(gpsCfg), rsp, 1024);

    LOGD("gnss_result = %d", gnss_result);

    gps_ubus_send_reply(ctx, req, (int)gnss_result);

    return 0;
}

#define ASR_GNSS_STATUS_LEN  128
static int ubus_gnss_get_state(struct ubus_context *ctx, struct ubus_object *obj,
            struct ubus_request_data *req, const char *method,
            struct blob_attr *msg)
{
    UNUSED(obj);
    UNUSED(req);
    UNUSED(method);
    UNUSED(msg);

    char    tmpBuf[ASR_GNSS_STATUS_LEN]  = {0};
    int              i   = 0;
    int              len = 0;
    int              ret = 0;
    int              num = 0;

    //ret = asr_gnss_get_gps_info(&param);
    if (0 == ret) {
        len = snprintf(tmpBuf, sizeof(tmpBuf), "%d, %d, %d, %d, %d;", 0, 0, 0, 0, 0);
        num = 6;
        for(i=0; i<num; i++)  {
            len += sprintf(&tmpBuf[len], " %d, %d;", 1, 2);

            if(len > ASR_GNSS_STATUS_LEN)
                break;
        }

        LOGD("[%d]tmpBuf=%s", len, tmpBuf);
        gps_ubus_string_reply(ctx, req, tmpBuf);
    }

    return 0;
}

static const struct ubus_method gps_ubus_methods[] = {
    UBUS_METHOD("gnss_init", ubus_gnss_init, gnss_init_policy),
    UBUS_METHOD_NOARG("gnss_deinit", ubus_gnss_deinit),
    UBUS_METHOD("gnss_get_agps", ubus_gnss_get_agps, get_agps_policy),
    UBUS_METHOD_NOARG("gnss_set_agps", ubus_gnss_set_agps),
    UBUS_METHOD("gnss_sleep", ubus_gnss_sleep, gnss_sleep_policy),
    UBUS_METHOD("gnss_setting", ubus_gnss_setting, gnss_setting_policy),
    UBUS_METHOD_NOARG("gnss_get_state", ubus_gnss_get_state),
};

static struct ubus_object_type gps_object_type =
    UBUS_OBJECT_TYPE("mbtk_gnss", gps_ubus_methods);

static struct ubus_object gps_ubus_obj = {
    .name = "mbtk_gnss",
    .type = &gps_object_type,
    .methods = gps_ubus_methods,
    .n_methods = ARRAY_SIZE(gps_ubus_methods),
};

int gnss_ubus_exit(struct ubus_context *ctx)
{
	if(!ctx) {
        return -1;
	}

	ubus_remove_object(ctx, &gps_ubus_obj);
	ubus_free(ctx);
	uloop_done();

	LOGD("ubus exit done");
	return 0;
}

struct ubus_context *gnss_ubus_init(void)
{
	struct ubus_context *ctx;

	uloop_init();

	ctx = ubus_connect(NULL);
	if (!ctx) {
		LOGE("Failed to connect to ubus");
		return NULL;
	}

	ubus_add_uloop(ctx);
	if (ubus_add_object(ctx, &gps_ubus_obj)) {
		LOGE("Failed to add server");
		ubus_free(ctx);
		uloop_done();
		return NULL;
	}

	LOGD("gps ubus init done!");

	return ctx;
}

