/*
*    ril_data_call.c
*
*  Implementation of APN and Data Call related functions.
*
*  The general structure is as follows:
*    API --- mbtk_rild --- atcmdsrv --- modem
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN        WHO       WHAT,WHERE,WHY
--------    --------    -------------------------------------------------------
2024/10/10    LiuBin     Initial version
2024/11/13    LiuBin     Add auto data call while network changed.
2024/12/31    LiuBin     Fix apn change for same name.
                         Disable deactivate default APN.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pthread.h>
#include <sys/epoll.h>
#include <fcntl.h>
#include <signal.h>
#include <cutils/properties.h>

#include "mbtk_type.h"
#include "mbtk_ril.h"
#include "atchannel.h"
#include "at_tok.h"
#include "mbtk_utils.h"
#include "ril_info.h"
#include "mbtk_ifc.h"
#include "mbtk_str.h"
#include "mbtk_file.h"

ril_cgact_wait_t cgact_wait;
static ril_data_call_info_t info_list[MBTK_APN_CID_MAX];
int ril_cid_start = MBTK_APN_CID_MIN;


static int req_apn_get(mbtk_sim_type_enum sim_id, ATPortType_enum port, bool get_def_cid, mbtk_apn_info_array_t *apns, int *cme_err);
static int req_apn_set(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_apn_info_t *apn, int *cme_err);
static void apn_prop_get(mbtk_sim_type_enum sim_id, mbtk_apn_info_array_t *apns);
void ril_rsp_pack_send(mbtk_sim_type_enum sim_id, ATPortType_enum port, int fd, int ril_id, int msg_index, const void* data, int data_len);
static int req_data_call_start(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_ril_cid_enum cid, int *cme_err);
static int req_data_call_state_get(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_ril_cid_enum cid, mbtk_ip_info_t *ip_info, int *cme_err);
void ril_state_change(ril_msg_id_enum msg_id, const void *data, int data_len);
bool asr_auto_data_call_enable();
ATPortId_enum portType_2_portId(mbtk_sim_type_enum sim_id, ATPortType_enum port);

/*
IPv4 : 10.255.74.26
IPv6 : 254.128.0.0.0.0.0.0.0.1.0.2.144.5.212.239
*/
bool is_ipv4(const char *ip)
{
    const char *ptr = ip;
    int count = 0;
    while(*ptr) {
        if(*ptr == '.')
            count++;
        ptr++;
    }

    if(count == 3) {
        return true;
    } else {
        return false;
    }
}

static int dns_cid_get()
{
    int cid = MBTK_RIL_CID_DEF;
    char def_cid[10] = {0};
    sprintf(def_cid, "%d", MBTK_RIL_CID_DEF);   // Default route and DNS is CID 1.
    char buffer[20] = {0};
    if(property_get(MBTK_DEF_DNS_CID, buffer, def_cid) > 0 && !str_empty(buffer)) {
        cid = atoi(buffer);
    }

    return cid;
}

static void net_if_as_def_route(mbtk_ril_cid_enum cid)
{
    if(cid != MBTK_RIL_CID_NUL) {
        char buf[100] = {0};

        // Delete all default route in the first.
        // route del default

        // Add default route.
        memset(buf, 0, sizeof(buf));
        sprintf(buf, "route add default dev ccinet%d", cid - 1);
        mbtk_system(buf);

        LOGD("Add CID %d to default route.", cid);
    }
}

static int net_if_as_dns(mbtk_ril_cid_enum cid, mbtk_ipv4_info_t *ipv4, mbtk_ipv6_info_t *ipv6)
{
    char buf[1024] = {0};
    char dns[128] = {0};
    int offset = 0;
    int fd = -1;

    memset(buf, 0x0, 1024);
    memset(dns, 0x0, 128);
    offset = sprintf(buf, "search lan\n");
    if(ipv4->valid)
    {
        if(inet_ntop(AF_INET, &(ipv4->PrimaryDNS), dns, 32) == NULL) {
            LOGE("PrimaryDNS error.");
        } else {
            LOGD("PrimaryDNS : %s", dns);
        }
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
        memset(dns, 0x0, 128);
        if(inet_ntop(AF_INET, &(ipv4->SecondaryDNS), dns, 32) == NULL) {
            LOGE("SecondaryDNS error.");
        } else {
            LOGD("SecondaryDNS : %s", dns);
        }
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
    }
    if(ipv6->valid)
    {
        memset(dns, 0x0, 128);
		if(ipv6_2_str(&(ipv6->PrimaryDNS), dns))
        {
			LOGE("PrimaryDNS error.");
		} else {
			LOGD("PrimaryDNS : %s", dns);
		}
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
        memset(dns, 0x0, 128);
		if(ipv6_2_str(&(ipv6->SecondaryDNS), dns))
        {
			LOGE("SecondaryDNS error.");
		} else {
			LOGD("SecondaryDNS : %s", dns);
		}
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
    }

    if(offset > 0)
    {
        int dns_cid = dns_cid_get();
        char *dns_file_path = NULL;

        // ASR auto data call enable and DNS server not change by MBTK.
        if(asr_auto_data_call_enable() && dns_cid == MBTK_RIL_CID_DEF) {
            dns_file_path = "/tmp/resolv.conf";

            if(file_link(dns_file_path, "/etc/resolv.conf")) {
                LOGE("Create link file fail.");
                return -1;
            }
        } else {
            dns_file_path = "/tmp/mbtk_resolv.conf";

            if(file_link(dns_file_path, "/etc/resolv.conf")) {
                LOGE("Create link file fail.");
                return -1;
            } else {
                // mbtk_system("chmod 644 /tmp/mbtk_resolv.conf");
                fd = open(dns_file_path, O_WRONLY | O_TRUNC | O_CREAT, 0644);
                if(fd < 0)
                {
                    LOGE("%s : open fail.", dns_file_path);
                    return -1;
                }

                int ret = write(fd, buf, offset);
                if(ret < 0)
                {
                    LOGE("%s : write fail.", dns_file_path);
                    close(fd);
                    return -1;
                }

                LOGD("Update DNS success.");
                close(fd);
                // mbtk_system("chmod 444 /tmp/mbtk_resolv.conf");
            }
        }
    }
    return 0;
}

int net_ifc_config(mbtk_sim_type_enum sim_id, mbtk_ril_cid_enum cid, bool def_route, bool as_dns, mbtk_ip_info_t *ip_info)
{
    int ret = -1;
    char dev[20] = {0};

    if(sim_id == MBTK_SIM_2) {
        cid += 8;//sim2 starts from ccinet8
    }
    sprintf(dev, "ccinet%d", cid - 1);
    if(ip_info) { // Config IP.
        // Config IPv4 address.
        if(ip_info->ipv4.valid) {
            char ip[20] = {0};
            if(inet_ntop(AF_INET, &(ip_info->ipv4.IPAddr), ip, 20) == NULL) {
                LOGE("inet_ntop ipv4 ip fail.");
                goto exit;
            }

            if(mbtk_ifc_configure2(dev, ip, 0, NULL, "255.255.255.0")) {
                LOGD("Config %s IPv4 %s fail.", dev, ip);
                goto exit;
            } else {
                LOGD("Config %s IPv4 %s success.", dev, ip);
            }
        }

        // Config IPv6 address.
        if(ip_info->ipv6.valid) {
            char ip[50] = {0};

            if(inet_ntop(AF_INET6, &(ip_info->ipv6.IPV6Addr), ip, 50) == NULL) {
                LOGE("inet_ntop ipv6 ip fail.");
                goto exit;
            }

            if(mbtk_ipv6_config(dev, ip, 64)) {
                LOGD("Config %s IPv6 %s fail.", dev, ip);
                goto exit;
            } else {
                LOGD("Config %s IPv6 %s success.", dev, ip);
            }
        }

        // mbtk_qser_route_config(cid, &ip_info->ipv4, &ip_info->ipv6);
        if(def_route && (ip_info->ipv4.valid || ip_info->ipv6.valid)) {
            net_if_as_def_route(cid);
        }

        if(as_dns) {
            ret = net_if_as_dns(cid, &(ip_info->ipv4), &(ip_info->ipv6));
        }
    } else { // Del IP
        if(mbtk_ifc_configure2(dev, NULL, 0, NULL, NULL)) {
            LOGD("Config %s IPv4 0 fail.", dev);
            goto exit;
        } else {
            LOGD("Config %s IPv4 0 success.", dev);
        }
#if 0
        if(mbtk_ipv6_config(dev, NULL, 64)) {
            LOGD("Config %s IPv6 0 fail.", dev);
            goto exit;
        } else {
            LOGD("Config %s IPv6 0 success.", dev);
        }
#endif
    }

    ret = 0;
exit:
    return ret;
}

int net_ifc_reconfig(mbtk_sim_type_enum sim_id, mbtk_ril_cid_enum cid, bool def_route, bool as_dns, mbtk_ip_info_t *ip_info)
{
    int ret = net_ifc_config(sim_id, cid, FALSE, FALSE, NULL);
    if(ret) {
        return ret;
    }

    return net_ifc_config(sim_id, cid, def_route, as_dns, ip_info);
}

static int apn_file_save(const char *file, char *data, int data_len)
{
    if(!file) {
        return -1;
    }

    if(str_empty(data) || data_len <= 0) { // Delete file
        return unlink(file);
    } else {
        int fd = open(file, O_CREAT | O_WRONLY | O_TRUNC, 0644);
        if(fd < 0) {
            LOGE("open(%s) fail:%d", file, errno);
            return -1;
        }

        if(write(fd, data, data_len) != data_len) {
            LOGE("write fail:%d", errno);
            close(fd);
            return -1;
        }
        close(fd);
        return 0;
    }
}

static int apn_file_read(const char *file, char *data, int data_len)
{
    if(!file) {
        LOGE("file is null");
        return -1;
    }

    if(data == NULL || data_len <= 100) {
        LOGE("apn_file_read() arg error.");
        return -1;
    } else {
        int len = -1;
        int fd = open(file, O_RDONLY, 0644);
        if(fd < 0) {
            LOGE("open(%s) fail:%d", file, errno);
            return -1;
        }

        memset(data, 0, data_len);
        if((len = read(fd, data, data_len)) < 0) {
            LOGE("read fail:%d", errno);
            close(fd);
            return -1;
        }
        close(fd);
        return len;
    }
}

void apn_auto_conf_from_prop(mbtk_sim_type_enum sim_id, ATPortType_enum port)
{
    mbtk_apn_info_array_t apns;
    int i = 0;
    memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
    apn_prop_get(sim_id, &apns);
    while(i < apns.num) {
        // Update apn info in buffer.
        int cid = apns.apns[i].cid;
        info_list[cid - 1].valid = true;
        info_list[cid - 1].manual_call = false;
//        info_list[cid - 1].act_state = RIL_ACT_STATE_DISCONNECTED;
        memcpy(&(info_list[cid - 1].apn_info), &(apns.apns[i]), sizeof(mbtk_apn_info_t));

        LOGD("[sim_id - %d] CID - %d, ip_type - %d, auth - %d, auto_save - %d, auto_boot_call - %d, def_route - %d, as_dns - %d, apn - %s",
            sim_id, cid, info_list[cid - 1].apn_info.ip_type,
            info_list[cid - 1].apn_info.auth, info_list[cid - 1].apn_info.auto_save,
            info_list[cid - 1].apn_info.auto_boot_call, info_list[cid - 1].apn_info.def_route,
            info_list[cid - 1].apn_info.as_dns, info_list[cid - 1].apn_info.apn);

        int cme_err = MBTK_RIL_ERR_CME_NON;
        if(req_apn_set(sim_id, port, &(apns.apns[i]), &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
        {
            LOGD("Set APN fail.");
        }
        else
        {
            LOGD("Set APN - %d success.", apns.apns[i].cid);
        }
        i++;
    }
}

static bool apn_conf_support(mbtk_ril_cid_enum cid)
{
    if(cid == MBTK_RIL_CID_DEF) {
        return !asr_auto_data_call_enable();
    } else {
        return TRUE;
    }
}

static int apn_check_and_cid_reset(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_apn_info_t *apn)
{
    // Delete apn
    if(str_empty(apn->apn)) {
        if(apn->cid == MBTK_RIL_CID_NUL)
        {
            LOGE("null cid %d can't be delete",apn->cid);
            return -1;
        }
            

        if(asr_auto_data_call_enable() && apn->cid == MBTK_RIL_CID_DEF)
        {
            LOGE("cid %d can't be delete",apn->cid);
            return -1;
        }

        // The cid no use,so can not delete.
        mbtk_apn_info_array_t apns;
        int cme_err = MBTK_RIL_ERR_CME_NON;
        memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
        if(req_apn_get(sim_id, port, FALSE, &apns, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
        {
            LOGW("Get APN fail.");
            return 0;
        }
        else
        {
            int index = 0;
            while(index < apns.num) {
                if(apns.apns[index].cid == apn->cid)
                    return 0;
                index++;
            }
            LOGE("local apn list  don't include cid %d to be delete.",apn->cid);
            return -1;
        }
    } else { // Add or change APN.
        int start_cid = ril_cid_start;
        mbtk_apn_info_array_t apns;
        int cme_err = MBTK_RIL_ERR_CME_NON;

        memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
        if(req_apn_get(sim_id, port, TRUE, &apns, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
        {
            LOGW("Get APN fail.");
            if(apn->cid == MBTK_RIL_CID_NUL) {
                apn->cid = start_cid;
            }
        }
        else
        {
            int index = 0;
            bool is_change = FALSE;  // Is add APN default.

            if(apn->cid == MBTK_RIL_CID_NUL) { // Is add (auto set cid).
                for(; start_cid <= MBTK_APN_CID_MAX; start_cid++) {
                    index = 0;
                    while(index < apns.num) {
                        if(apns.apns[index].cid == start_cid)
                            break;
                        index++;
                    }

                    if(index == apns.num) { // Found not used cid : start_cid.
                        LOGD("Change CID : %d -> %d", apn->cid, start_cid);
                        apn->cid = start_cid;
                        // return 0;
                        break;
                    }
                }

                if(start_cid > MBTK_APN_CID_MAX) {
                    LOGE("APN full.");
                    return -1;
                }
                is_change = FALSE;
            } else {
                index = 0;
                while(index < apns.num) {
                    if(apns.apns[index].cid == apn->cid) {
                        is_change = TRUE;
                        break;
                    }
                    index++;
                }
            }

            if(apn->cid == MBTK_RIL_CID_1) {
#if 1
                apn->auto_boot_call = 0;    // Reset to 0 for CID 1.
#else
                if(apn->auto_boot_call) {
                    LOGE("CID 1 not support auto boot data call(Please open \"wan_default.default.enable\").");
                    return -1;
                }
#endif
            }

            // Is add,the APN can't same.
            // Is change, not change apn name to other cid.
#if 0
            if(!is_change) {
                index = 0;
                while(index < apns.num) {
                    if(strcmp((char*)apns.apns[index].apn,(char*)apn->apn) == 0) {
                        LOGW("APN : %s exist.", apn->apn);
                        return -1;
                    }
                    index++;
                }
            }
#else
            index = 0;
            while(index < apns.num) {
                if(strcmp((char*)apns.apns[index].apn,(char*)apn->apn) == 0
                    || (strlen((char*)apns.apns[index].type) > 0 && strcmp((char*)apns.apns[index].type,(char*)apn->type) == 0)) {
                    if(!is_change) { // Is add,the APN name/type can't same.
                        LOGW("APN : %s/%s exist, can't add.", apn->apn, apn->type);
                        return -1;
                    } else { // Is change, not change apn name/type to other cid.
                        if(apns.apns[index].cid != apn->cid) {
                            LOGW("APN : %s/%s exist in cid[%d], can't change to cid[%d]", apn->apn, apn->type,
                                apns.apns[index].cid, apn->cid);
                            return -1;
                        }
                    }
                }
                index++;
            }
#endif
        }
    }
    return 0;
}

static void apn_prop_get(mbtk_sim_type_enum sim_id, mbtk_apn_info_array_t *apns)
{
    char prop_name[128] = {0};
    char prop_data[1024] = {0};
    int cid = ril_cid_start;
    memset(apns, 0, sizeof(mbtk_apn_info_array_t));

    char def_cid[10] = {0};
    sprintf(def_cid, "%d", MBTK_RIL_CID_DEF);   // Default route and DNS is CID 1.
    if(property_get(MBTK_DEF_ROUTE_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_route = (mbtk_ril_cid_enum)atoi(prop_data);
    }
    memset(prop_data, 0, sizeof(prop_data));
    if(property_get(MBTK_DEF_DNS_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_dns = (mbtk_ril_cid_enum)atoi(prop_data);
    }

    for(; cid <= MBTK_APN_CID_MAX; cid++) {
        memset(prop_name, 0, sizeof(prop_name));
        memset(prop_data, 0, sizeof(prop_data));

        sprintf(prop_name, "%s_%d_%d",MBTK_APN_PROP,cid, sim_id);
        // ip_type,auth,auto_data_call,redial_enable,apn,user,pass,type
#if 0
        if(property_get(prop_name, prop_data, "") > 0 && !str_empty(prop_data)) {
#else
        if(apn_file_read(prop_name, prop_data, sizeof(prop_data)) > 0 && !str_empty(prop_data)) {
#endif
            apns->apns[apns->num].auto_save = 1;
            apns->apns[apns->num].cid = (mbtk_ril_cid_enum)cid;

            if(apns->cid_for_def_dns == apns->apns[apns->num].cid)
                apns->apns[apns->num].as_dns = 1;

            if(apns->cid_for_def_route == apns->apns[apns->num].cid)
                apns->apns[apns->num].def_route = 1;

            char *ptr_1 = prop_data;
            apns->apns[apns->num].ip_type = (mbtk_ip_type_enum)atoi(ptr_1);
            ptr_1 = strstr(ptr_1, ",");
            if(!ptr_1) {
                continue;
            }
            ptr_1++; // Jump ',' to auth

            apns->apns[apns->num].auth = (mbtk_apn_auth_type_enum)atoi(ptr_1);
            ptr_1 = strstr(ptr_1, ",");
            if(!ptr_1) {
                continue;
            }
            ptr_1++; // Jump ',' to auto_data_call

            apns->apns[apns->num].auto_boot_call = (uint8)atoi(ptr_1);
            ptr_1 = strstr(ptr_1, ",");
            if(!ptr_1) {
                continue;
            }
            ptr_1++; // Jump ',' to redial_enable

            apns->apns[apns->num].redial_enable = (uint8)atoi(ptr_1);
            ptr_1 = strstr(ptr_1, ",");
            if(!ptr_1) {
                continue;
            }
            ptr_1++; // Jump ',' to apn

            // Reset auto_boot_call to 0 for CID 1.
            if(apns->apns[apns->num].cid == MBTK_RIL_CID_1) {
                apns->apns[apns->num].auto_boot_call = 0;
            }

            char *ptr_2 = strstr(ptr_1, ",");
            if(!ptr_2) {
                continue;
            }
            if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].apn, ptr_1, ptr_2 - ptr_1); // apn
            }

            ptr_2++; // Jump ',' to user
            ptr_1 = strstr(ptr_2, ",");
            if(!ptr_1) {
                continue;
            }
            if(memcmp(ptr_2, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].user, ptr_2, ptr_1 - ptr_2); // user
            }

            ptr_1++; // Jump ',' to pass
            ptr_2 = strstr(ptr_1, ",");
            if(!ptr_2) {
                continue;
            }
            if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].pass, ptr_1, ptr_2 - ptr_1); // pass
            }

            ptr_2++; // Jump ',' to type
            if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].type, ptr_2, strlen(ptr_2)); // type
            }

            apns->num++;
        }
    }
}

static int apn_prop_get_by_cid(mbtk_sim_type_enum sim_id, mbtk_ril_cid_enum cid, mbtk_apn_info_t *apn)
{
    char prop_name[128] = {0};
    char prop_data[1024] = {0};
    memset(apn, 0, sizeof(mbtk_apn_info_t));

    sprintf(prop_name, "%s_%d_%d", MBTK_APN_PROP, cid, sim_id);
    // ip_type,auth,auto_data_call,redial_enable,apn,user,pass,type
#if 0
    if(property_get(prop_name, prop_data, "") > 0 && !str_empty(prop_data)) {
#else
    if(apn_file_read(prop_name, prop_data, sizeof(prop_data)) > 0 && !str_empty(prop_data)) {
#endif
        apn->cid = cid;
        apn->auto_save = (uint8)1;
        char *ptr_1 = prop_data;
        apn->ip_type = (mbtk_ip_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auth

        apn->auth = (mbtk_apn_auth_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auto_data_call

        apn->auto_boot_call = (uint8)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to redial_enable

        apn->redial_enable = (uint8)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to apn

        // Reset auto_boot_call to 0 for CID 1.
        if(apn->cid == MBTK_RIL_CID_1) {
            apn->auto_boot_call = 0;
        }

        char *ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->apn, ptr_1, ptr_2 - ptr_1); // apn
        }

        ptr_2++; // Jump ',' to user
        ptr_1 = strstr(ptr_2, ",");
        if(!ptr_1) {
            return -1;
        }
        if(memcmp(ptr_2, "NULL", 4)) { // Not "NULL"
            memcpy(apn->user, ptr_2, ptr_1 - ptr_2); // user
        }

        ptr_1++; // Jump ',' to pass
        ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->pass, ptr_1, ptr_2 - ptr_1); // pass
        }

        ptr_2++; // Jump ',' to type
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->type, ptr_2, strlen(ptr_2)); // pass
        }
        return 0;
    }
    return -1;
}

static int apn_prop_get_by_cid_without_cgdcont(mbtk_sim_type_enum sim_id, mbtk_ril_cid_enum cid, mbtk_apn_info_t *apn)
{
    char prop_name[128] = {0};
    char prop_data[1024] = {0};

    sprintf(prop_name, "%s_%d_%d",MBTK_APN_PROP,cid, sim_id);
    // ip_type,auth,auto_data_call,redial_enable,apn,user,pass,type
#if 0
    if(property_get(prop_name, prop_data, "") > 0 && !str_empty(prop_data)) {
#else
    if(apn_file_read(prop_name, prop_data, sizeof(prop_data)) > 0 && !str_empty(prop_data)) {
#endif
        LOGD("APN : %s", prop_data);
        char *ptr_1 = prop_data;
        apn->auto_save = (uint8)1;
        //apn->ip_type = (mbtk_ip_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auth

        apn->auth = (mbtk_apn_auth_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auto_data_call

        apn->auto_boot_call = (uint8)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to redial_enable

        apn->redial_enable = (uint8)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to apn

        // Reset auto_boot_call to 0 for CID 1.
        if(cid == MBTK_RIL_CID_1) {
            apn->auto_boot_call = 0;
        }

        char *ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
#if 0
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->apn, ptr_1, ptr_2 - ptr_1); // apn
        }
#endif

        ptr_2++; // Jump ',' to user
        ptr_1 = strstr(ptr_2, ",");
        if(!ptr_1) {
            return -1;
        }
        if(memcmp(ptr_2, "NULL", 4)) { // Not "NULL"
            memcpy(apn->user, ptr_2, ptr_1 - ptr_2); // user
        }

        ptr_1++; // Jump ',' to pass
        ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->pass, ptr_1, ptr_2 - ptr_1); // pass
        }

        ptr_2++; // Jump ',' to type
        if(memcmp(ptr_2, "NULL", 4)) { // Not "NULL"
            memcpy(apn->type, ptr_2, strlen(ptr_2)); // pass
        }

        return 0;
    } else {
        apn->auto_save = (uint8)0;
    }
    return -1;
}

static int apn_prop_set(mbtk_sim_type_enum sim_id, mbtk_apn_info_t *apn)
{
    char prop_name[20] = {0};
    char prop_data[1024] = {0};
    int ret = 0;
    if(apn->auto_save) {
        // auto_boot_call must be 0 for CID 1.
        if(apn->cid == MBTK_RIL_CID_1) {
            apn->auto_boot_call = 0;
        }

        sprintf(prop_name, "%s_%d_%d", MBTK_APN_PROP, apn->cid, sim_id);
        // Delete apn
        if(!str_empty(apn->apn)) {
            snprintf(prop_data, 1024, "%d,%d,%d,%d,%s,%s,%s,%s", apn->ip_type, apn->auth, apn->auto_boot_call,
                apn->redial_enable,
                apn->apn,
                str_empty(apn->user) ? "NULL" : (char*)apn->user,
                str_empty(apn->pass) ? "NULL" : (char*)apn->pass,
                str_empty(apn->type) ? "NULL" : (char*)apn->type);
        }

#if 0
        ret = property_set(prop_name, prop_data);
#else
        if(str_empty(apn->apn)) { // Delete APN
            ret = apn_file_save(prop_name, NULL, 0);
        } else {
            ret = apn_file_save(prop_name, prop_data, strlen(prop_data));
        }
#endif
    }

    if(!ret) {
        memset(prop_data, 0, sizeof(prop_data));
        if(apn->def_route) {
            prop_data[0] = '0' + apn->cid;
            ret = property_set(MBTK_DEF_ROUTE_CID, prop_data);
        } else {
            char def_cid[10] = {0};
            int cid = -1;
            sprintf(def_cid, "%d", MBTK_RIL_CID_DEF);   // Default route and DNS is CID 1.
            memset(prop_data, 0, sizeof(prop_data));
            if(property_get(MBTK_DEF_ROUTE_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
                cid = atoi(prop_data);
            }

            if(apn->cid == cid) { // The route cid should reset to default.
                prop_data[0] = '0' + MBTK_RIL_CID_DEF;
                ret = property_set(MBTK_DEF_ROUTE_CID, prop_data);
            }
        }
    }
    if(!ret) {
        memset(prop_data, 0, sizeof(prop_data));
        if(apn->as_dns) {
            prop_data[0] = '0' + apn->cid;
            ret = property_set(MBTK_DEF_DNS_CID, prop_data);
        } else {
            char def_cid[10] = {0};
            int cid = -1;
            sprintf(def_cid, "%d", MBTK_RIL_CID_DEF);   // Default route and DNS is CID 1.
            memset(prop_data, 0, sizeof(prop_data));
            if(property_get(MBTK_DEF_DNS_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
                cid = atoi(prop_data);
            }

            if(apn->cid == cid) { // The dns cid should reset to default.
                prop_data[0] = '0' + MBTK_RIL_CID_DEF;
                ret = property_set(MBTK_DEF_DNS_CID, prop_data);
            }
        }
    }

    // Update apn info in buffer.
    if(str_empty(apn->apn)) {
        memset(&(info_list[apn->cid - 1]), 0, sizeof(ril_data_call_info_t));
    } else {
        info_list[apn->cid - 1].valid = true;
        memcpy(&(info_list[apn->cid - 1].apn_info), apn, sizeof(mbtk_apn_info_t));
    }

    return ret;
}

static int apn_prop_reset(mbtk_sim_type_enum sim_id, mbtk_data_call_info_t *data_info)
{
    if(data_info->auto_boot_call != MBTK_DATA_CALL_ITEM_STATE_NON
        || data_info->def_route != MBTK_DATA_CALL_ITEM_STATE_NON
        || data_info->as_dns != MBTK_DATA_CALL_ITEM_STATE_NON) {
        mbtk_apn_info_t apn;
        if(apn_prop_get_by_cid(sim_id, data_info->cid, &apn)) {
            if(data_info->cid == MBTK_RIL_CID_1) {
                memset(&apn, 0, sizeof(mbtk_apn_info_t));
                apn.cid = MBTK_RIL_CID_1;
                memcpy(apn.apn, "NULL", 4);
                if(data_info->auto_boot_call != MBTK_DATA_CALL_ITEM_STATE_NON)
                    apn.auto_boot_call = (data_info->auto_boot_call == MBTK_DATA_CALL_ITEM_STATE_ENABLE) ? 1 : 0;

                if(data_info->def_route != MBTK_DATA_CALL_ITEM_STATE_NON)
                    apn.def_route = (data_info->def_route == MBTK_DATA_CALL_ITEM_STATE_ENABLE) ? 1 : 0;

                if(data_info->as_dns != MBTK_DATA_CALL_ITEM_STATE_NON)
                    apn.as_dns = (data_info->as_dns == MBTK_DATA_CALL_ITEM_STATE_ENABLE) ? 1 : 0;
                return apn_prop_set(sim_id, &apn);
            } else {
                return -1;
            }
        } else {
            if(data_info->auto_boot_call != MBTK_DATA_CALL_ITEM_STATE_NON)
                apn.auto_boot_call = (data_info->auto_boot_call == MBTK_DATA_CALL_ITEM_STATE_ENABLE) ? 1 : 0;

            if(data_info->def_route != MBTK_DATA_CALL_ITEM_STATE_NON)
                apn.def_route = (data_info->def_route == MBTK_DATA_CALL_ITEM_STATE_ENABLE) ? 1 : 0;

            if(data_info->as_dns != MBTK_DATA_CALL_ITEM_STATE_NON)
                apn.as_dns = (data_info->as_dns == MBTK_DATA_CALL_ITEM_STATE_ENABLE) ? 1 : 0;
            return apn_prop_set(sim_id, &apn);
        }
    }

    return 0;
}

static int wait_cgact_complete(int timeout)
{
    int count = timeout * 10; // timeout * 1000 / 100
    int i = 0;

    while(cgact_wait.waitting && i < count) {
        i++;
        usleep(100000); // 100ms
    }

    memset(&cgact_wait, 0, sizeof(ril_cgact_wait_t));
    if(i == count) { // Timeout
        return -1;
    } else {
        return 0;
    }
}

void data_call_retry(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_ril_net_reg_state_info_t *reg_state)
{
    LOGV("NET_REG_STATE_CHANGE : type - %d, tech - %d, reg_state - %d", reg_state->type, reg_state->tech, reg_state->reg_state);
    // Only for 2g/3g/4g data domain.
    if(reg_state->type == MBTK_NET_REG_TYPE_DATA_GSM_WCDMA
        || reg_state->type == MBTK_NET_REG_TYPE_DATA_LTE) {
        if(reg_state->tech >= MBTK_RADIO_TECH_UTRAN) { // No for GSM(No GPRS)
            if(reg_state->reg_state == MBTK_NET_REG_STATE_HOME
                || reg_state->reg_state == MBTK_NET_REG_STATE_ROAMING) { // Only Home and Roaming network.
                int cid = ril_cid_start;
                // mbtk_ip_info_t ip_info;
                mbtk_ril_pdp_state_info_t pdp_info;
                for(; cid <= MBTK_APN_CID_MAX; cid++) {
                    LOGV("cid - %d, valid - %d, act_state - %d", cid, info_list[cid - 1].valid,
                        info_list[cid - 1].act_state);
                    if(info_list[cid - 1].valid &&
                        ((info_list[cid - 1].manual_call && info_list[cid - 1].act_state == RIL_ACT_STATE_CONNECTED_RETRY)
                        || info_list[cid - 1].apn_info.auto_boot_call)) {   // Auto data call when boot.
                        memset(&pdp_info, 0, sizeof(mbtk_ril_pdp_state_info_t));
                        pdp_info.action = TRUE;
                        pdp_info.cid = cid;

                        if(!apn_conf_support(pdp_info.cid)) {
                            LOGD("Can not data call for CID : %d", pdp_info.cid);
                            continue;
                        }

                        // Reset ip information.
                        char dev[20] = {0};
                        if(sim_id == MBTK_SIM_2) {
                            sprintf(dev, "ccinet%d", cid + MBTK_APN_CID_MAX); //sim2 starts from ccinet8
                        } else {
                            sprintf(dev, "ccinet%d", cid - MBTK_APN_CID_MIN); //sim1 starts from ccinet0
                        }
                        if(mbtk_ifc_configure2(dev, NULL, 0, NULL, NULL)) {
                            LOGD("Config %s IPv4 0 fail.", dev);
                        } else {
                            LOGD("Config %s IPv4 0 success.", dev);
                        }

                        // PDP active
                        int cme_err = MBTK_RIL_ERR_CME_NON;
                        if(req_data_call_start(sim_id, port, cid, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                        {
                            LOGE("Restart data call fail.");
                            return;
                        }
                        else
                        {
                            // Wait for "CONNECT" or "+CGEV:"
                            if(wait_cgact_complete(5000)) { // Timeout
                                LOGE("PDP active timeout.");
                                return;
                            }

                            // Get Ip informations.
                            // memset(&ip_info, 0, sizeof(ip_info));
                            cme_err = MBTK_RIL_ERR_CME_NON;
                            if(req_data_call_state_get(sim_id, port, cid, &(pdp_info.ip_info), &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                            {
                                LOGE("Get net informations fail.");
                                return;
                            }
                            else
                            {
                                // Config network informations.
                                LOGD("def_route - %d, as_dns - %d, ipv4_valid - %d, ipv6_valid - %d",
                                    info_list[cid - 1].apn_info.def_route,
                                    info_list[cid - 1].apn_info.as_dns,
                                    pdp_info.ip_info.ipv4.valid, pdp_info.ip_info.ipv6.valid);
                                if(net_ifc_reconfig(sim_id, cid, info_list[cid - 1].apn_info.def_route,
                                    info_list[cid - 1].apn_info.as_dns, &(pdp_info.ip_info))) {
                                    LOGE("Set net informations fail.");
                                    return;
                                }

                                info_list[cid - 1].act_state = RIL_ACT_STATE_CONNECTED;

                                pdp_info.ip_info_valid = TRUE;

                                ril_state_change(RIL_MSG_ID_IND_PDP_STATE_CHANGE, &pdp_info, sizeof(mbtk_ril_pdp_state_info_t));
                            }
                        }
                    }
                }
            }
        }
    }
}

/*
AT+CGDCONT?

+CGDCONT: 1,"IPV4V6","ctnet","10.142.64.116 254.128.0.0.0.0.0.0.0.0.0.0.0.0.0.1",0,0,0,2,0,0

+CGDCONT: 8,"IPV4V6","IMS","254.128.0.0.0.0.0.0.0.0.0.0.0.0.0.1",0,0,0,2,1,1

OK

*/
static int req_apn_get(mbtk_sim_type_enum sim_id, ATPortType_enum port, bool get_def_cid, mbtk_apn_info_array_t *apns, int *cme_err)
{
    ATResponse *response = NULL;
    int err = at_send_command_multiline(portType_2_portId(sim_id, port), "AT+CGDCONT?", "+CGDCONT:", &response);

    if (err < 0 || response->success == 0 || !response->p_intermediates){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

    ATLine* lines_ptr = response->p_intermediates;
    char *line = NULL;
    int tmp_int;
    char *tmp_str = NULL;
    int cid_start;
    if(get_def_cid) {
        cid_start = MBTK_RIL_CID_DEF;
    } else {
        cid_start = ril_cid_start;
    }
    /*
    <apn_num[1]><cid[1]><ip_type[1]><apn_len[2]><apn><user_len[2]><user><pass_len[2]><pass><auth_len[2]><auth>...
                <cid[1]><ip_type[1]><apn_len[2]><apn><user_len[2]><user><pass_len[2]><pass><auth_len[2]><auth>
    */
    while(lines_ptr)
    {
        line = lines_ptr->line;
        err = at_tok_start(&line);
        if (err < 0)
        {
            goto exit;
        }

        err = at_tok_nextint(&line, &tmp_int); // cid
        if (err < 0)
        {
            goto exit;
        }
        // Only get CID 1-7
        if(tmp_int >= cid_start && tmp_int <= MBTK_APN_CID_MAX) {
            apns->apns[apns->num].cid = (mbtk_ril_cid_enum)tmp_int;

            err = at_tok_nextstr(&line, &tmp_str);// ip type
            if (err < 0)
            {
                goto exit;
            }
            if(!strcasecmp(tmp_str, "IP")) {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_IP;
            } else if(!strcasecmp(tmp_str, "IPV6")) {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_IPV6;
            } else if(!strcasecmp(tmp_str, "IPV4V6")) {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_IPV4V6;
            } else {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_PPP;
            }

            err = at_tok_nextstr(&line, &tmp_str); // apn
            if (err < 0)
            {
                goto exit;
            }
            if(!str_empty(tmp_str)) {
                memcpy(apns->apns[apns->num].apn, tmp_str, strlen(tmp_str));
            }

            // Get other arg from proc or file.
            apn_prop_get_by_cid_without_cgdcont(sim_id, apns->apns[apns->num].cid, &(apns->apns[apns->num]));

            apns->num++;
        }

        lines_ptr = lines_ptr->p_next;
    }

 //   char prop_name[128] = {0};
    char prop_data[1024] = {0};
    char def_cid[10] = {0};
    sprintf(def_cid, "%d", MBTK_RIL_CID_DEF);  // Default route and DNS is CID 1.

    if(property_get(MBTK_DEF_ROUTE_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_route = (mbtk_ril_cid_enum)atoi(prop_data);
    }
    memset(prop_data, 0, sizeof(prop_data));
    if(property_get(MBTK_DEF_DNS_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_dns = (mbtk_ril_cid_enum)atoi(prop_data);
    }

    goto exit;
exit:
    at_response_free(response);
    return err;
}

/*
AT+CGDCONT=1,"IPV4V6","cmnet"
OK

AT*AUTHReq=1,1,marvell,123456
OK

*/
static int req_apn_set(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_apn_info_t *apn, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[400] = {0};
    int index = 0;
    int err = 0;

    // Delete apn
    if(str_empty(apn->apn)) {
        sprintf(cmd, "AT+CGDCONT=%d", apn->cid);
        err = at_send_command(portType_2_portId(sim_id, port), cmd, &response);
        if (err < 0 || response->success == 0){
            if(cme_err) {
                *cme_err = at_get_cme_error(response);
            }
            goto exit;
        }
    } else {
        index += sprintf(cmd, "AT+CGDCONT=%d,", apn->cid);
        switch(apn->ip_type) {
            case MBTK_IP_TYPE_IP: {
                index += sprintf(cmd + index,"\"IP\",");
                break;
            }
            case MBTK_IP_TYPE_IPV6: {
                index += sprintf(cmd + index,"\"IPV6\",");
                break;
            }
            case MBTK_IP_TYPE_IPV4V6: {
                index += sprintf(cmd + index,"\"IPV4V6\",");
                break;
            }
            default: {
                index += sprintf(cmd + index,"\"PPP\",");
                break;
            }
        }
        if(strlen((char*)apn->apn) > 0) {
            index += sprintf(cmd + index,"\"%s\"", apn->apn);
        }

        err = at_send_command(portType_2_portId(sim_id, port), cmd, &response);
        if (err < 0 || response->success == 0){
            if(cme_err) {
                *cme_err = at_get_cme_error(response);
            }
            goto exit;
        }

        if(!str_empty(apn->user) || !str_empty(apn->pass)) {
            at_response_free(response);
            response = NULL;

            memset(cmd,0,400);
            int cmd_auth=0;
            if(apn->auth == MBTK_APN_AUTH_PROTO_NONE)
                cmd_auth = 0;
            else if(apn->auth == MBTK_APN_AUTH_PROTO_PAP)
                cmd_auth = 1;
            else if(apn->auth == MBTK_APN_AUTH_PROTO_CHAP)
                cmd_auth = 2;
            else
                goto exit;

            sprintf(cmd, "AT*AUTHREQ=%d,%d,%s,%s",apn->cid,cmd_auth,apn->user,apn->pass);
            err = at_send_command(portType_2_portId(sim_id, port), cmd, &response);
            if (err < 0 || response->success == 0){
                if(cme_err) {
                    *cme_err = at_get_cme_error(response);
                }
                goto exit;
            }
        }
    }

exit:
    at_response_free(response);
    return err;
}

/*
AT+CGACT?
+CGACT: 1,1
+CGACT: 8,1
OK

AT+CGACT=1,<cid>
OK

*/
static int req_data_call_start(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_ril_cid_enum cid, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[400] = {0};
    int err = 0;
    memset(&cgact_wait, 0, sizeof(ril_cgact_wait_t));
    cgact_wait.waitting = true;
    cgact_wait.cid = cid;
    cgact_wait.act = true;

    sprintf(cmd, "AT+CGACT=1,%d", cid);
    err = at_send_command(portType_2_portId(sim_id, port), cmd, &response);
    if (err < 0 || response->success == 0){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

    info_list[cid - 1].manual_call = true;
exit:
    at_response_free(response);
    return err;
}

/*
AT+CGACT=0,<cid>
OK

*/
static int req_data_call_stop(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_ril_cid_enum cid, int timeout, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[400] = {0};
    int err = 0;

    if(cid == MBTK_RIL_CID_DEF) {
        cgact_wait.waitting = false;
        info_list[cid - 1].manual_call = false;
        LOGD("Can't deactivate default apn.");
        return 0;
    }

    memset(&cgact_wait, 0, sizeof(ril_cgact_wait_t));
    cgact_wait.waitting = true;
    cgact_wait.cid = cid;
    cgact_wait.act = false;

    sprintf(cmd, "AT+CGACT=0,%d", cid);
    err = at_send_command(portType_2_portId(sim_id, port), cmd, &response);
    if (err < 0 || response->success == 0){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

    info_list[cid - 1].manual_call = false;
exit:
    at_response_free(response);
    return err;
}

void get_gate_way_v4(uint32 ip_v4,  void* gateWay)
{
     uint32 local_gateWay=ip_v4;
     if((local_gateWay & 0xff000000) < 0xfe000000)
     {
         local_gateWay+=(1 << 24);
     }
     else
     {
         local_gateWay-=(1 << 24);
     }
     (* (uint32*) gateWay) = local_gateWay;
}

void get_gate_way_v6(void* ip_v6, void* gateWay)
{
    memcpy(gateWay,ip_v6,32);
    if((((uint32*)gateWay)[3] & 0xff000000)!=0xff000000)
    {
        ((uint32*)gateWay)[3] += (1 << 24);
    }
    else
    {
        ((uint32*)gateWay)[3] -= (1 << 24);
    }
}

/*
AT+CGCONTRDP=1
+CGCONTRDP: 1,7,"cmnet-2.MNC000.MCC460.GPRS","10.255.74.26","","223.87.253.100","223.87.253.253","","",0,0
+CGCONTRDP: 1,7,"cmnet-2.MNC000.MCC460.GPRS","254.128.0.0.0.0.0.0.0.1.0.2.144.5.212.239","","36.9.128.98.32.0.0.2.0.0.0.0.0.0.0.1","36.9.128.98.32.0.0.2.0.0.0.0.0.0.0.2","","",0,0

OK

*/
static int req_data_call_state_get(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_ril_cid_enum cid, mbtk_ip_info_t *ip_info, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[50] = {0};
    int err = 0;

    sprintf(cmd, "AT+CGCONTRDP=%d", cid);

    err = at_send_command_multiline(portType_2_portId(sim_id, port), cmd, "+CGCONTRDP:", &response);
    if (err < 0 || response->success == 0 || !response->p_intermediates){
        *cme_err = at_get_cme_error(response);
        goto exit;
    }
    ATLine* lines_ptr = response->p_intermediates;
    char *line = NULL;
    int tmp_int;
    char *tmp_ptr = NULL;
    while(lines_ptr)
    {
        line = lines_ptr->line;
        err = at_tok_start(&line);
        if (err < 0)
        {
            goto exit;
        }

        err = at_tok_nextint(&line, &tmp_int); // cid
        if (err < 0)
        {
            goto exit;
        }
        err = at_tok_nextint(&line, &tmp_int); // bearer_id
        if (err < 0)
        {
            goto exit;
        }
        err = at_tok_nextstr(&line, &tmp_ptr); // APN
        if (err < 0)
        {
            goto exit;
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // IP
        if (err < 0 || str_empty(tmp_ptr))
        {
            goto exit;
        }
        if(is_ipv4(tmp_ptr)) {
            if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.IPAddr)) < 0) {
                LOGE("inet_pton() fail.");
                err = -1;
                goto exit;
            }

            ip_info->ipv4.valid = true;
            //log_hex("IPv4", &(ipv4->IPAddr), sizeof(struct in_addr));
        } else {
            if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.IPV6Addr))) {
                LOGE("str_2_ipv6() fail.");
                err = -1;
                goto exit;
            }

            ip_info->ipv6.valid = true;
            //log_hex("IPv6", &(ipv6->IPV6Addr), 16);
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // Gateway
        if (err < 0)
        {
            goto exit;
        }
        if(!str_empty(tmp_ptr)) { // No found gateway
            if(is_ipv4(tmp_ptr)) {
                if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.GateWay)) < 0) {
                    LOGE("inet_pton() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv4", &(ipv4->GateWay), sizeof(struct in_addr));
            } else {
                if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.GateWay))) {
                    LOGE("str_2_ipv6() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv6", &(ipv6->GateWay), 16);
            }
        }
        else
        {
            if(ip_info->ipv4.valid)
            {
                get_gate_way_v4(ip_info->ipv4.IPAddr,&(ip_info->ipv4.GateWay));
                ip_info->ipv4.NetMask=0xffffff;
            }
            else if(ip_info->ipv6.valid)
            {
                get_gate_way_v6(ip_info->ipv6.IPV6Addr,ip_info->ipv6.GateWay);
            }
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // prim_DNS
        if (err < 0)
        {
            goto exit;
        }
        if(!str_empty(tmp_ptr)) { // No found Primary DNS
            if(is_ipv4(tmp_ptr)) {
                if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.PrimaryDNS)) < 0) {
                    LOGE("inet_pton() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv4", &(ipv4->PrimaryDNS), sizeof(struct in_addr));
            } else {
                if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.PrimaryDNS))) {
                    LOGE("str_2_ipv6() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv6", &(ipv6->PrimaryDNS), 16);
            }
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // sec_DNS
        if (err < 0)
        {
            goto exit;
        }
        if(!str_empty(tmp_ptr)) { // No found Secondary DNS
            if(is_ipv4(tmp_ptr)) {
                if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.SecondaryDNS)) < 0) {
                    LOGE("inet_pton() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv4", &(ipv4->SecondaryDNS), sizeof(struct in_addr));
            } else {
                if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.SecondaryDNS))) {
                    LOGE("str_2_ipv6() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv6", &(ipv6->SecondaryDNS), 16);
            }
        }

        lines_ptr = lines_ptr->p_next;
    }

exit:
    at_response_free(response);
    return err;
}

void data_call_state_change_cb(mbtk_sim_type_enum sim_id, int cid, bool action, bool auto_change, int reason)
{
    // Will restart data call.
    if(auto_change && !action) {
        info_list[cid - 1].act_state = RIL_ACT_STATE_CONNECTED_RETRY;

        // Reset ip information.
        char dev[20] = {0};
        sprintf(dev, "ccinet%d", cid - 1);
        if(mbtk_ifc_configure2(dev, NULL, 0, NULL, NULL)) {
            LOGD("Config %s IPv4 0 fail.", dev);
        } else {
            LOGD("Config %s IPv4 0 success.", dev);
        }
    }
}

int req_apn_init_attach_apn_inner(mbtk_sim_type_enum sim_id, ATPortType_enum port, mbtk_apn_info_t *apn, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[400];
    int index = 0;
    int err = 0;

    index += sprintf(cmd, "AT*CGDFLT=1,");//save to nvm
    switch(apn->ip_type) {
        case MBTK_IP_TYPE_IP: {
            index += sprintf(cmd + index,"\"IP\",");
            break;
        }
        case MBTK_IP_TYPE_IPV6: {
            index += sprintf(cmd + index,"\"IPV6\",");
            break;
        }
        case MBTK_IP_TYPE_IPV4V6: {
            index += sprintf(cmd + index,"\"IPV4V6\",");
            break;
        }
        default: {
            index += sprintf(cmd + index,"\"PPP\",");
            break;
        }
    }
    if(strlen((char*)apn->apn) > 0) {
        index += sprintf(cmd + index,"\"%s\",,,,,,,,,,,,,,,,,,1", apn->apn);
    }
    else
    {
        index += sprintf(cmd + index,"\"%s\",,,,,,,,,,,,,,,,,,0", "");
    }
    
    err = at_send_command(portType_2_portId(sim_id, port), cmd, &response);
    if (err < 0 || response->success == 0){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }
  
    if(!str_empty(apn->user) || !str_empty(apn->pass)) {
        at_response_free(response);
        response = NULL;

        int cmd_auth=0;
        if(apn->auth == MBTK_APN_AUTH_PROTO_NONE)
            cmd_auth = 0;
        else if(apn->auth == MBTK_APN_AUTH_PROTO_PAP)
            cmd_auth = 1;
        else if(apn->auth == MBTK_APN_AUTH_PROTO_CHAP)
            cmd_auth = 2;
        else
            goto exit;
  
        sprintf(cmd, "AT*CGDFAUTH=%d,%d,%s,%s",1,cmd_auth,apn->user,apn->pass);
        err = at_send_command(portType_2_portId(sim_id, port), cmd, &response);
        if (err < 0 || response->success == 0){
            if(cme_err) {
                *cme_err = at_get_cme_error(response);
            }
            goto exit;
        }
    }
  exit:
      at_response_free(response);
      return err;

}

int req_apn_init_attach_apn(sock_cli_info_t* cli_info, ril_msg_pack_info_t* pack,mbtk_apn_info_t *apn)
{
    int err = 0;
    int cme_err = MBTK_RIL_ERR_CME_NON;
    if(req_apn_init_attach_apn_inner(cli_info->sim_id, cli_info->port, apn, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
   {
       if(cme_err != MBTK_RIL_ERR_CME_NON) {
           err = MBTK_RIL_ERR_CME + cme_err;
       } else {
           err = MBTK_RIL_ERR_UNKNOWN;
       }
       LOGE("Set initial attach APN fail, err is %d",err);
   } 
   else
   {
       mbtk_ril_cid_enum return_cid = apn->cid;
       ril_rsp_pack_send(cli_info->sim_id, cli_info->port, cli_info->fd, pack->msg_id, pack->msg_index, &return_cid, sizeof(mbtk_ril_cid_enum));
   }
   return err;

}

//void net_list_free(void *data);
// Return MBTK_INFO_ERR_SUCCESS,will call pack_error_send() to send RSP.
// Otherwise, do not call pack_error_send().
mbtk_ril_err_enum data_call_pack_req_process(sock_cli_info_t* cli_info, ril_msg_pack_info_t* pack)
{
    mbtk_ril_err_enum err = MBTK_RIL_ERR_SUCCESS;
    int cme_err = MBTK_RIL_ERR_CME_NON;
    switch(pack->msg_id)
    {
        case RIL_MSG_ID_DATA_CALL_APN:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                mbtk_apn_info_array_t apns;
                memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
                if(req_apn_get(cli_info->sim_id, cli_info->port, FALSE, &apns, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOGD("Get APN fail.");
                }
                else
                {
                    LOGD("size - %d", sizeof(mbtk_apn_info_array_t));
                    ril_rsp_pack_send(cli_info->sim_id, cli_info->port, cli_info->fd, pack->msg_id, pack->msg_index, &apns, sizeof(mbtk_apn_info_array_t));
                }
            }
            else     // Set
            {
                mbtk_apn_info_t *apn = (mbtk_apn_info_t*)pack->data;
                if(apn->b_initial_attach_apn)
                {
                    err=req_apn_init_attach_apn(cli_info,pack,apn);
                    break;
                }
                mbtk_ril_cid_enum return_cid = apn->cid;
                if(apn_check_and_cid_reset(cli_info->sim_id, cli_info->port, apn)) {
                    err = MBTK_RIL_ERR_CID;
                } else {
                    if(apn_conf_support(apn->cid)) {
                        if(req_apn_set(cli_info->sim_id, cli_info->port, apn, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                        {
                            if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                err = MBTK_RIL_ERR_CME + cme_err;
                            } else {
                                err = MBTK_RIL_ERR_UNKNOWN;
                            }
                            LOGD("Set APN fail.");
                        }
                        else
                        {
                            if(apn_prop_set(cli_info->sim_id, apn)) {
                                LOGE("Save APN fail.");
                            }

                            return_cid = apn->cid;
                            ril_rsp_pack_send(cli_info->sim_id, cli_info->port, cli_info->fd, pack->msg_id, pack->msg_index, &return_cid, sizeof(mbtk_ril_cid_enum));
                        }
                    } else {
                        err = MBTK_RIL_ERR_UNSUPPORTED;
                        LOGD("Can not set APN for CID : %d", apn->cid);
                    }
                }
            }
            break;
        }
        case RIL_MSG_ID_DATA_CALL_OPT:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
            }
            else     // Set
            {
                mbtk_data_call_info_t *call_info = (mbtk_data_call_info_t*)pack->data;
                // Check for cid 1.
                if(!apn_conf_support(call_info->cid)) {
                    err = MBTK_RIL_ERR_UNSUPPORTED;
                    LOGD("Can not data call for CID : %d", call_info->cid);
                } else {
                    if(call_info->type == MBTK_DATA_CALL_START) {
                        mbtk_ip_info_t ip_info;
                        memset(&ip_info, 0, sizeof(ip_info));

                        if(!info_list[call_info->cid - 1].apn_info.redial_enable
                            && info_list[call_info->cid - 1].act_state == RIL_ACT_STATE_CONNECTED) {
                            err = MBTK_RIL_ERR_DATA_CONNECTED;
                            LOGW("Data call is connected.");
                            break;
                        }
#if 0
                        if(apn_prop_reset(call_info)) {
                            err = MBTK_RIL_ERR_REQ_UNKNOWN;
                            LOG("apn_prop_reset() fail.");
                        } else
#else
                        if(apn_prop_reset(cli_info->sim_id, call_info)) {
                            LOG("apn_prop_reset() fail.");
                        }
#endif
                        {
                            int index = 0;
                            info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_CONNECTING;
data_call_retry:
                            if(req_data_call_start(cli_info->sim_id, cli_info->port, call_info->cid, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                            {
                                if(call_info->retry_interval[index] > 0) {
                                    LOGD("Start data call fail, will retry in %d s.", call_info->retry_interval[index]);
                                    sleep(call_info->retry_interval[index]);
                                    index++;
                                    goto data_call_retry;
                                } else {
                                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                        err = MBTK_RIL_ERR_CME + cme_err;
                                    } else {
                                        err = MBTK_RIL_ERR_UNKNOWN;
                                    }
                                    LOGD("Start data call fail.");
                                }
                            }
                            else
                            {
                                // Wait for "CONNECT" or "+CGEV:"
                                if(wait_cgact_complete(call_info->timeout)) { // Timeout
                                    err = MBTK_RIL_ERR_TIMEOUT;
                                    break;
                                }

                                // Get Ip informations.
                                cme_err = MBTK_RIL_ERR_CME_NON;
                                if(req_data_call_state_get(cli_info->sim_id, cli_info->port, call_info->cid ,&ip_info, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                                {
                                    LOGE("Get net informations fail.");
                                    err = MBTK_RIL_ERR_NET_CONF;
                                }
                                else
                                {
                                    // Config network informations.
                                    if(net_ifc_reconfig(cli_info->sim_id, call_info->cid, info_list[call_info->cid - 1].apn_info.def_route,
                                                        info_list[call_info->cid - 1].apn_info.as_dns, &ip_info)) {
                                        err = MBTK_RIL_ERR_NET_CONF;
                                        break;
                                    }

                                    info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_CONNECTED;

                                    // Data call success, valid will be true.
                                    info_list[call_info->cid - 1].valid = TRUE;

                                    ril_rsp_pack_send(cli_info->sim_id, cli_info->port, cli_info->fd, pack->msg_id, pack->msg_index, &ip_info, sizeof(mbtk_ip_info_t));
                                }
                            }
                        }
                    } else if(call_info->type == MBTK_DATA_CALL_STOP) {
                        if(!info_list[call_info->cid - 1].apn_info.redial_enable
                            && info_list[call_info->cid - 1].act_state == RIL_ACT_STATE_DISCONNECTED) {
                            err = MBTK_RIL_ERR_DATA_DISCONNECTED;
                            LOGW("Data call is disconnected.");
                            break;
                        }

                        info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_DISCONNECTING;
                        if(req_data_call_stop(cli_info->sim_id, cli_info->port, call_info->cid, call_info->timeout, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                        {
                            if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                err = MBTK_RIL_ERR_CME + cme_err;
                            } else {
                                err = MBTK_RIL_ERR_UNKNOWN;
                            }
                            LOGD("Stop data call fail.");
                        }
                        else
                        {
                            // Wait for "CONNECT" or "+CGEV:"
                            if(wait_cgact_complete(call_info->timeout)) { // Timeout
                                err = MBTK_RIL_ERR_TIMEOUT;
                                break;
                            }

                            // Clean network config.
                            if(net_ifc_config(cli_info->sim_id, call_info->cid, FALSE, FALSE, NULL)) {
                                err = MBTK_RIL_ERR_NET_CONF;
                                break;
                            }

                            info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_DISCONNECTED;
                            ril_rsp_pack_send(cli_info->sim_id, cli_info->port, cli_info->fd, pack->msg_id, pack->msg_index, NULL, 0);
                        }
                    } else {
                        mbtk_ip_info_t ip_info;
                        memset(&ip_info, 0, sizeof(ip_info));
                        if(req_data_call_state_get(cli_info->sim_id, cli_info->port, call_info->cid ,&ip_info, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                        {
                            if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                err = MBTK_RIL_ERR_CME + cme_err;
                            } else {
                                err = MBTK_RIL_ERR_UNKNOWN;
                            }
                            LOGD("Get data call state fail.");
                        }
                        else
                        {
                            ril_rsp_pack_send(cli_info->sim_id, cli_info->port, cli_info->fd, pack->msg_id, pack->msg_index, &ip_info, sizeof(mbtk_ip_info_t));
                        }
                    }
                }
            }
            break;
        }
        default:
        {
            err = MBTK_RIL_ERR_REQ_UNKNOWN;
            LOG("Unknown request : %s", id2str(pack->msg_id));
            break;
        }
    }

    return err;
}


