/*
*    mbtk_gnss.c
*
*    MBTK GNSS API source.
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN        WHO       WHAT,WHERE,WHY
--------    --------    -------------------------------------------------------
2024/7/11     LiuBin      Initial version

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <sys/epoll.h>
#include <time.h>

#include "mbtk_gnss_inter.h"
#include "mbtk_log.h"
#include "mbtk_utils.h"

#define GNSS_BUFF_SIZE 2048

static int gnss_cli_fd = -1;
static pthread_t read_thread_id;
static int exit_fd[2] = {-1};
static mbtk_gnss_callback_func gnss_cb = NULL;
static bool gnss_busy = FALSE;
static pthread_cond_t gnss_cond;
static pthread_mutex_t gnss_mutex;
static gnss_err_enum gnss_result;

#if 1//MBTK_GNSS_LOCATION_INFO
static mbtk_gnss_location_info_t locl_info;

extern long timezone;
#endif

static int sock_read(int fd, uint8 *msg, int data_len)
{
    memset(msg, 0, data_len);
    int len = 0;

    int read_len = 0;
    while(1)
    {
        len = read(fd, msg + read_len, data_len - read_len);
        if(len > 0)
        {
            read_len += len;
        }
        else if(len == 0)
        {
            LOGE("read() end.");
            break;
        }
        else
        {
            if(EAGAIN == errno)
            {
                LOGV("Read end, lenght = %d", read_len);
            }
            else
            {
                LOGE("read() error[%d].", errno);
            }
            break;
        }
    }

    if(read_len > 0)
    {
        // log_hex("DATA_RECV", msg, read_len);
        return read_len;
    }
    else
    {
        return -1;
    }
}

static int sock_write(int fd, uint8 *msg, int data_len)
{
    int len = 0;
    int write_len = 0;
    while(write_len < data_len)
    {
        len = write(fd, msg + write_len, data_len - write_len);
        if(len > 0)
        {
            write_len += len;
        }
        else if(len == 0)
        {
            LOGE("write() end.");
            break;
        }
        else
        {
            LOGE("write() error[%d].", errno);
            break;
        }
    }

    if(write_len > 0)
    {
        //log_hex("DATA_SEND", msg, write_len);
        return write_len;
    }
    else
    {
        return -1;
    }
}

static void gnss_timer_cb(int signo)
{
    if(gnss_busy) {
        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_signal(&gnss_cond);
        pthread_mutex_unlock(&gnss_mutex);
        gnss_result = GNSS_ERR_TIMEOUT;
    }
    return;
}

#if 1//MBTK_GNSS_LOCATION_INFO
static int strstr_n(const char *s1, const char *s2)
{
    int n;
    int strlen = 0;

    if(*s2)
    {
        while(*s1)
        {
            for(n = 0; *(s1+n) == *(s2 + n); n++)
            {
                if(!*(s2 + n + 1))
                {
                    strlen++;
                    return strlen;
                }
            }
            s1++;
            strlen++;
        }
        return 0;
    }

    return 0;
}

static int str2int( const char* head, const char* end )
{
    int result = 0;
    int len    = end - head;

    for(; len > 0; len--, head++)
    {
        int  c;
        if (head >= end)
        {
            goto Fail;
        }

        c = *head - '0';
        if ((unsigned)c >= 10)
        {
            goto Fail;
        }
        result = result * 10 + c;
    }
    return result;
Fail:
    return -1;
}

static double str2float( const char* head, const char* end )
{
    int len = end - head;
    char temp[16];

    if(len >= (int)sizeof(temp))
    {
        return 0;
    }

    memcpy( temp, head, len );
    temp[len] = 0;
    return strtod(temp, NULL);
}

static mbtk_token nmea_tokenizer_get(mbtk_nmeatokenizer* t, int  index)
{
    mbtk_token tok;
    static const char*  dummy = "";

    if (index < 0 || index >= t->count)
    {
        tok.head = tok.end = dummy;
    }
    else
    {
        tok = t->tokens[index];
    }
    return tok;
}

static int nmea_tokenizer_init(mbtk_nmeatokenizer* t, const char* head, const char* end, int param_num)
{
    int count = 0;
    const char* p = head;
    const char* q = end;
    const char* tmp = NULL;
    // the initial '$' is optional
    if (p < q && p[0] == '$')
    {
        p += 1;
    }
    else
    {
        return -1;
    }

    //find '*',del '*25\r\n'
    // get rid of checksum at the end of the sentecne
    if (q >= p + 5 && q[-5] == '*')
    {
        q -= 5;
    }
    else
    {
        return -1;
    }

    while (p <= q)
    {
        tmp = memchr(p, ',', q-p);
        if (tmp == NULL)
        {
            tmp = q;
        }
        // if (q > p) {
        // q >= p include empty token: ,,
        if (tmp >= p)
        {
            if (count < MAX_NMEA_TOKENS)
            {
                t->tokens[count].head = p;
                t->tokens[count].end = tmp;
                count += 1;
            }
        }

        if (tmp <= q)
        {
            tmp += 1;
        }

        p = tmp;
    }

    if(count != param_num)
    {
        LOGD("count [%d], param_num [%d]", count, param_num);
        return -1;
    }

    t->count = count;
    return count;
}

static int nmea_update_date_time(mbtk_gnss_location_info_t* locl_info, mbtk_token date, mbtk_token time)
{
    char tmp_char[4] = {0};
    struct tm tmp_time;

    memset(&tmp_time, 0x0, sizeof(struct tm));
    if (date.head + 6 > date.end)
    {
        LOGD("date get fail");
        return -1;
    }

    memcpy(tmp_char, date.head, 2);
    tmp_time.tm_mday = atoi(tmp_char);
    memcpy(tmp_char, date.head + 2, 2);
    tmp_time.tm_mon = atoi(tmp_char) - 1;
    memcpy(tmp_char, date.head + 4, 2);
    tmp_time.tm_year = 100 + atoi(tmp_char);

    if (time.head + 6 > time.end)
    {
        LOGD("time get fail");
        return -1;
    }

    memcpy(tmp_char, time.head, 2);
    tmp_time.tm_hour = atoi(tmp_char);
    memcpy(tmp_char, time.head + 2, 2);
    tmp_time.tm_min = atoi(tmp_char);
    memcpy(tmp_char, time.head + 4, 2);
    tmp_time.tm_sec = atoi(tmp_char);
    tmp_time.tm_isdst = -1;

#if MBTK_GNSS_LOG_ENABLED
    LOGD("data:%d-%d-%d %d:%d:%d", tmp_time.tm_year + 1900,
                                    tmp_time.tm_mon,
                                    tmp_time.tm_mday,
                                    tmp_time.tm_hour,
                                    tmp_time.tm_min,
                                    tmp_time.tm_sec);
#endif

    time_t _t = mktime(&tmp_time);//parse location tmp_time
#if MBTK_GNSS_LOG_ENABLED
    LOGD("time: %ld", _t);
#endif
    tzset();   // auto set tz
    _t = _t - timezone;
    locl_info->timestamp = (int64_t)_t;
#if MBTK_GNSS_LOG_ENABLED
    LOGD("timestamp: %ld, %ld", locl_info->timestamp, timezone);
#endif
    return 0;
}

static int nmea_update_bearing(mbtk_gnss_location_info_t* locl_info, mbtk_token         bearing)
{
    mbtk_token tok = bearing;

    if (tok.head >= tok.end)
    {
        LOGD("bearing get fail");
        return -1;
    }

    locl_info->bearing = str2float(tok.head, tok.end);
    return 0;
}

static int nmea_update_speed(mbtk_gnss_location_info_t* locl_info,        mbtk_token speed)
{
    mbtk_token tok = speed;

    if (tok.head >= tok.end)
    {
        LOGD("speed get fail");
        return -1;
    }

    locl_info->speed = str2float(tok.head, tok.end);
    return 0;
}

static int nmea_update_latlong(mbtk_gnss_location_info_t* locl_info, mbtk_token         latitude, mbtk_token longitude)
{
    double   lat, lon;
    mbtk_token    tok;
    tok = latitude;
    if (tok.head + 6 > tok.end)
    {
        LOGD("latitude get fail");
        return -1;
    }
    locl_info->latitude = str2float(tok.head, tok.end);

    tok = longitude;
    if (tok.head + 6 > tok.end)
    {
        LOGD("longitude get fail");
        return -1;
    }
    locl_info->longitude = str2float(tok.head, tok.end);
    return 0;
}

static int nmea_update_altitude(mbtk_gnss_location_info_t* locl_info, mbtk_token          altitude)
{
    mbtk_token tok = altitude;

    if (tok.head >= tok.end)
    {
        LOGD("altitude get fail");
        return -1;
    }

    locl_info->altitude = str2float(tok.head, tok.end);
    return 0;
}

static int ind_nmea_parse(const char *data, int data_len, mbtk_gnss_location_info_t* locl_info)
{
    int ret;
    mbtk_nmeatokenizer tzer = {0};
    if(strstr_n(data + 3, "RMC"))
    {
#if MBTK_GNSS_LOG_ENABLED
        LOGD("ind data_len: [%d] data: %s", data_len, data);
#endif
        ret = nmea_tokenizer_init(&tzer, data, data + data_len, NMEA_RMC_PARAM_NUM);
        if(ret < 0)
        {
            LOGD("nmea_tokenizer_init fail");
            return -1;
        }

        mbtk_token  tok_time      = nmea_tokenizer_get(&tzer,1);
        mbtk_token  tok_fixStatus = nmea_tokenizer_get(&tzer,2);
        mbtk_token  tok_speed     = nmea_tokenizer_get(&tzer,7);
        mbtk_token  tok_bearing   = nmea_tokenizer_get(&tzer,8);
        mbtk_token  tok_date      = nmea_tokenizer_get(&tzer,9);

        if(tok_fixStatus.head[0] == 'A')
        {
            ret = nmea_update_date_time(locl_info, tok_date, tok_time);
            if(ret < 0)
            {
                LOGD("nmea_update_date_time fail");
                return -1;
            }
            locl_info->flags |= GNSS_LOCATION_HAS_TIMESTAMP;

            ret = nmea_update_bearing(locl_info, tok_bearing);
            if(ret < 0)
            {
                LOGD("nmea_update_bearing fail");
                return -1;
            }
            locl_info->flags |= GNSS_LOCATION_HAS_BEARING;

            ret = nmea_update_speed(locl_info, tok_speed);
            if(ret < 0)
            {
                LOGD("nmea_update_speed fail");
                return -1;
            }
            locl_info->flags |= GNSS_LOCATION_HAS_SPEED;
        }
    }
    else if(strstr_n(data + 3, "GGA"))
    {
#if MBTK_GNSS_LOG_ENABLED
        LOGD("ind data_len: [%d] data: %s", data_len, data);
#endif
        ret = nmea_tokenizer_init(&tzer, data, data + data_len, NMEA_GGA_PARAM_NUM);
        if(ret < 0)
        {
            LOGD("nmea_tokenizer_init fail");
            return -1;
        }

        mbtk_token  tok_latitude  = nmea_tokenizer_get(&tzer,2);
        mbtk_token  tok_longitude = nmea_tokenizer_get(&tzer,4);
        mbtk_token  tok_isPix     = nmea_tokenizer_get(&tzer,6);
        mbtk_token  tok_altitude  = nmea_tokenizer_get(&tzer,9);

        if(tok_isPix.head[0] > '0' && tok_isPix.head[0] < '9')
        {
            ret = nmea_update_latlong(locl_info, tok_latitude, tok_longitude);
            if(ret < 0)
            {
                LOGD("nmea_update_latlong fail");
                return -1;
            }
            locl_info->flags |= GNSS_LOCATION_HAS_LAT_LONG;

            ret = nmea_update_altitude(locl_info, tok_altitude);
            if(ret < 0)
            {
                LOGD("nmea_update_altitude fail");
                return -1;
            }
            locl_info->flags |= GNSS_LOCATION_HAS_ALTITUDE;
        }
    }
    else
    {
#if MBTK_GNSS_LOG_ENABLED
        LOGD("ind data_len: [%d] data: %s", data_len, data);
#endif
    }
    return 0;
}
#endif


static void gnss_rsp_process(const char *data, int data_len)
{
    int index = 0;
    char buff[GNSS_BUFF_SIZE];
    int buff_len = 0;
    int ret = 0;
    int tag_len = 0;
    while(index < data_len) {
        if(data[index] == MBTK_IND_START_FLAG) {
            memset(buff, 0, sizeof(buff));
            buff_len = 0;
        } else if(data[index] == MBTK_IND_END_FLAG) {
            buff[buff_len] = '\0';
            if(memcmp(MBTK_IND_LOCATION_TAG, buff, strlen(MBTK_IND_LOCATION_TAG)) == 0) {
                if(gnss_cb) {
#if 1//MBTK_GNSS_LOCATION_INFO
                    tag_len = strlen(MBTK_IND_LOCATION_TAG);
                    ret = ind_nmea_parse(buff + tag_len, buff_len - tag_len, &locl_info);
#if MBTK_GNSS_LOG_ENABLED
                    LOGD("gnss_cb: [%p], locl_info.flags [%d]", gnss_cb, locl_info.flags);
#endif
                    if(locl_info.flags == GNSS_LOCATION_HAS_ALL) {
                        gnss_cb(MBTK_GNSS_IND_LOCATION, &locl_info, sizeof(mbtk_gnss_location_info_t));
                        locl_info.flags = 0;
                    }
#endif
                }
            } else if(memcmp(MBTK_IND_NMEA_TAG, buff, strlen(MBTK_IND_NMEA_TAG)) == 0) {
                if(gnss_cb) {
                    tag_len = strlen(MBTK_IND_NMEA_TAG);
                    gnss_cb(MBTK_GNSS_IND_NMEA, buff + tag_len, buff_len - tag_len);
                }
            } else {
                LOGD("RSP[len - %d] : %s", buff_len, buff);
                if(gnss_busy) {
                    // XXXXXX:<result>
                    char *ptr = strstr(buff, ":");
                    if(ptr) {
                        gnss_result = atoi(ptr + 1);
                    }
                    pthread_mutex_lock(&gnss_mutex);
                    pthread_cond_signal(&gnss_cond);
                    pthread_mutex_unlock(&gnss_mutex);
                }
            }
        } else {
            buff[buff_len++] = data[index];
        }
        index++;
    }
}

static void* gnss_read_run(void* arg)
{
    int epoll_fd = epoll_create(5);
    if(epoll_fd < 0)
    {
        LOGE("epoll_create() fail[%d].", errno);
        return NULL;
    }

    uint32 event = EPOLLIN | EPOLLET;
    struct epoll_event ev_cli, ev_exit;
    ev_cli.data.fd = gnss_cli_fd;
    ev_cli.events = event; //EPOLLIN | EPOLLERR | EPOLLET;
    epoll_ctl(epoll_fd,EPOLL_CTL_ADD,gnss_cli_fd,&ev_cli);

    ev_exit.data.fd = exit_fd[0];
    ev_exit.events = event; //EPOLLIN | EPOLLERR | EPOLLET;
    epoll_ctl(epoll_fd,EPOLL_CTL_ADD,exit_fd[0],&ev_exit);

    int nready = -1;
    struct epoll_event epoll_events[EPOLL_LISTEN_MAX];
    while(1)
    {
        nready = epoll_wait(epoll_fd, epoll_events, EPOLL_LISTEN_MAX, -1);
        if(nready > 0)
        {
            int i;
            for(i = 0; i < nready; i++)
            {
                LOGV("fd[%d] event = %x",epoll_events[i].data.fd, epoll_events[i].events);
                if(epoll_events[i].events & EPOLLHUP)   // Closed by server.
                {

                }
                else if(epoll_events[i].events & EPOLLIN)
                {
                    if(gnss_cli_fd == epoll_events[i].data.fd)  // Server data arrive.
                    {
                        char buff[GNSS_BUFF_SIZE + 1];
                        int len = sock_read(gnss_cli_fd, buff, GNSS_BUFF_SIZE);
                        if(len > 0) {
                            gnss_rsp_process(buff, len);
                        }
                    }
                    else if(exit_fd[0] == epoll_events[i].data.fd) //
                    {
                        char buff[100] = {0};
                        int len = read(exit_fd[0], buff, 100);
                        if(len > 0) {
                            LOGI("CMD : %s", buff);
                            if(strcmp(buff, "EXIT") == 0) {
                                goto read_thread_exit;
                            } else {
                                LOGD("Unkonw cmd : %s", buff);
                            }
                        } else {
                            LOGE("sock_read() fail.");
                        }
                    }
                    else
                    {
                        LOGW("Unknown socket : %d", epoll_events[i].data.fd);
                    }
                }
                else
                {
                    LOGW("Unknown event : %x", epoll_events[i].events);
                }
            }
        }
        else
        {
            LOGW("epoll_wait() fail[%d].", errno);
        }
    }

read_thread_exit:
    LOGD("info_read thread exit.");
    return NULL;
}

gnss_err_enum mbtk_gnss_init(mbtk_gnss_callback_func cb)
{
    if(gnss_cli_fd > 0) {
        LOGW("GNSS client has inited.");
        return GNSS_ERR_OK;
    }

    gnss_cli_fd = socket(AF_LOCAL, SOCK_STREAM, 0);
    if(gnss_cli_fd < 0)
    {
        LOGE("socket() fail[%d].", errno);
        goto error;
    }

    // Set O_NONBLOCK
    int flags = fcntl(gnss_cli_fd, F_GETFL, 0);
    if (flags < 0)
    {
        LOGE("Get flags error:%d", errno);
        goto error;
    }
    flags |= O_NONBLOCK;
    if (fcntl(gnss_cli_fd, F_SETFL, flags) < 0)
    {
        LOGE("Set flags error:%d", errno);
        goto error;
    }

    struct sockaddr_un cli_addr;
    memset(&cli_addr, 0, sizeof(cli_addr));
    cli_addr.sun_family = AF_LOCAL;
    strcpy(cli_addr.sun_path, SOCK_GNSS_PATH);
    if(connect(gnss_cli_fd, (struct sockaddr *)&cli_addr, sizeof(cli_addr)))
    {
        LOGE("connect() fail[%d].", errno);
        goto error;
    }

    if(pipe(exit_fd)) {
        LOGE("pipe() fail[%d].", errno);
        goto error;
    }

    if(pthread_create(&read_thread_id, NULL, gnss_read_run, NULL))
    {
        LOGE("pthread_create() fail.");
        goto error;
    }
    pthread_mutex_init(&gnss_mutex, NULL);
    pthread_cond_init(&gnss_cond, NULL);
    gnss_cb = cb;
    return GNSS_ERR_OK;

error:
    if(gnss_cli_fd > 0) {
        close(gnss_cli_fd);
        gnss_cli_fd = -1;
    }
    if(exit_fd[0] > 0) {
        close(exit_fd[0]);
        exit_fd[0] = -1;
    }
    if(exit_fd[1] > 0) {
        close(exit_fd[1]);
        exit_fd[1] = -1;
    }

    return GNSS_ERR_UNKNOWN;
}

gnss_err_enum mbtk_gnss_deinit()
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    }

    if(exit_fd[1] > 0) {
        write(exit_fd[1], "EXIT", 4);
    }

    int ret = pthread_join(read_thread_id, NULL);
    if(ret){
        LOGE("pthrad_join fail(%d)",ret);
        return GNSS_ERR_UNKNOWN;
    }

    close(gnss_cli_fd);
    gnss_cli_fd = -1;
    close(exit_fd[0]);
    exit_fd[0] = -1;
    close(exit_fd[1]);
    exit_fd[1] = -1;
    gnss_cb = NULL;
    gnss_busy = FALSE;
    pthread_mutex_destroy(&gnss_mutex);
    pthread_cond_destroy(&gnss_cond);

    return GNSS_ERR_OK;
}

/*===========================================================================
FUNCTION mbtk_gnss_open

DESCRIPTION:
  Open/Close GNSS.

PARAMETERS:
  type [IN]: How to turn on or off GNSS,The available values are as follows:
           0 : Close GNSS;
           1-15 : Open GNSS with NMEA print;
              (GNSS_PRINT_PORT_UART1 / GNSS_PRINT_PORT_USB_NMEA / GNSS_PRINT_PORT_USB_AT / GNSS_PRINT_PORT_TTY_AT)
           Other : Open GNSS without print NMEA
  timeout [IN]: Timeout with second.

RETURN VALUE:
  gnss_err_enum

===========================================================================*/
gnss_err_enum mbtk_gnss_open(int type, int timeout)
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    } else {
        gnss_result = GNSS_ERR_OK;
        gnss_busy = TRUE;
        if(timeout > 0) {
            mbtk_timer_set(gnss_timer_cb, timeout * 1000);
        }

        // gnss_init:x
        char cmd[32] = {0};
        snprintf(cmd, sizeof(cmd), "gnss_init:%d", type);
        sock_write(gnss_cli_fd, cmd, strlen(cmd));

        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_wait(&gnss_cond, &gnss_mutex);
        pthread_mutex_unlock(&gnss_mutex);

        gnss_busy = FALSE;

        return gnss_result;
    }
}

gnss_err_enum mbtk_gnss_close(int timeout)
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    } else {
        gnss_result = GNSS_ERR_OK;
        gnss_busy = TRUE;
        if(timeout > 0) {
            mbtk_timer_set(gnss_timer_cb, timeout * 1000);
        }

        // gnss_deinit
        char *cmd = "gnss_deinit";
        sock_write(gnss_cli_fd, cmd, strlen(cmd));

        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_wait(&gnss_cond, &gnss_mutex);
        pthread_mutex_unlock(&gnss_mutex);

        gnss_busy = FALSE;

        return gnss_result;
    }
}


gnss_err_enum mbtk_gnss_setting(const char *setting_cmd, int timeout)
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    } else {
        gnss_result = GNSS_ERR_OK;
        gnss_busy = TRUE;
        if(timeout > 0) {
            mbtk_timer_set(gnss_timer_cb, timeout * 1000);
        }

        // gnss_setting:cmd
        char cmd[1024] = {0};
        snprintf(cmd, sizeof(cmd), "gnss_setting:%s", setting_cmd);
        sock_write(gnss_cli_fd, cmd, strlen(cmd));

        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_wait(&gnss_cond, &gnss_mutex);
        pthread_mutex_unlock(&gnss_mutex);

        gnss_busy = FALSE;

        return gnss_result;
    }
}

gnss_err_enum mbtk_gnss_dl(const char *fw_path, int timeout)
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    } else {
        gnss_result = GNSS_ERR_OK;
        gnss_busy = TRUE;
        if(timeout > 0) {
            mbtk_timer_set(gnss_timer_cb, timeout * 1000);
        }

        // gnss_dl:fw_name
        char cmd[32] = {0};
        snprintf(cmd, sizeof(cmd), "gnss_dl:%s", fw_path);
        sock_write(gnss_cli_fd, cmd, strlen(cmd));

        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_wait(&gnss_cond, &gnss_mutex);
        pthread_mutex_unlock(&gnss_mutex);

        gnss_busy = FALSE;

        return gnss_result;
    }
}

/**
* Set GNSS data callback.
*
* gnss_ind : MBTK_GNSS_IND_LOCATION / MBTK_GNSS_IND_NMEA
* timeout : Timeout with second.
*
*/
gnss_err_enum mbtk_gnss_ind_set(uint32 gnss_ind, int timeout)
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    } else {
        gnss_result = GNSS_ERR_OK;
        gnss_busy = TRUE;
        if(timeout > 0) {
            mbtk_timer_set(gnss_timer_cb, timeout * 1000);
        }

        // gnss_ind:ind_type
        char cmd[32] = {0};
        snprintf(cmd, sizeof(cmd), "gnss_ind:%d", gnss_ind);
        sock_write(gnss_cli_fd, cmd, strlen(cmd));

        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_wait(&gnss_cond, &gnss_mutex);
        pthread_mutex_unlock(&gnss_mutex);

        gnss_busy = FALSE;

        return gnss_result;
    }
}

gnss_err_enum mbtk_gnss_eph_download(int timeout)
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    } else {
        gnss_result = GNSS_ERR_OK;
        gnss_busy = TRUE;
        if(timeout > 0) {
            mbtk_timer_set(gnss_timer_cb, timeout * 1000);
        }

        // gnss_agnss_get:<eph_data>,<alam_flag>
        char cmd[32] = {0};
        snprintf(cmd, sizeof(cmd), "gnss_agnss_get:%d,0", (int)GNSS_EPH_CFG);
        sock_write(gnss_cli_fd, cmd, strlen(cmd));

        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_wait(&gnss_cond, &gnss_mutex);
        pthread_mutex_unlock(&gnss_mutex);

        gnss_busy = FALSE;

        return gnss_result;
    }
}

gnss_err_enum mbtk_gnss_eph_inject(int timeout)
{
    if(gnss_cli_fd < 0) {
        LOGW("GNSS client not inited.");
        return GNSS_ERR_UNKNOWN;
    }

    if(gnss_busy) {
        LOGE("BUSY");
        return GNSS_ERR_BUSY;
    } else {
        gnss_result = GNSS_ERR_OK;
        gnss_busy = TRUE;
        if(timeout > 0) {
            mbtk_timer_set(gnss_timer_cb, timeout * 1000);
        }

        // gnss_agnss_set
        char cmd[32] = {0};
        snprintf(cmd, sizeof(cmd), "gnss_agnss_set");
        sock_write(gnss_cli_fd, cmd, strlen(cmd));

        pthread_mutex_lock(&gnss_mutex);
        pthread_cond_wait(&gnss_cond, &gnss_mutex);
        pthread_mutex_unlock(&gnss_mutex);

        gnss_busy = FALSE;

        return gnss_result;
    }
}

void mbtk_gnss_lib_info_print()
{
    MBTK_SOURCE_INFO_PRINT("mbtk_gnss_lib");
}

