#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include "log_config.h"

#define  ALOG_DEV "/dev/log_radio"
#define LOG_CONFIG_LEN 50

typedef enum android_LogPriority {
    ANDROID_LOG_UNKNOWN = 0,
    ANDROID_LOG_DEFAULT,    /* only for SetMinPriority() */
    ANDROID_LOG_VERBOSE,
    ANDROID_LOG_DEBUG,
    ANDROID_LOG_INFO,
    ANDROID_LOG_WARN,
    ANDROID_LOG_ERROR,
    ANDROID_LOG_FATAL,
    ANDROID_LOG_SILENT,     /* only for SetMinPriority(); must be last */
} android_LogPriority;

typedef struct AndroidLogEntry_t {
    time_t tv_sec;
    long tv_nsec;
    android_LogPriority priority;
    int32_t pid;
    int32_t tid;
    const char * tag;
    size_t messageLen;
    const char * message;
} AndroidLogEntry;

//static const char *log_file, *log_ip, *log_port, *log_prefix, *pid_file, *hostname;
static char log_file[LOG_CONFIG_LEN], log_ip[LOG_CONFIG_LEN], log_port[LOG_CONFIG_LEN], log_prefix[LOG_CONFIG_LEN], pid_file[LOG_CONFIG_LEN], hostname[LOG_CONFIG_LEN];
static int log_size = 1 * 1024 * 1024;

static log_config_entry *config = NULL;
static char tmp_log[48] = {0};

static int fd_radio = 0;

void hex_print(char* buf, int len)
{
    int i;

    for (i = 0; i < len; i++) {
        printf("%x ", buf[i]);
    }
}

static char filterPriToChar(android_LogPriority pri)
{
    switch (pri) {
        case ANDROID_LOG_VERBOSE:
            return 'V';
        case ANDROID_LOG_DEBUG:
            return 'D';
        case ANDROID_LOG_INFO:
            return 'I';
        case ANDROID_LOG_WARN:
            return 'W';
        case ANDROID_LOG_ERROR:
            return 'E';
        case ANDROID_LOG_FATAL:
            return 'F';
        case ANDROID_LOG_SILENT:
            return 'S';

        case ANDROID_LOG_DEFAULT:
        case ANDROID_LOG_UNKNOWN:
        default:
            return '?';
    }
}

static android_LogPriority filterCharToPri(char c)
{
    switch (c) {
        case 'v':
            return ANDROID_LOG_VERBOSE;
        case 'd':
            return ANDROID_LOG_DEBUG;
        case 'i':
            return ANDROID_LOG_INFO;
        case 'w':
            return ANDROID_LOG_WARN;
        case 'e':
            return ANDROID_LOG_ERROR;
        case 'f':
            return ANDROID_LOG_FATAL;
        case 's':
            return ANDROID_LOG_SILENT;
        case '*':
        default:
            return ANDROID_LOG_VERBOSE;
    }
}

int fileter_log(int pri, char *tag, struct filter_list_t *filter)
{
    struct filter_list_t *_filter = filter;

    while(_filter)
    {
        int p = filterCharToPri(_filter->priority);
        if(_filter->tag)
        {
            int len = strlen(_filter->tag);
            // tag and priority
            if(0 == memcmp(_filter->tag, tag, len) && ((pri > p) || (pri == p)))
                return 0;
        }else{ // have no tag
            if(pri < p)
                return -1;
            else
                return 0;
        }
        _filter = _filter->next;
    }

    return -1;
}

int alog_process(char* data, int len, AndroidLogEntry *entry)
{
    int i, n = 0;
    int tmp_len;

    for (i = 0; i < len;) {
        if (data[i] == '\0') {
            i += 1;
        } else {
            switch (n) {
                case 0: {
                    // printf("%d - %d: %x\n", i, tmp_len, data[i]);
                    i++;
                    break;
                }
                case 1: {
                    int* pid = (int*)&data[i];
                    entry->pid = *pid;
                    // printf("%d pid: %d\n", i, entry->pid);
                    i += 4;
                    break;
                }
                case 2: {
                    int* tid = (int*)&data[i];
                    entry->tid = *tid;
                    // printf("%d tid: %d\n", i, entry->tid);
                    i += 4;
                    break;
                }
                case 3: {
                    // printf("%d - %d: %x %x %x %x\n", i, tmp_len, data[i], data[i + 1], data[i + 2], data[i + 3]);
                    time_t* _t = (time_t*)&data[i];
                    entry->tv_sec = *_t;
                    i += 8;
                    break;
                }
                case 4: {
                    entry->priority = data[i];
                    entry->tag = &data[i + 1];
                    i += strlen(&data[i]);
                    break;
                }
                //* format: <priority:1><tag:N>\0<message:N>\0
                case 5: {
                    entry->message = &data[i];
                    entry->messageLen = strlen(&data[i]);
                    i += entry->messageLen;
                    break;
                }
                default:
                    printf("process error \n");
                    break;
            }
            n++;
        }
    }
    return 0;
}

int android_log_printLogLine(
    struct file_list_t *_file_list,
    const AndroidLogEntry *entry)
{
    char priChar;
    char timeBuf[32];
    char defaultBuffer[512];
    size_t totalLen;
    int index = 0;
    int len = 0;
    struct stat s;
    char * ret = NULL;
    
    if(access(tmp_log, W_OK) != 0)
    {
        fd_radio = open(tmp_log, O_CREAT | O_WRONLY| O_APPEND, 0600);
        if (fd_radio < 0) 
        {
            fprintf(stderr, "failed to open %s: %s\n", tmp_log, strerror(errno));
            exit(-1);
        }
        
    }
    
    if (log_size && (!stat(tmp_log, &s)) && (s.st_size > log_size))
    {
        fd_radio = get_rotate_file(fd_radio, log_file, _file_list);
        if (fd_radio < 0) 
        {
            fprintf(stderr, "failed to open %s: %s\n", log_file, strerror(errno));
            exit(-1);
        }
    }

    if(fileter_log(entry->priority, entry->tag, config->filter_list))
    {
        // printf("%s %d: fileter pri:%d tag:%s!\n", __FUNCTION__, __LINE__, entry->priority, entry->tag);
        return -1;
    }

    priChar = filterPriToChar(entry->priority);
    struct tm* ptm = localtime(&entry->tv_sec);
    strftime(timeBuf, sizeof(timeBuf), "%Y-%m-%d %H:%M:%S", ptm);

    totalLen = snprintf(defaultBuffer, sizeof(defaultBuffer),
    "%s %c/%s (%d): %s\n", timeBuf, priChar, entry->tag, entry->pid, entry->message);
    
    len = strlen(defaultBuffer);
    if(access("/etc/syslog_encrypt_flag", F_OK) == 0)
    {
        for(index = 0; index < len; index++)
        {
            defaultBuffer[index] ^= 1;
        }
    }
    ret = write(fd_radio, defaultBuffer, totalLen);

    return ret;
}

void* alog_thread(void* argv)
{
    int dev_fd, ret;
    int log_fd;
    AndroidLogEntry entry;
    char buf[512] = {0};
    static struct file_list_t file_list;
    config = (log_config_entry *)argv;

    pthread_detach(pthread_self());
    if (NULL == argv)
        return NULL;

    dev_fd = open(ALOG_DEV, O_RDONLY, 0600);
    if (dev_fd < 0) {
        fprintf(stderr, "failed to open %s: %s\n", ALOG_DEV, strerror(errno));
        exit(-1);
    }

    memset(&file_list, 0, sizeof(struct file_list_t));
    file_list.total = config->rotate_file_count;
    //log_file = config->out_path;
	memset(log_file, 0, sizeof(log_file));
	memset(log_ip, 0, sizeof(log_ip));
	memset(log_port, 0, sizeof(log_port));
	memset(log_prefix, 0, sizeof(log_prefix));
	memset(pid_file, 0, sizeof(pid_file));
	memset(hostname, 0, sizeof(hostname));
    if(config->out_path != NULL)
	{
		strncpy(log_file, config->out_path, LOG_CONFIG_LEN - 1);
	}

    if (config->ip && config->port) 
    {
        int port = atoi(config->port);
        printf("%s %d : %s:%s\n", __FUNCTION__, __LINE__, config->ip, config->port);
        log_fd = tcp_connect(config->ip, port);
    } 
    else if (strlen(log_file) > 0) 
    {
        //sprintf(tmp_log, "/tmp/log%s", strstr_tail(log_file, "/"));
        
        snprintf(tmp_log,sizeof(tmp_log), "%s", log_file);
        // 先将文件保存到 /tmp/log/ 目录下，后面到达 rotate_file_size 后，转移到out_path
       
    } 
    else 
    {
        log_fd = STDOUT_FILENO;
    }
    if(config->rotate_file_size)
    {
        log_size = config->rotate_file_size;
    }
    printf("android log start...\n");
    while (1) 
    {
        ret = read(dev_fd, buf, sizeof(buf));
        if (ret < 0) 
        {
            printf("read error\n");
            break;
        }
        alog_process(buf, ret, &entry);
        android_log_printLogLine(&file_list, &entry);
        memset(buf, 0, sizeof(buf));
    }
    close(dev_fd);
   
    printf("%s exit \n", __FUNCTION__);
    pthread_exit(NULL);
	return NULL;
}
