#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <pthread.h>
#include <sys/epoll.h>
#include <fcntl.h>
#include <signal.h>

#include "mbtk_type.h"
#include "mbtk_ril.h"
#include "atchannel.h"
#include "at_tok.h"
#include "mbtk_utils.h"
#include "ril_info.h"

ril_cgact_wait_t cgact_wait;
static ril_data_call_info_t info_list[MBTK_APN_CID_MAX];

static int req_apn_get(bool get_def_cid, mbtk_apn_info_array_t *apns, int *cme_err);
static int req_apn_set(mbtk_apn_info_t *apn, int *cme_err);
static void apn_prop_get(mbtk_apn_info_array_t *apns);

/*
IPv4 : 10.255.74.26
IPv6 : 254.128.0.0.0.0.0.0.0.1.0.2.144.5.212.239
*/
bool is_ipv4(const char *ip)
{
    const char *ptr = ip;
    int count = 0;
    while(*ptr) {
        if(*ptr == '.')
            count++;
        ptr++;
    }

    if(count == 3) {
        return true;
    } else {
        return false;
    }
}

static void net_if_as_def_route(mbtk_ril_cid_enum cid)
{
    if(cid != MBTK_RIL_CID_NUL) {
        char buf[100] = {0};

        // Delete all default route in the first.
        // route del default

        // Add default route.
        memset(buf, 0, sizeof(buf));
        sprintf(buf, "route add default dev ccinet%d", cid - 1);
        system(buf);
    }
}

static int net_if_as_dns(mbtk_ril_cid_enum cid, mbtk_ipv4_info_t *ipv4, mbtk_ipv6_info_t *ipv6)
{
    char buf[1024] = {0};
    char dns[128] = {0};
    int offset = 0;
    int fd = -1;

    memset(buf, 0x0, 1024);
    memset(dns, 0x0, 128);
    offset = sprintf(buf, "search lan\n");
    if(ipv4->valid)
    {
        if(inet_ntop(AF_INET, &(ipv4->PrimaryDNS), dns, 32) == NULL) {
            LOGE("PrimaryDNS error.");
        } else {
            LOGD("PrimaryDNS : %s", dns);
        }
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
        memset(dns, 0x0, 128);
        if(inet_ntop(AF_INET, &(ipv4->SecondaryDNS), dns, 32) == NULL) {
            LOGE("SecondaryDNS error.");
        } else {
            LOGD("SecondaryDNS : %s", dns);
        }
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
    }
    if(ipv6->valid)
    {
        memset(dns, 0x0, 128);
		if(ipv6_2_str(&(ipv6->PrimaryDNS), dns))
        {
			LOGE("PrimaryDNS error.");
		} else {
			LOGD("PrimaryDNS : %s", dns);
		}
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
        memset(dns, 0x0, 128);
		if(ipv6_2_str(&(ipv6->SecondaryDNS), dns))
        {
			LOGE("SecondaryDNS error.");
		} else {
			LOGD("SecondaryDNS : %s", dns);
		}
        offset += sprintf(buf + offset, "nameserver %s\n", dns);
    }

    if(offset > 0)
    {
        fd = open("/tmp/resolv.conf", O_WRONLY | O_TRUNC);
        if(fd < 0)
        {
            LOGE("/tmp/resolv.conf : open fail.");
            return -1;
        }

        int ret = write(fd, buf, offset);
        if(ret < 0)
        {
            LOGE("/tmp/resolv.conf : write fail.");
            close(fd);
            return -1;
        }

        close(fd);
    }
    return 0;
}

int net_ifc_config(mbtk_ril_cid_enum cid, bool def_route, bool as_dns, mbtk_ip_info_t *ip_info)
{
    int ret = -1;
    char dev[20] = {0};
    sprintf(dev, "ccinet%d", cid - 1);
    if(ip_info) { // Config IP.
        // Config IPv4 address.
        if(ip_info->ipv4.valid) {
            char ip[20] = {0};
            if(inet_ntop(AF_INET, &(ip_info->ipv4.IPAddr), ip, 20) == NULL) {
                LOGE("inet_ntop ipv4 ip fail.");
                goto exit;
            }

            if(mbtk_ifc_configure2(dev, ip, 0, NULL, "255.255.255.0")) {
                LOGD("Config %s IPv4 %s fail.", dev, ip);
                goto exit;
            } else {
                LOGD("Config %s IPv4 %s success.", dev, ip);
            }
        }

        // Config IPv6 address.
        if(ip_info->ipv6.valid) {
            char ip[50] = {0};

            if(inet_ntop(AF_INET6, &(ip_info->ipv6.IPV6Addr), ip, 50) == NULL) {
                LOGE("inet_ntop ipv6 ip fail.");
                goto exit;
            }

            if(mbtk_ipv6_config(dev, ip, 64)) {
                LOGD("Config %s IPv6 %s fail.", dev, ip);
                goto exit;
            } else {
                LOGD("Config %s IPv6 %s success.", dev, ip);
            }
        }

        // mbtk_qser_route_config(cid, &ip_info->ipv4, &ip_info->ipv6);
        if(def_route && (ip_info->ipv4.valid || ip_info->ipv6.valid)) {
            net_if_as_def_route(cid);
        }

        if(as_dns) {
            ret = net_if_as_dns(cid, &(ip_info->ipv4), &(ip_info->ipv6));
        }
    } else { // Del IP
        if(mbtk_ifc_configure2(dev, NULL, 0, NULL, NULL)) {
            LOGD("Config %s IPv4 0 fail.", dev);
            goto exit;
        } else {
            LOGD("Config %s IPv4 0 success.", dev);
        }
#if 0
        if(mbtk_ipv6_config(dev, NULL, 64)) {
            LOGD("Config %s IPv6 0 fail.", dev);
            goto exit;
        } else {
            LOGD("Config %s IPv6 0 success.", dev);
        }
#endif
    }

    ret = 0;
exit:
    return ret;
}

int net_ifc_reconfig(mbtk_ril_cid_enum cid, bool def_route, bool as_dns, mbtk_ip_info_t *ip_info)
{
    int ret = net_ifc_config(cid, FALSE, FALSE, NULL);
    if(ret) {
        return ret;
    }

    return net_ifc_config(cid, def_route, as_dns, ip_info);
}

static int apn_file_save(const char *file, char *data, int data_len)
{
    if(!file) {
        return -1;
    }

    if(str_empty(data) || data_len <= 0) { // Delete file
        return unlink(file);
    } else {
        int fd = open(file, O_CREAT | O_WRONLY | O_TRUNC, 0644);
        if(fd < 0) {
            LOGE("open(%s) fail:%d", file, errno);
            return -1;
        }

        if(write(fd, data, data_len) != data_len) {
            LOGE("write fail:%d", errno);
            close(fd);
            return -1;
        }
        close(fd);
        return 0;
    }
}

static int apn_file_read(const char *file, char *data, int data_len)
{
    if(!file) {
        LOGE("file is null");
        return -1;
    }

    if(data == NULL || data_len <= 100) {
        LOGE("apn_file_read() arg error.");
        return -1;
    } else {
        int len = -1;
        int fd = open(file, O_RDONLY, 0644);
        if(fd < 0) {
            LOGE("open(%s) fail:%d", file, errno);
            return -1;
        }

        memset(data, 0, data_len);
        if((len = read(fd, data, data_len)) < 0) {
            LOGE("read fail:%d", errno);
            close(fd);
            return -1;
        }
        close(fd);
        return len;
    }
}


void apn_auto_conf_from_prop()
{
    mbtk_apn_info_array_t apns;
    int i = 0;
    memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
    apn_prop_get(&apns);
    while(i < apns.num) {
        int cme_err = MBTK_RIL_ERR_CME_NON;
        if(req_apn_set(&(apns.apns[i]), &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
        {
            LOGD("Set APN fail.");
        }
        else
        {
            LOGD("Set APN - %d success.", apns.apns[i].cid);
        }
        i++;
    }
}

static bool apn_conf_support(mbtk_ril_cid_enum cid)
{
    if(cid == MBTK_RIL_CID_DEF) {
        /*
        uci show wan_default.default.enable
        wan_default.default.enable='1'

        uci get wan_default.default.enable
        1
        */
        char buff[128] = {0};
        if(mbtk_cmd_line("uci get wan_default.default.enable", buff, sizeof(buff)) && strlen(buff) > 0) {
            return buff[0] == '1' ? FALSE : TRUE;
        }
    }
    return TRUE;
}

static int apn_check_and_cid_reset(mbtk_apn_info_t *apn)
{
    // Delete apn
    if(str_empty(apn->apn)) {
        if(apn->cid == MBTK_RIL_CID_NUL)
            return -1;

        if(!apn_conf_support(MBTK_RIL_CID_DEF) && apn->cid == MBTK_RIL_CID_DEF)
            return -1;

        // The cid no use,so can not delete.
        mbtk_apn_info_array_t apns;
        int cme_err = MBTK_RIL_ERR_CME_NON;
        memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
        if(req_apn_get(FALSE, &apns, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
        {
            LOGW("Get APN fail.");
            return 0;
        }
        else
        {
            int index = 0;
            while(index < apns.num) {
                if(apns.apns[index].cid == apn->cid)
                    return 0;
                index++;
            }
            return -1;
        }
    } else { // Add or change APN.
        int start_cid;
        bool asr_auto_call_open = !apn_conf_support(MBTK_RIL_CID_DEF);
        mbtk_apn_info_array_t apns;
        int cme_err = MBTK_RIL_ERR_CME_NON;

        if(asr_auto_call_open) {
            start_cid = MBTK_RIL_CID_2;
        } else {
            start_cid = MBTK_APN_CID_MIN;
        }
        memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
        if(req_apn_get(TRUE, &apns, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
        {
            LOGW("Get APN fail.");
            if(apn->cid == MBTK_RIL_CID_NUL) {
                apn->cid = start_cid;
            }
        }
        else
        {
            int index = 0;
            bool is_change = FALSE;  // Is add APN default.

            if(apn->cid == MBTK_RIL_CID_NUL) { // Is add (auto set cid).
                for(; start_cid <= MBTK_APN_CID_MAX; start_cid++) {
                    index = 0;
                    while(index < apns.num) {
                        if(apns.apns[index].cid == start_cid)
                            break;
                        index++;
                    }

                    if(index == apns.num) { // Found not used cid : start_cid.
                        LOGD("Change CID : %d -> %d", apn->cid, start_cid);
                        apn->cid = start_cid;
                        // return 0;
                        break;
                    }
                }

                if(start_cid > MBTK_APN_CID_MAX) {
                    LOGE("APN full.");
                    return -1;
                }
                is_change = FALSE;
            } else {
                index = 0;
                while(index < apns.num) {
                    if(apns.apns[index].cid == apn->cid) {
                        is_change = TRUE;
                        break;
                    }
                    index++;
                }
            }

            // Is add,the APN can't same.
            if(!is_change) {
                index = 0;
                while(index < apns.num) {
                    if(strcmp(apns.apns[index].apn,apn->apn) == 0) {
                        LOGW("APN : %s exist.", apn->apn);
                        return -1;
                    }
                    index++;
                }
            }
        }
    }
    return 0;
}

static void apn_prop_get(mbtk_apn_info_array_t *apns)
{
    char prop_name[128] = {0};
    char prop_data[1024] = {0};
    int cid;
    memset(apns, 0, sizeof(mbtk_apn_info_array_t));
    bool asr_auto_call_open = !apn_conf_support(MBTK_RIL_CID_DEF);

    // If auto data call is open,the default route is CID 1.
    if(asr_auto_call_open) {
        //apns->cid_for_def_route = MBTK_RIL_CID_DEF;
        //apns->cid_for_def_dns = MBTK_RIL_CID_DEF;
        cid = MBTK_RIL_CID_2;
    } else {
        cid = MBTK_APN_CID_MIN;
    }

    char def_cid[10] = {0};
    sprintf(def_cid, "%d", MBTK_RIL_CID_DEF);
    if(property_get(MBTK_DEF_ROUTE_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_route = (mbtk_ril_cid_enum)atoi(prop_data);
    }
    memset(prop_data, 0, sizeof(prop_data));
    if(property_get(MBTK_DEF_DNS_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_dns = (mbtk_ril_cid_enum)atoi(prop_data);
    }

    for(; cid <= MBTK_APN_CID_MAX; cid++) {
        memset(prop_name, 0, sizeof(prop_name));
        memset(prop_data, 0, sizeof(prop_data));

        sprintf(prop_name, "%s_%d",MBTK_APN_PROP,cid);
        // ip_type,auth,auto_data_call,apn,user,pass
#if 0
        if(property_get(prop_name, prop_data, "") > 0 && !str_empty(prop_data)) {
#else
        if(apn_file_read(prop_name, prop_data, sizeof(prop_data)) > 0 && !str_empty(prop_data)) {
#endif
            apns->apns[apns->num].cid = (mbtk_ril_cid_enum)cid;
            char *ptr_1 = prop_data;
            apns->apns[apns->num].ip_type = (mbtk_ip_type_enum)atoi(ptr_1);
            ptr_1 = strstr(ptr_1, ",");
            if(!ptr_1) {
                continue;
            }
            ptr_1++; // Jump ',' to auth

            apns->apns[apns->num].auth = (mbtk_apn_auth_type_enum)atoi(ptr_1);
            ptr_1 = strstr(ptr_1, ",");
            if(!ptr_1) {
                continue;
            }
            ptr_1++; // Jump ',' to auto_data_call

            apns->apns[apns->num].auto_boot_call = (uint8)atoi(ptr_1);
            ptr_1 = strstr(ptr_1, ",");
            if(!ptr_1) {
                continue;
            }
            ptr_1++; // Jump ',' to apn

            char *ptr_2 = strstr(ptr_1, ",");
            if(!ptr_2) {
                continue;
            }
            if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].apn, ptr_1, ptr_2 - ptr_1); // apn
            }

            ptr_2++; // Jump ',' to user
            ptr_1 = strstr(ptr_2, ",");
            if(!ptr_1) {
                continue;
            }
            if(memcmp(ptr_2, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].user, ptr_2, ptr_1 - ptr_2); // user
            }

            ptr_1++; // Jump ',' to pass
            ptr_2 = strstr(ptr_1, ",");
            if(!ptr_2) {
                continue;
            }
            if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].pass, ptr_1, ptr_2 - ptr_1); // pass
            }

            ptr_2++; // Jump ',' to type
            if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
                memcpy(apns->apns[apns->num].type, ptr_2, strlen(ptr_2)); // type
            }

            apns->num++;
        }
    }
}

static int apn_prop_get_by_cid(mbtk_ril_cid_enum cid, mbtk_apn_info_t *apn)
{
    char prop_name[128] = {0};
    char prop_data[1024] = {0};
    memset(apn, 0, sizeof(mbtk_apn_info_t));

    sprintf(prop_name, "%s_%d",MBTK_APN_PROP,cid);
    // ip_type,auth,auto_data_call,apn,user,pass
#if 0
    if(property_get(prop_name, prop_data, "") > 0 && !str_empty(prop_data)) {
#else
    if(apn_file_read(prop_name, prop_data, sizeof(prop_data)) > 0 && !str_empty(prop_data)) {
#endif
        apn->cid = cid;
        apn->auto_save = (uint8)1;
        char *ptr_1 = prop_data;
        apn->ip_type = (mbtk_ip_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auth

        apn->auth = (mbtk_apn_auth_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auto_data_call

        apn->auto_boot_call = (uint8)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to apn

        char *ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->apn, ptr_1, ptr_2 - ptr_1); // apn
        }

        ptr_2++; // Jump ',' to user
        ptr_1 = strstr(ptr_2, ",");
        if(!ptr_1) {
            return -1;
        }
        if(memcmp(ptr_2, "NULL", 4)) { // Not "NULL"
            memcpy(apn->user, ptr_2, ptr_1 - ptr_2); // user
        }

        ptr_1++; // Jump ',' to pass
        ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->user, ptr_1, ptr_2 - ptr_1); // pass
        }

        ptr_2++; // Jump ',' to type
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->type, ptr_2, strlen(ptr_2)); // pass
        }
        return 0;
    }
    return -1;
}

static int apn_prop_get_by_cid_without_cgdcont(mbtk_ril_cid_enum cid, mbtk_apn_info_t *apn)
{
    char prop_name[128] = {0};
    char prop_data[1024] = {0};

    sprintf(prop_name, "%s_%d",MBTK_APN_PROP,cid);
    // ip_type,auth,auto_data_call,apn,user,pass
#if 0
    if(property_get(prop_name, prop_data, "") > 0 && !str_empty(prop_data)) {
#else
    if(apn_file_read(prop_name, prop_data, sizeof(prop_data)) > 0 && !str_empty(prop_data)) {
#endif
        LOGD("APN : %s", prop_data);
        char *ptr_1 = prop_data;
        apn->auto_save = (uint8)1;
        //apn->ip_type = (mbtk_ip_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auth

        apn->auth = (mbtk_apn_auth_type_enum)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to auto_data_call

        apn->auto_boot_call = (uint8)atoi(ptr_1);
        ptr_1 = strstr(ptr_1, ",");
        if(!ptr_1) {
            return -1;
        }
        ptr_1++; // Jump ',' to apn

        char *ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
#if 0
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->apn, ptr_1, ptr_2 - ptr_1); // apn
        }
#endif

        ptr_2++; // Jump ',' to user
        ptr_1 = strstr(ptr_2, ",");
        if(!ptr_1) {
            return -1;
        }
        if(memcmp(ptr_2, "NULL", 4)) { // Not "NULL"
            memcpy(apn->user, ptr_2, ptr_1 - ptr_2); // user
        }

        ptr_1++; // Jump ',' to pass
        ptr_2 = strstr(ptr_1, ",");
        if(!ptr_2) {
            return -1;
        }
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->user, ptr_1, ptr_2 - ptr_1); // pass
        }

        ptr_2++; // Jump ',' to type
        if(memcmp(ptr_1, "NULL", 4)) { // Not "NULL"
            memcpy(apn->type, ptr_2, strlen(ptr_2)); // pass
        }
        return 0;
    } else {
        apn->auto_save = (uint8)0;
    }
    return -1;
}

static int apn_prop_set(mbtk_apn_info_t *apn)
{
    char prop_name[20] = {0};
    char prop_data[1024] = {0};
    int ret = -1;
    if(apn->auto_save) {
        sprintf(prop_name, "%s_%d", MBTK_APN_PROP, apn->cid);
        // Delete apn
        if(!str_empty(apn->apn)) {
            snprintf(prop_data, 1024, "%d,%d,%d,%s,%s,%s,%s", apn->ip_type, apn->auth, apn->auto_boot_call,
                apn->apn,
                str_empty(apn->user) ? "NULL" : apn->user,
                str_empty(apn->pass) ? "NULL" : apn->pass,
                str_empty(apn->pass) ? "NULL" : apn->type);
        }

#if 0
        ret = property_set(prop_name, prop_data);
#else
        if(str_empty(apn->apn)) { // Delete APN
            ret = apn_file_save(prop_name, NULL, 0);
        } else {
            ret = apn_file_save(prop_name, prop_data, strlen(prop_data));
        }
#endif
    }

    if(!ret && apn->def_route) {
        memset(prop_data, 0, sizeof(prop_data));
        prop_data[0] = '0' + apn->cid;
        ret = property_set(MBTK_DEF_ROUTE_CID, prop_data);
    }
    if(!ret && apn->as_dns) {
        memset(prop_data, 0, sizeof(prop_data));
        prop_data[0] = '0' + apn->cid;
        ret = property_set(MBTK_DEF_DNS_CID, prop_data);
    }
    return ret;
}

static int apn_prop_reset(mbtk_data_call_info_t *data_info)
{
    mbtk_apn_info_t apn;
    if(apn_prop_get_by_cid(data_info->cid, &apn)) {
        return -1;
    } else {
        apn.auto_boot_call = data_info->auto_boot_call;
        apn.def_route = data_info->def_route;
        apn.as_dns = data_info->as_dns;
        return apn_prop_set(&apn);
    }
}

static int wait_cgact_complete(int timeout)
{
    int count = timeout * 10; // timeout * 1000 / 100
    int i = 0;

    while(cgact_wait.waitting && i < count) {
        i++;
        usleep(100000); // 100ms
    }

    memset(&cgact_wait, 0, sizeof(ril_cgact_wait_t));
    if(i == count) { // Timeout
        return -1;
    } else {
        return 0;
    }
}

/*
AT+CGDCONT?

+CGDCONT: 1,"IPV4V6","ctnet","10.142.64.116 254.128.0.0.0.0.0.0.0.0.0.0.0.0.0.1",0,0,0,2,0,0

+CGDCONT: 8,"IPV4V6","IMS","254.128.0.0.0.0.0.0.0.0.0.0.0.0.0.1",0,0,0,2,1,1

OK

*/
static int req_apn_get(bool get_def_cid, mbtk_apn_info_array_t *apns, int *cme_err)
{
    ATResponse *response = NULL;
    int err = at_send_command_multiline("AT+CGDCONT?", "+CGDCONT:", &response);

    if (err < 0 || response->success == 0 || !response->p_intermediates){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

    ATLine* lines_ptr = response->p_intermediates;
    char *line = NULL;
    int tmp_int;
    char *tmp_str = NULL;
    int cid_start;
    if(apn_conf_support(MBTK_RIL_CID_DEF)) {
        cid_start = MBTK_RIL_CID_DEF;
    } else {
        if(get_def_cid) {
            cid_start = MBTK_RIL_CID_DEF;
        } else {
            cid_start = MBTK_RIL_CID_2;
        }
    }
    /*
    <apn_num[1]><cid[1]><ip_type[1]><apn_len[2]><apn><user_len[2]><user><pass_len[2]><pass><auth_len[2]><auth>...
                <cid[1]><ip_type[1]><apn_len[2]><apn><user_len[2]><user><pass_len[2]><pass><auth_len[2]><auth>
    */
    while(lines_ptr)
    {
        line = lines_ptr->line;
        err = at_tok_start(&line);
        if (err < 0)
        {
            goto exit;
        }

        err = at_tok_nextint(&line, &tmp_int); // cid
        if (err < 0)
        {
            goto exit;
        }
        // Only get CID 1-7
        if(tmp_int >= cid_start && tmp_int <= MBTK_APN_CID_MAX) {
            apns->apns[apns->num].cid = (mbtk_ril_cid_enum)tmp_int;

            err = at_tok_nextstr(&line, &tmp_str);// ip type
            if (err < 0)
            {
                goto exit;
            }
            if(!strcasecmp(tmp_str, "IP")) {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_IP;
            } else if(!strcasecmp(tmp_str, "IPV6")) {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_IPV6;
            } else if(!strcasecmp(tmp_str, "IPV4V6")) {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_IPV4V6;
            } else {
                apns->apns[apns->num].ip_type = MBTK_IP_TYPE_PPP;
            }

            err = at_tok_nextstr(&line, &tmp_str); // apn
            if (err < 0)
            {
                goto exit;
            }
            if(!str_empty(tmp_str)) {
                memcpy(apns->apns[apns->num].apn, tmp_str, strlen(tmp_str));
            }

            // Get other arg from proc or file.
            apn_prop_get_by_cid_without_cgdcont(apns->apns[apns->num].cid, &(apns->apns[apns->num]));

            apns->num++;
        }

        lines_ptr = lines_ptr->p_next;
    }

    char prop_name[128] = {0};
    char prop_data[1024] = {0};
    char def_cid[10] = {0};
    sprintf(def_cid, "%d", MBTK_RIL_CID_DEF);

    if(property_get(MBTK_DEF_ROUTE_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_route = (mbtk_ril_cid_enum)atoi(prop_data);
    }
    memset(prop_data, 0, sizeof(prop_data));
    if(property_get(MBTK_DEF_DNS_CID, prop_data, def_cid) > 0 && !str_empty(prop_data)) {
        apns->cid_for_def_dns = (mbtk_ril_cid_enum)atoi(prop_data);
    }

    goto exit;
exit:
    at_response_free(response);
    return err;
}

/*
AT+CGDCONT=1,"IPV4V6","cmnet"
OK

AT*AUTHReq=1,1,marvell,123456
OK

*/
static int req_apn_set(mbtk_apn_info_t *apn, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[400] = {0};
    int index = 0;
    int err = 0;

    // Delete apn
    if(str_empty(apn->apn)) {
        sprintf(cmd, "AT+CGDCONT=%d", apn->cid);
        err = at_send_command(cmd, &response);
        if (err < 0 || response->success == 0){
            if(cme_err) {
                *cme_err = at_get_cme_error(response);
            }
            goto exit;
        }
    } else {
        index += sprintf(cmd, "AT+CGDCONT=%d,", apn->cid);
        switch(apn->ip_type) {
            case MBTK_IP_TYPE_IP: {
                index += sprintf(cmd + index,"\"IP\",");
                break;
            }
            case MBTK_IP_TYPE_IPV6: {
                index += sprintf(cmd + index,"\"IPV6\",");
                break;
            }
            case MBTK_IP_TYPE_IPV4V6: {
                index += sprintf(cmd + index,"\"IPV4V6\",");
                break;
            }
            default: {
                index += sprintf(cmd + index,"\"PPP\",");
                break;
            }
        }
        if(strlen(apn->apn) > 0) {
            index += sprintf(cmd + index,"\"%s\"", apn->apn);
        }

        err = at_send_command(cmd, &response);
        if (err < 0 || response->success == 0){
            if(cme_err) {
                *cme_err = at_get_cme_error(response);
            }
            goto exit;
        }

        if(!str_empty(apn->user) || !str_empty(apn->pass)) {
            at_response_free(response);

            memset(cmd,0,400);
            int cmd_auth=0;
            if(apn->auth == MBTK_APN_AUTH_PROTO_NONE)
                cmd_auth = 0;
            else if(apn->auth == MBTK_APN_AUTH_PROTO_PAP)
                cmd_auth = 1;
            else if(apn->auth == MBTK_APN_AUTH_PROTO_CHAP)
                cmd_auth = 2;
            else
                goto exit;

            sprintf(cmd, "AT*AUTHREQ=%d,%d,%s,%s",apn->cid,cmd_auth,apn->user,apn->pass);
            err = at_send_command(cmd, &response);
            if (err < 0 || response->success == 0){
                if(cme_err) {
                    *cme_err = at_get_cme_error(response);
                }
                goto exit;
            }
        }
    }

exit:
    at_response_free(response);
    return err;
}

/*
AT+CGACT?
+CGACT: 1,1
+CGACT: 8,1
OK

AT+CGACT=1,<cid>
OK

*/
static int req_data_call_start(mbtk_ril_cid_enum cid, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[400] = {0};
    int err = 0;
    memset(&cgact_wait, 0, sizeof(ril_cgact_wait_t));
    cgact_wait.waitting = true;
    cgact_wait.cid = cid;
    cgact_wait.act = true;

    sprintf(cmd, "AT+CGACT=1,%d", cid);
    err = at_send_command(cmd, &response);
    if (err < 0 || response->success == 0){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

exit:
    at_response_free(response);
    return err;
}

/*
AT+CGACT=0,<cid>
OK

*/
static int req_data_call_stop(mbtk_ril_cid_enum cid, int timeout, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[400] = {0};
    int err = 0;

    memset(&cgact_wait, 0, sizeof(ril_cgact_wait_t));
    cgact_wait.waitting = true;
    cgact_wait.cid = cid;
    cgact_wait.act = false;

    sprintf(cmd, "AT+CGACT=0,%d", cid);
    err = at_send_command(cmd, &response);
    if (err < 0 || response->success == 0){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

exit:
    at_response_free(response);
    return err;
}

/*
AT+CGCONTRDP=1
+CGCONTRDP: 1,7,"cmnet-2.MNC000.MCC460.GPRS","10.255.74.26","","223.87.253.100","223.87.253.253","","",0,0
+CGCONTRDP: 1,7,"cmnet-2.MNC000.MCC460.GPRS","254.128.0.0.0.0.0.0.0.1.0.2.144.5.212.239","","36.9.128.98.32.0.0.2.0.0.0.0.0.0.0.1","36.9.128.98.32.0.0.2.0.0.0.0.0.0.0.2","","",0,0

OK

*/
static int req_data_call_state_get(mbtk_ril_cid_enum cid, mbtk_ip_info_t *ip_info, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[50] = {0};
    int err = 0;

    sprintf(cmd, "AT+CGCONTRDP=%d", cid);

    err = at_send_command_multiline(cmd, "+CGCONTRDP:", &response);
    if (err < 0 || response->success == 0 || !response->p_intermediates){
        *cme_err = at_get_cme_error(response);
        goto exit;
    }
    ATLine* lines_ptr = response->p_intermediates;
    char *line = NULL;
    int tmp_int;
    char *tmp_ptr = NULL;
    while(lines_ptr)
    {
        line = lines_ptr->line;
        err = at_tok_start(&line);
        if (err < 0)
        {
            goto exit;
        }

        err = at_tok_nextint(&line, &tmp_int); // cid
        if (err < 0)
        {
            goto exit;
        }
        err = at_tok_nextint(&line, &tmp_int); // bearer_id
        if (err < 0)
        {
            goto exit;
        }
        err = at_tok_nextstr(&line, &tmp_ptr); // APN
        if (err < 0)
        {
            goto exit;
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // IP
        if (err < 0 || str_empty(tmp_ptr))
        {
            goto exit;
        }
        if(is_ipv4(tmp_ptr)) {
            if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.IPAddr)) < 0) {
                LOGE("inet_pton() fail.");
                err = -1;
                goto exit;
            }

            ip_info->ipv4.valid = true;
            //log_hex("IPv4", &(ipv4->IPAddr), sizeof(struct in_addr));
        } else {
            if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.IPV6Addr))) {
                LOGE("str_2_ipv6() fail.");
                err = -1;
                goto exit;
            }

            ip_info->ipv6.valid = true;
            //log_hex("IPv6", &(ipv6->IPV6Addr), 16);
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // Gateway
        if (err < 0)
        {
            goto exit;
        }
        if(!str_empty(tmp_ptr)) { // No found gateway
            if(is_ipv4(tmp_ptr)) {
                if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.GateWay)) < 0) {
                    LOGE("inet_pton() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv4", &(ipv4->GateWay), sizeof(struct in_addr));
            } else {
                if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.GateWay))) {
                    LOGE("str_2_ipv6() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv6", &(ipv6->GateWay), 16);
            }
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // prim_DNS
        if (err < 0)
        {
            goto exit;
        }
        if(!str_empty(tmp_ptr)) { // No found Primary DNS
            if(is_ipv4(tmp_ptr)) {
                if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.PrimaryDNS)) < 0) {
                    LOGE("inet_pton() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv4", &(ipv4->PrimaryDNS), sizeof(struct in_addr));
            } else {
                if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.PrimaryDNS))) {
                    LOGE("str_2_ipv6() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv6", &(ipv6->PrimaryDNS), 16);
            }
        }

        err = at_tok_nextstr(&line, &tmp_ptr); // sec_DNS
        if (err < 0)
        {
            goto exit;
        }
        if(!str_empty(tmp_ptr)) { // No found Secondary DNS
            if(is_ipv4(tmp_ptr)) {
                if(inet_pton(AF_INET, tmp_ptr, &(ip_info->ipv4.SecondaryDNS)) < 0) {
                    LOGE("inet_pton() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv4", &(ipv4->SecondaryDNS), sizeof(struct in_addr));
            } else {
                if(str_2_ipv6(tmp_ptr, &(ip_info->ipv6.SecondaryDNS))) {
                    LOGE("str_2_ipv6() fail.");
                    err = -1;
                    goto exit;
                }

                //log_hex("IPv6", &(ipv6->SecondaryDNS), 16);
            }
        }

        lines_ptr = lines_ptr->p_next;
    }

exit:
    at_response_free(response);
    return err;
}

void data_call_state_change_cb(int cid, bool action, bool auto_change, int reason)
{
    if(auto_change && !action) {
        info_list[cid - 1].act_state = RIL_ACT_STATE_CONNECTED_RETRY;
    }
}

//void net_list_free(void *data);
// Return MBTK_INFO_ERR_SUCCESS,will call pack_error_send() to send RSP.
// Otherwise, do not call pack_error_send().
mbtk_ril_err_enum data_call_pack_req_process(sock_cli_info_t* cli_info, ril_msg_pack_info_t* pack)
{
    mbtk_ril_err_enum err = MBTK_RIL_ERR_SUCCESS;
    int cme_err = MBTK_RIL_ERR_CME_NON;
    switch(pack->msg_id)
    {
        case RIL_MSG_ID_DATA_CALL_APN:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                mbtk_apn_info_array_t apns;
                memset(&apns, 0, sizeof(mbtk_apn_info_array_t));
                if(req_apn_get(FALSE, &apns, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOGD("Get APN fail.");
                }
                else
                {
                    LOGD("size - %d", sizeof(mbtk_apn_info_array_t));
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &apns, sizeof(mbtk_apn_info_array_t));
                }
            }
            else     // Set
            {
                mbtk_apn_info_t *apn = (mbtk_apn_info_t*)pack->data;
                if(apn_check_and_cid_reset(apn)) {
                    err = MBTK_RIL_ERR_CID;
                } else {
                    if(apn_conf_support(apn->cid)) {
                        if(req_apn_set(apn, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                        {
                            if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                err = MBTK_RIL_ERR_CME + cme_err;
                            } else {
                                err = MBTK_RIL_ERR_UNKNOWN;
                            }
                            LOGD("Set APN fail.");
                        }
                        else
                        {
                            if(apn_prop_set(apn)) {
                                LOGE("Save APN fail.");
                            }

                            memcpy(&(info_list[apn->cid - 1].apn_info), apn, sizeof(mbtk_apn_info_t));

                            ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, NULL, 0);
                        }
                    } else {
                        err = MBTK_RIL_ERR_UNSUPPORTED;
                        LOGD("Can not set APN for CID : %d", apn->cid);
                    }
                }
            }
            break;
        }
        case RIL_MSG_ID_DATA_CALL_OPT:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
            }
            else     // Set
            {
                mbtk_data_call_info_t *call_info = (mbtk_data_call_info_t*)pack->data;
                // Check for cid 1.
                if(!apn_conf_support(call_info->cid)) {
                    err = MBTK_RIL_ERR_UNSUPPORTED;
                    LOGD("Can not data call for CID : %d", call_info->cid);
                } else {
                    if(call_info->type == MBTK_DATA_CALL_START) {
                        mbtk_ip_info_t ip_info;
                        memset(&ip_info, 0, sizeof(ip_info));
#if 0
                        if(apn_prop_reset(call_info)) {
                            err = MBTK_RIL_ERR_REQ_UNKNOWN;
                            LOG("apn_prop_reset() fail.");
                        } else
#else
                        if(apn_prop_reset(call_info)) {
                            LOG("apn_prop_reset() fail.");
                        }
#endif
                        {
                            int index = 0;
                            info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_CONNECTING;
data_call_retry:
                            if(req_data_call_start(call_info->cid, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                            {
                                if(call_info->retry_interval[index] > 0) {
                                    LOGD("Start data call fail, will retry in %d s.", call_info->retry_interval[index]);
                                    sleep(call_info->retry_interval[index]);
                                    index++;
                                    goto data_call_retry;
                                } else {
                                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                        err = MBTK_RIL_ERR_CME + cme_err;
                                    } else {
                                        err = MBTK_RIL_ERR_UNKNOWN;
                                    }
                                    LOGD("Start data call fail.");
                                }
                            }
                            else
                            {
                                // Wait for "CONNECT" or "+CGEV:"
                                if(wait_cgact_complete(call_info->timeout)) { // Timeout
                                    err = MBTK_RIL_ERR_TIMEOUT;
                                    break;
                                }

                                // Get Ip informations.
                                cme_err = MBTK_RIL_ERR_CME_NON;
                                if(req_data_call_state_get(call_info->cid ,&ip_info, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                                {
                                    LOGD("Get net informations fail.");
                                    err = MBTK_RIL_ERR_NET_CONF;
                                }
                                else
                                {
                                    // Config network informations.
                                    if(net_ifc_reconfig(call_info->cid, call_info->def_route, call_info->as_dns, &ip_info)) {
                                        err = MBTK_RIL_ERR_NET_CONF;
                                        break;
                                    }

                                    info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_CONNECTED;

                                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &ip_info, sizeof(mbtk_ip_info_t));
                                }
                            }
                        }
                    } else if(call_info->type == MBTK_DATA_CALL_STOP) {
                        info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_DISCONNECTING;
                        if(req_data_call_stop(call_info->cid, call_info->timeout, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                        {
                            if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                err = MBTK_RIL_ERR_CME + cme_err;
                            } else {
                                err = MBTK_RIL_ERR_UNKNOWN;
                            }
                            LOGD("Stop data call fail.");
                        }
                        else
                        {
                            // Wait for "CONNECT" or "+CGEV:"
                            if(wait_cgact_complete(call_info->timeout)) { // Timeout
                                err = MBTK_RIL_ERR_TIMEOUT;
                                break;
                            }

                            // Clean network config.
                            if(net_ifc_config(call_info->cid, FALSE, FALSE, NULL)) {
                                err = MBTK_RIL_ERR_NET_CONF;
                                break;
                            }

                            info_list[call_info->cid - 1].act_state = RIL_ACT_STATE_DISCONNECTED;
                            ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, NULL, 0);
                        }
                    } else {
                        mbtk_ip_info_t ip_info;
                        memset(&ip_info, 0, sizeof(ip_info));
                        if(req_data_call_state_get(call_info->cid ,&ip_info, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                        {
                            if(cme_err != MBTK_RIL_ERR_CME_NON) {
                                err = MBTK_RIL_ERR_CME + cme_err;
                            } else {
                                err = MBTK_RIL_ERR_UNKNOWN;
                            }
                            LOGD("Get data call state fail.");
                        }
                        else
                        {
                            ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &ip_info, sizeof(mbtk_ip_info_t));
                        }
                    }
                }
            }
            break;
        }
        default:
        {
            err = MBTK_RIL_ERR_REQ_UNKNOWN;
            LOG("Unknown request : %s", id2str(pack->msg_id));
            break;
        }
    }

    return err;
}


