#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <pthread.h>
#include <sys/epoll.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>

#include "mbtk_ril_api.h"

#define CLI_THREAD_MAX 3

typedef struct {
    pthread_t pid;
    bool is_running;
    char name[20];
} ril_cli_thread_info_t;

static ril_cli_thread_info_t cli_threads[CLI_THREAD_MAX];
static int cli_pid_index = 1;

static void help()
{
    printf("msdcfg <item> <value>: Set MSD item value.\n");
    printf("msdgen: Generate MSD by items.\n");
    printf("msd <MSD>: Get/Set MSD.\n");
    printf("msdpush: PUSH MSD.\n");
    printf("only <0-5> <test_num> <reconfig_num>: Get/Set ecall only mode.\n");
    printf("reg 0/1: Set ecall reg mode.\n");
    printf("dial <0-5>: Get/Start ecall dial.\n");
    printf("mode EU/ERA: Get/Set ecall mode\n");
    printf("cfg <type> <value>: Get/Set ecall config item.\n");
    printf("sms_num <number>: Get/Set sms number.\n");
    printf("mute_spk <0/1>: Set spk mute or not.\n");
    printf("gain <0/1/2> <gain>: Set ecall DSP gain.\n");
}

static void thread_exit_with_wait()
{
    int i = 0;
    while(i < CLI_THREAD_MAX) {
        cli_threads[i].is_running = FALSE;
        i++;
    }

    i = 0;
    while(i < CLI_THREAD_MAX) {
        if(cli_threads[i].pid) {
            pthread_join(cli_threads[i].pid, NULL);
            printf("Thread (%s) exit.\n", cli_threads[i].name);
        }
        i++;
    }
}

static void ecall_state_change_cb(const void* data, int data_len)
{
    if(data)
    {
        mbtk_ril_ecall_state_info_t *ecall_data = (mbtk_ril_ecall_state_info_t*)data;
        printf("ecall state change : urc_id - %d, urc_data - %s\n", ecall_data->urc_id, ecall_data->urc_data);
    }
}


static void sig_process(int sig)
{
    LOGI("I got signal %d\n", sig);
    switch(sig)
    {
        case SIGINT: // Ctrl + C
        {
            LOGI("Exit by SIGINT.\n");
            thread_exit_with_wait();
            mbtk_ril_deinit();
            exit(0);
        }
        case SIGQUIT: // Ctrl + \ (类似 SIGINT ，但要产生core文件)
        {
            LOGI("Exit by SIGQUIT.\n");
            thread_exit_with_wait();
            mbtk_ril_deinit();
            exit(0);
        }
        case SIGTERM:// 默认kill   (同 SIGKILL ，但 SIGKILL 不可捕获)
        {
            LOGI("Exit by SIGTERM.\n");
            thread_exit_with_wait();
            mbtk_ril_deinit();
            exit(0);
        }
        case SIGTSTP:// Ctrl + Z (同 SIGSTOP ，但 SIGSTOP 不可捕获)
        {
            LOGI("Exit by SIGTSTP.\n");
            exit(0);
        }
        case SIGSEGV: // 如空指针
        {
            LOGI("Exit by SIGSEGV.\n");
            exit(0);
        }
        default:
        {
            LOGI("Unknown sig:%d\n",sig);
            break;
        }
    }
}

static char* thread_id2name(pthread_t pid)
{
    int i = 0;
    while(i < CLI_THREAD_MAX) {
        if(pid == cli_threads[i].pid) {
            return cli_threads[i].name;
        }
        i++;
    }

    return "UNKNOWN";
}

static void* sub_thread_run(void *arg)
{
    ril_cli_thread_info_t *cli = (ril_cli_thread_info_t*)arg;
    cli->pid = pthread_self();
    cli->is_running = TRUE;
    sprintf(cli->name, "PID-%d", cli_pid_index++);

    printf("[%s] enter.\n", thread_id2name(cli->pid));
    while(cli->is_running) {
        srand((int)(time(0) + cli->pid));
        int time_sec = 1 + (int)(10.0 * rand() / ( RAND_MAX + 1.0));
        char version[50] = {0};
        mbtk_ril_err_enum err = mbtk_version_get(version);
        if(err != MBTK_RIL_ERR_SUCCESS) {
            printf("[%s : %ds]Error : %d\n", thread_id2name(cli->pid), time_sec, err);
        } else {
            printf("[%s : %ds]Version : %s\n", thread_id2name(cli->pid), time_sec, version);
        }

        sleep(time_sec);
    }
    printf("[%s] exit.\n", thread_id2name(cli->pid));
    return NULL;
}

int main(int argc, char *argv[])
{
    signal(SIGINT, sig_process);
    signal(SIGQUIT, sig_process);
    signal(SIGTERM, sig_process);
    //signal(SIGTSTP, sig_process);
    //signal(SIGSEGV, sig_process);

    mbtk_log_init("radio","ECALL_CLI");

#ifdef MBTK_DUMP_SUPPORT
    mbtk_debug_open(NULL, TRUE);
#endif

    //test2(0, "192.168.1.198");
    //test2(1, "2409:8162:140:cd3c:1:2:1494:72ba");
    //test2(1, "254.128.0.0.0.0.0.0.0.1.0.2.144.5.212.239");
    //test2(1, "2400:3200::1");

    if(mbtk_ril_init() != MBTK_RIL_ERR_SUCCESS)
    {
        return -1;
    }

    mbtk_ecall_state_change_cb_reg(ecall_state_change_cb);

    printf(">>>>>>>>>>>>>>>>>>>>>>>>Enter cmd:\n");
    char cmd[1024];
    while(1)
    {
        memset(cmd, 0, sizeof(cmd));
        mbtk_ril_err_enum err;
        if(fgets(cmd, sizeof(cmd), stdin))
        {
            char *ptr = cmd + strlen(cmd) - 1;
            while(ptr >= cmd && (*ptr == '\r' || *ptr == '\n'))
            {
                *ptr-- = '\0';
            }

            if(!strncasecmp(cmd, "msdcfg", 6)) // msdcfg <item> <value>
            {
                mbtk_ecall_msd_cfg_info_t cfg;
                memset(&cfg, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
                int item;
                int count = sscanf(cmd, "msdcfg %d %s", &item, cfg.data);
                if(count == 2 && strlen(cfg.data) > 0) {
                    cfg.item_type = (mbtk_ecall_msd_item_enum)item;
                    err = mbtk_ecall_msd_item_set(&cfg);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("MSD item set success\n");
                    }
                }
            }
            else if(!strncasecmp(cmd, "msdgen", 6)){
                err = mbtk_ecall_msd_gen();
                if(err != MBTK_RIL_ERR_SUCCESS) {
                    printf("Error : %d\n", err);
                } else {
                    printf("MSD gen success\n");
                }
            } else if(!strncasecmp(cmd, "msdpush", 7)){
                err = mbtk_ecall_push();
                if(err != MBTK_RIL_ERR_SUCCESS) {
                    printf("Error : %d\n", err);
                } else {
                    printf("MSD PUSH success\n");
                }
            } else if(!strncasecmp(cmd, "msd", 3)){ // msd <MSD>
                uint8 msd[500] = {0};
                if(!strcasecmp(cmd, "msd")) { // Get
                    err = mbtk_ecall_msd_get(msd);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("MSD : %s\n", msd);
                    }
                } else { // Set
                    int count = sscanf(cmd, "msd %s", msd);
                    if(count == 1 && strlen(msd) > 0) { // set
                        err = mbtk_ecall_msd_set(msd);
                        if(err != MBTK_RIL_ERR_SUCCESS) {
                            printf("Error : %d\n", err);
                        } else {
                            printf("MSD set success\n");
                        }
                    }
                }
            } else if(!strncasecmp(cmd, "only", 4)){ // only <0-5> <test_num> <reconfig_num>
                mbtk_ecall_only_info_t only_info;
                if(strcasecmp(cmd, "only") == 0) { // Get
                    memset(&only_info, 0, sizeof(mbtk_ecall_only_info_t));
                    err = mbtk_ecall_only_get(&only_info);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Only : active-%d,sim_type-%d,test_num-%s,reconfig_num-%s\n", only_info.active, only_info.sim_type, only_info.test_num, only_info.reconfig_num);
                    }
                } else { // Set
                    memset(&only_info, 0, sizeof(mbtk_ecall_only_info_t));
                    int tmp_int = -1;
                    int count = sscanf(cmd, "only %d %s %s", &tmp_int, only_info.test_num, only_info.reconfig_num);
                    if(count > 0 && tmp_int >= 0) { // set
                        only_info.active = (mbtk_ecall_only_type_enum)tmp_int;
                        err = mbtk_ecall_only_set(&only_info);
                        if(err != MBTK_RIL_ERR_SUCCESS) {
                            printf("Error : %d\n", err);
                        } else {
                            printf("MSD set success\n");
                        }
                    }
                }
            } else if(!strncasecmp(cmd, "dial", 4)){ // dial <0-5>
                mbtk_ecall_dial_type_enum type;
                if(!strcasecmp(cmd, "dial")) { // Get
                    err = mbtk_ecall_dial_state_get(&type);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Type : %d\n", type);
                    }
                } else { // Set
                    int reset;
                    int count = sscanf(cmd, "dial %d", &reset);
                    if(count == 1) {
                        err = mbtk_ecall_dial_start((mbtk_ecall_dial_type_enum)reset);
                        if(err != MBTK_RIL_ERR_SUCCESS) {
                            printf("Error : %d\n", err);
                        } else {
                            printf("Start ecall dial success\n");
                        }
                    }
                }
            } else if(!strncasecmp(cmd, "reg", 3)){ // reg <0/1>
                if(!strcasecmp(cmd, "reg 0")) {
                    err = mbtk_ecall_reg_set(0);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set net reg 0\n");
                    }
                } else if(!strcasecmp(cmd, "reg 1")) {
                    err = mbtk_ecall_reg_set(1);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set net reg 1\n");
                    }
                }
            } else if(!strncasecmp(cmd, "mode", 4)){ // mode EU/ERA
                if(!strcasecmp(cmd, "mode")) {
                    mbtk_ecall_mode_type_enum mode;
                    err = mbtk_ecall_mode_get(&mode);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Ecall mode[%d] : %s\n", mode, mode == MBTK_ECALL_MODE_TYPE_EU ? "EU" : "ERA");
                    }
                } else if(!strcasecmp(cmd, "mode EU")) {
                    err = mbtk_ecall_mode_set(MBTK_ECALL_MODE_TYPE_EU);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set mode EU\n");
                    }
                } else if(!strcasecmp(cmd, "mode ERA")) {
                    err = mbtk_ecall_mode_set(MBTK_ECALL_MODE_TYPE_ERA);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set mode ERA\n");
                    }
                }
            } else if(!strncasecmp(cmd, "cfg", 3)){ // cfg <type> <value>
                int tmp1 = -1, tmp2 = -1;
                int count = sscanf(cmd, "cfg %d %d", &tmp1, &tmp2);
                mbtk_ecall_cfg_info_t cfg;
                if(count == 1 && tmp1 >= 0) { // Get
                    memset(&cfg, 0, sizeof(cfg));
                    cfg.type = (mbtk_ecall_cfg_item_enum)tmp1;
                    err = mbtk_ecall_cfg_get(&cfg);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Ecall config[%d] is %d\n", cfg.type, cfg.data);
                    }
                } else if(count == 2 && tmp1 >= 0 && tmp2 >= 0) { // Set
                    memset(&cfg, 0, sizeof(cfg));
                    cfg.type = (mbtk_ecall_cfg_item_enum)tmp1;
                    cfg.data = (uint32)tmp2;
                    err = mbtk_ecall_cfg_set(&cfg);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set ecall config[%d] to %d success\n", cfg.type, cfg.data);
                    }
                }
            } else if(!strncasecmp(cmd, "sms_num", 7)){ // sms_num <number>
                uint8 number[RIL_MAX_NUMBER_LEN] = {0};
                if(!strcasecmp(cmd, "sms_num")) { // Get
                    err = mbtk_ecall_sms_number_get(number);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("SMS number : %s\n", number);
                    }
                } else { // Set
                    int count = sscanf(cmd, "sms_num %s", number);
                    if(count == 1 && strlen(number) > 0) {
                        err = mbtk_ecall_sms_number_set(number);
                        if(err != MBTK_RIL_ERR_SUCCESS) {
                            printf("Error : %d\n", err);
                        } else {
                            printf("Set SMS number[%s] success\n", number);
                        }
                    }
                }
            } else if(!strncasecmp(cmd, "mute_spk", 8)){ // mute_spk <0/1>
                if(!strcasecmp(cmd, "mute_spk 0")) {
                    err = mbtk_ecall_mute_spk_set(0);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set mute spk to 0\n");
                    }
                } else if(!strcasecmp(cmd, "mute_spk 1")) {
                    err = mbtk_ecall_mute_spk_set(1);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set mute spk to 1\n");
                    }
                }
            } else if(!strncasecmp(cmd, "gain", 4)){    // gain <0/1/2> <gain>
                int tmp1 = -1, tmp2 = -1;
                int count = sscanf(cmd, "gain %d %d", &tmp1, &tmp2);
                mbtk_ecall_gain_info_t gain;
                if(count == 2 && tmp1 >= 0) { // Get
                    memset(&gain, 0, sizeof(gain));
                    gain.mode = (mbtk_ecall_gain_mode_enum)tmp1;
                    gain.gain = (int8)tmp2;
                    err = mbtk_ecall_dsp_gain_set(&gain);
                    if(err != MBTK_RIL_ERR_SUCCESS) {
                        printf("Error : %d\n", err);
                    } else {
                        printf("Set DSP gain[%d] to %d\n", gain.mode, gain.gain);
                    }
                }
            }
            else if(!strcasecmp(cmd, "h") || !strcasecmp(cmd, "help")) {
                help();
            } else if(!strcasecmp(cmd, "q")) {
                mbtk_ril_deinit();
                break;
            } else {
                printf("\n");
            }
        }
    }

    thread_exit_with_wait();

exit:
    mbtk_ril_deinit();

    LOG("Client exec complete.");

    return 0;
}

