#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>

#include "mbtk_log.h"
#include "mbtk_audio_internal.h"

#define WAV_PLAY_BUFF 2048

typedef struct {
    const unsigned char *pcm_data;
    int data_size;
} audio_buff_t;

static int wav_play_fd = -1;
static audio_play_state_enum play_state = AUDIO_PLAY_STATE_STOP;
static pthread_cond_t play_cond;
static pthread_mutex_t play_mutex;
static pthread_t play_thread_play;

static int wav_recorder_fd = -1;
static struct wav_header recorder_header;
static uint32 recorver_data_count = 0;

static audio_buff_t audio_buff;

static int current_loopback_state = 0;
static int current_loopback_device = -1;
static mbtk_audio_service_error_cb_f service_error_cb = NULL;

static void audio_play_thread(void *arg)
{
    int rc, len, frames = 0;
    char buf[WAV_PLAY_BUFF];

    play_state = AUDIO_PLAY_STATE_RUNNING;
    pthread_mutex_init(&play_mutex, NULL);
    pthread_cond_init(&play_cond, NULL);

    int data_send = 0;
    while (TRUE) {
        /* Playback loop */
        pthread_mutex_lock(&play_mutex);
        if(play_state == AUDIO_PLAY_STATE_STOP) {
            LOGD("Stop play...");
            pthread_mutex_unlock(&play_mutex);
            break;
        } else if(play_state == AUDIO_PLAY_STATE_PAUSE) {
            pthread_cond_wait(&play_cond, &play_mutex);
            pthread_mutex_unlock(&play_mutex);
            continue;
        } else {
            pthread_mutex_unlock(&play_mutex);
        }

        memset(buf, 0x00, sizeof(buf));
        if(wav_play_fd > 0) { // Play file.
            len = read(wav_play_fd, buf, WAV_PLAY_BUFF);
            if (len == -1) {
                LOGE("%s: error reading from file", __FUNCTION__);
                goto thread_end;
            }

            if (len == 0) {
                /* reached EOF */
                LOGE("%s: Read wav file end.", __FUNCTION__);
                break;
            }
        } else { // Play buffer.
            if(data_send >= audio_buff.data_size) {
                /* reached EOF */
                LOGE("%s: Read buffer end.", __FUNCTION__);
                break;
            }
            if(audio_buff.data_size - data_send >= WAV_PLAY_BUFF) {
                memcpy(buf, audio_buff.pcm_data + data_send, WAV_PLAY_BUFF);
                len = WAV_PLAY_BUFF;
            } else {
                memcpy(buf, audio_buff.pcm_data + data_send, audio_buff.data_size - data_send);
                len = audio_buff.data_size - data_send;
            }
            data_send += len;
        }

        if((rc = mbtk_audio_pcm_play_data_send(buf, len)) < len) {
            LOGE("Send data %d/%d", rc, len);
            goto thread_end;
        }

        LOGD("%s: No.%d frame playback", __FUNCTION__, ++frames);
    }

    play_state = AUDIO_PLAY_STATE_STOP;

thread_end:
    pthread_mutex_destroy(&play_mutex);
    pthread_cond_destroy(&play_cond);
    mbtk_audio_pcm_play_stop();

    if(wav_play_fd > 0) {
        if (close(wav_play_fd))
            LOGE("%s: error closing file", __FUNCTION__);
    }

    wav_play_fd = -1;
    LOGD("%s: finished pcm playback.", __FUNCTION__);
    return;
}

static void audio_recorder_cb(void *data, uint32 data_len)
{
    if(data_len > 0) {
        LOGD("Recorver data:%d, count:%d", data_len, recorver_data_count);
        if (write(wav_recorder_fd, data, data_len) < data_len) {
            LOGE("%s: error writing to file!", __FUNCTION__);
        }
        recorver_data_count += data_len;
    } else {
        LOGD("Recorver data end.");
        recorder_header.data_sz = recorver_data_count;
        recorder_header.riff_sz = recorder_header.data_sz + sizeof(recorder_header) - 8;
        lseek(wav_recorder_fd, 0, SEEK_SET);
        write(wav_recorder_fd, &recorder_header, sizeof(struct wav_header));

        close(wav_recorder_fd);
        wav_recorder_fd = -1;
    }
}

int mbtk_register_error_callback(mbtk_audio_service_error_cb_f cb)
{
    if (cb == NULL) {
        LOGE("Error: Callback function is NULL.");
        return -1;
    }
    service_error_cb = cb;
    LOGD("Callback function registered successfully.");
    return 0;
}

int mbtk_audio_wav_init()
{
    //mbtk_log_init("radio", "MBTK_AUDIO");
    return mbtk_audio_pcm_init();
}

int mbtk_audio_wav_play_start(const void *wav_file)
{
    struct stat st;
    const char *path = (const char *)wav_file;
    struct riff_wave_header riff_wave_header;
    struct chunk_header chunk_header;
    struct chunk_fmt chunk_fmt = {0};
    unsigned int more_chunks = 1;

    if(play_state != AUDIO_PLAY_STATE_STOP) {
        LOGW("Audio is playing...");
        return -1;
    }

    /* Check and open source file */
    if (access(path, F_OK) || stat(path, &st)) {
        LOGE("%s: error reading from file %s", __FUNCTION__, path);
        return -1;
    }

    if (!st.st_size) {
        LOGE("%s: empty file %s", __FUNCTION__, path);
        return -1;
    }

    wav_play_fd = open(path, O_RDONLY);
    if (wav_play_fd < 0) {
        LOGE("%s: error opening file %s", __FUNCTION__, path);
        return -1;
    }

    read(wav_play_fd, &riff_wave_header, sizeof(riff_wave_header));
    if ((riff_wave_header.riff_id != ID_RIFF) || (riff_wave_header.wave_id != ID_WAVE)) {
        LOGE("Error: '%s' is not a riff/wave file", path);
        close(wav_play_fd);
        return -1;
    }

    do {
        read(wav_play_fd, &chunk_header, sizeof(chunk_header));

        switch (chunk_header.id) {
            case ID_FMT:
                read(wav_play_fd, &chunk_fmt, sizeof(chunk_fmt));
                /* If the format header is larger, skip the rest */
                if (chunk_header.sz > sizeof(chunk_fmt))
                    lseek(wav_play_fd, chunk_header.sz - sizeof(chunk_fmt), SEEK_CUR);
                break;
            case ID_DATA:
                /* Stop looking for chunks */
                more_chunks = 0;
                break;
            default:
                /* Unknown chunk, skip bytes */
                lseek(wav_play_fd, chunk_header.sz, SEEK_CUR);
        }
    } while (more_chunks);

    //Support 8k/16k & mono wave file
    if (((chunk_fmt.sample_rate != 8000) && (chunk_fmt.sample_rate != 16000))
            || (chunk_fmt.num_channels != 1) ) {
        LOGD("%s: error wave file:sample_rate = %d, num_channels = %d!!",
                __FUNCTION__,chunk_fmt.sample_rate, chunk_fmt.num_channels);
        close(wav_play_fd);
        return -1;
    }

    LOGE("%s: success open wave file:%s, sample_rate = %d, num_channels = %d.",
            __FUNCTION__, path, chunk_fmt.sample_rate, chunk_fmt.num_channels);

    if ((8000 == chunk_fmt.sample_rate) && (1 == chunk_fmt.num_channels)) {
        mbtk_audio_pcm_sample_rate_set(MBTK_AUDIO_SAMPLE_RATE_8000);
    } else if ((16000 == chunk_fmt.sample_rate) && (1 == chunk_fmt.num_channels)) {
        mbtk_audio_pcm_sample_rate_set(MBTK_AUDIO_SAMPLE_RATE_16000);
    }

    if (mbtk_audio_pcm_play_start()) {
        LOGE("%s: error opening output device.", __FUNCTION__);
        return -1;
    }

    LOGD("Start play wav file...");

    pthread_attr_t thread_attr;
    pthread_attr_init(&thread_attr);
    if(pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))
    {
        LOGE("pthread_attr_setdetachstate() fail.");
        return -1;
    }

    if (pthread_create(&play_thread_play, NULL, (void *)&audio_play_thread, NULL) < 0) {
        LOGE("%s: error creating thread_play!", __FUNCTION__);
        return -1;
    }

    return 0;
}

int mbtk_audio_wav_stream_play_start(const unsigned char *pcm_data, int data_size, int sample_rate, int num_channels)
{
    if(play_state != AUDIO_PLAY_STATE_STOP) {
        LOGW("Audio is playing...");
        return -1;
    }

    //Support 8k/16k & mono wave file
    if (((sample_rate != 8000) && (sample_rate != 16000))
            || (num_channels != 1) ) {
        LOGD("%s: error wave file:sample_rate = %d, num_channels = %d!!",
                __FUNCTION__,sample_rate, num_channels);
        return -1;
    }

    LOGE("%s: success open wave stream, sample_rate = %d, num_channels = %d.",
            __FUNCTION__,  sample_rate, num_channels);

    if ((8000 == sample_rate) && (1 == num_channels)) {
        mbtk_audio_pcm_sample_rate_set(MBTK_AUDIO_SAMPLE_RATE_8000);
    } else if ((16000 == sample_rate) && (1 == num_channels)) {
        mbtk_audio_pcm_sample_rate_set(MBTK_AUDIO_SAMPLE_RATE_16000);
    }

    if (mbtk_audio_pcm_play_start()) {
        LOGE("%s: error opening output device.", __FUNCTION__);
        return -1;
    }

    audio_buff.pcm_data = pcm_data;
    audio_buff.data_size = data_size;
    LOGD("Start play wav stream...");

    pthread_attr_t thread_attr;
    pthread_attr_init(&thread_attr);
    if(pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))
    {
        LOGE("pthread_attr_setdetachstate() fail.");
        return -1;
    }

    if (pthread_create(&play_thread_play, NULL, (void *)&audio_play_thread, NULL) < 0) {
        LOGE("%s: error creating thread_play!", __FUNCTION__);
        return -1;
    }

    return 0;
}


int mbtk_audio_wav_play_pause()
{
    int result = 0;
    pthread_mutex_lock(&play_mutex);
    if(play_state == AUDIO_PLAY_STATE_RUNNING) {
        play_state = AUDIO_PLAY_STATE_PAUSE;
    } else {
        // result = -1;
        LOGW("Audio state : %d", play_state);
    }
    pthread_mutex_unlock(&play_mutex);
    return result;
}

int mbtk_audio_wav_play_resume()
{
    int result = 0;
    pthread_mutex_lock(&play_mutex);
    if(play_state == AUDIO_PLAY_STATE_PAUSE) {
        play_state = AUDIO_PLAY_STATE_RUNNING;
        pthread_cond_signal(&play_cond);
    } else {
        // result = -1;
        LOGW("Audio state : %d", play_state);
    }
    pthread_mutex_unlock(&play_mutex);
    return result;
}


int mbtk_audio_wav_play_stop()
{
    int result = 0;
    pthread_mutex_lock(&play_mutex);
    if(play_state == AUDIO_PLAY_STATE_PAUSE || play_state == AUDIO_PLAY_STATE_RUNNING) {
        if(play_state == AUDIO_PLAY_STATE_PAUSE) {
            pthread_cond_signal(&play_cond);
        }
        play_state = AUDIO_PLAY_STATE_STOP;
        pthread_mutex_unlock(&play_mutex);

        LOGD("Waitting play thread exit...");
        if (pthread_join(play_thread_play, NULL)) {
            LOGE("error join thread_play!");
            // abort();
        }
        LOGD("Play thread exit success.");
    } else {
        pthread_mutex_unlock(&play_mutex);
        // result = -1;
        LOGW("Audio state : %d", play_state);
    }

    return result;
}

int mbtk_audio_wav_recorder_start(const void *wav_file, mbtk_audio_sample_rate_enum sample_rate)
{
    int rc;
    const char *path = (const char *)wav_file;

    LOGD("wav_file is %s.", path);
    if(wav_recorder_fd > 0) {
        LOGW("Audio is recorder...");
    }

    memset(&recorder_header, 0x0, sizeof(struct wav_header));
    recorder_header.riff_id = ID_RIFF;
    recorder_header.riff_sz = 0;
    recorder_header.riff_fmt = ID_WAVE;
    recorder_header.fmt_id = ID_FMT;
    recorder_header.fmt_sz = 16;
    recorder_header.audio_format = 1;        //FORMAT_PCM;
    recorder_header.num_channels = 1;        //Modem ONLY support mono recording
    recorder_header.sample_rate = (sample_rate == MBTK_AUDIO_SAMPLE_RATE_8000 ? 8000 : 16000);
    recorder_header.bits_per_sample = 16;    //PCM_SAMPLEBITS_S16_LE;
    recorder_header.byte_rate = (recorder_header.bits_per_sample / 8) * recorder_header.num_channels * recorder_header.sample_rate;
    recorder_header.block_align = recorder_header.num_channels * (recorder_header.bits_per_sample / 8);
    recorder_header.data_id = ID_DATA;

    mbtk_audio_pcm_sample_rate_set(sample_rate);

    wav_recorder_fd = open(path, O_WRONLY | O_CREAT | O_TRUNC, 0666);
    if (wav_recorder_fd < 0) {
        LOGE("%s: error opening file %s!", __FUNCTION__, path);
        return -1;
    }

    //leave enough room for header
    lseek(wav_recorder_fd, sizeof(struct wav_header), SEEK_SET);

    recorver_data_count = 0;

    return mbtk_audio_pcm_recorder_start(audio_recorder_cb);
}

int mbtk_audio_wav_recorder_pause()
{
    return mbtk_audio_pcm_recorder_pause();
}

int mbtk_audio_wav_recorder_resume()
{
    return mbtk_audio_pcm_recorder_resume();
}


int mbtk_audio_wav_recorder_stop()
{
    return mbtk_audio_pcm_recorder_stop();
}

int mbtk_audio_wav_deinit()
{
    if(play_state != AUDIO_PLAY_STATE_STOP) {
        if(mbtk_audio_wav_play_stop()) {
            LOGE("mbtk_audio_wav_play_stop() fail.");
        }
    }

    if(wav_recorder_fd > 0) {
        mbtk_audio_wav_recorder_stop();
    }

    return mbtk_audio_pcm_deinit();
}

int mbtk_audio_set_loopback_enable_state(int device, int enable_state)
{
    char command[128];

    if (device < 0 || device > 2 || (enable_state != 0 && enable_state != 1)) {
        LOGE("Invalid device or enable_state");
        return -1;
    }

    snprintf(command, sizeof(command), "ubus call audio_if audio_mode_set '{\"param0\":0}'");
    if (system(command) != 0) {
        LOGE("Failed to set audio mode");
        return -1;
    }

    if (enable_state == 1) {
        snprintf(command, sizeof(command), "ubus call audio_if loopback_enable '{\"param0\":%d}'", device);
        if (system(command) != 0) {
            LOGE("Failed to enable loopback");
            return -1;
        }
    } else {
        if (system("ubus call audio_if loopback_disable") != 0) {
            LOGE("Failed to disable loopback");
            return -1;
        }
    }

    current_loopback_device = device;
    current_loopback_state = enable_state;

    return 0;
}

int mbtk_audio_get_loopback_enable_state(int *device, int *enable_state)
{
    if (device == NULL || enable_state == NULL) {
        LOGE("Null pointer provided for device or enable_state");
        return -1;
    }

    *device = current_loopback_device;
    *enable_state = current_loopback_state;

    return 0;
}
