#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include "mbtk_log.h"
#include "mbtk_audio_internal.h"
#include "audio_if_audio_hw_mrvl.h"

#define LOCK_FILE "/var/run/mbtk_audio.lock"
#define AUDIO_DEBUG 0

#define AUDIO_LOG(fmt, args ...) \
    do{ \
        if(AUDIO_DEBUG) {LOGD(fmt, ##args);} \
    } while(0)

extern void* audio_hal_install(void);
extern void audio_hal_uninstall(void);
extern void configure_vcm(unsigned int data[]);

static audio_inter_info_t *audio_info = NULL;
static pthread_t recorder_thread_play;
static mbtk_audio_pa_switch_func audio_pa_switch_func = NULL;
static mbtk_audio_volume_set_func audio_volume_set_func = NULL;
static int audio_pcm_write_count = 0;

static int lock_get() {
    int fd = open(LOCK_FILE, O_WRONLY | O_CREAT | O_TRUNC, 0666);
    if (fd == -1) {
        LOGE("Open(%s) fail:%d", LOCK_FILE, errno);
        return -1;
    }
    // 尝试对文件进行加锁
    struct flock fl;
    fl.l_type = F_WRLCK;
    fl.l_whence = SEEK_SET;
    fl.l_start = 0;
    fl.l_len = 1;
    int ret = fcntl(fd, F_SETLK, &fl);
    if (ret == -1) {
        LOGE("Another instance is running, exiting...");
        close(fd);
        return -1;
    }
    close(fd);
    LOGD("Get file lock.");
    return 0;
}

static int lock_free() {
    int fd = open(LOCK_FILE, O_WRONLY | O_CREAT | O_TRUNC, 0666);
    if (fd == -1) {
        LOGE("Open(%s) fail:%d", LOCK_FILE, errno);
        return -1;
    }
    // 释放文件锁
    struct flock fl;
    fl.l_type = F_UNLCK;
    fl.l_whence = SEEK_SET;
    fl.l_start = 0;
    fl.l_len = 1;
    int ret = fcntl(fd, F_SETLK, &fl);
    if (ret == -1) {
        LOGE("Another instance is running, exiting...");
        close(fd);
        return -1;
    }

    LOGD("Free file lock.");
    return 0;
}

static void audio_recorder_thread(void *arg)
{
    int rc, len, frames = 0;
    char buff[MBTK_PCM_WB_BUF_SIZE];

    audio_info->info.recorder.state = AUDIO_RECORDER_STATE_RUNNING;
    pthread_mutex_init(&audio_info->info.recorder.mutex, NULL);
    pthread_cond_init(&audio_info->info.recorder.cond, NULL);

    while (TRUE) {
        /* Playback loop */
        pthread_mutex_lock(&audio_info->info.recorder.mutex);
        if(audio_info->info.recorder.state == AUDIO_RECORDER_STATE_STOP) {
            LOGD("Stop recorder...");
            pthread_mutex_unlock(&audio_info->info.recorder.mutex);
            break;
        } else if(audio_info->info.recorder.state == AUDIO_RECORDER_STATE_PAUSE) {
            pthread_cond_wait(&audio_info->info.recorder.cond, &audio_info->info.recorder.mutex);
            pthread_mutex_unlock(&audio_info->info.recorder.mutex);
            continue;
        } else {
            pthread_mutex_unlock(&audio_info->info.recorder.mutex);
        }

        //record the needed format stream from the device.
        //only read pcm stream, no send command.
        len = audio_info->info.recorder.stream_in->read(audio_info->info.recorder.stream_in, buff,
                    audio_info->playback_size);
        if (len <= 0) {
            LOGE("%s: error reading!", __FUNCTION__);
            goto thread_end;
        }

        AUDIO_LOG("Recorder data : len - %d", len);

        if(audio_info->info.recorder.recorder_cb) {
            audio_info->info.recorder.recorder_cb(buff, len);
        }

        LOGD("%s: No.%d frame playback.", __FUNCTION__, ++frames);
    }


thread_end:
    pthread_mutex_destroy(&audio_info->info.recorder.mutex);
    pthread_cond_destroy(&audio_info->info.recorder.cond);

    audio_info->info.recorder.stream_in->common.standby(&audio_info->info.recorder.stream_in->common);
    audio_info->audio_ahw_dev_ubus->close_input_stream(audio_info->audio_ahw_dev_ubus, audio_info->info.recorder.stream_in);
    VCMDeinit();//close the fd of audiostub_ctl when exit the thread.
    audio_info->info.recorder.stream_in = NULL;
    LOGD("%s: finished pcm playback.", __FUNCTION__);

    // Notify audio recorder data end.
    if(audio_info->info.recorder.recorder_cb) {
        audio_info->info.recorder.recorder_cb(NULL, 0);
    }
    return;
}


static int config_parameters(mbtk_audio_direction_enum direction, mbtk_audio_sample_rate_enum NBWB)
{
    unsigned int srcdst, priority, dest;
    char kvpair[128];
    struct str_parms *param = NULL;
    int data[5];
    const char *key = NULL;
    bool update_vcm = false;

    srcdst = 1;/* 0-None, 1-Near end, 2-Far end, 3-Both ends */
    priority = 1;/* 0-Do not combine(override), 1-Combine */
    dest = 1;/* 0-Near codec, 1-Near Vocoder */

    if(direction == MBTK_AUDIO_DIRECTION_OUTPUT){//output
        if(NBWB == MBTK_AUDIO_SAMPLE_RATE_8000)
            audio_info->playback_size = MBTK_PCM_NB_BUF_SIZE;
        else
            audio_info->playback_size = MBTK_PCM_WB_BUF_SIZE;

        LOGD("config playback parameters.");
    }
    else if(direction == MBTK_AUDIO_DIRECTION_INPUT){//input
        if(NBWB == MBTK_AUDIO_SAMPLE_RATE_8000)
            audio_info->playback_size = MBTK_PCM_NB_BUF_SIZE;
        else
            audio_info->playback_size = MBTK_PCM_WB_BUF_SIZE;

        LOGD("config record parameters.");
    }

    memset(kvpair, 0x00, sizeof(kvpair));
    sprintf(kvpair, "%s=%d;%s=%d;%s=%d;%s=%d;%s=%d", VCM_CONFIG_DIRECTION, direction,
            VCM_CONFIG_TYPE, NBWB, VCM_CONFIG_SRC_DST, srcdst,
            VCM_CONFIG_PRIORITY, priority, VCM_CONFIG_DEST, dest);

    LOGD("%s: config information kvpair is %s.\n", __FUNCTION__, kvpair);

    //extract the parameter and config from string
    param = str_parms_create_str(kvpair);
    if (!param) {
        LOGE("%s: param create str is null!", __FUNCTION__);
        return -1;
    }

    //set vcm configurations
    key = VCM_CONFIG_DIRECTION;
    if (str_parms_get_int(param, key, &data[0]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_TYPE;
    if (str_parms_get_int(param, key, &data[1]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_SRC_DST;
    if (str_parms_get_int(param, key, &data[2]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_PRIORITY;
    if (str_parms_get_int(param, key, &data[3]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_DEST;
    if (str_parms_get_int(param, key, &data[4]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }

    //printf("Direction is %d, Type is %d, Src_Dst is %d, Priority is %d, Dest is %d. \n",data[0], data[1], data[2], data[3], data[4]);

    if (update_vcm) {
        configure_vcm(data);   /*TODO check if all inputs got all values successfully*/
    }

    return 0;
}

int mbtk_audio_pcm_init()
{
    //mbtk_log_init("radio", "MBTK_AUDIO");
    // Audio is running...
    if(lock_get()) {
        return -1;
    }

    if(audio_info) {
        return 0;
    }

    audio_info = (audio_inter_info_t*)malloc(sizeof(audio_inter_info_t));
    if(audio_info == NULL) {
        LOGE("malloc() fail:%d", errno);
        return -1;
    }
    memset(audio_info, 0x00, sizeof(audio_inter_info_t));

    // Set default audio parameter.
    audio_info->channel = 1;
    audio_info->sample_rate = MBTK_AUDIO_SAMPLE_RATE_8000;
    audio_info->playback_size = MBTK_PCM_NB_BUF_SIZE;

    audio_info->audio_ahw_dev_ubus = audio_hal_install();
    if (audio_info->audio_ahw_dev_ubus == NULL) {
        LOGE("audio_hal_install() failed!");
        goto init_fail;
    }
    return 0;
init_fail:
    free(audio_info);
    audio_info = NULL;
    return -1;
}

void mbtk_audio_pa_func_set(mbtk_audio_pa_switch_func pa_switch_func)
{
	audio_pa_switch_func = pa_switch_func;
}

void mbtk_audio_volume_set_func_set(mbtk_audio_volume_set_func volume_set_func)
{
	audio_volume_set_func = volume_set_func;
}

int mbtk_audio_pcm_sample_rate_set(mbtk_audio_sample_rate_enum sample_rate)
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    audio_info->sample_rate = sample_rate;

    return 0;
}

int mbtk_audio_pcm_play_start()
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->audio_ahw_dev_ubus) {
        LOGE("audio_info->audio_ahw_dev_ubus is NULL.");
        return -1;
    }

    config_parameters(MBTK_AUDIO_DIRECTION_OUTPUT, audio_info->sample_rate);

    int rc = audio_info->audio_ahw_dev_ubus->open_output_stream(audio_info->audio_ahw_dev_ubus, 0,
            audio_info->audio_ahw_dev_ubus->get_supported_devices(audio_info->audio_ahw_dev_ubus),
            AUDIO_OUTPUT_FLAG_DIRECT, NULL, &(audio_info->info.play.stream_out), 0);
    if (rc < 0) {
        LOGE("Open output device fail:%d", rc);
        goto play_start_fail;
    }

    audio_info->direction = MBTK_AUDIO_DIRECTION_OUTPUT;
    audio_info->info.play.buff_remain_len = 0;
    audio_pcm_write_count = 0;

    return 0;
play_start_fail:

    return -1;
}

int mbtk_audio_pcm_play_data_send(const void* data,uint32 data_len)
{
    UNUSED(data);
    UNUSED(data_len);

    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->info.play.stream_out) {
        LOGE("Output device not open.");
        return -1;
    }

    uint32 index = 0;
    // There are remaining data from the previous package。
    if(audio_info->info.play.buff_remain_len > 0) {
        // Too less for one package.
        if(data_len + audio_info->info.play.buff_remain_len < audio_info->playback_size) {
            AUDIO_LOG("Save remain data : len - %d", data_len);
            memcpy(audio_info->info.play.buff_remain + audio_info->info.play.buff_remain_len, data, data_len);
            audio_info->info.play.buff_remain_len += data_len;
            return data_len;
        } else {
            AUDIO_LOG("Write remain data : %d + %d", audio_info->info.play.buff_remain_len, audio_info->playback_size - audio_info->info.play.buff_remain_len);
            memcpy(audio_info->info.play.buff_remain + audio_info->info.play.buff_remain_len, data, audio_info->playback_size - audio_info->info.play.buff_remain_len);

            int rc = audio_info->info.play.stream_out->write(audio_info->info.play.stream_out, audio_info->info.play.buff_remain, audio_info->playback_size);
            if (rc < 0) {
                LOGE("%s: error writing (child).", __FUNCTION__);
                goto send_fail;
            } else if (rc < (signed int)audio_info->playback_size) {
                LOGW("%s: wrote less than buffer size, rc=%d.", __FUNCTION__, rc);
                index += rc;
                goto send_fail;
            }

            if(audio_pcm_write_count < 3) {
                audio_pcm_write_count++;
                if(audio_pcm_write_count == 3) {
                    // Set audio volume.
                    if(audio_volume_set_func) {
                        audio_volume_set_func();
                    }

                    // Open PA
                    if(audio_pa_switch_func) {
                        audio_pa_switch_func(TRUE);
                    }
                }
            }

            index += (audio_info->playback_size - audio_info->info.play.buff_remain_len);
            audio_info->info.play.buff_remain_len = 0;
        }
    }

    while(data_len - index >= audio_info->playback_size) {
        AUDIO_LOG("Package : %d -> %d", index, index + audio_info->playback_size - 1);
        int rc = audio_info->info.play.stream_out->write(audio_info->info.play.stream_out, (const char*)data + index, audio_info->playback_size);
        if (rc < 0) {
            LOGE("%s: error writing (child).", __FUNCTION__);
            goto send_fail;
        } else if (rc < (signed int)audio_info->playback_size) {
            LOGW("%s: wrote less than buffer size, rc=%d.", __FUNCTION__, rc);
            goto send_fail;
        }

        index += rc;
    }

    // Last package.( less then audio_info->playback_size)
    // Save to buffer audio_info->play_buff_remain
    if(data_len - index > 0) {
        AUDIO_LOG("Save remain data : len - %d", data_len - index);
        memcpy(audio_info->info.play.buff_remain, data + index, data_len - index);
        audio_info->info.play.buff_remain_len = data_len - index;
    }

    return data_len;
send_fail:
    return -1;
}

int mbtk_audio_pcm_play_stop()
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->info.play.stream_out) {
        LOGE("Output device not open.");
        return -1;
    }

    // Write last package.
    if(audio_info->info.play.buff_remain_len > 0) {
        char buf[MBTK_PCM_WB_BUF_SIZE];
        memset(buf, 0x00, sizeof(buf));
        memcpy(buf, audio_info->info.play.buff_remain, audio_info->info.play.buff_remain_len);

        LOGD("len %d is smaller than needed %d, so fill the buffer with 0.", audio_info->info.play.buff_remain_len, audio_info->playback_size);

        int rc = audio_info->info.play.stream_out->write(audio_info->info.play.stream_out, buf, audio_info->playback_size);
        if (rc < 0) {
            LOGE("%s: error writing (child).", __FUNCTION__);
            //goto send_fail;
        } else if (rc < (signed int)audio_info->playback_size) {
            LOGW("%s: wrote less than buffer size, rc=%d.", __FUNCTION__, rc);
            //goto send_fail;
        }
        audio_info->info.play.buff_remain_len = 0;
    }

    // Close PA
    if(audio_pa_switch_func) {
        audio_pa_switch_func(FALSE);
    }

    vcm_playback_drain(80);//wait for drain the AP audiostub queue.
    usleep(80000);//delay 80ms until DSP play out its buffered data.
    audio_info->info.play.stream_out->common.standby(&(audio_info->info.play.stream_out->common));
    audio_info->audio_ahw_dev_ubus->close_output_stream(audio_info->audio_ahw_dev_ubus, audio_info->info.play.stream_out);

    audio_info->info.play.stream_out = NULL;
    return 0;
}


int mbtk_audio_pcm_recorder_start(mbtk_recorder_callback_func recorder_cb)
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->audio_ahw_dev_ubus) {
        LOGE("audio_info->audio_ahw_dev_ubus is NULL.");
        return -1;
    }

    if(audio_info->info.recorder.state != AUDIO_RECORDER_STATE_STOP) {
        LOGW("Audio is recorder...");
        return -1;
    }

    config_parameters(MBTK_AUDIO_DIRECTION_INPUT, audio_info->sample_rate);

    VCMInit();

    int rc = audio_info->audio_ahw_dev_ubus->open_input_stream(audio_info->audio_ahw_dev_ubus, 0,
                audio_info->audio_ahw_dev_ubus->get_supported_devices(audio_info->audio_ahw_dev_ubus),
                NULL, &(audio_info->info.recorder.stream_in), 0, 0, AUDIO_SOURCE_VOICE_CALL);
    if (rc < 0) {
        LOGE("Open input device fail:%d", rc);
        goto recorder_start_fail;
    }

    audio_info->direction = MBTK_AUDIO_DIRECTION_INPUT;
    audio_info->info.recorder.recorder_cb = recorder_cb;

    pthread_attr_t thread_attr;
    pthread_attr_init(&thread_attr);
    if(pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))
    {
        LOGE("pthread_attr_setdetachstate() fail.");
        return -1;
    }

    if (pthread_create(&recorder_thread_play, NULL, (void *)&audio_recorder_thread, NULL) < 0) {
        LOGE("%s: error creating thread_recorder!", __FUNCTION__);
        return -1;
    }

    return 0;
recorder_start_fail:

    return -1;
}

int mbtk_audio_pcm_recorder_pause()
{
    int result = 0;
    pthread_mutex_lock(&audio_info->info.recorder.mutex);
    if(audio_info->info.recorder.state == AUDIO_RECORDER_STATE_RUNNING) {
        audio_info->info.recorder.state = AUDIO_RECORDER_STATE_PAUSE;
    } else {
        result = -1;
        LOGW("Audio state : %d", audio_info->info.recorder.state);
    }
    pthread_mutex_unlock(&audio_info->info.recorder.mutex);
    return result;
}

int mbtk_audio_pcm_recorder_resume()
{
    int result = 0;
    pthread_mutex_lock(&audio_info->info.recorder.mutex);
    if(audio_info->info.recorder.state == AUDIO_RECORDER_STATE_PAUSE) {
        audio_info->info.recorder.state = AUDIO_RECORDER_STATE_RUNNING;
        pthread_cond_signal(&audio_info->info.recorder.cond);
    } else {
        result = -1;
        LOGW("Audio state : %d", audio_info->info.recorder.state);
    }
    pthread_mutex_unlock(&audio_info->info.recorder.mutex);
    return result;
}


int mbtk_audio_pcm_recorder_stop()
{
    int result = 0;
    pthread_mutex_lock(&audio_info->info.recorder.mutex);
    if(audio_info->info.recorder.state == AUDIO_RECORDER_STATE_PAUSE || audio_info->info.recorder.state == AUDIO_RECORDER_STATE_RUNNING) {
        if(audio_info->info.recorder.state == AUDIO_RECORDER_STATE_PAUSE) {
            pthread_cond_signal(&audio_info->info.recorder.cond);
        }
        audio_info->info.recorder.state = AUDIO_RECORDER_STATE_STOP;
        pthread_mutex_unlock(&audio_info->info.recorder.mutex);

        LOGD("Waitting recorder thread exit...");
        if (pthread_join(recorder_thread_play, NULL)) {
            LOGE("error join thread_recorder!");
            // abort();
        }
        LOGD("Recorder thread exit success.");
    } else {
        pthread_mutex_unlock(&audio_info->info.recorder.mutex);
        result = -1;
        LOGW("Audio state : %d", audio_info->info.recorder.state);
    }

    return result;
}

int mbtk_audio_pcm_deinit()
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    audio_hal_uninstall();

    if(lock_free()) {
        return -1;
    }

    free(audio_info);
    audio_info = NULL;
    return 0;
}


typedef enum {
	AUD_PLAYER_ERROR = -1,
	AUD_PLAYER_START = 0,
	AUD_PLAYER_PAUSE,
	AUD_PLAYER_RESUME,
	AUD_PLAYER_NODATA,  //Buff no data and play tread will sleep
    AUD_PLAYER_LESSDATA, //Buff has less data
	AUD_PLAYER_FINISHED,
} Enum_AudPlayer_State;

struct mopen_audio_t
{
    int device;
    audio_hw_device_t *audio_ahw_dev_ubus;
    struct audio_stream_in *stream_in;
    struct audio_stream_out *stream_out;
    int pcm_packet_size;     //320:NB, 640:WB
    int pipe_data;
    int fd[2];
    pthread_t pid;
    mbtk_audio_state_enum state;
    pthread_mutex_t _cond_mutex;
    pthread_mutex_t _stream_mutex;
    void *usrData;
};

static struct mopen_audio_t *internal_hdl = NULL;
typedef int (*_play_callback)(int hdl, int result);


static int config_parameters_new(int in_out, int NBWB)
{
    unsigned int direction = 0xFF, type, srcdst, priority, dest;
    char kvpair[128];
    struct str_parms *param = NULL;
    int data[5];
    const char *key = NULL;
    bool update_vcm = false;

    direction = in_out;/* 0-play, 1-record */
    type = NBWB; /* 0:PCM_NB_BUF_SIZE, 1:PCM_WB_BUF_SIZE */
    srcdst = 1;/* 0-None, 1-Near end, 2-Far end, 3-Both ends */
    priority = 1;/* 0-Do not combine(override), 1-Combine */
    dest = 1;/* 0-Near codec, 1-Near Vocoder */

    memset(kvpair, 0x00, sizeof(kvpair));
    sprintf(kvpair, "%s=%d;%s=%d;%s=%d;%s=%d;%s=%d", VCM_CONFIG_DIRECTION, direction,
            VCM_CONFIG_TYPE, type, VCM_CONFIG_SRC_DST, srcdst,
            VCM_CONFIG_PRIORITY, priority, VCM_CONFIG_DEST, dest);

    LOGD("%s: config information kvpair is %s.\n", __FUNCTION__, kvpair);

    //extract the parameter and config from string
    param = str_parms_create_str(kvpair);
    if (!param) {
        printf("%s: param create str is null!", __FUNCTION__);
        return -1;
    }

    //set vcm configurations
    key = VCM_CONFIG_DIRECTION;
    if (str_parms_get_int(param, key, &data[0]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_TYPE;
    if (str_parms_get_int(param, key, &data[1]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_SRC_DST;
    if (str_parms_get_int(param, key, &data[2]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_PRIORITY;
    if (str_parms_get_int(param, key, &data[3]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_DEST;
    if (str_parms_get_int(param, key, &data[4]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }

    //printf("Direction is %d, Type is %d, Src_Dst is %d, Priority is %d, Dest is %d. \n",data[0], data[1], data[2], data[3], data[4]);

    if (update_vcm) {
        configure_vcm(data);   /*TODO check if all inputs got all values successfully*/
    }

    return 0;
}

static int audio_open_pcm_new(void *dev_hdl, int num_channels, int rate, int in_out)
{
    struct mopen_audio_t *pcxt = (struct mopen_audio_t *)dev_hdl;
    int nb_wb;

    //Support 8k/16k & mono wave file
    if (((rate != 8000) && (rate != 16000))
            || (num_channels != 1) ) {
        LOGD("%s: error wave file:rate = %d, num_channels = %d!! \n",
                __FUNCTION__,rate, num_channels);
        return -1;
    }

    LOGD("%s: success open, rate = %d, num_channels = %d.\n",
            __FUNCTION__, rate, num_channels);

    if ((8000 == rate) && (1 == num_channels)) {
        nb_wb = 0;//NB
        pcxt->pcm_packet_size = 320;
    } else if ((16000 == rate) && (1 == num_channels)) {
        nb_wb = 1;//WB
        pcxt->pcm_packet_size = 640;
    }

    //config playback parameters.
    config_parameters_new(in_out, nb_wb);
    pcxt->device = in_out;
    return 0;
}

static void mbtk_audio_set_status_new(void* hdl, mbtk_audio_state_enum _status)
{
    struct mopen_audio_t *pcxt = (struct mopen_audio_t *)hdl;
    if (NULL == hdl || NULL == internal_hdl)
        return 0;
    pthread_mutex_lock(&pcxt->_cond_mutex);
    pcxt->state = _status;
    pthread_mutex_unlock(&pcxt->_cond_mutex);
}

mbtk_audio_handle mbtk_audio_open_new(mbtk_audio_dev_enum dev, int flag, int rate, void *usrData)
{

    int value = 1;
    struct mopen_audio_t *aud_hdl = NULL;
    int ret;

    if(internal_hdl)
    {
        printf("Audio device inited\n");
        return internal_hdl;
    }

    aud_hdl = (struct mopen_audio_t *) calloc(1, sizeof(struct mopen_audio_t));
    if (!aud_hdl)
    {
        LOGD("\n%s:Failed to allocate audio hdl!, errno=%d\n", __FUNCTION__, errno);
        return NULL;
    }
    memset(aud_hdl, 0, sizeof(struct mopen_audio_t));
    LOGD("mbtk_audio_open() success  aud_hdl:%p\n", aud_hdl);

    //init global variables
    aud_hdl->audio_ahw_dev_ubus = audio_hal_install();
    if (aud_hdl->audio_ahw_dev_ubus == NULL) {
        LOGD("%s: audio_hal_install failed!\n", __FUNCTION__);
        goto error;
    }

    ret = audio_open_pcm_new(aud_hdl, flag, rate, dev);
    if (ret)
    {
        LOGD("\n%s: pcm open error, errno=%d\n", __FUNCTION__, errno);
        goto error;
    }

    ret = aud_hdl->audio_ahw_dev_ubus->open_output_stream(aud_hdl->audio_ahw_dev_ubus, 0,
        aud_hdl->audio_ahw_dev_ubus->get_supported_devices(aud_hdl->audio_ahw_dev_ubus),
        AUDIO_OUTPUT_FLAG_DIRECT, NULL, &aud_hdl->stream_out, 0);

    if (ret < 0) {
        LOGD("%s: error opening output device. rc = %d\n", __FUNCTION__, ret);
        goto error;
    }

    pthread_mutex_init(&aud_hdl->_cond_mutex, NULL);
    aud_hdl->usrData = usrData;
    aud_hdl->state = AUDIO_OPEN;
    /* printf("Mbtk_Audio_Open aud_hdl[%x][%x][%x]\n", aud_hdl, aud_hdl->_mixer_ctl, aud_hdl->_pcm); */
    internal_hdl = aud_hdl;

    LOGD("mbtk_audio_open() success  aud_hdl:%p\n", aud_hdl);
    return (void *)aud_hdl;

error:
    value = 0;
    free(aud_hdl);
    return NULL;
}

int mbtk_audio_play_file_new(void *dev_hdl, int file_fd, int offset)
{
    unsigned bufsize = 0;
    char *data = NULL;
    int first_set = 0;
    int file_data_sz = 0;
    int res = 0;
    int ret;

    char cmd[128] = {0};
    bool flay_flag = TRUE;
    int i = 0;


    struct mopen_audio_t *pcxt = (struct mopen_audio_t *)dev_hdl;
    _play_callback audio_play_cb = (_play_callback)pcxt->usrData;
    if (NULL == dev_hdl || NULL == internal_hdl)
        return -1;


    LOGD("mbtk_audio_play_file_new()  dev_hdl:%p\n", dev_hdl);

    if(AUDIO_RUNNING == pcxt->state)
        return -2;

    // file_data_sz = mbtk_wav_pcm16Le_check(file_fd);
    bufsize = pcxt->pcm_packet_size;
    data = calloc(1, bufsize);
    if (!data) {
        fprintf(stderr, "\n%s:could not allocate %d bytes\n",__FUNCTION__,bufsize);
        pthread_exit(NULL);
    }
    if(offset)
    {
        lseek(file_fd, offset, SEEK_SET);
    }

    mbtk_audio_set_status_new(dev_hdl, AUDIO_RUNNING);
    int all_size = 0;

 //   system("echo 1 >/sys/class/gpio/gpio19/value");
    while (pcxt->state != AUDIO_STOP)
    {
        res = read(file_fd, data, bufsize);
      //   printf("%s:read : %d bytes\n", __FUNCTION__, res);
        if(res == 0 || res < 0)
        {
             LOGD("read:[%d]", res);
            break;
        }

        all_size += res;
//        if (file_data_sz < all_size || file_data_sz == all_size) {
//            printf("aplay size :%d - %d\n", file_data_sz, all_size);
 //           break;
 //       }

        while(AUDIO_PAUSE == pcxt->state)
        {
            usleep(80000);
            audio_play_cb(pcxt, AUD_PLAYER_PAUSE);
        }

        if ((0 == first_set || AUDIO_RESUME == pcxt->state))
        {
            first_set = 1;
            mbtk_audio_set_status_new(dev_hdl, AUDIO_RUNNING);
            audio_play_cb(pcxt, AUD_PLAYER_RESUME);
        }

        ret = pcxt->stream_out->write(pcxt->stream_out, data, bufsize);
        if (ret < 0) {
            LOGD("%s: error writing (child).\n", __FUNCTION__);
            audio_play_cb(pcxt, AUD_PLAYER_ERROR);
            break;
        } else if (ret < (signed int)pcxt->pcm_packet_size) {
            LOGD("%s: wrote less than buffer size, rc=%d.\n", __FUNCTION__, ret);
            audio_play_cb(pcxt, AUD_PLAYER_LESSDATA);
            break;
        }

    }
    if (audio_play_cb)
        audio_play_cb(pcxt, AUD_PLAYER_FINISHED);

    LOGD("file_data_sz :%d - all_size: %d\n", file_data_sz, all_size);
    mbtk_audio_set_status_new(dev_hdl, AUDIO_OPEN);
    free(data);
//    system("echo 0 >/sys/class/gpio/gpio19/value");

    return 0;
}

int mbtk_audio_play_stream_new(void *dev_hdl, const void *pData, int len, int gain )
{
    unsigned bufsize = 0;
    char *data = NULL;
    int first_set = 0;
    int res = 0;
    int ret = 0;
    int read_size = 0;
    char *p = (char *)pData;

    char cmd[128] = {0};
    bool flay_flag = TRUE;
    int i = 0;

    struct mopen_audio_t *pcxt = (struct mopen_audio_t *)dev_hdl;

    if (NULL == dev_hdl || NULL == internal_hdl || pData == NULL)
        return -1;

    LOGD("mbtk_audio_play_stream_new()  dev_hdl:%p\n", dev_hdl);

    if(AUDIO_RUNNING == pcxt->state)
        return -2;

    bufsize = pcxt->pcm_packet_size;
    data = calloc(1, bufsize);
    if (!data) {
        fprintf(stderr, "\n%s:could not allocate %d bytes\n",__FUNCTION__,bufsize);
        pthread_exit(NULL);
    }

    if(pcxt->state == AUDIO_OPEN)
    {
        mbtk_audio_set_status_new(dev_hdl, AUDIO_RUNNING);

    }

 //   system("echo 1 >/sys/class/gpio/gpio19/value");
    while (pcxt->state != AUDIO_STOP)
    {

        memcpy(data,p+read_size, bufsize );
        read_size += bufsize;

        if(read_size > len)
        {
         //    printf(">[%d]\n", read_size);
            break;
        }

        while(AUDIO_PAUSE == pcxt->state)
        {
            usleep(80000);
        }

        if ((0 == first_set || AUDIO_RESUME == pcxt->state))
        {
            first_set = 1;
            mbtk_audio_set_status_new(dev_hdl, AUDIO_RUNNING);
        }

        ret = pcxt->stream_out->write(pcxt->stream_out, data, bufsize);
        if (ret < 0) {
            LOGD("%s: error writing (child).\n", __FUNCTION__);
            break;
        } else if (ret < (signed int)pcxt->pcm_packet_size) {
            LOGD("%s: wrote less than buffer size, rc=%d.\n", __FUNCTION__, ret);
            break;
        }

        if(read_size == len)
        {
         //    printf("=[%d]", read_size);
            break;
        }

        i++;
        if(i == 3)  // 该值可以自行调节
        {
            if(flay_flag && gain != 50)
            {
                sprintf(cmd, "ubus call audio_if config_dspgain \"{\'type\':1, \'gain\':%d}\"", gain);
                system(cmd);

                flay_flag = FALSE;
                usleep(80000);
            }
        }

    }

    if(pcxt->state != AUDIO_STOP)
    {
        mbtk_audio_set_status_new(dev_hdl, AUDIO_OPEN);
    }
 //   system("echo 0 >/sys/class/gpio/gpio19/value");
    free(data);
    return 0;
}

int mbtk_audio_close_new(void *dev_hdl)
{
    LOGD("mbtk_audio_close()\n");
    int value = 0;
    struct mopen_audio_t *_hdl = (struct mopen_audio_t *)dev_hdl;

    if (NULL == _hdl || NULL == internal_hdl )
    {
        LOGD("mbtk_audio_close() fail dev_hdl is NULL\n");
        return -1;
    }

    mbtk_audio_set_status_new(_hdl, AUDIO_STOP);

    vcm_playback_drain(0);//wait for drain the AP audiostub queue.
    usleep(80000);//delay 80ms until DSP play out its buffered data.
    _hdl->stream_out->common.standby(&_hdl->stream_out->common);
    _hdl->audio_ahw_dev_ubus->close_output_stream(_hdl->audio_ahw_dev_ubus, _hdl->stream_out);

    audio_hal_uninstall();
    pthread_mutex_destroy(&_hdl->_cond_mutex);
    free(_hdl);
    _hdl = NULL;
    internal_hdl = NULL;

    return 0;
}

void mbtk_audio_lib_info_print()
{
    MBTK_SOURCE_INFO_PRINT("mbtk_audio_lib");
}

