/*!
 *****************************************************************************
 **
 ** \file        fwup.c
 ** \brief       firmware update function
 **
 ** \attention   This sample code is provided "as is" without warranty of any
 **              kind. We disclaim all warranties, either express or implied,
 **              including the warranties of merchantability and fitness for a
 **              particular purpose.
 **
 **
 *****************************************************************************/
//#include "hd8040_upgrade.h"
#include "port.h"

#define READ_LENGTH 128
#define VERSION_MAX_LEN 16

const uint8_t cmdMONVER[] = {0xf1, 0xd9, 0x0a, 0x04, 0x00, 0x00, 0x0e, 0x34};
const uint8_t cmdSETFRQ[] = {0xf1, 0xd9, 0xf4, 0x00, 0x04, 0x00, 0x80, 0xba, 0x8c, 0x01, 0xbf, 0xff};
uint8_t cmdSETFRQ_ONLY[] = {0xF1, 0xD9, 0xF4, 0x03, 0x08, 0x00, 0x00, 0x08, 0x07, 0x00, 0x00, 0x08, 0x07, 0x00, 0x1D, 0x69};
const uint8_t cmdCFGFWUP[] = {0xf1, 0xd9, 0x06, 0x50, 0x01, 0x00, 0x10, 0x67, 0x71, 0x00};
uint8_t cmdOpenWrtPro[] = 	{0xF1, 0xD9, 0xF4, 0x08, 0x04, 0x00, 0x00, 0x02, 0x00, 0x80, 0x82, 0x76};
uint8_t cmdOpenQe_and_wrtPro[] = {0xF1, 0xD9, 0xF4, 0x13, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x0E, 0x55};
const uint8_t cmdBOOTERASE[] = {0xf1, 0xd9, 0xf4, 0x05, 0x06, 0x00, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x8f, 0x95};
uint8_t cmdBOOTERASE_100[] = {0xf1, 0xd9, 0xf4, 0x05, 0x06, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x0f, 0x25};
const uint8_t cmdCFGRST[] = {0xf1, 0xd9, 0x06, 0x40, 0x01, 0x00, 0x00, 0x47, 0x21};
uint8_t cmdBOOTERASE_900[] = {0xf1, 0xd9, 0xf4, 0x05, 0x06, 0x00, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x8f, 0x95};
uint8_t cmd_RadFlash[] = {0xF1, 0xD9, 0xF4, 0x07, 0x01, 0x00, 0x02, 0xFE, 0xE5};

const uint8_t cmdBOOTBAUD[] = {0xf1, 0xd9, 0xf4, 0x03, 0x08, 0x00, 0x00, 0xc2, 0x01, 0x00, 0x00, 0xc2, 0x01, 0x00, 0x85, 0x7d};
uint8_t cmdNMEAOFF[] = {
0xf1, 0xd9, 0x06, 0x01, 0x03, 0x00, 0xf0, 0x00, 0x00, 0xfa, 0x0f,
0xf1, 0xd9, 0x06, 0x01, 0x03, 0x00, 0xf0, 0x02, 0x00, 0xfc, 0x13,
0xf1, 0xd9, 0x06, 0x01, 0x03, 0x00, 0xf0, 0x03, 0x00, 0xfd, 0x15,
0xf1, 0xd9, 0x06, 0x01, 0x03, 0x00, 0xf0, 0x04, 0x00, 0xfe, 0x17,
0xf1, 0xd9, 0x06, 0x01, 0x03, 0x00, 0xf0, 0x05, 0x00, 0xff, 0x19,
0xf1, 0xd9, 0x06, 0x01, 0x03, 0x00, 0xf0, 0x06, 0x00, 0x00, 0x1b,
0xf1, 0xd9, 0x06, 0x01, 0x03, 0x00, 0xf0, 0x07, 0x00, 0x01, 0x1d
};

void uart_clear_buf(int32_t fd)
{
    tcflush(fd, TCIFLUSH);
	//clearBuffer(fd, TCIOFLUSH);
	return;
}

static int
byte_array_to_hex_string(char *pResult, uint8_t const * pData, int dataLen)
{
    int strLen = 0;
    int res = 0;
    char * pMsg = pResult;
    int i = 0;

    if(!pData || !pResult || dataLen <= 0)
    {
        HDBD_LOG("[%s %d]Para err, return -1\n", __FUNCTION__, __LINE__);
        return -1;
    }

    for(i = 0; i < dataLen; ++i)
    {
        /*  The first Hex value does not need a leading '-' */
        if(i != 0)
        {
            res = snprintf(pMsg + strLen, 2, "-");
            /* increase the total strLength */
            if(res >= 0)
            {
                strLen += res;
            }
            else
            {
                HDBD_LOG("[%s %d]add leading Error, return -1\n", __FUNCTION__, __LINE__);
                return -1;
            }
        }

        /*  Print the hex character to the string */
        res = snprintf(pMsg + strLen, 3, "%02X", pData[i]);
        if(res >= 0)
        {
            strLen += res;
        }
        else
        {
            HDBD_LOG("[%s %d]Print the hex character to the string Error, return -1\n", __FUNCTION__, __LINE__);
            return -1;
        }
    }

    return strLen;
}


/**
  * @brief  send cmd to chip wait result
  * @param  uart_fd : uart fd
  * @param  cmd : send data command
  * @param  recv_buf : send result
  * @param  send_length: send cmd length
  * @param  recv_length : result length
  * @param  timeout : wait result time
  * @retval SUCCESS/FAIL
  */
static int
send_cmd_and_recv_result(int uart_fd, uint8_t *cmd, uint8_t *recv_buf, int send_length, int *recv_length, int timeout)
{
    int ret = -1;

    ret = uart_send(uart_fd, cmd, send_length);
    if(ret == -1)
    {
        HDBD_LOG("[%s %d] uart send failed\n", __FUNCTION__, __LINE__);
        return  ret;
    }

    ret = uart_recv(uart_fd, recv_buf, recv_length, timeout);
    if(ret == -1)
    {
        HDBD_LOG("[%s %d] uart recv failed\n", __FUNCTION__, __LINE__);
        return  ret;
    }

    return 0;
}

/**
  * @brief  Boot mode send one block data of firmware
  * @param  addr: address of the data in HD8030 chip memory
  * @param  cnt : the number of the block
  * @param  data: the pointer of the block data
  * @param  len : the length of data
  * @retval None
  */
int sendfwboot(int uart_fd, uint32_t addr, uint16_t cnt, uint8_t *data, uint32_t len, uint8_t *recv_buf)
{
    uint8_t cmd[] = {0xf1, 0xd9, 0xf4, 0x05, 0x06, 0x08, 0x00, 0x08, 0x00, 0x00, 0x01, 0x00};
	uint8_t ck[2] = { 0 };
    uint32_t i = 0;
	uint32_t ck1 = 0, ck2 = 0;
    int actual_length = 10;//READ_LENGTH-1;
    int time_out = 3; /* 1 second */

	*((uint16_t *)(cmd + 4)) = len + 6;
	*((uint32_t *)(cmd + 6)) = addr;
	*((uint16_t *)(cmd + 10)) = cnt;
	//check sum
	for(i = 0; i < 12; ++i)
	{
		if(i >= 2)
		{
			ck1 += cmd[i];
			ck2 += ck1;
		}
	}
	for(i = 0; i < len; ++i)
	{
		ck1 += data[i];
		ck2 += ck1;
	}
	ck[0] = ck1;
	ck[1] = ck2;

	uart_send(uart_fd, (uint8_t *)cmd, 12);
	uart_send(uart_fd, (uint8_t *)data, len);
	//uart_send((uint8_t *)ck, 2);
    send_cmd_and_recv_result(uart_fd, (uint8_t *)ck, recv_buf, 2, &actual_length, time_out);
    HDBD_LOG("[%s %d]boot up: cnt is <%d>, recv [%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x]\n", __FUNCTION__, __LINE__, cnt, recv_buf[0], recv_buf[1], recv_buf[2], recv_buf[3],recv_buf[4], recv_buf[5], recv_buf[6], recv_buf[7], recv_buf[8], recv_buf[9]);
	if( (recv_buf[0] == 0xF1) && (recv_buf[1] == 0xD9) && (recv_buf[2] == 0x05) && (recv_buf[3] == 0x01) )
	{
		return 0;
	}

	return -1;
}




/**
  * @brief  get version from chip,if user do not save version, so get_mon_ver(NULL, 0)
  * @param  ver_buf:save version buffer
  * @param  ver_len:length of version buffer
  * @retval SUCCESS/FAIL
  */
int get_mon_ver(int uart_fd,uint8_t *ver_buf, int ver_len)
{
    int ret = -1;
    int actual_length = 40;
    uint8_t cur_sw_ver[VERSION_MAX_LEN + 1] = {0};
    uint8_t cur_hw_ver[VERSION_MAX_LEN + 1] = {0};
    uint8_t recv_buf[READ_LENGTH] = {0};
    //uint8_t *p1 = recv_buf;
	uint16_t f1StarAddr = 0;
	uint8_t read_count = 0;
    int time_out = 3; /* 1 second */
    char str_hex[512] = {0};

    if(NULL == ver_buf || ver_len < 41)
    {
        HDBD_LOG("[%s %d]Para mybe error, FAIL\n", __FUNCTION__, __LINE__);
        return -1;
    }

    memset(recv_buf, 0x0, sizeof(recv_buf));
    memset(cur_sw_ver, 0x0, VERSION_MAX_LEN);
    memset(cur_hw_ver, 0x0, VERSION_MAX_LEN);

	HDBD_LOG("[%s %d] enter\n", __FUNCTION__, __LINE__);
    //ret = send_cmd_and_recv_result(uart_fd,cmdMONVER, recv_buf, sizeof(cmdMONVER), &actual_length, time_out);
    ret = uart_send(uart_fd, cmdMONVER, sizeof(cmdMONVER));
    if(ret == -1)
    {
        HDBD_LOG("[%s %d] uart send failed\n", __FUNCTION__, __LINE__);
        return  ret;
    }

	read_count = 1;
	while(1)
	{
		if((read_count % 10) == 0)
		{
			ret = uart_send(uart_fd, cmdMONVER, sizeof(cmdMONVER));
			if(ret == -1)
			{
				HDBD_LOG("[%s %d] uart send failed\n", __FUNCTION__, __LINE__);
				return	ret;
			}
		}

	    ret = uart_recv(uart_fd, recv_buf, &actual_length, time_out);
	    if(ret == -1)
	    {
	        HDBD_LOG("[%s %d] uart recv failed\n", __FUNCTION__, __LINE__);
	        return  ret;
	    }

		f1StarAddr = 0;
		HDBD_LOG("[%s %d] actual_length:%d, read_count:%d\n", __FUNCTION__, __LINE__, actual_length, read_count);
		memset(str_hex, 0x0, sizeof(str_hex));
		byte_array_to_hex_string(str_hex, recv_buf+f1StarAddr, actual_length);
		HDBD_LOG("[%s %d]str_hex : %s\n", __FUNCTION__, __LINE__, str_hex);
		while(f1StarAddr < (actual_length-16))
		{
			if( (recv_buf[f1StarAddr] == 0xF1) && (recv_buf[f1StarAddr+1] == 0xD9) )
			{
				goto READ_VER;
			}

			f1StarAddr ++;
		}

		read_count ++;
		if(read_count >= 25)
		{
	        return  -1;
		}

	}

READ_VER:
	if(f1StarAddr >= (actual_length-16))
	{
		HDBD_LOG("[%s %d] get ver fail\n", __FUNCTION__, __LINE__);
		return -1;
	}

	//memset(str_hex, 0x0, sizeof(str_hex));
    //byte_array_to_hex_string(str_hex, recv_buf+f1StarAddr, actual_length);
    //ALOGI("[%s %d]Version : %s\n", __FUNCTION__, __LINE__, str_hex);
    memcpy(cur_sw_ver, recv_buf + f1StarAddr + 6, VERSION_MAX_LEN);
    memcpy(cur_hw_ver, recv_buf + f1StarAddr + 6 + VERSION_MAX_LEN, VERSION_MAX_LEN);
    HDBD_LOG("[%s %d]=====> sw_ver:%s\n", __FUNCTION__, __LINE__, cur_sw_ver);
    HDBD_LOG("[%s %d]=====> hw_ver:%s\n", __FUNCTION__, __LINE__, cur_hw_ver);

    if ( (strncmp((char *)(recv_buf + f1StarAddr + 6), "FB03BOOT", 8) != 0)
		&& (strncmp((char *)(recv_buf + f1StarAddr + 6), "FB02BOOT", 8) != 0) )
    {
        HDBD_LOG("[%s %d] user mode\n", __FUNCTION__, __LINE__);

        if(strncmp((char *)(recv_buf + f1StarAddr + 6 + VERSION_MAX_LEN), "HD", 2) == 0)
		{
			HDBD_LOG("[%s %d] get user ver sucess\n", __FUNCTION__, __LINE__);
		}
		else
		{
			HDBD_LOG("[%s %d] get user ver fail\n", __FUNCTION__, __LINE__);
			return -1;
		}
    }
	else
    {
        HDBD_LOG("[%s %d]boot_mode\n", __FUNCTION__, __LINE__);
    }

    /* COPY version to user, the length of recv_buf is more than 40 bytes */
    if (ver_buf != NULL && (ver_len > 16))
    {
        memcpy(ver_buf, cur_sw_ver, 16);
    }

    return 0;
}


/**
  * @brief  Boot mode firmware update
  * @param  data: the pointer of the firmware data
  * @param  len : the length of firmware data
  * @retval None
  */
  //8040update
int fw_update_boot(int uart_fd, uint8_t *data, uint32_t len)
{
	int i = 1;
//	uint8_t dataAck[10];
//  1st 1KB send at last
	uint8_t *data1st1k = data;
    uint8_t recvBuf[READ_LENGTH] = {0};
	uint8_t errorCount = 0;
	int32_t ret = 0;
    if (NULL == data )
    {
        HDBD_LOG("[%s %d]firmware data error\n", __FUNCTION__, __LINE__);
        return -1;
    }

	uart_clear_buf(uart_fd);
	errorCount = 0;
	while(1)
	{
		ret = get_mon_ver(uart_fd, (uint8_t *)recvBuf, READ_LENGTH - 1);
		errorCount ++;
		if (ret != 0)
		{
			HDBD_LOG("[%s %d]boot: Get version FAIL in baudrate 460800, return\n", __FUNCTION__, __LINE__);
			if(errorCount >= 4)
			{
				HDBD_LOG("[%s %d] enter boot failed!\n", __FUNCTION__, __LINE__);
				return -1;
			}
		}
		else
		{
			break;
		}
	}

	i = 0;
	while(i < 15)
	{
		if( (recvBuf[i] == 'F') && (recvBuf[i+1] == 'B') && (recvBuf[i+2] == '0') && (recvBuf[i+3] == '3'))
		{
			goto BOOT;
		}
		HDBD_LOG("[%s %d] enter boot failed!\n", __FUNCTION__, __LINE__);
		return -1;
	}

BOOT:
	ret = uart_send(uart_fd, (uint8_t *)cmdSETFRQ, sizeof(cmdSETFRQ));
	usleep(50000);

	uart_clear_buf(uart_fd);
	memset(recvBuf, 0x00, sizeof(recvBuf));
	i = 10;

	ret = send_cmd_and_recv_result(uart_fd,cmdBOOTERASE_100, recvBuf, sizeof(cmdBOOTERASE_100), &i, 3);
	HDBD_LOG("[%s %d]ret:%d,cmdOpenQe_and_wrtPro(%d)(%02x,%x,%x,%x,%x,%x,%x,%x,%x,%x)\n", __FUNCTION__, __LINE__, ret, i,
		recvBuf[0],recvBuf[1],recvBuf[2],recvBuf[3],recvBuf[4],recvBuf[5],
		recvBuf[6],recvBuf[7],recvBuf[8],recvBuf[9]);
	if( (recvBuf[0] == 0xF1) && (recvBuf[1] == 0xD9) && (recvBuf[2] == 0x05) && (recvBuf[3] != 0x01) )
	{
		HDBD_LOG("[%s %d] cmdBOOTERASE error!\n", __FUNCTION__, __LINE__);
		return -1;
	}

	uart_clear_buf(uart_fd);

	data += 1024;
	len -= 1024;
	for(i = 1 ;len > 1024; ++i, data += 1024, len -= 1024)
	{
		ret = sendfwboot(uart_fd, 0x00100000 + i * 0x400, i, data, 1024, recvBuf);
		if (ret != 0)
        {
            HDBD_LOG("[%s %d] upgrade error,return!\n", __FUNCTION__, __LINE__);
			return -1;
        }
	}

	ret = sendfwboot(uart_fd, 0x00100000 + i * 0x400, i, data, len, recvBuf);
	if (ret != 0)
	{
		HDBD_LOG("[%s %d] upgrade error,return!\n", __FUNCTION__, __LINE__);
		return -1;
	}
	i++;

	ret = sendfwboot(uart_fd, 0x00100000, i, data1st1k, 1024, recvBuf);
	if (ret != 0)
	{
		HDBD_LOG("[%s %d] upgrade error,return!\n", __FUNCTION__, __LINE__);
		return -1;
	}

	//uart_send((uint8_t *)cmdCFGRST, 9);
	HDBD_LOG("[%s %d] upgrade success!\n", __FUNCTION__, __LINE__);
	return 0;
}


//8122update
int fw_update_boot_8122(int uart_fd, uint8_t *data, uint32_t len)
{
	int i = 1;
//	uint8_t dataAck[10];
//  1st 1KB send at last
	uint8_t *data1st1k = data;
    uint8_t recvBuf[READ_LENGTH] = {0};
	uint8_t errorCount = 0;
	int32_t ret = 0;
	int j = 0;
    if (NULL == data )
    {
        HDBD_LOG("[%s %d]firmware data error\n", __FUNCTION__, __LINE__);
        return -1;
    }

	uart_clear_buf(uart_fd);
	errorCount = 0;
	while(1)
	{
		ret = get_mon_ver(uart_fd, (uint8_t *)recvBuf, READ_LENGTH - 1);
		errorCount ++;
		if (ret != 0)
		{
			HDBD_LOG("[%s %d]boot: Get version FAIL in baudrate 460800, return\n", __FUNCTION__, __LINE__);
			if(errorCount >= 3)
			{
				HDBD_LOG("[%s %d] enter boot failed!\n", __FUNCTION__, __LINE__);
				return -1;
			}
		}
		else
		{
			break;
		}
	}

	i = 0;
	while(i < 15)
	{
		if( (recvBuf[i] == 'F') && (recvBuf[i+1] == 'B') && (recvBuf[i+2] == '0') && (recvBuf[i+3] == '3'))
		{
			goto BOOT;
		}
		HDBD_LOG("[%s %d] enter boot failed!\n", __FUNCTION__, __LINE__);
		return -1;
	}

BOOT:
	ret = uart_send(uart_fd, (uint8_t *)cmdSETFRQ, sizeof(cmdSETFRQ));
	usleep(50000);

	uart_clear_buf(uart_fd);
	memset(recvBuf, 0x00, sizeof(recvBuf));
	i = 10;
	ret = send_cmd_and_recv_result(uart_fd,cmdOpenWrtPro, recvBuf, sizeof(cmdOpenWrtPro), &i, 3);
	HDBD_LOG("[%s %d]ret:%d,recvBuf(%d)(%02x,%x,%x,%x,%x,%x,%x,%x,%x,%x)\n", __FUNCTION__, __LINE__, ret, i,
		recvBuf[0],recvBuf[1],recvBuf[2],recvBuf[3],recvBuf[4],recvBuf[5],
		recvBuf[6],recvBuf[7],recvBuf[8],recvBuf[9]);
	if( (recvBuf[8] != 0x04) || (recvBuf[9] != 0x1b))
	{
		HDBD_LOG("[%s %d] upgrade write:open flash write protect failed!\n", __FUNCTION__, __LINE__);
		return -1;
	}


	i = 10;
	ret = send_cmd_and_recv_result(uart_fd, cmdOpenQe_and_wrtPro, recvBuf, sizeof(cmdOpenQe_and_wrtPro), &i, 3);
	HDBD_LOG("[%s %d]ret:%d,cmdOpenQe_and_wrtPro(%d)(%02x,%x,%x,%x,%x,%x,%x,%x,%x,%x)\n", __FUNCTION__, __LINE__, ret, i,
		recvBuf[0],recvBuf[1],recvBuf[2],recvBuf[3],recvBuf[4],recvBuf[5],
		recvBuf[6],recvBuf[7],recvBuf[8],recvBuf[9]);
	if((recvBuf[2] == 0x05) && (recvBuf[3] == 0x00))
	{

		i = 10;
		ret = send_cmd_and_recv_result(uart_fd, cmd_RadFlash, recvBuf, sizeof(cmd_RadFlash), &i, 3);
		HDBD_LOG("[%s %d]ret:%d,cmd_RadFlash(%d)(%02x,%x,%x,%x,%x,%x,%x,%x,%x,%x)\n", __FUNCTION__, __LINE__, ret, i,
			recvBuf[0],recvBuf[1],recvBuf[2],recvBuf[3],recvBuf[4],recvBuf[5],
			recvBuf[6],recvBuf[7],recvBuf[8],recvBuf[9]);
		if((recvBuf[2] != 0xF4) || (recvBuf[3] != 0x07))
		{
			HDBD_LOG("[%s %d] cmd_RadFlash error!\n", __FUNCTION__, __LINE__);
			return -1;
		}

		i = 10;
		ret = send_cmd_and_recv_result(uart_fd, cmdOpenQe_and_wrtPro, recvBuf, sizeof(cmdOpenQe_and_wrtPro), &i, 3);
		HDBD_LOG("[%s %d]ret:%d,cmdOpenQe_and_wrtPro(%d)(%02x,%x,%x,%x,%x,%x,%x,%x,%x,%x)\n", __FUNCTION__, __LINE__, ret, i,
			recvBuf[0],recvBuf[1],recvBuf[2],recvBuf[3],recvBuf[4],recvBuf[5],
			recvBuf[6],recvBuf[7],recvBuf[8],recvBuf[9]);
		j = 0;
		while(j < i)
		{
			if((recvBuf[j] == 0xF1) && (recvBuf[j+1] == 0xD9))
			{
				if((recvBuf[j+2] != 0x05) || (recvBuf[j+3] != 0x01))
				{
					HDBD_LOG("[%s %d] cmdOpenQe_and_wrtPro error!\n", __FUNCTION__, __LINE__);
					return -1;
				}
			}

			j++;

			if(j > 20)
			{
				break;
			}
		}
	}

	ret = send_cmd_and_recv_result(uart_fd,cmdBOOTERASE_900, recvBuf, sizeof(cmdBOOTERASE_900), &i, 3);
	HDBD_LOG("[%s %d]ret:%d,cmdOpenQe_and_wrtPro(%d)(%02x,%x,%x,%x,%x,%x,%x,%x,%x,%x)\n", __FUNCTION__, __LINE__, ret, i,
		recvBuf[0],recvBuf[1],recvBuf[2],recvBuf[3],recvBuf[4],recvBuf[5],
		recvBuf[6],recvBuf[7],recvBuf[8],recvBuf[9]);
	if( (recvBuf[0] == 0xF1) && (recvBuf[1] == 0xD9) && (recvBuf[2] == 0x05) && (recvBuf[3] != 0x01) )
	{
		HDBD_LOG("[%s %d] cmdBOOTERASE error!\n", __FUNCTION__, __LINE__);
		return -1;
	}

	uart_clear_buf(uart_fd);

	data += 1024;
	len -= 1024;
	for(i = 1 ;len > 1024; ++i, data += 1024, len -= 1024)
	{
		ret = sendfwboot(uart_fd, 0x90000000 + i * 0x400, i, data, 1024, recvBuf);
		if (ret != 0)
        {
            HDBD_LOG("[%s %d] upgrade error,return!\n", __FUNCTION__, __LINE__);
			return -1;
        }
	}

	ret = sendfwboot(uart_fd, 0x90000000 + i * 0x400, i, data, len, recvBuf);
	if (ret != 0)
	{
		HDBD_LOG("[%s %d] upgrade error,return!\n", __FUNCTION__, __LINE__);
		return -1;
	}
	i++;

	ret = sendfwboot(uart_fd, 0x90000000, i, data1st1k, 1024, recvBuf);
	if (ret != 0)
	{
		HDBD_LOG("[%s %d] upgrade error,return!\n", __FUNCTION__, __LINE__);
		return -1;
	}

	//uart_send((uint8_t *)cmdCFGRST, 9);
	HDBD_LOG("[%s %d] upgrade success!\n", __FUNCTION__, __LINE__);
	return 0;
}

