#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mbtk_type.h"
#include "mbtk_device.h"

static mbtk_device_info_header_t info_header = {
    .tag = MBTK_DEVICE_INFO_PARTITION_TAG,
    .version = MBTK_DEVICE_INFO_CURR_VERSION,
    .item_count = MBTK_DEVICE_INFO_ITEM_NUM,
    .item_header = {
        {MBTK_DEVICE_INFO_ITEM_BASIC, MBTK_DEVICE_INFO_ITEM_ADDR_BASIC},
        {MBTK_DEVICE_INFO_ITEM_FOTA, MBTK_DEVICE_INFO_ITEM_ADDR_FOTA},
        {MBTK_DEVICE_INFO_ITEM_MODEM, MBTK_DEVICE_INFO_ITEM_ADDR_MODEM},
        {MBTK_DEVICE_INFO_ITEM_LOG, MBTK_DEVICE_INFO_ITEM_ADDR_LOG},
    }
};

static mbtk_device_info_basic_t item_basic = {
    .name = MBTK_DEVICE_INFO_ITEM_STR_BASIC,
    .version = MBTK_DEVICE_INFO_CURR_VERSION,
    .project = {0},
    .project_cust = {0},
    .ab_support = 1,            // Default for ab system.
    .reboot_flag = MBTK_REBOOT_FLAG_NORMAL,
    .revision_out = {0},
    .revision_in = {0},
    .build_time = {0},
    .asr_baseline = {0}
};

static mbtk_device_info_fota_t item_fota = {
    .name = MBTK_DEVICE_INFO_ITEM_STR_FOTA,
    .version = MBTK_DEVICE_INFO_CURR_VERSION,
    .state = 0
};

static mbtk_device_info_modem_t item_modem = {
    .name = MBTK_DEVICE_INFO_ITEM_STR_MODEM,
    .version = MBTK_DEVICE_INFO_CURR_VERSION,
    .band_area = MBTK_MODEM_BAND_AREA_ALL,  // Default for all bands.
    .net_pref = 15,        // Default *band is 15
    .net_support = MBTK_NET_SUPPORT_2G | MBTK_NET_SUPPORT_3G | MBTK_NET_SUPPORT_4G,      // Default support 2G/3G/4G
    .band_gsm = MBTK_BAND_ALL_GSM_DEFAULT,
    .band_wcdma = MBTK_BAND_ALL_WCDMA_DEFAULT,
    .band_tdlte = MBTK_BAND_ALL_TDLTE_DEFAULT,
    .band_fddlte = MBTK_BAND_ALL_FDDLTE_DEFAULT,
    .band_lte_ext = MBTK_BAND_ALL_EXT_LTE_DEFAULT,
    .band_nr_3 = MBTK_BAND_ALL_NR_3_DEFAULT,
    .band_nr_2 = MBTK_BAND_ALL_NR_2_DEFAULT,
    .band_nr_1 = MBTK_BAND_ALL_NR_1_DEFAULT,
    .band_nr_0 = MBTK_BAND_ALL_NR_0_DEFAULT
};

static mbtk_device_info_log_t item_log = {
    .name = MBTK_DEVICE_INFO_ITEM_STR_LOG,
    .version = MBTK_DEVICE_INFO_CURR_VERSION,
    .state = 0
};

static void help()
{
    printf("device_info_generate -a [a/ab] -b [revision_out] -c [revision_in] -d [project] -e [project_cust] -f [cn/eu/all] -g [build_time] -h [net_pref] -i [net_support] -j [asr_baseline] -o [out_bin]\n");
}

static int update_and_write_header(int fd, mbtk_device_info_header_t *header)
{
    if(sizeof(mbtk_device_info_header_t) != write(fd, header, sizeof(mbtk_device_info_header_t))) {
        printf("Write header fail:%d\n", errno);
        return -1;
    }

    return 0;
}

static int write_item_basic(int fd, uint32 addr, mbtk_device_info_basic_t *item_basic)
{
    if(-1 == lseek(fd, addr, SEEK_SET)) {
        printf("lseek() fail:%d\n", errno);
        return -1;
    }

    if(sizeof(mbtk_device_info_basic_t) != write(fd, item_basic, sizeof(mbtk_device_info_basic_t))) {
        printf("Write item basic fail:%d\n", errno);
        return -1;
    }

    return 0;
}

static int write_item_fota(int fd, uint32 addr, mbtk_device_info_fota_t *item_fota)
{
    if(-1 == lseek(fd, addr, SEEK_SET)) {
        printf("lseek() fail:%d\n", errno);
        return -1;
    }

    if(sizeof(mbtk_device_info_fota_t) != write(fd, item_fota, sizeof(mbtk_device_info_fota_t))) {
        printf("Write item fota fail:%d\n", errno);
        return -1;
    }

    return 0;
}

static int write_item_modem(int fd, uint32 addr, mbtk_device_info_modem_t *item_modem)
{
    if(-1 == lseek(fd, addr, SEEK_SET)) {
        printf("lseek() fail:%d\n", errno);
        return -1;
    }

    if(sizeof(mbtk_device_info_modem_t) != write(fd, item_modem, sizeof(mbtk_device_info_modem_t))) {
        printf("Write item modem fail:%d\n", errno);
        return -1;
    }

    return 0;
}

static int write_item_log(int fd, uint32 addr, mbtk_device_info_log_t *item_log)
{
    if(-1 == lseek(fd, addr, SEEK_SET)) {
        printf("lseek() fail:%d\n", errno);
        return -1;
    }

    if(sizeof(mbtk_device_info_log_t) != write(fd, item_log, sizeof(mbtk_device_info_log_t))) {
        printf("Write item log fail:%d\n", errno);
        return -1;
    }

    return 0;
}

static char* net_support_str_get(uint32 net_support)
{
    static char net_str[100] = {0};

    if(net_support & 0x01) { // GSM
        if(strlen(net_str) > 0) {
            strcat(net_str, "/2G");
        } else {
            strcat(net_str, "2G");
        }
    }

    if(net_support & 0x02) { // WCDMA
        if(strlen(net_str) > 0) {
            strcat(net_str, "/3G");
        } else {
            strcat(net_str, "3G");
        }
    }

    if(net_support & 0x04) { // LTE
        if(strlen(net_str) > 0) {
            strcat(net_str, "/4G");
        } else {
            strcat(net_str, "4G");
        }
    }

    if(net_support & 0x08) { // NR
        if(strlen(net_str) > 0) {
            strcat(net_str, "/5G");
        } else {
            strcat(net_str, "5G");
        }
    }
    return net_str;
}

/*
*
* device_info_generate -a [a/ab] -b [revision_out] -c [revision_in] -d [project] -e [project_cust] -f [cn/eu/sa/all] -o [out_bin]
*
*/
int main(int argc, char *argv[])
{
    int ch;
    char out_bin[128] = {0};
    while((ch = getopt(argc, argv, "a:b:c:d:e:f:g:h:i:j:o:"))!= -1){
        switch(ch)
        {
            case 'a':
                if(strcmp(optarg, "ab") == 0) {
                    item_basic.ab_support = 1;
                } else if(strcmp(optarg, "a") == 0) {
                    item_basic.ab_support = 0;
                } else {
                    printf("Must be a/ab.\n");
                    return -1;
                }
                break;
            case 'b':
                if(strlen(optarg) > 0)
                    memcpy(item_basic.revision_out, optarg, strlen(optarg));
                break;
            case 'c':
                if(strlen(optarg) > 0)
                    memcpy(item_basic.revision_in, optarg, strlen(optarg));
                break;
            case 'd':
                if(strlen(optarg) > 0)
                    memcpy(item_basic.project, optarg, strlen(optarg));
                break;
            case 'e':
                if(strlen(optarg) > 0)
                    memcpy(item_basic.project_cust, optarg, strlen(optarg));
                break;
            case 'f':
                if(strcmp(optarg, "cn") == 0) {
                    item_modem.band_area = MBTK_MODEM_BAND_AREA_CN;
                    item_modem.band_gsm = MBTK_BAND_CN_GSM_DEFAULT;
                    item_modem.band_wcdma = MBTK_BAND_CN_WCDMA_DEFAULT;
                    item_modem.band_tdlte = MBTK_BAND_CN_TDLTE_DEFAULT;
                    item_modem.band_fddlte = MBTK_BAND_CN_FDDLTE_DEFAULT;
                    item_modem.band_lte_ext = MBTK_BAND_CN_EXT_LTE_DEFAULT;
                    item_modem.band_nr_3 = MBTK_BAND_CN_NR_3_DEFAULT;
                    item_modem.band_nr_2 = MBTK_BAND_CN_NR_2_DEFAULT;
                    item_modem.band_nr_1 = MBTK_BAND_CN_NR_1_DEFAULT;
                    item_modem.band_nr_0 = MBTK_BAND_CN_NR_0_DEFAULT;
                } else if(strcmp(optarg, "eu") == 0) {
                    item_modem.band_area = MBTK_MODEM_BAND_AREA_EU;
                    item_modem.band_gsm = MBTK_BAND_EU_GSM_DEFAULT;
                    item_modem.band_wcdma = MBTK_BAND_EU_WCDMA_DEFAULT;
                    item_modem.band_tdlte = MBTK_BAND_EU_TDLTE_DEFAULT;
                    item_modem.band_fddlte = MBTK_BAND_EU_FDDLTE_DEFAULT;
                    item_modem.band_lte_ext = MBTK_BAND_EU_EXT_LTE_DEFAULT;
                    item_modem.band_nr_3 = MBTK_BAND_EU_NR_3_DEFAULT;
                    item_modem.band_nr_2 = MBTK_BAND_EU_NR_2_DEFAULT;
                    item_modem.band_nr_1 = MBTK_BAND_EU_NR_1_DEFAULT;
                    item_modem.band_nr_0 = MBTK_BAND_EU_NR_0_DEFAULT;
                } else if(strcmp(optarg, "sa") == 0) {
                    item_modem.band_area = MBTK_MODEM_BAND_AREA_SA;
                    item_modem.band_gsm = MBTK_BAND_SA_GSM_DEFAULT;
                    item_modem.band_wcdma = MBTK_BAND_SA_WCDMA_DEFAULT;
                    item_modem.band_tdlte = MBTK_BAND_SA_TDLTE_DEFAULT;
                    item_modem.band_fddlte = MBTK_BAND_SA_FDDLTE_DEFAULT;
                    item_modem.band_lte_ext = MBTK_BAND_SA_EXT_LTE_DEFAULT;
                    item_modem.band_nr_3 = MBTK_BAND_SA_NR_3_DEFAULT;
                    item_modem.band_nr_2 = MBTK_BAND_SA_NR_2_DEFAULT;
                    item_modem.band_nr_1 = MBTK_BAND_SA_NR_1_DEFAULT;
                    item_modem.band_nr_0 = MBTK_BAND_SA_NR_0_DEFAULT;
                } else {
                    item_modem.band_area = MBTK_MODEM_BAND_AREA_ALL;
                    item_modem.band_gsm = MBTK_BAND_ALL_GSM_DEFAULT;
                    item_modem.band_wcdma = MBTK_BAND_ALL_WCDMA_DEFAULT;
                    item_modem.band_tdlte = MBTK_BAND_ALL_TDLTE_DEFAULT;
                    item_modem.band_fddlte = MBTK_BAND_ALL_FDDLTE_DEFAULT;
                    item_modem.band_lte_ext = MBTK_BAND_ALL_EXT_LTE_DEFAULT;
                    item_modem.band_nr_3 = MBTK_BAND_ALL_NR_3_DEFAULT;
                    item_modem.band_nr_2 = MBTK_BAND_ALL_NR_2_DEFAULT;
                    item_modem.band_nr_1 = MBTK_BAND_ALL_NR_1_DEFAULT;
                    item_modem.band_nr_0 = MBTK_BAND_ALL_NR_0_DEFAULT;
                    printf("Set to default band.\n");
                }
                break;
            case 'h':
                item_modem.net_pref = (uint32)atoi(optarg);
                printf("Set net_pref to %d success.\n", item_modem.net_pref);
                break;
            case 'i':
                item_modem.net_support = (uint32)atoi(optarg);
                printf("Set net_support to %d(%s) success.\n", item_modem.net_support,
                    net_support_str_get(item_modem.net_support));
                break;
            case 'g':
                if(strlen(optarg) > 0)
                    memcpy(item_basic.build_time, optarg, strlen(optarg));
                break;
            case 'j':
                if(strlen(optarg) > 0)
                    memcpy(item_basic.asr_baseline, optarg, strlen(optarg));
                break;
            case 'o':
                if(strlen(optarg) > 0)
                    memcpy(out_bin, optarg, strlen(optarg));
                break;
            default:
                help();
                return -1;
        }
    }
    if(strlen(item_basic.revision_out) == 0 || strlen(out_bin) == 0) {
        help();
        return -1;
    }

    printf("Version:%s, Bin:%s\n", item_basic.revision_out, out_bin);

    int fd = open(out_bin, O_WRONLY | O_TRUNC | O_CREAT, 0644);
    if(fd < 0) {
        printf("Open(%s) fail:%d\n", out_bin, errno);
        return -1;
    }

    if(update_and_write_header(fd, &info_header)) {
        printf("update_and_write_header() fail.");
        goto fail;
    }

    if(write_item_basic(fd, info_header.item_header[MBTK_DEVICE_INFO_ITEM_BASIC].addr, &item_basic)) {
        printf("update_and_write_item_basic() fail.");
        goto fail;
    }

    if(write_item_fota(fd, info_header.item_header[MBTK_DEVICE_INFO_ITEM_FOTA].addr, &item_fota)) {
        printf("update_and_write_item_fota() fail.");
        goto fail;
    }

    if(write_item_modem(fd, info_header.item_header[MBTK_DEVICE_INFO_ITEM_MODEM].addr, &item_modem)) {
        printf("update_and_write_item_modem() fail.");
        goto fail;
    }

    if(write_item_log(fd, info_header.item_header[MBTK_DEVICE_INFO_ITEM_LOG].addr, &item_log)) {
        printf("update_and_write_item_log() fail.");
        goto fail;
    }

    printf("Success generate device_info bin:%s\n", out_bin);
    close(fd);
    return 0;
fail:
    close(fd);
    return -1;
}

