/*
*    mbtk_device_info.c
*
*    MBTK device partition information utils.
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN        WHO       WHAT,WHERE,WHY
--------    --------    -------------------------------------------------------
2024/2/27     LiuBin      Initial version

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <mtd/mtd-user.h>

#include "mbtk_type.h"
#include "mbtk_log.h"
#include "mbtk_str.h"
#include "mbtk_mtd.h"

#ifdef MBTK_DEV_INFO_VERSION_2
#include "mbtk_device_v2.h"
#else
#include "mbtk_device.h"
#endif


/*
* MBTK_DEVICE_INFO_ITEM_BASIC - mbtk_device_info_basic_t
* MBTK_DEVICE_INFO_ITEM_FOTA - mbtk_device_info_fota_t
* MBTK_DEVICE_INFO_ITEM_MODEM - mbtk_device_info_modem_t
* MBTK_DEVICE_INFO_ITEM_LOG - mbtk_device_info_log_t
*/
int mbtk_dev_info_read(mbtk_device_info_item_enum item_type, void *item_ptr, int item_size)
{
    int fd, len, i;
    mbtk_device_info_header_t info_header;
    memset(&info_header, 0, sizeof(mbtk_device_info_header_t));

    switch(item_type) {
        case MBTK_DEVICE_INFO_ITEM_BASIC:
        {
            if(item_ptr == NULL || item_size != sizeof(mbtk_device_info_basic_t)) {
                LOGE("ARG error:item-%d, item_size-%d", item_type, item_size);
                return -1;
            }
            break;
        }
        case MBTK_DEVICE_INFO_ITEM_FOTA:
        {
            if(item_ptr == NULL || item_size != sizeof(mbtk_device_info_fota_t)) {
                LOGE("ARG error:item-%d, item_size-%d", item_type, item_size);
                return -1;
            }
            break;
        }
        case MBTK_DEVICE_INFO_ITEM_MODEM:
        {
            if(item_ptr == NULL || item_size != sizeof(mbtk_device_info_modem_t)) {
                LOGE("ARG error:item-%d, item_size-%d", item_type, item_size);
                return -1;
            }
            break;
        }
        case MBTK_DEVICE_INFO_ITEM_LOG:
        {
            if(item_ptr == NULL || item_size != sizeof(mbtk_device_info_log_t)) {
                LOGE("ARG error:item-%d, item_size-%d", item_type, item_size);
                return -1;
            }
            break;
        }
        default:
        {
            LOGE("Item type[%d] error.", item_type);
            return -1;
        }
    }

    mbtk_partition_info_t *partition_info = mbtk_partition_get();
    if(partition_info == NULL) {
        LOGE("mbtk_partition_get() fail.");
        return -1;
    }

    i = 0;
    char dev[32] = {0};
    while(i < MBTK_PARTITION_NUM_MAX) {
        if(partition_info[i].used && strcmp(partition_info[i].name, MBTK_DEVICE_INFO_PARTITION_NAME) == 0) {
            snprintf(dev, 32, "/dev/%s", partition_info[i].dev);
            LOGD("%s -> %s", partition_info[i].name, dev);
            break;
        }
        i++;
    }
    if(strlen(dev) == 0) {
        LOGE("DEV is null.");
        return -1;
    }

    fd = open(dev, O_RDONLY);
    if (fd < 0) {
        LOGE("Fatal error: can't open device_info %s\n", dev);
        return -1;
    }

    len = read(fd, &info_header, sizeof(mbtk_device_info_header_t));
    if (len != sizeof(mbtk_device_info_header_t)) {
        LOGE("Fatal error: read %d bytes(expect %d)\n", len, sizeof(mbtk_device_info_header_t));
        close(fd);
        goto fail;
    }

    if(info_header.tag != MBTK_DEVICE_INFO_PARTITION_TAG) {
        LOGE("TAG error : %08x", info_header.tag);
        goto fail;
    }

    if(info_header.version != MBTK_DEVICE_INFO_CURR_VERSION) {
        LOGE("Version error : %d", info_header.version);
        goto fail;
    }

    LOGD("Item count:%d", info_header.item_count);

    if(info_header.item_header[item_type].addr == 0) {
        LOGE("No found item : %d", item_type);
        goto fail;
    }

    lseek(fd, info_header.item_header[item_type].addr, SEEK_SET);
    if (read(fd, item_ptr, item_size) != item_size) {
        LOGE("Read fail:%d", errno);
        goto fail;
    }

    close(fd);
    return 0;

fail:
    close(fd);
    return -1;
}

int mbtk_dev_info_write(mbtk_device_info_item_enum item_type, void *item_ptr, int item_size)
{
    if(item_ptr == NULL) {
        LOGE("ARG error.");
        return -1;
    }

    switch(item_type) {
        case MBTK_DEVICE_INFO_ITEM_BASIC:
        {
            if(item_size != sizeof(mbtk_device_info_basic_t)) {
                LOGE("item_size != sizeof(mbtk_device_info_basic_t)\n\r");
                return -1;
            }
            break;
        }
        case MBTK_DEVICE_INFO_ITEM_FOTA:
        {
            if(item_size != sizeof(mbtk_device_info_fota_t)) {
                LOGE("item_size != sizeof(mbtk_device_info_fota_t)\n\r");
                return -1;
            }
            break;
        }
        case MBTK_DEVICE_INFO_ITEM_MODEM:
        {
            if(item_size != sizeof(mbtk_device_info_modem_t)) {
                LOGE("item_size != sizeof(mbtk_device_info_modem_t)\n\r");
                return -1;
            }
            break;
        }
        case MBTK_DEVICE_INFO_ITEM_LOG:
        {
            if(item_size != sizeof(mbtk_device_info_log_t)) {
                LOGE("item_size != sizeof(mbtk_device_info_log_t)\n\r");
                return -1;
            }
            break;
        }
        default:
        {
            LOGE("Item type[%d] error.\n\r", item_type);
            return -1;
        }
    }

    mbtk_partition_info_t info;
    memset(&info, 0x0, sizeof(mbtk_partition_info_t));
    if(mbtk_partition_get_by_name("device_info", &info)) {
        LOGE("mbtk_partition_get_by_name() fail.");
        return -1;
    }

    LOGD("device_info name : %s, dev : %s, addr : %08x, size : %08x, erase_size : %08x", info.name,
        info.dev, info.partition_start, info.partition_size, info.erase_size);

    if(info.erase_size <= 0 || info.partition_size <= 0) {
        LOGE("partition info error.");
        return -1;
    }

    int fd = open(info.dev, O_RDWR);
    if (fd < 0) {
        LOGE("Fatal error: can't open device_info %s\n", info.dev);
        return -1;
    }

    char *mtd_buff = (char*)malloc(info.erase_size);
    if(mtd_buff == NULL) {
        LOGE("malloc() failed\n");
		return -1;
    }
    memset(mtd_buff, 0xFF, info.erase_size);
    int len = read(fd, mtd_buff, info.erase_size);
    if (len != info.erase_size) {
        LOGE("Fatal error: read %d[%d] bytes(expect %d)\n", len, errno, info.erase_size);
        goto fail;
    }

    struct erase_info_user mtdEraseInfo;
	if (lseek(fd, 0, SEEK_SET) < 0)
	{
		LOGE("seek failed\n");
		return -1;
	}

	mtdEraseInfo.length = info.partition_size;
	mtdEraseInfo.start = 0;
	ioctl(fd, MEMUNLOCK, &mtdEraseInfo);
	ioctl(fd, MEMERASE, &mtdEraseInfo);

    mbtk_device_info_header_t *info_header = (mbtk_device_info_header_t*)mtd_buff;
    memcpy(mtd_buff + info_header->item_header[item_type].addr, item_ptr, item_size);

    lseek(fd, 0, SEEK_SET);
    if (write(fd, mtd_buff, info.erase_size) != info.erase_size) {
        LOGE("Write fail:%d", errno);
        goto fail;
    }

    if(mtd_buff) {
        free(mtd_buff);
    }

    close(fd);
    return 0;

fail:
    close(fd);
    return -1;
}


