#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdint.h>

#include "logo_partition.h"

#pragma pack(push, 1)  // 取消结构体对齐
typedef struct {
    uint16_t bfType;      // 文件类型，应为"BM"（0x4D42）
    uint32_t bfSize;      // 文件总大小
    uint16_t bfReserved1; // 保留字段
    uint16_t bfReserved2;
    uint32_t bfOffBits;   // 像素数据偏移量
} BITMAPFILEHEADER;

typedef struct {
    uint32_t biSize;         // 信息头大小（通常为40）
    int32_t  biWidth;        // 图像宽度（像素）
    int32_t  biHeight;       // 图像高度（像素）
    uint16_t biPlanes;       // 颜色平面数（必须为1）
    uint16_t biBitCount;     // 每像素位数（如24表示RGB）
    uint32_t biCompression;  // 压缩类型（0表示无压缩）
    uint32_t biSizeImage;    // 像素数据大小（含填充）
    int32_t  biXPelsPerMeter;// 水平分辨率（像素/米）
    int32_t  biYPelsPerMeter;// 垂直分辨率
    uint32_t biClrUsed;      // 实际使用的颜色数
    uint32_t biClrImportant; // 重要颜色数
} BITMAPINFOHEADER;
#pragma pack(pop)

#define ANIM_PIC_MAX 100      // 开机动画的最大图片数量 (1.bmp - 100.bmp)

static mbtk_logo_header_t logo_header = {
    .addr = LOGO_ADDR_START,
    .pic_width = 240,
    .pic_height = 320,
    .bg_rgb = 0x00000000        // black
};

static mbtk_animation_header_t animation_header = {
    .addr = ANIMATION_ADDR_START,
    .pic_num = 0,               // Max is ANIM_PIC_MAX
    .pic_width = 240,
    .pic_height = 320,
    .waitting_time = 0,         // ms
    .frame_show_num = 0,        // show forever
    .bg_rgb = 0x00000000        // black
};

static int is_directory_exists(const char *path) {
    struct stat stats;
    if (stat(path, &stats) == 0 && S_ISDIR(stats.st_mode)) {
        return 1; // 目录存在
    }
    printf("%s not exists.\n", path);
    return 0; // 目录不存在或路径是文件
}

static void help()
{
    //printf("logo_partition_generate [pic_dir] -a [logo_w] -b [logo_h] -c [logo_bg] -d [anim_num] -e [anim_w] -f [anim_h] -g [anim_rate] -i [anim_bg]\n");
    printf("logo_partition_generate [pic_dir] -a [logo_bg] -b [waitting_time(ms)] -c [anim_bg] -d [frame_show_num]\n");
}

static int bmp_read(const char* file_name, uint32 *width, uint32 *height, char *buff, int buff_size)
{
    FILE* file = fopen(file_name, "rb");
    if (!file) return 0;    // 文件不存在

    // 读取文件头和信息头
    BITMAPFILEHEADER file_header;
    BITMAPINFOHEADER info_header;
    fread(&file_header, sizeof(file_header), 1, file);
    fread(&info_header, sizeof(info_header), 1, file);

    // 验证是否为BMP文件
    if (file_header.bfType != 0x4D42) {
        fclose(file);
        return -1;
    }

    // 仅支持24位无压缩格式
    if (info_header.biBitCount != 24 || info_header.biCompression != 0) {
        fclose(file);
        return -1;
    }

    *width = info_header.biWidth;
    *height = abs(info_header.biHeight); // 高度可能为负（表示从上到下存储）

    // 计算每行实际字节数（含4字节对齐填充）
    uint32_t row_size = ((info_header.biWidth * 3 + 3) / 4) * 4; // [[9, 18]]
    uint32_t data_size = row_size * abs(info_header.biHeight);   // [[9, 17]]
    if(buff_size < data_size) {
        printf("buff_size too small...");
        return -1;
    }

    // 分配内存并读取像素数据
    fseek(file, file_header.bfOffBits, SEEK_SET);  // 定位到数据区 [[2, 8, 15]]
#if 0
    fread(buff, 1, data_size, file);
#else
    // bmp中y轴是反的，所以从最下面一行开始依次读取一行数据。
    int line_size = info_header.biWidth * 3;
    for(int y = info_header.biHeight - 1; y >= 0; y--) {
        fread(buff + y * line_size, 1, line_size, file);
    }
#endif
    fclose(file);

    return data_size;
}

static int write_header(int fd, mbtk_logo_header_t *logo_h, mbtk_animation_header_t *anim_h)
{
    if(-1 == lseek(fd, 0, SEEK_SET)) {
        printf("lseek() fail:%d\n", errno);
        return -1;
    }

    if(sizeof(mbtk_logo_header_t) != write(fd, logo_h, sizeof(mbtk_logo_header_t))) {
        printf("Write logo header fail:%d\n", errno);
        return -1;
    }

    if(sizeof(mbtk_animation_header_t) != write(fd, anim_h, sizeof(mbtk_animation_header_t))) {
        printf("Write anim header fail:%d\n", errno);
        return -1;
    }

    return 0;
}

static int write_item_logo(int fd, const char *dir, mbtk_logo_header_t *logo_h)
{
    if(-1 == lseek(fd, logo_h->addr, SEEK_SET)) {
        printf("lseek() fail:%d\n", errno);
        return -1;
    }

    char pic_name[256] = {0};
    char buff[4 * 1024 * 1024] = {0}; // 4MB
    int len = 0;
    snprintf(pic_name, sizeof(pic_name), "%s/%s", dir, "logo.bmp");
    if((len = bmp_read(pic_name, &(logo_h->pic_width), &(logo_h->pic_height), buff, sizeof(buff))) <= 0) {
        printf("bmp_read() fail.\n");
        return -1;
    }

    if(len != write(fd, buff, len)) {
        printf("Write logo buffer fail:%d\n", errno);
        return -1;
    }

    return 0;
}

static int write_item_anim(int fd, const char *dir, mbtk_animation_header_t *anim_h)
{
    if(-1 == lseek(fd, anim_h->addr, SEEK_SET)) {
        printf("lseek() fail:%d\n", errno);
        return -1;
    }

    char pic_name[256] = {0};
    char buff[4 * 1024 * 1024] = {0}; // 4MB
    int len, i = 1;     // 1.bmp - ANIM_PIC_MAX.bmp

    for(; i <= ANIM_PIC_MAX; i++) {
        memset(pic_name, 0, sizeof(pic_name));
        snprintf(pic_name, sizeof(pic_name), "%s/%d.bmp", dir, i);

/*
        if(!is_directory_exists(pic_name)) {
            anim_h->pic_num = i - 1;
            printf("Anim pic num is : %d\n", i - 1);
            return 0;
        }
*/

        len = bmp_read(pic_name, &(anim_h->pic_width), &(anim_h->pic_height), buff, sizeof(buff));
        if(len == 0) { // 文件不存在
            anim_h->pic_num = i - 1;
            printf("Anim pic num is : %d\n", i - 1);
            return 0;
        } else if(len < 0) {
            printf("bmp_read() fail.\n");
            return -1;
        }

        if(len != write(fd, buff, len)) {
            printf("Write logo buffer fail:%d\n", errno);
            return -1;
        }
    }

    anim_h->pic_num = ANIM_PIC_MAX;
    return 0;
}


/*
*
* logo_partition_generate [pic_dir] -a [logo_w] -b [logo_h] -c [logo_bg] -d [anim_num] -e [anim_w] -f [anim_h] -g [anim_rate] -i [anim_bg]
* logo_partition_generate [pic_dir] -a [logo_bg] -b [waitting_time(ms)] -c [anim_bg] -d [frame_show_num]
*/
int main(int argc, char *argv[])
{
    int ch;
    char *out_bin = "logo.img";
    uint32 temp_int;
    if(argc < 2) {
        help();
        return -1;
    }

    char *dir = argv[1];
    if(!is_directory_exists(dir)) {
        printf("Dir %s not exists.\n", dir);
        return -1;
    }

    while((ch = getopt(argc, argv, "a:b:c:d:"))!= -1){
        switch(ch)
        {
#if 0
            case 'a':
                temp_int = (uint32)atoi(optarg);
                if(temp_int > 0) {
                    logo_header.pic_width = temp_int;
                } else {
                    printf("logo width error.");
                }
                break;
            case 'b':
                temp_int = (uint32)atoi(optarg);
                if(temp_int > 0) {
                    logo_header.pic_height = temp_int;
                } else {
                    printf("logo height error.");
                }
                break;
            case 'c':
                temp_int = (uint32)strtoul(optarg, NULL, 0); // 自动识别
                logo_header.bg_rgb = temp_int & 0x00FFFFFF;
                break;
            case 'd':
                temp_int = (uint32)atoi(optarg);
                if(temp_int > 0) {
                    animation_header.pic_num = temp_int;
                } else {
                    printf("animation number error.");
                }
                break;
            case 'e':
                temp_int = (uint32)atoi(optarg);
                if(temp_int > 0) {
                    animation_header.pic_width = temp_int;
                } else {
                    printf("animation width error.");
                }
                break;
            case 'f':
                temp_int = (uint32)atoi(optarg);
                if(temp_int > 0) {
                    animation_header.pic_height = temp_int;
                } else {
                    printf("animation height error.");
                }
                break;
            case 'g':
                temp_int = (uint32)atoi(optarg);
                if(temp_int > 0) {
                    animation_header.frame_rate = temp_int;
                } else {
                    printf("animation frame_rate error.");
                }
                break;
            case 'i':
                temp_int = (uint32)strtoul(optarg, NULL, 0); // 自动识别
                animation_header.bg_rgb = temp_int & 0x00FFFFFF;
                break;
#else
            case 'a':
                temp_int = (uint32)strtoul(optarg, NULL, 0); // 自动识别
                logo_header.bg_rgb = temp_int << 8;
                break;
            case 'b':
                temp_int = (uint32)atoi(optarg);
                if(temp_int > 0) {
                    animation_header.waitting_time = temp_int;
                } else {
                    printf("animation waitting_time error.");
                }
                break;
            case 'c':
                temp_int = (uint32)strtoul(optarg, NULL, 0); // 自动识别
                animation_header.bg_rgb = temp_int << 8;
                break;
            case 'd':
                temp_int = (uint32)atoi(optarg);
                if(temp_int >= 0) {
                    animation_header.frame_show_num = temp_int;
                } else {
                    printf("animation frame_show_num error.");
                }
                break;
#endif
            default:
                help();
                return -1;
        }
    }

    printf("Before Logo : addr - 0x%x, w - %d, h - %d, bg - 0x%06x\n", logo_header.addr ,logo_header.pic_width, logo_header.pic_height, logo_header.bg_rgb);
    printf("Before Anim : addr - 0x%x, w - %d, h - %d, bg - 0x%06x, num - %d, waitting_time(ms) - %d, frame_show_num - %d\n", animation_header.addr ,animation_header.pic_width, animation_header.pic_height, animation_header.bg_rgb,
                    animation_header.pic_num, animation_header.waitting_time, animation_header.frame_show_num);

    int fd = open(out_bin, O_WRONLY | O_TRUNC | O_CREAT, 0644);
    if(fd < 0) {
        printf("Open(%s) fail:%d\n", out_bin, errno);
        return -1;
    }

    if(write_item_logo(fd, dir, &logo_header)) {
        printf("write_item_logo() fail.");
        goto fail;
    }

    if(write_item_anim(fd, dir, &animation_header)) {
        printf("write_item_anim() fail.");
        goto fail;
    }

    // Write header in the last.
    if(write_header(fd, &logo_header, &animation_header)) {
        printf("write_header() fail.");
        goto fail;
    }

    printf("After Logo : addr - 0x%x, w - %d, h - %d, bg - 0x%06x\n", logo_header.addr ,logo_header.pic_width, logo_header.pic_height, logo_header.bg_rgb);
    printf("After Anim : addr - 0x%x, w - %d, h - %d, bg - 0x%06x, num - %d, waitting_time(ms) - %d, frame_show_num - %d\n", animation_header.addr ,animation_header.pic_width, animation_header.pic_height, animation_header.bg_rgb,
                    animation_header.pic_num, animation_header.waitting_time, animation_header.frame_show_num);

    printf("Success generate logo_partition bin:%s\n", out_bin);
    close(fd);
    return 0;
fail:
    close(fd);
    return -1;
}

