/*
*    gnss_log.c
*
*    gnss test log.
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN           WHO           WHAT,WHERE,WHY
--------    --------------    -------------------------------------------------------
2024/9/6     wangyouqiang      Initial version

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/time.h>
#include <errno.h>
#include <time.h>

#include "gnss_log.h"
#include "mbtk_log.h"
#include "mbtk_utils.h"

#define GNSS_LOG_FOLDER_PATH      "/user_data/mbtk_log"
#define GNSS_LOG_FILE_PATH        "/user_data/mbtk_log/mbtk_gnss_log.txt"
#define GNSS_LOG_LAST_FILE_PATH   "/user_data/mbtk_log/mbtk_gnss_last_log.txt"
#define GNSS_LOG_GNSS_FILE_HEAD   "\n[-------------------mbtk gnss log V1.0-------------------][%ld]\n"
#define GNSS_LOG_FILE_MAX_SIZE    10485760 //10 * 1024 * 1024 10M
#define GNSS_TIME_CHECK_NUM       10

static int log_file_fd = -1;
static long log_file_size = 0;

static int mbtk_rmc_num = 0;

static int log_file_init(int *fd, long *size, bool mv_flag)
{
    int head_len = 0;
    int ret = -1;
    if(access(GNSS_LOG_FOLDER_PATH, F_OK) != 0)
    {
        ret = mkdir(GNSS_LOG_FOLDER_PATH, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
        if (ret != 0)
        {
            LOGE("Failed to create folder.");
            return -1;
        }
    }

    if(access(GNSS_LOG_FILE_PATH, F_OK) == 0 && mv_flag)
    {
        char cmd[128] = {0};
        memset(cmd, 0x0, 128);
        snprintf(cmd, 128, "mv %s %s", GNSS_LOG_FILE_PATH, GNSS_LOG_LAST_FILE_PATH);
        LOGE("cmd: [%s].", cmd);
        mbtk_system(cmd);
    }

    *fd = open(GNSS_LOG_FILE_PATH, O_CREAT | O_WRONLY | O_APPEND | O_NONBLOCK, 0644);
    if(*fd < 0)
    {
        LOGE("Open file fail [errno = %d].", errno);
        return -1;
    }
    else
    {
        char temp[128] = {0};
        struct stat st;
        stat(GNSS_LOG_FILE_PATH, &st);
        *size = st.st_size;
        snprintf(temp, 128, GNSS_LOG_GNSS_FILE_HEAD, *size);
        head_len = (int)strlen(temp);
        ret = write(*fd, temp, head_len);
        if(ret == head_len)
        {
            *size += ret;
        }
        else
        {
            LOGE("write head fail [errno = %d].", errno);
            goto error;
        }
    }

    return 0;
error:
    if(*fd > 0) {
        close(*fd);
        *fd = -1;
    }
    *size = 0;
    return -1;
}

void mbtk_gnss_time_check(time_t _time)
{
    long diff = 0;
    struct timeval time;
    gettimeofday(&time, NULL);
    time_t sys_time = time.tv_sec;
    time_t gps_time = _time;

    if(gps_time > sys_time)
    {
        diff = gps_time - sys_time;
    }
    else
    {
        diff = sys_time - gps_time;
    }

    gnss_test_log("sys time[%ld], gps time[%ld], diff time[%ld]", sys_time, gps_time, diff);
}

int mbtk_check_num(void)
{
    if(mbtk_rmc_num < GNSS_TIME_CHECK_NUM)
    {
        mbtk_rmc_num++;
        return 1;
    }

    mbtk_rmc_num = 0;
    return 0;
}

void gnss_test_log(const char* format, ...)
{
    int ret;

    if(log_file_fd < 0)
    {
        ret = log_file_init(&log_file_fd, &log_file_size, false);
        if(ret < 0)
        {
            LOGE("log_file_init fail.");
            return;
        }
        else
        {
            LOGE("log fd: [%d], log size: [%ld].", log_file_fd, log_file_size);
        }
    }

    char tmp[50] = {0};
    char param_buf[1024] = {0};
    int length = 0;
    char write_buf[1096] = {0};
    int write_length = 0;

    va_list ap;
    struct timeval log_time;

    va_start(ap, format);
    length = vsnprintf(param_buf, 1023, format, ap);
    va_end(ap);

    if (length < 0 || 0 == length)
    {
        LOGE("vsnprintf fail.");
        return;
    }
    else
    {
        if(param_buf[length - 1] != '\n')
        {
            param_buf[length] = '\n';
        }
    }

    gettimeofday(&log_time, NULL);
    struct tm* tm_t = localtime(&(log_time.tv_sec));
    strftime(tmp, 50, "%F %T", tm_t);

    write_length = snprintf(write_buf, 1096, "[%s %03d]--->%s", tmp, (int)(log_time.tv_usec / 1000), param_buf);
    if(write_length <= 0)
    {
        LOGE("snprintf fail.");
        return;
    }

    if((log_file_size + write_length) > GNSS_LOG_FILE_MAX_SIZE)
    {
        close(log_file_fd);
        log_file_fd = -1;
        log_file_size = 0;

        ret = log_file_init(&log_file_fd, &log_file_size, true);
        if(ret < 0)
        {
            LOGE("log_file_init fail.");
            goto error;
        }
        else
        {
            LOGE("log fd: [%d], log size: [%ld].", log_file_fd, log_file_size);
        }
    }

    ret = write(log_file_fd, write_buf, write_length);
    if(ret == write_length)
    {
        log_file_size += ret;
    }
    else
    {
        goto error;
    }


    if (log_file_fd > 2)
    {
        fsync(log_file_fd);
    }
    return;
error:
    if(log_file_fd > 0)
    {
        close(log_file_fd);
        log_file_fd = -1;
    }
    log_file_size = 0;
    return;
}