/*
*    mbtk_pcm_stream.c
*
*    MBTK audio playback/record PCM source.
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN        WHO       WHAT,WHERE,WHY
--------    --------    -------------------------------------------------------
2024/03/23     LiuBin      Initial version.
2024/10/15     LiuBin      Add PA and volume control API.
2024/11/30     LiuBin      Add get/send PCM from/to the end of voice(RTP).

******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <cutils/str_parms.h>

#include "mbtk_log.h"
#include "mbtk_audio_internal.h"
#include "audio_if_audio_hw_mrvl.h"

#define LOCK_FILE "/var/run/mbtk_audio.lock"
#define AUDIO_DEBUG 0

#define AUDIO_LOG(fmt, args ...) \
    do{ \
        if(AUDIO_DEBUG) {LOGD(fmt, ##args);} \
    } while(0)

extern void* audio_hal_install(void);
extern void audio_hal_uninstall(void);
extern void configure_vcm(unsigned int data[]);

static audio_inter_info_t *audio_info = NULL;
static pthread_t recorder_thread_play;
static mbtk_audio_pa_switch_func audio_pa_switch_func = NULL;
static mbtk_audio_volume_set_func audio_volume_set_func = NULL;
static int audio_pcm_write_count = 0;

static int lock_get() {
    int fd = open(LOCK_FILE, O_WRONLY | O_CREAT | O_TRUNC, 0666);
    if (fd == -1) {
        LOGE("Open(%s) fail:%d", LOCK_FILE, errno);
        return -1;
    }
    // 尝试对文件进行加锁
    struct flock fl;
    fl.l_type = F_WRLCK;
    fl.l_whence = SEEK_SET;
    fl.l_start = 0;
    fl.l_len = 1;
    int ret = fcntl(fd, F_SETLK, &fl);
    if (ret == -1) {
        LOGE("Another instance is running, exiting...");
        close(fd);
        return -1;
    }
    close(fd);
    LOGD("Get file lock.");
    return 0;
}

static int lock_free() {
    int fd = open(LOCK_FILE, O_WRONLY | O_CREAT | O_TRUNC, 0666);
    if (fd == -1) {
        LOGE("Open(%s) fail:%d", LOCK_FILE, errno);
        return -1;
    }
    // 释放文件锁
    struct flock fl;
    fl.l_type = F_UNLCK;
    fl.l_whence = SEEK_SET;
    fl.l_start = 0;
    fl.l_len = 1;
    int ret = fcntl(fd, F_SETLK, &fl);
    if (ret == -1) {
        LOGE("Another instance is running, exiting...");
        close(fd);
        return -1;
    }

    LOGD("Free file lock.");
    return 0;
}

static void audio_recorder_thread(void *arg)
{
    int len, frames = 0;
    char buff[MBTK_PCM_WB_BUF_SIZE];

    audio_info->recorder.state = AUDIO_RECORDER_STATE_RUNNING;
    pthread_mutex_init(&audio_info->recorder.mutex, NULL);
    pthread_cond_init(&audio_info->recorder.cond, NULL);

    while (TRUE) {
        /* Playback loop */
        pthread_mutex_lock(&audio_info->recorder.mutex);
        if(audio_info->recorder.state == AUDIO_RECORDER_STATE_STOP) {
            LOGD("Stop recorder...");
            pthread_mutex_unlock(&audio_info->recorder.mutex);
            break;
        } else if(audio_info->recorder.state == AUDIO_RECORDER_STATE_PAUSE) {
            pthread_cond_wait(&audio_info->recorder.cond, &audio_info->recorder.mutex);
            pthread_mutex_unlock(&audio_info->recorder.mutex);
            continue;
        } else {
            pthread_mutex_unlock(&audio_info->recorder.mutex);
        }

        //record the needed format stream from the device.
        //only read pcm stream, no send command.
        len = audio_info->recorder.stream_in->read(audio_info->recorder.stream_in, buff,
                    audio_info->playback_size);
        if (len <= 0) {
            LOGE("%s: error reading!", __FUNCTION__);
            goto thread_end;
        }

        AUDIO_LOG("Recorder data : len - %d", len);

        if(audio_info->recorder.recorder_cb) {
            audio_info->recorder.recorder_cb(buff, len);
        }

        AUDIO_LOG("%s: No.%d frame playback.", __FUNCTION__, ++frames);
    }


thread_end:
    pthread_mutex_destroy(&audio_info->recorder.mutex);
    pthread_cond_destroy(&audio_info->recorder.cond);

    audio_info->recorder.stream_in->common.standby(&audio_info->recorder.stream_in->common);
    audio_info->audio_ahw_dev_ubus->close_input_stream(audio_info->audio_ahw_dev_ubus, audio_info->recorder.stream_in);
    VCMDeinit();//close the fd of audiostub_ctl when exit the thread.
    audio_info->recorder.stream_in = NULL;
    LOGD("%s: finished pcm playback.", __FUNCTION__);

    // Notify audio recorder data end.
    if(audio_info->recorder.recorder_cb) {
        audio_info->recorder.recorder_cb(NULL, 0);
    }
    return;
}


// srcdst = 1;/* 0-None, 1-Near end, 2-Far end, 3-Both ends */
// priority = 1;/* 0-Do not combine(override), 1-Combine */
// dest = 1;/* 0-Near codec, 1-Near Vocoder */
static int config_parameters(mbtk_audio_direction_enum direction, mbtk_audio_sample_rate_enum NBWB,
                unsigned int srcdst, unsigned int priority, unsigned int dest)
{
    char kvpair[128];
    struct str_parms *param = NULL;
    int data[5];
    const char *key = NULL;
    bool update_vcm = false;

    if(direction == MBTK_AUDIO_DIRECTION_OUTPUT){//output
        if(NBWB == MBTK_AUDIO_SAMPLE_RATE_8000)
            audio_info->playback_size = MBTK_PCM_NB_BUF_SIZE;
        else
            audio_info->playback_size = MBTK_PCM_WB_BUF_SIZE;

        LOGD("config playback parameters.");
    }
    else if(direction == MBTK_AUDIO_DIRECTION_INPUT){//input
        if(NBWB == MBTK_AUDIO_SAMPLE_RATE_8000)
            audio_info->playback_size = MBTK_PCM_NB_BUF_SIZE;
        else
            audio_info->playback_size = MBTK_PCM_WB_BUF_SIZE;

        LOGD("config record parameters.");
    }

    memset(kvpair, 0x00, sizeof(kvpair));
    sprintf(kvpair, "%s=%d;%s=%d;%s=%d;%s=%d;%s=%d", VCM_CONFIG_DIRECTION, direction,
            VCM_CONFIG_TYPE, NBWB, VCM_CONFIG_SRC_DST, srcdst,
            VCM_CONFIG_PRIORITY, priority, VCM_CONFIG_DEST, dest);

    LOGD("%s: config information kvpair is %s.\n", __FUNCTION__, kvpair);

    //extract the parameter and config from string
    param = str_parms_create_str(kvpair);
    if (!param) {
        LOGE("%s: param create str is null!", __FUNCTION__);
        return -1;
    }

    //set vcm configurations
    key = VCM_CONFIG_DIRECTION;
    if (str_parms_get_int(param, key, &data[0]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_TYPE;
    if (str_parms_get_int(param, key, &data[1]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_SRC_DST;
    if (str_parms_get_int(param, key, &data[2]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_PRIORITY;
    if (str_parms_get_int(param, key, &data[3]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }
    key = VCM_CONFIG_DEST;
    if (str_parms_get_int(param, key, &data[4]) == 0) {
        update_vcm = true;
        str_parms_del(param, key);
    }

    //printf("Direction is %d, Type is %d, Src_Dst is %d, Priority is %d, Dest is %d. \n",data[0], data[1], data[2], data[3], data[4]);

    if (update_vcm) {
        configure_vcm((unsigned int*)data);   /*TODO check if all inputs got all values successfully*/
    }

    return 0;
}

int mbtk_audio_pcm_init()
{
    //mbtk_log_init("radio", "MBTK_AUDIO");
    // Audio is running...
    if(lock_get()) {
        return -1;
    }

    if(audio_info) {
        return 0;
    }

    audio_info = (audio_inter_info_t*)malloc(sizeof(audio_inter_info_t));
    if(audio_info == NULL) {
        LOGE("malloc() fail:%d", errno);
        return -1;
    }
    memset(audio_info, 0x00, sizeof(audio_inter_info_t));

    // Set default audio parameter.
    audio_info->channel = 1;
    audio_info->sample_rate = MBTK_AUDIO_SAMPLE_RATE_8000;
    audio_info->playback_size = MBTK_PCM_NB_BUF_SIZE;

    audio_info->audio_ahw_dev_ubus = audio_hal_install();
    if (audio_info->audio_ahw_dev_ubus == NULL) {
        LOGE("audio_hal_install() failed!");
        goto init_fail;
    }
    return 0;
init_fail:
    free(audio_info);
    audio_info = NULL;
    return -1;
}

void mbtk_audio_pa_func_set(mbtk_audio_pa_switch_func pa_switch_func)
{
	audio_pa_switch_func = pa_switch_func;
}

void mbtk_audio_volume_set_func_set(mbtk_audio_volume_set_func volume_set_func)
{
	audio_volume_set_func = volume_set_func;
}

int mbtk_audio_pcm_sample_rate_set(mbtk_audio_sample_rate_enum sample_rate)
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    audio_info->sample_rate = sample_rate;

    return 0;
}

int audio_pcm_play_start(unsigned int srcdst, unsigned int priority, unsigned int dest)
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->audio_ahw_dev_ubus) {
        LOGE("audio_info->audio_ahw_dev_ubus is NULL.");
        return -1;
    }

    config_parameters(MBTK_AUDIO_DIRECTION_OUTPUT, audio_info->sample_rate, srcdst, priority, dest);

    int rc = audio_info->audio_ahw_dev_ubus->open_output_stream(audio_info->audio_ahw_dev_ubus, 0,
            audio_info->audio_ahw_dev_ubus->get_supported_devices(audio_info->audio_ahw_dev_ubus),
            AUDIO_OUTPUT_FLAG_DIRECT, NULL, &(audio_info->play.stream_out), 0);
    if (rc < 0) {
        LOGE("Open output device fail:%d", rc);
        goto play_start_fail;
    }

    audio_info->direction = MBTK_AUDIO_DIRECTION_OUTPUT;
    audio_info->play.buff_remain_len = 0;
    audio_pcm_write_count = 0;

    return 0;
play_start_fail:

    return -1;
}

// Send PCM to the other end of the voice.
int mbtk_audio_voice_pcm_playback_start()
{
    // direction = 0;/* 0-play, 1-record */
    // type = 0/1; /* 0:PCM_NB_BUF_SIZE, 1:PCM_WB_BUF_SIZE */
    // srcdst = 2;/* 0-None, 1-Near end, 2-Far end, 3-Both ends */
    // priority = 0;/* 0-Do not combine(override), 1-Combine */
    // dest = 0;/* 0-Near codec, 1-Near Vocoder */
    return audio_pcm_play_start(2, 0, 0);
#if 0
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->audio_ahw_dev_ubus) {
        LOGE("audio_info->audio_ahw_dev_ubus is NULL.");
        return -1;
    }

    config_parameters(MBTK_AUDIO_DIRECTION_OUTPUT, audio_info->sample_rate, 2, 0, 0);
    audio_info->play.is_voip = TRUE;

    return 0;
#endif
}

// Send PCM to location SPK.
int mbtk_audio_pcm_play_start()
{
    return audio_pcm_play_start(1, 1, 1);
}

int mbtk_audio_pcm_play_data_send(const void* data,uint32 data_len)
{
    UNUSED(data);
    UNUSED(data_len);

    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->play.stream_out) {
        LOGE("Output device not open.");
        return -1;
    }

    uint32 index = 0;
    // There are remaining data from the previous package。
    if(audio_info->play.buff_remain_len > 0) {
        // Too less for one package.
        if(data_len + audio_info->play.buff_remain_len < audio_info->playback_size) {
            AUDIO_LOG("Save remain data : len - %d", data_len);
            memcpy(audio_info->play.buff_remain + audio_info->play.buff_remain_len, data, data_len);
            audio_info->play.buff_remain_len += data_len;
            return data_len;
        } else {
            AUDIO_LOG("Write remain data : %d + %d", audio_info->play.buff_remain_len, audio_info->playback_size - audio_info->play.buff_remain_len);
            memcpy(audio_info->play.buff_remain + audio_info->play.buff_remain_len, data, audio_info->playback_size - audio_info->play.buff_remain_len);

            int rc = audio_info->play.stream_out->write(audio_info->play.stream_out, audio_info->play.buff_remain, audio_info->playback_size);
            if (rc < 0) {
                LOGE("%s: error writing (child).", __FUNCTION__);
                goto send_fail;
            } else if (rc < (signed int)audio_info->playback_size) {
                LOGW("%s: wrote less than buffer size, rc=%d.", __FUNCTION__, rc);
                index += rc;
                goto send_fail;
            }

            if(audio_pcm_write_count < 3) {
                audio_pcm_write_count++;
                if(audio_pcm_write_count == 3) {
                    // Set audio volume.
                    if(audio_volume_set_func) {
                        audio_volume_set_func();
                    }

                    // Open PA
                    if(audio_pa_switch_func) {
                        audio_pa_switch_func(TRUE);
                    }
                }
            }

            index += (audio_info->playback_size - audio_info->play.buff_remain_len);
            audio_info->play.buff_remain_len = 0;
        }
    }

    while(data_len - index >= audio_info->playback_size) {
        AUDIO_LOG("Package : %d -> %d", index, index + audio_info->playback_size - 1);
        int rc = audio_info->play.stream_out->write(audio_info->play.stream_out, (const char*)data + index, audio_info->playback_size);
        if (rc < 0) {
            LOGE("%s: error writing (child).", __FUNCTION__);
            goto send_fail;
        } else if (rc < (signed int)audio_info->playback_size) {
            LOGW("%s: wrote less than buffer size, rc=%d.", __FUNCTION__, rc);
            goto send_fail;
        }

        index += rc;
    }

    // Last package.( less then audio_info->playback_size)
    // Save to buffer audio_info->play_buff_remain
    if(data_len - index > 0) {
        AUDIO_LOG("Save remain data : len - %d", data_len - index);
        memcpy(audio_info->play.buff_remain, data + index, data_len - index);
        audio_info->play.buff_remain_len = data_len - index;
    }

    return data_len;
send_fail:
    return -1;
}

int mbtk_audio_pcm_play_stop()
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->play.stream_out) {
        LOGE("Output device not open.");
        return -1;
    }

    // Write last package.
    if(audio_info->play.buff_remain_len > 0) {
        char buf[MBTK_PCM_WB_BUF_SIZE];
        memset(buf, 0x00, sizeof(buf));
        memcpy(buf, audio_info->play.buff_remain, audio_info->play.buff_remain_len);

        LOGD("len %d is smaller than needed %d, so fill the buffer with 0.", audio_info->play.buff_remain_len, audio_info->playback_size);

        int rc = audio_info->play.stream_out->write(audio_info->play.stream_out, buf, audio_info->playback_size);
        if (rc < 0) {
            LOGE("%s: error writing (child).", __FUNCTION__);
            //goto send_fail;
        } else if (rc < (signed int)audio_info->playback_size) {
            LOGW("%s: wrote less than buffer size, rc=%d.", __FUNCTION__, rc);
            //goto send_fail;
        }
        audio_info->play.buff_remain_len = 0;
    }

    // Close PA
    if(audio_pa_switch_func) {
        audio_pa_switch_func(FALSE);
    }

    vcm_playback_drain(80);//wait for drain the AP audiostub queue.
    usleep(80000);//delay 80ms until DSP play out its buffered data.
    audio_info->play.stream_out->common.standby(&(audio_info->play.stream_out->common));
    audio_info->audio_ahw_dev_ubus->close_output_stream(audio_info->audio_ahw_dev_ubus, audio_info->play.stream_out);

    audio_info->play.stream_out = NULL;
    return 0;
}

static int audio_pcm_recorder_start(mbtk_recorder_callback_func recorder_cb,
                unsigned int srcdst, unsigned int priority, unsigned int dest)
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->audio_ahw_dev_ubus) {
        LOGE("audio_info->audio_ahw_dev_ubus is NULL.");
        return -1;
    }

    if(audio_info->recorder.state != AUDIO_RECORDER_STATE_STOP) {
        LOGW("Audio is recorder...");
        return -1;
    }

    config_parameters(MBTK_AUDIO_DIRECTION_INPUT, audio_info->sample_rate, srcdst, priority, dest);

    VCMInit();

    int rc = audio_info->audio_ahw_dev_ubus->open_input_stream(audio_info->audio_ahw_dev_ubus, 0,
                audio_info->audio_ahw_dev_ubus->get_supported_devices(audio_info->audio_ahw_dev_ubus),
                NULL, &(audio_info->recorder.stream_in), 0, 0, AUDIO_SOURCE_VOICE_CALL);
    if (rc < 0) {
        LOGE("Open input device fail:%d", rc);
        goto recorder_start_fail;
    }

    audio_info->direction = MBTK_AUDIO_DIRECTION_INPUT;
    audio_info->recorder.recorder_cb = recorder_cb;

    pthread_attr_t thread_attr;
    pthread_attr_init(&thread_attr);
    if(pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))
    {
        LOGE("pthread_attr_setdetachstate() fail.");
        return -1;
    }

    if (pthread_create(&recorder_thread_play, NULL, (void *)&audio_recorder_thread, NULL) < 0) {
        LOGE("%s: error creating thread_recorder!", __FUNCTION__);
        return -1;
    }

    return 0;
recorder_start_fail:

    return -1;
}

// Get PCM from the other end of the voice.
int mbtk_audio_voice_pcm_record_start(mbtk_recorder_callback_func recorder_cb)
{
    // direction = 1;/* 0-play, 1-record */
    // type = 0/1; /* 0:PCM_NB_BUF_SIZE, 1:PCM_WB_BUF_SIZE */
    // srcdst = 2;/* 0-None, 1-Near end, 2-Far end, 3-Both ends */
    // priority = 0;/* 0-Do not combine(override), 1-Combine */
    // dest = 0;/* 0-Near codec, 1-Near Vocoder */
    return audio_pcm_recorder_start(recorder_cb, 2, 0, 0);
#if 0
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    if(!audio_info->audio_ahw_dev_ubus) {
        LOGE("audio_info->audio_ahw_dev_ubus is NULL.");
        return -1;
    }

    if(audio_info->recorder.state != AUDIO_RECORDER_STATE_STOP) {
        LOGW("Audio is recorder...");
        return -1;
    }

    config_parameters(MBTK_AUDIO_DIRECTION_INPUT, audio_info->sample_rate, 2, 0, 0);
    audio_info->recorder.is_voip = TRUE;

    return 0;
#endif
}

// Get PCM from location MIC.
int mbtk_audio_pcm_recorder_start(mbtk_recorder_callback_func recorder_cb)
{
    return audio_pcm_recorder_start(recorder_cb, 1, 1, 1);
}

int mbtk_audio_pcm_recorder_pause()
{
    int result = 0;
    pthread_mutex_lock(&audio_info->recorder.mutex);
    if(audio_info->recorder.state == AUDIO_RECORDER_STATE_RUNNING) {
        audio_info->recorder.state = AUDIO_RECORDER_STATE_PAUSE;
    } else {
        result = -1;
        LOGW("Audio state : %d", audio_info->recorder.state);
    }
    pthread_mutex_unlock(&audio_info->recorder.mutex);
    return result;
}

int mbtk_audio_pcm_recorder_resume()
{
    int result = 0;
    pthread_mutex_lock(&audio_info->recorder.mutex);
    if(audio_info->recorder.state == AUDIO_RECORDER_STATE_PAUSE) {
        audio_info->recorder.state = AUDIO_RECORDER_STATE_RUNNING;
        pthread_cond_signal(&audio_info->recorder.cond);
    } else {
        result = -1;
        LOGW("Audio state : %d", audio_info->recorder.state);
    }
    pthread_mutex_unlock(&audio_info->recorder.mutex);
    return result;
}


int mbtk_audio_pcm_recorder_stop()
{
    int result = 0;
    pthread_mutex_lock(&audio_info->recorder.mutex);
    if(audio_info->recorder.state == AUDIO_RECORDER_STATE_PAUSE || audio_info->recorder.state == AUDIO_RECORDER_STATE_RUNNING) {
        if(audio_info->recorder.state == AUDIO_RECORDER_STATE_PAUSE) {
            pthread_cond_signal(&audio_info->recorder.cond);
        }
        audio_info->recorder.state = AUDIO_RECORDER_STATE_STOP;
        pthread_mutex_unlock(&audio_info->recorder.mutex);

        LOGD("Waitting recorder thread exit...");
        if (pthread_join(recorder_thread_play, NULL)) {
            LOGE("error join thread_recorder!");
            // abort();
        }
        LOGD("Recorder thread exit success.");
    } else {
        pthread_mutex_unlock(&audio_info->recorder.mutex);
        result = -1;
        LOGW("Audio state : %d", audio_info->recorder.state);
    }

    return result;
}

int mbtk_audio_pcm_deinit()
{
    if(!audio_info) {
        LOGE("Not inited.");
        return -1;
    }

    audio_hal_uninstall();

    if(lock_free()) {
        return -1;
    }

    free(audio_info);
    audio_info = NULL;
    return 0;
}

