/*-----------------------------------------------------------------------------------------------*/
/**
  @file mbtk_ds_ril_test.c
  @brief MBTK DS API TEST
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2024 mobiletek Wireless Solution, Co., Ltd. All Rights Reserved.
  mobiletek Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---------    -----------------------------------------------------------------
  20250425    yq.wang      Created .
-------------------------------------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <pthread.h>
#include <unistd.h>
#include <dlfcn.h>
#include <stdint.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <sys/epoll.h>
#include <fcntl.h>
#include <signal.h>
#include <arpa/inet.h>

#include "mbtk_log.h"
#include "mbtk_ril_api.h"
#include "mbtk_str.h"

#define BUF_SIZE_128 128
#define BUF_SIZE_1024 1024

typedef void (*cmd_func)(void);

typedef struct
{
    const char *funcName;
	cmd_func excute_func;
} api_test_info_t;

static void item_mbtk_ril_close(void);
static void item_print_help(void);
static void item_exit(void);

static int cases_count = 0;
static mbtk_ril_handle* handle_def = NULL;
static mbtk_ril_handle* handle_data = NULL;
static mbtk_ril_handle* handle_voice = NULL;

static void item_sim_type_printf(mbtk_sim_type_enum *sim_id)
{
    char cmd[BUF_SIZE_128 + 1] = {0};
    printf("please input sim type([0: sim1] [1: sim2] [255: sim_auto]):\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    *sim_id = (mbtk_sim_type_enum)atoi(cmd);
}

static void item_mbtk_ril_open(void)
{
    if(NULL == handle_def)
    {
        handle_def = mbtk_ril_open(MBTK_AT_PORT_DEF);
        if(NULL == handle_def)
        {
            printf("mbtk_ril_open(MBTK_AT_PORT_DEF) fail.\n");
            goto error;
        }
    }

    if(NULL == handle_data)
    {
        handle_data = mbtk_ril_open(MBTK_AT_PORT_DATA);
        if(NULL == handle_data)
        {
            printf("mbtk_ril_open(MBTK_AT_PORT_DATA) fail.\n");
            goto error;
        }
    }

    if(NULL == handle_voice)
    {
        handle_voice = mbtk_ril_open(MBTK_AT_PORT_VOICE);
        if(NULL == handle_voice)
        {
            printf("mbtk_ril_open(MBTK_AT_PORT_VOICE) fail.\n");
            goto error;
        }
    }
    
    printf("mbtk_ril_open() success.\n");
    return;
error:
    item_mbtk_ril_close();
}

static void item_mbtk_ril_close(void)
{
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    if(handle_def)
    {
        ret = mbtk_ril_close(MBTK_AT_PORT_DEF);
        if(MBTK_RIL_ERR_SUCCESS != ret)
        {
            printf("mbtk_ril_close(MBTK_AT_PORT_DEF) fail.[%d]\n", ret);
            return;
        }
        handle_def = NULL;
    }

    if(handle_data)
    {
        ret = mbtk_ril_close(MBTK_AT_PORT_DATA);
        if(MBTK_RIL_ERR_SUCCESS != ret)
        {
            printf("mbtk_ril_close(MBTK_AT_PORT_DATA) fail.[%d]\n", ret);
            return;
        }
        handle_data = NULL;
    }

    if(handle_voice)
    {
        ret = mbtk_ril_close(MBTK_AT_PORT_VOICE);
        if(MBTK_RIL_ERR_SUCCESS != ret)
        {
            printf("mbtk_ril_close(MBTK_AT_PORT_VOICE) fail.[%d]\n", ret);
            return;
        }
        handle_voice = NULL;
    }
    
    printf("mbtk_ril_close() success.\n");
}

static void item_mbtk_version_get(void)
{
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char version[BUF_SIZE_128] = {0};
    
    memset(version, 0x00, BUF_SIZE_128);
    ret = mbtk_version_get(handle_def, version);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_version_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("version: [%s]\n", version);
    }
}

static void item_mbtk_ds_model_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char model[BUF_SIZE_128] = {0};
    
    item_sim_type_printf(&sim_id);

    memset(model, 0x00, BUF_SIZE_128);
    ret = mbtk_ds_model_get(handle_def, sim_id, model);;
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_model_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("model: [%s]\n", model);
    }
}

static void item_mbtk_ds_radio_state_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_radio_state_enum radio_state = MBTK_RADIO_STATE_UNKNOWN;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    
    item_sim_type_printf(&sim_id);
    
    ret = mbtk_ds_radio_state_get(handle_def, sim_id, &radio_state);;
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_radio_state_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("radio_state: [%d]\n", radio_state);
    }
}

static void item_mbtk_ds_radio_state_set(void)
{
    mbtk_sim_type_enum sim_id;
    int reset_modem = 0;
    mbtk_radio_state_enum radio_state = MBTK_RADIO_STATE_UNKNOWN;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <radio_state----->-<reset_modem-->\n");
    printf("exampl ([0: MINI] [1: FULL])-([0: NO] [1: YES])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    ret = sscanf(cmd, "%d %d", &radio_state, &reset_modem);
    
    ret = mbtk_ds_radio_state_set(handle_def, sim_id, radio_state, reset_modem == 1 ? true : false);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_radio_state_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_radio_state_set() success\n");
    }
}

static void item_mbtk_ds_cell_time_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cell_time[BUF_SIZE_128] = {0};
    
    item_sim_type_printf(&sim_id);

    memset(cell_time, 0x00, BUF_SIZE_128);
    ret = mbtk_ds_cell_time_get(handle_def, sim_id, cell_time);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_cell_time_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("cell_time: [%s]\n", cell_time);
    }
}

static void item_mbtk_ds_sim_state_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_sim_state_enum sim_state = MBTK_SIM_STATE_UNKNOWN;
    
    item_sim_type_printf(&sim_id);
    
    ret = mbtk_ds_sim_state_get(handle_def, sim_id, &sim_state);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sim_state_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("sim_state: [%d]\n", sim_state);
    }
}

static void item_mbtk_ds_sim_type_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_sim_card_type_enum sim_card_type = MBTK_UNKNOWN;
    
    item_sim_type_printf(&sim_id);
    
    ret = mbtk_ds_sim_type_get(handle_def, sim_id, &sim_card_type);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sim_type_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("sim_card_type: [%d]\n", sim_card_type);
    }
}

static void item_mbtk_ds_imsi_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char imsi[BUF_SIZE_128] = {0};
    
    item_sim_type_printf(&sim_id);

    memset(imsi, 0x00, BUF_SIZE_128);
    ret = mbtk_ds_imsi_get(handle_def, sim_id, imsi);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_imsi_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("imsi: [%s]\n", imsi);
    }
}

static void item_mbtk_ds_iccid_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char iccid[BUF_SIZE_128] = {0};
    
    item_sim_type_printf(&sim_id);

    memset(iccid, 0x00, BUF_SIZE_128);
    ret = mbtk_ds_iccid_get(handle_def, sim_id, iccid);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_iccid_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("iccid: [%s]\n", iccid);
    }
}

static void item_mbtk_ds_phone_number_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char phone_number[BUF_SIZE_128] = {0};
    
    item_sim_type_printf(&sim_id);

    memset(phone_number, 0x00, BUF_SIZE_128);
    ret = mbtk_ds_phone_number_get(handle_def, sim_id, phone_number);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_phone_number_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("phone_number: [%s]\n", phone_number);
    }
}

static void item_mbtk_ds_sim_lock_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int lock_state = 0;
    
    item_sim_type_printf(&sim_id);

    ret = mbtk_ds_sim_lock_get(handle_def, sim_id, &lock_state);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sim_lock_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("lock_state: [%d]\n", lock_state);
    }
}

static void item_mbtk_ds_sim_lock_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_sim_lock_info_t lock_info = {0};
    char cmd[BUF_SIZE_1024 + 1] = {0};
    int type = 0;
    
    item_sim_type_printf(&sim_id);

    printf("exampl <lock_type>-<pin1>-<pin2>-<puk>\n");
    printf("lock_type ([0: DISABLE] [1: ENABLE] [3: CHANGE] [4: VERIFY_PIN] [5: VERIFY_PUK])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    memset(&lock_info, 0x00, sizeof(mbtk_sim_lock_info_t));
    ret = sscanf(cmd, "%d %s %s %s", &type, lock_info.pin1, lock_info.pin2, lock_info.puk);
    lock_info.type = (mbtk_sim_lock_type_enum)type;
    
    ret = mbtk_ds_sim_lock_set(handle_def, sim_id, &lock_info);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sim_lock_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_sim_lock_set() success\n");
    }
}

static void item_mbtk_ds_sim_lock_retry_times_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_pin_puk_last_times_t retry_times = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&retry_times, 0x00, sizeof(mbtk_pin_puk_last_times_t));
    ret = mbtk_ds_sim_lock_retry_times_get(handle_def, sim_id, &retry_times);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sim_lock_retry_times_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("p1_retry: [%d], p2_retry: [%d], puk1_retry: [%d], puk2_retry: [%d]\n", retry_times.p1_retry, retry_times.p2_retry, retry_times.puk1_retry, retry_times.puk2_retry);
    }
}

static void item_mbtk_ds_plmn_list_get(void)
{
    int i = 0;
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_plmn_info plmn_list = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&plmn_list, 0x00, sizeof(mbtk_plmn_info));
    ret = mbtk_ds_plmn_list_get(handle_def, sim_id, &plmn_list);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_plmn_list_get() fail.[%d]\n", ret);
    }
    else
    {
        for(i = 0; i < plmn_list.count; i++)
        {
            printf("format: [%d], plmn: [%s]\n", plmn_list.plmn[i].format, plmn_list.plmn[i].name);
        }
    }
}

static void item_mbtk_ds_available_net_get(void)
{
    int i = 0;
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_net_info_array_t net_array = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&net_array, 0x00, sizeof(mbtk_net_info_array_t));
    ret = mbtk_ds_available_net_get(handle_def, sim_id, &net_array);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_available_net_get() fail.[%d]\n", ret);
    }
    else
    {
        for(i = 0; i < net_array.num; i++)
        {
            printf("net_sel_mode: [%d], net_type: [%d], net_state: [%d], plmn: [%d]\n", net_array.net_info[i].net_sel_mode,
                                                                                        net_array.net_info[i].net_type,
                                                                                        net_array.net_info[i].net_state,
                                                                                        net_array.net_info[i].plmn);
        }
    }
}

static void item_mbtk_ds_net_sel_mode_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_net_info_t net = {0};
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <net_sel_mode>-<net_type>-<net_state>-<plmn>\n");
    printf("net_sel_mode ([0: automativ] [1: manual])\n");
    printf("net_type ([0: GSM] [1: GSM Compact] [2: UTRAN] [3: GSM w/EGPRS] [4: UTRAN w/HSDPA])\n");
    printf("         ([5: UTRAN w/HSDPA] [6: UTRAN w/HSDUPA] [7: E-UTRAN] [8: UTRAN HSPA+])\n");
    printf("net_state ([0: unknown] [1: available] [2: current] [3: forbidden])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    memset(&net, 0x00, sizeof(mbtk_net_info_t));
    ret = sscanf(cmd, "%hhd %hhd %hhd %d", &(net.net_sel_mode), &(net.net_type), &(net.net_state), &(net.plmn));

    ret = mbtk_ds_net_sel_mode_set(handle_def, sim_id, &net);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_net_sel_mode_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_net_sel_mode_set() success\n");
    }
}

static void item_mbtk_ds_net_sel_mode_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_net_info_t net = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&net, 0x00, sizeof(mbtk_net_info_t));    
    ret = mbtk_ds_net_sel_mode_get(handle_def, sim_id, &net);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_net_sel_mode_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("net_sel_mode: [%d], net_type: [%d], net_state: [%d], plmn: [%d]\n", net.net_sel_mode, net.net_type, net.net_state, net.plmn);
    }
}

static void item_mbtk_ds_current_band_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_band_info_t band = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&band, 0x00, sizeof(mbtk_band_info_t));    
    ret = mbtk_ds_current_band_get(handle_def, sim_id, &band);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_current_band_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("net_pref: [%d], gsm_band: [%d], umts_band: [%d], tdlte_band: [%d], fddlte_band: [%d], lte_ext_band: [%d]\n", band.net_pref, band.gsm_band,
                                                                                                                             band.umts_band, band.tdlte_band,
                                                                                                                             band.fddlte_band, band.lte_ext_band);
        printf("nr_3_band: [%d], nr_2_band: [%d], nr_1_band: [%d], nr_0_band: [%d]\n", band.nr_3_band, band.nr_2_band, band.nr_1_band, band.nr_0_band);
    }
}

static void item_mbtk_ds_current_band_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_band_info_t band = {0};
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <net_pref>-<gsm_band>-<umts_band>-<tdlte_band>-<fddlte_band>-<lte_ext_band>-<nr_3_band>-<nr_2_band>-<nr_1_band>-<nr_0_band>\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    memset(&band, 0x00, sizeof(mbtk_band_info_t));
    ret = sscanf(cmd, "%d %d %d %d %d %d %d %d %d %d", &(band.net_pref), &(band.gsm_band), &(band.umts_band), &(band.tdlte_band), &(band.fddlte_band), &(band.lte_ext_band), 
                                                        &(band.nr_3_band), &(band.nr_2_band), &(band.nr_1_band), &(band.nr_0_band));
 
    ret = mbtk_ds_current_band_set(handle_def, sim_id, &band);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_current_band_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_current_band_set() success\n");
    }
}

static void item_mbtk_ds_cell_get(void)
{
    int i = 0;
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_cell_info_array_t cell_array = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&cell_array, 0x00, sizeof(mbtk_cell_info_array_t));    
    ret = mbtk_ds_cell_get(handle_def, sim_id, &cell_array);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_cell_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("type [%d]:\n", cell_array.type);
        for(i = 0; i < cell_array.num; i++)
        {
            printf("value[%d]: [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%d] [%lld] [%lld] [%lld] [%lld] [%lld]\n", i, cell_array.cell[i].value1, cell_array.cell[i].value2,
                                                                                        cell_array.cell[i].value3, cell_array.cell[i].value4,
                                                                                        cell_array.cell[i].value5, cell_array.cell[i].value6,
                                                                                        cell_array.cell[i].value7, cell_array.cell[i].value8,
                                                                                        cell_array.cell[i].value9, cell_array.cell[i].value10,
                                                                                        cell_array.cell[i].value11, cell_array.cell[i].value12,
                                                                                        cell_array.cell[i].value13, cell_array.cell[i].value14,
                                                                                        cell_array.cell[i].value15, cell_array.cell[i].value64_1,
                                                                                        cell_array.cell[i].value64_2, cell_array.cell[i].value64_3,
                                                                                        cell_array.cell[i].value64_4, cell_array.cell[i].value64_5);
        }
    }
}

static void item_mbtk_ds_cell_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmd[BUF_SIZE_128 + 1] = {0};
    char response[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);
    
    printf("exampl 2,3,,40936,430\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);

    memset(response, 0x00, BUF_SIZE_1024 + 1);    
    ret = mbtk_ds_cell_set(handle_def, sim_id, cmd, response);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_cell_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("response: [%s]\n", response);
    }
}

static void item_mbtk_ds_net_signal_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_signal_info_t signal = {0};
    
    item_sim_type_printf(&sim_id);
    
    memset(&signal, 0x00, sizeof(mbtk_signal_info_t));    
    ret = mbtk_ds_net_signal_get(handle_def, sim_id, &signal);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_net_signal_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("type: [%d], rssi: [%d], rxlev: [%d], ber: [%d], rscp: [%d], ecno: [%d], rsrq: [%d], rsrp: [%d], ss_rsrq: [%d], ss_rsrp: [%d], ss_sinr: [%d]\n", 
                signal.type, signal.rssi, signal.rxlev, signal.ber, signal.rscp, signal.ecno, signal.rsrq, signal.rsrp, signal.ss_rsrq, signal.ss_rsrp, signal.ss_sinr);
    }
}

static void item_mbtk_ds_net_reg_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_net_reg_info_t reg = {0};
    
    item_sim_type_printf(&sim_id);
    
    memset(&reg, 0x00, sizeof(mbtk_net_reg_info_t));    
    ret = mbtk_ds_net_reg_get(handle_def, sim_id, &reg);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_net_reg_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("call_state: [%d], data_state: [%d], ims_state: [%d], type: [%d], lac: [%d], ci: [%lld]\n",
                reg.call_state, reg.data_state, reg.ims_state, reg.type, reg.lac, reg.ci);
    }
}

static void item_mbtk_ds_oos_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_ril_oos_info_t oos_info = {0};
    
    item_sim_type_printf(&sim_id);
    
    memset(&oos_info, 0x00, sizeof(mbtk_ril_oos_info_t));    
    ret = mbtk_ds_oos_get(handle_def, sim_id, &oos_info);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_oos_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("state: [%d], OosPhase1: [%d], OosPhase2: [%d], OosPhase3: [%d]\n",
                oos_info.state, oos_info.oosPhase[0], oos_info.oosPhase[1], oos_info.oosPhase[2]);
    }
}

static void item_mbtk_ds_oos_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_ril_oos_info_t oos_info = {0};
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <state>-<OosOhase1>-<OosOhase2>-<OosOhase3>\n");
    printf("state ([0: disable] [1:enable])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    memset(&oos_info, 0x00, sizeof(mbtk_ril_oos_info_t));
    ret = sscanf(cmd, "%d %d %d %d", &(oos_info.state), &(oos_info.oosPhase[0]), &(oos_info.oosPhase[1]), &(oos_info.oosPhase[2]));
      
    ret = mbtk_ds_oos_set(handle_def, sim_id, &oos_info);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_oos_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_oos_get() success\n");
    }
}

static void item_mbtk_ds_wakeup_state_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int wakeup_state = 0;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <wakeup_state>\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    wakeup_state = atoi(cmd);
      
    ret = mbtk_ds_wakeup_state_set(handle_def, sim_id, wakeup_state);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_wakeup_state_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_wakeup_state_set() success\n");
    }
}

static void item_mbtk_ds_apn_get(void)
{
    int i = 0;
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_apn_info_array_t apns = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&apns, 0x00, sizeof(mbtk_apn_info_array_t));
    ret = mbtk_ds_apn_get(handle_data, sim_id, &apns);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_apn_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("cid_for_def_route: [%d], cid_for_def_dns: [%d]\n", apns.cid_for_def_route, apns.cid_for_def_dns);
        for(i = 0; i < apns.num; i++)
        {
            printf("cid: [%d], ip_type: [%d], auth: [%d], auto_save: [%d], auto_boot_call: [%d], redial_enable: [%d], def_route: [%d], as_dns: [%d], apn: [%s], user: [%s], pass: [%s], type: [%s]\n",
                    apns.apns[i].cid, apns.apns[i].ip_type, apns.apns[i].auth, apns.apns[i].auto_save, apns.apns[i].auto_boot_call, apns.apns[i].redial_enable, apns.apns[i].def_route, apns.apns[i].as_dns, 
                                                            str_empty(apns.apns[i].apn) ? "NULL" : (char *)apns.apns[i].apn, 
                                                            str_empty(apns.apns[i].user) ? "NULL" : (char *)apns.apns[i].user, 
                                                            str_empty(apns.apns[i].pass) ? "NULL" : (char *)apns.apns[i].pass, 
                                                            str_empty(apns.apns[i].type) ? "NULL" : (char *)apns.apns[i].type);
        }
    }
}

static void item_mbtk_ds_apn_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_apn_info_t apn = {0};
    int cid = 0;
    int ip_type = 0;
    int auth = 0;
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <cid>-<ip_type>-<auth>-<auto_save>-<auto_boot_call>-<redial_enable>-<def_route>-<as_dns>-<apn>-<user>-<pass>-<type>\n");
    printf("cid ([1 - 7])\n");
    printf("ip_type ([0: ipv4] [1: ipv6] [2: ipv4v6] [3: ppp])\n");
    printf("auth ([0: default] [1: NONE] [2: PAP] [3: CHAP])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    memset(&apn, 0x00, sizeof(mbtk_apn_info_t));
    ret = sscanf(cmd, "%d %d %d %d %d %d %d %d %s %s %s %s", &cid, &ip_type, &auth, &(apn.auto_save),&(apn.auto_boot_call),
                                                            &(apn.redial_enable), &(apn.def_route), &(apn.as_dns), apn.apn, apn.user, apn.pass, apn.type);
    apn.cid = (mbtk_ril_cid_enum)cid;
    apn.ip_type = (mbtk_ip_type_enum)ip_type;
    apn.auth = (mbtk_apn_auth_type_enum)auth;
    
    ret = mbtk_ds_apn_set(handle_data, sim_id, &apn);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_apn_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_apn_set() success\n");
    }
}

static void item_mbtk_ds_data_call_start(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int cid = 0;
    int timeout = 0;
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <cid>-<timeout>\n");
    printf("cid ([1 - 7])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    ret = sscanf(cmd, "%d %d", &cid, &timeout);
    
    ret = mbtk_ds_data_call_start(handle_data, sim_id, (mbtk_ril_cid_enum)cid, MBTK_DATA_CALL_ITEM_STATE_NON,
            MBTK_DATA_CALL_ITEM_STATE_NON, MBTK_DATA_CALL_ITEM_STATE_NON, NULL, 0, timeout, NULL);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_data_call_start() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_data_call_start() success\n");
    }
}

static void item_mbtk_ds_data_call_stop(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int cid = 0;
    int timeout = 0;
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <cid>-<timeout>\n");
    printf("cid ([1 - 7])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    ret = sscanf(cmd, "%d %d", &cid, &timeout);
    
    ret = mbtk_ds_data_call_stop(handle_data, sim_id, (mbtk_ril_cid_enum)cid, timeout);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_data_call_stop() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_data_call_stop() success\n");
    }
}

static void item_mbtk_ds_data_call_state_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int cid = 0;
    mbtk_ip_info_t ip = {0};
    char ip_tmp[BUF_SIZE_128 + 1] = {0};
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <cid>\n");
    printf("cid ([1 - 7])\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    cid = atoi(cmd);

    memset(&ip, 0x00, sizeof(mbtk_ip_info_t));
    ret = mbtk_ds_data_call_state_get(handle_data, sim_id, (mbtk_ril_cid_enum)cid, &ip);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_data_call_state_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("ip.ipv4.valid: [%d]\n", ip.ipv4.valid);
        if(ip.ipv4.valid)
        {
            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(inet_ntop(AF_INET, &(ip.ipv4.IPAddr), ip_tmp, BUF_SIZE_128) == NULL)
            {
                printf("IP error.\n");
            }
            else
            {
                printf("IP: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(inet_ntop(AF_INET, &(ip.ipv4.PrimaryDNS), ip_tmp, BUF_SIZE_128) == NULL)
            {
                printf("PrimaryDNS error.\n");
            } 
            else 
            {
                printf("PrimaryDNS: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(inet_ntop(AF_INET, &(ip.ipv4.SecondaryDNS), ip_tmp, BUF_SIZE_128) == NULL)
            {
                printf("SecondaryDNS error.\n");
            } 
            else 
            {
                printf("SecondaryDNS: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(inet_ntop(AF_INET, &(ip.ipv4.GateWay), ip_tmp, BUF_SIZE_128) == NULL)
            {
                printf("GateWay error.\n");
            } 
            else 
            {
                printf("GateWay: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(inet_ntop(AF_INET, &(ip.ipv4.NetMask), ip_tmp, BUF_SIZE_128) == NULL)
            {
                printf("NetMask error.\n");
            } 
            else 
            {
                printf("NetMask: [%s]\n", ip_tmp);
            }
        }

        printf("ip.ipv6.valid: [%d]\n", ip.ipv6.valid);
        if(ip.ipv6.valid)
        {
            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(ipv6_2_str(&(ip.ipv6.IPV6Addr), ip_tmp))
            {
                printf("IPV6 error.\n");
            } 
            else 
            {
                printf("IPV6: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(ipv6_2_str(&(ip.ipv6.PrimaryDNS), ip_tmp))
            {
                printf("PrimaryDNS_V6 error.\n");
            } 
            else 
            {
                printf("PrimaryDNS_V6: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(ipv6_2_str(&(ip.ipv6.SecondaryDNS), ip_tmp))
            {
                printf("SecondaryDNS_V6 error.\n");
            } 
            else 
            {
                printf("SecondaryDNS_V6: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(ipv6_2_str(&(ip.ipv6.GateWay), ip_tmp)) 
            {
                printf("GateWay_V6 error.\n");
            } 
            else 
            {
                printf("GateWay_V6: [%s]\n", ip_tmp);
            }

            memset(ip_tmp, 0x00, BUF_SIZE_128 + 1);
            if(ipv6_2_str(&(ip.ipv6.NetMask), ip_tmp))
            {
                printf("NetMask_V6 error.\n");
            } 
            else 
            {
                printf("NetMask_V6: [%s]\n", ip_tmp);
            }
        }
    }
}

static void item_mbtk_ds_sms_cmgf_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int sms_state = 0;
    
    item_sim_type_printf(&sim_id);
       
    ret = mbtk_ds_sms_cmgf_get(handle_def, sim_id, &sms_state);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgf_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("sms_state； [%d]\n", sms_state);
    }
}

static void item_mbtk_ds_sms_cmgf_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int mode = 0;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(mode):\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    mode = atoi(cmd);
       
    ret = mbtk_ds_sms_cmgf_set(handle_def, sim_id, mode);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgf_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_sms_cmgf_set() success\n");
    }
}

static void item_mbtk_ds_sms_cmgs_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char resp[BUF_SIZE_128 + 1] = {0};
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(cmgs):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);

    memset(resp, 0x00, BUF_SIZE_128 + 1);
    ret = mbtk_ds_sms_cmgs_set(handle_def, sim_id, cmd, resp);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgs_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("resp: [%s]\n", resp);
    }
}

static void item_mbtk_ds_sms_cmgw_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char resp[BUF_SIZE_128 + 1] = {0};
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(cmgw):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);

    memset(resp, 0x00, BUF_SIZE_128 + 1);
    ret = mbtk_ds_sms_cmgw_set(handle_def, sim_id, cmd, resp);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgw_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("resp: [%s]\n", resp);
    }
}

static void item_mbtk_ds_sms_cmgd_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(cmdg):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);

    ret = mbtk_ds_sms_cmgd_set(handle_def, sim_id, cmd);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgd_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_sms_cmgd_set() success\n");
    }
}

static void item_mbtk_ds_sms_cmgd_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmdg[BUF_SIZE_1024 + 1] = {0};

    item_sim_type_printf(&sim_id);

    memset(cmdg, 0x00, BUF_SIZE_1024 + 1);
    ret = mbtk_ds_sms_cmgd_get(handle_def, sim_id, cmdg);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgd_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("cmdg: [%s]\n", cmdg);
    }
}

static void item_mbtk_ds_sms_cmgl_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char resp[BUF_SIZE_128 + 1] = {0};
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(cmgw):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);

    memset(resp, 0x00, BUF_SIZE_128 + 1);
    ret = mbtk_ds_sms_cmgl_set(handle_def, sim_id, cmd, resp);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgl_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("resp: [%s]\n", resp);
    }
}

static void item_mbtk_ds_sms_csca_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char buff[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    memset(buff, 0x00, BUF_SIZE_1024 + 1);
    ret = mbtk_ds_sms_csca_get(handle_def, sim_id, buff);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_csca_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("buff: [%s]\n", buff);
    }
}

static void item_mbtk_ds_sms_csca_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(csca):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    
    ret = mbtk_ds_sms_csca_set(handle_def, sim_id, cmd);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_csca_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_sms_csca_set() success\n");
    }
}

static void item_mbtk_ds_sms_csmp_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(csmp):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    
    ret = mbtk_ds_sms_csmp_set(handle_def, sim_id, cmd);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_csmp_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_sms_csmp_set() success\n");
    }
}

static void item_mbtk_ds_sms_cscb_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(csmp):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    
    ret = mbtk_ds_sms_cscb_set(handle_def, sim_id, cmd);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cscb_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_sms_cscb_set() success\n");
    }
}

static void item_mbtk_ds_sms_cnmi_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    
    item_sim_type_printf(&sim_id);
    
    ret = mbtk_ds_sms_cnmi_set(handle_def, sim_id);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cnmi_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_sms_cnmi_set() success\n");
    }
}

static void item_mbtk_ds_sms_cmss_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char resp[BUF_SIZE_128 + 1] = {0};
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(cmss):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);

    memset(resp, 0x00, BUF_SIZE_128 + 1);    
    ret = mbtk_ds_sms_cmss_set(handle_def, sim_id, cmd, resp);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmss_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("resp: [%s]\n", resp);
    }
}

static void item_mbtk_ds_sms_cpms_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char resp[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    memset(resp, 0x00, BUF_SIZE_128 + 1);    
    ret = mbtk_ds_sms_cpms_get(handle_def, sim_id, resp);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cpms_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("resp: [%s]\n", resp);
    }
}

static void item_mbtk_ds_sms_cpms_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char resp[BUF_SIZE_128 + 1] = {0};
    char cmd[BUF_SIZE_1024 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(cmss):\n");
    if(NULL == fgets(cmd, BUF_SIZE_1024, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);

    memset(resp, 0x00, BUF_SIZE_128 + 1);     
    ret = mbtk_ds_sms_cpms_set(handle_def, sim_id, cmd, resp);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cpms_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("resp: [%s]\n", resp);
    }
}

static void item_mbtk_ds_sms_cmgr_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char resp[BUF_SIZE_128 + 1] = {0};
    int index = 0;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(cmss):\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    index = atoi(cmd);

    memset(resp, 0x00, BUF_SIZE_128 + 1);     
    ret = mbtk_ds_sms_cmgr_set(handle_def, sim_id, index, resp);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_sms_cmgr_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("resp: [%s]\n", resp);
    }
}

static void item_mbtk_ds_call_start(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(phone_number):\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
 
    ret = mbtk_ds_call_start(handle_voice, sim_id, cmd);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_call_start() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_call_start() success\n");
    }
}

static void item_mbtk_ds_call_answer(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    
    item_sim_type_printf(&sim_id);
 
    ret = mbtk_ds_call_answer(handle_voice, sim_id);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_call_answer() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_call_answer() success\n");
    }
}

static void item_mbtk_ds_call_hang(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    
    item_sim_type_printf(&sim_id);
 
    ret = mbtk_ds_call_hang(handle_voice, sim_id);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_call_hang() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_call_hang() success\n");
    }
}

static void item_mbtk_ds_a_call_hang(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int phone_id = 0;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(phone_id):\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    phone_id = atoi(cmd);
 
    ret = mbtk_ds_a_call_hang(handle_voice, sim_id, phone_id);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_a_call_hang() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_a_call_hang() success\n");
    }
}

static void item_mbtk_ds_waiting_or_background_call_hang(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    
    item_sim_type_printf(&sim_id);
 
    ret = mbtk_ds_waiting_or_background_call_hang(handle_voice, sim_id);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_waiting_or_background_call_hang() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_waiting_or_background_call_hang() success\n");
    }
}

static void item_mbtk_ds_foreground_resume_background_call_hang(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    
    item_sim_type_printf(&sim_id);
 
    ret = mbtk_ds_foreground_resume_background_call_hang(handle_voice, sim_id);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_foreground_resume_background_call_hang() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_foreground_resume_background_call_hang() success\n");
    }
}

static void item_mbtk_ds_call_reg_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_call_info_t reg = {0};
    
    item_sim_type_printf(&sim_id);

    memset(&reg, 0x00, sizeof(mbtk_call_info_t));
    ret = mbtk_ds_call_reg_get(handle_voice, sim_id, &reg);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_call_reg_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("RING: [%hhd] [%hhd] [%hhd] [%hhd] [%hhd] [%hhd] [%s] [%hhd] [%hhd] [%hhd] [%s]\n",
                reg.call_wait, reg.dir1, reg.dir, reg.state, reg.mode, reg.mpty, reg.phone_number, reg.type, reg.pas, reg.disconnected_id, (char *)reg.end_reason);
    }
}

static void item_mbtk_ds_mute_state_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int mute_state = 0;
    
    item_sim_type_printf(&sim_id);

    ret = mbtk_ds_mute_state_get(handle_def, sim_id, &mute_state);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_mute_state_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("mute_state: [%d]\n", mute_state);
    }
}

static void item_mbtk_ds_mute_state_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int mute_state = 0;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(mute_state):\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    mute_state = atoi(cmd);

    ret = mbtk_ds_mute_state_set(handle_def, sim_id, mute_state);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_mute_state_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_mute_state_set() success\n");
    }
}

static void item_mbtk_ds_dtmf_send(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    mbtk_call_dtmf_info_t dtmf_character = {0};
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("exampl <character>-<duration>\n");
    printf("please input param:\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    ret = sscanf(cmd, "%c %d", &(dtmf_character.character), &(dtmf_character.duration));

    ret = mbtk_ds_dtmf_send(handle_def, sim_id, &dtmf_character);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_dtmf_send() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_dtmf_send() success\n");
    }
}

static void item_mbtk_ds_centric_set(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int centric = 0;
    char cmd[BUF_SIZE_128 + 1] = {0};
    
    item_sim_type_printf(&sim_id);

    printf("please input param(centric):\n");
    if(NULL == fgets(cmd, BUF_SIZE_128, stdin))
    {
        fflush(stdin);
        printf("fgets cmd fail.\n");
        return;
    }
    fflush(stdin);
    centric = atoi(cmd);

    ret = mbtk_ds_centric_set(handle_def, sim_id, centric);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_centric_set() fail.[%d]\n", ret);
    }
    else
    {
        printf("mbtk_ds_centric_set() success\n");
    }
}

static void item_mbtk_ds_centric_get(void)
{
    mbtk_sim_type_enum sim_id;
    mbtk_ril_err_enum ret = MBTK_RIL_ERR_SUCCESS;
    int centric = 0;
    
    item_sim_type_printf(&sim_id);

    ret = mbtk_ds_centric_get(handle_def, sim_id, &centric);
    if(MBTK_RIL_ERR_SUCCESS != ret)
    {
        printf("mbtk_ds_centric_get() fail.[%d]\n", ret);
    }
    else
    {
        printf("centric: [%d]\n", centric);
    }
}

//for server test
static api_test_info_t api_test_cases[] =
{
    {"[API]", NULL},
    {"mbtk_ril_open", item_mbtk_ril_open},
    {"mbtk_ril_close", item_mbtk_ril_open},
    {"mbtk_version_get", item_mbtk_version_get},
    {"mbtk_ds_model_get", item_mbtk_ds_model_get},
    {"mbtk_ds_radio_state_get", item_mbtk_ds_radio_state_get},
    {"mbtk_ds_radio_state_set", item_mbtk_ds_radio_state_set},
    {"mbtk_ds_cell_time_get", item_mbtk_ds_cell_time_get},
    {"mbtk_ds_sim_state_get", item_mbtk_ds_sim_state_get},
    {"mbtk_ds_sim_type_get", item_mbtk_ds_sim_type_get},
    {"mbtk_ds_imsi_get", item_mbtk_ds_imsi_get},
    {"mbtk_ds_iccid_get", item_mbtk_ds_iccid_get},
    {"mbtk_ds_phone_number_get", item_mbtk_ds_phone_number_get},
    {"mbtk_ds_sim_lock_get", item_mbtk_ds_sim_lock_get},
    {"mbtk_ds_sim_lock_set", item_mbtk_ds_sim_lock_set},
    {"mbtk_ds_sim_lock_retry_times_get", item_mbtk_ds_sim_lock_retry_times_get},
    {"mbtk_ds_plmn_list_get", item_mbtk_ds_plmn_list_get},
    {"mbtk_ds_available_net_get", item_mbtk_ds_available_net_get},
    {"mbtk_ds_net_sel_mode_set", item_mbtk_ds_net_sel_mode_set},
    {"mbtk_ds_net_sel_mode_get", item_mbtk_ds_net_sel_mode_get},
    {"mbtk_ds_current_band_get", item_mbtk_ds_current_band_get},
    {"mbtk_ds_current_band_set", item_mbtk_ds_current_band_set},
    {"mbtk_ds_cell_get", item_mbtk_ds_cell_get},
    {"mbtk_ds_cell_set", item_mbtk_ds_cell_set},
    {"mbtk_ds_net_signal_get", item_mbtk_ds_net_signal_get},
    {"mbtk_ds_net_reg_get", item_mbtk_ds_net_reg_get},
    {"mbtk_ds_oos_get", item_mbtk_ds_oos_get},
    {"mbtk_ds_oos_set", item_mbtk_ds_oos_set},
    {"mbtk_ds_wakeup_state_set", item_mbtk_ds_wakeup_state_set},
    {"mbtk_ds_apn_get", item_mbtk_ds_apn_get},
    {"mbtk_ds_apn_set", item_mbtk_ds_apn_set},
    {"mbtk_ds_data_call_start", item_mbtk_ds_data_call_start},
    {"mbtk_ds_data_call_stop", item_mbtk_ds_data_call_stop},
    {"mbtk_ds_data_call_state_get", item_mbtk_ds_data_call_state_get},
    {"mbtk_ds_sms_cmgf_get", item_mbtk_ds_sms_cmgf_get},
    {"mbtk_ds_sms_cmgf_set", item_mbtk_ds_sms_cmgf_set},
    {"mbtk_ds_sms_cmgs_set", item_mbtk_ds_sms_cmgs_set},
    {"mbtk_ds_sms_cmgw_set", item_mbtk_ds_sms_cmgw_set},
    {"mbtk_ds_sms_cmgd_set", item_mbtk_ds_sms_cmgd_set},
    {"mbtk_ds_sms_cmgd_get", item_mbtk_ds_sms_cmgd_get},
    {"mbtk_ds_sms_cmgl_set", item_mbtk_ds_sms_cmgl_set},
    {"mbtk_ds_sms_csca_get", item_mbtk_ds_sms_csca_get},
    {"mbtk_ds_sms_csca_set", item_mbtk_ds_sms_csca_set},
    {"mbtk_ds_sms_csmp_set", item_mbtk_ds_sms_csmp_set},
    {"mbtk_ds_sms_cscb_set", item_mbtk_ds_sms_cscb_set},
    {"mbtk_ds_sms_cnmi_set", item_mbtk_ds_sms_cnmi_set},
    {"mbtk_ds_sms_cmss_set", item_mbtk_ds_sms_cmss_set},
    {"mbtk_ds_sms_cpms_get", item_mbtk_ds_sms_cpms_get},
    {"mbtk_ds_sms_cpms_set", item_mbtk_ds_sms_cpms_set},
    {"mbtk_ds_sms_cmgr_set", item_mbtk_ds_sms_cmgr_set},
    {"mbtk_ds_call_start", item_mbtk_ds_call_start},
    {"mbtk_ds_call_answer", item_mbtk_ds_call_answer},
    {"mbtk_ds_call_hang", item_mbtk_ds_call_hang},
    {"mbtk_ds_a_call_hang", item_mbtk_ds_a_call_hang},
    {"mbtk_ds_waiting_or_background_call_hang", item_mbtk_ds_waiting_or_background_call_hang},
    {"mbtk_ds_foreground_resume_background_call_hang", item_mbtk_ds_foreground_resume_background_call_hang},
    {"mbtk_ds_call_reg_get", item_mbtk_ds_call_reg_get},
    {"mbtk_ds_mute_state_get", item_mbtk_ds_mute_state_get},
    {"mbtk_ds_mute_state_set", item_mbtk_ds_mute_state_set},
    {"mbtk_ds_dtmf_send", item_mbtk_ds_dtmf_send},
    {"mbtk_ds_centric_set", item_mbtk_ds_centric_set},
    {"mbtk_ds_centric_get", item_mbtk_ds_centric_get},
    {"[OTHER OPERATIONS]", NULL},
    {"qser_help", item_print_help},
    {"qser_exit", item_exit}
};

static void item_print_help(void)
{
    int i;
    printf("test supported %d cases:\n", cases_count);
    for(i = 0; i < cases_count; i++)
    {
        if(api_test_cases[i].excute_func == NULL)
        {
            printf("%s\n", api_test_cases[i].funcName);
        }
        else
        {
            printf("%d:\t%s\n", i, api_test_cases[i].funcName);
        }
    }
}

static void item_exit(void)
{
    printf("mbtk ds test exit.\n");
}

int main(int argc, char const *argv[])
{
    int cmdidx = 0;
    char cases[BUF_SIZE_128] = {0};
    cases_count = (sizeof(api_test_cases) / sizeof(api_test_info_t));

    item_print_help();
    while(1)
    {
        printf("please input cmd index:\n");
        memset(cases, 0x0, BUF_SIZE_128);
        if(NULL == fgets(cases, BUF_SIZE_128, stdin))
        {
            fflush(stdin);
            printf("get cases fail.\n");
            continue;
        }
        fflush(stdin);
        
        cmdidx = atoi(cases);
        if(cmdidx < 0 || cmdidx >= cases_count)
        {
            printf("cmdidx no exist.\n");
            continue;
        }
        else if(cmdidx == cases_count - 1)
        {
            api_test_cases[cmdidx].excute_func();
            break;
        }

        if(api_test_cases[cmdidx].excute_func != NULL)
        {
            api_test_cases[cmdidx].excute_func();
        }
        else
        {
            printf("excute func is NULL.\n");
        }
    }

   return 0;
}
