/*
*    mbtk_ril_api.c
*
*    MBTK ril API for v2(Only for DS).
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN        WHO       WHAT,WHERE,WHY
--------    --------    -------------------------------------------------------
2024/08/15     LiuBin      Initial version
2024/12/31     LiuBin      Auto open default port.
                           Add port index for open/close,close port only index is 0.
2025/03/26     LiuBin      Add timeout for API(Default timeout is 10s).
2025/04/22     LiuBin      Add DS(Dual Sim) support.
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <pthread.h>
#include <sys/epoll.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>

#include "mbtk_ril.h"
#include "mbtk_list.h"
#include "mbtk_utils.h"
#include "mbtk_log.h"
#include "mbtk_str.h"

typedef struct {
    ATPortType_enum port;
//    bool enable;
    int open_count;
} ril_at_port_info_t;

typedef struct {
    int cli_fd;
    bool ril_ready;

    ril_at_port_info_t ports[ATPORTTYPE_NUM];
    int exit_fd[2];
    mbtk_ril_callback_func cb[IND_REGISTER_MAX];
    pthread_t read_thread_id;
    pthread_mutex_t send_mutex;
    list_node_t* msg_list;           // ril_msg_info_t
    list_node_t* cli_thread_list;     // ril_cli_thread_info_t
} ril_cli_info_t;

#define EPOLL_LISTEN_MAX 100
#define RIL_TIMEOUT_DEF 10  // Second

#define INIT_DEFAULT_0 0

static ril_cli_info_t ril_cli;
static int pthread_id_index = 1;

static bool ril_cid_check(mbtk_ril_cid_enum cid)
{
    if(cid < MBTK_APN_CID_MIN || cid > MBTK_APN_CID_MAX)
        return FALSE;

    return TRUE;
}

static ril_cli_thread_info_t* thread_find_by_pid(pthread_t pid)
{
    ril_cli_thread_info_t* cli_thread = NULL;
    list_first(ril_cli.cli_thread_list);
    while ((cli_thread = (ril_cli_thread_info_t*)list_next(ril_cli.cli_thread_list)))
    {
        if(pid == cli_thread->pid) {
            break;
        }
    }

    if(!cli_thread) { // No found cli thread in list.
        cli_thread = (ril_cli_thread_info_t*)malloc(sizeof(ril_cli_thread_info_t));
        memset(cli_thread, 0, sizeof(ril_cli_thread_info_t));
        cli_thread->pid = pid;
        sprintf(cli_thread->name, "PID-%d", pthread_id_index++);
        pthread_mutex_init(&cli_thread->mutex, NULL);
        pthread_cond_init(&cli_thread->cond, NULL);
        cli_thread->is_waitting = FALSE;

        list_add(ril_cli.cli_thread_list, cli_thread);
    }

    return cli_thread;
}

static void timer_start(ril_cli_thread_info_t* cli_thread, int initial_sec) {
    struct sigevent sev;
    struct itimerspec its;

    // 设置信号通知方式
    sev.sigev_notify = SIGEV_SIGNAL;
    sev.sigev_signo = SIGRTMIN;  // 使用实时信号
    sev.sigev_value.sival_ptr = &(cli_thread->pid);

    // 创建计时器
    if (timer_create(CLOCK_MONOTONIC, &sev, &(cli_thread->timer_id)) == -1) {
        LOGE("timer_create() fail.");
        return;
    }

    // 配置计时器时间
    its.it_value.tv_sec = initial_sec;
    its.it_value.tv_nsec = 0;
    its.it_interval.tv_sec = 0;  // 0 表示单次触发
    its.it_interval.tv_nsec = 0;

    // 启动计时器
    if (timer_settime(cli_thread->timer_id, 0, &its, NULL) == -1) {
        LOGE("timer_settime() fail.");
    } else {
        LOGD("Timer start : %s, msg_index - %d, timeout - %ds", cli_thread->name, cli_thread->msg_index, initial_sec);
    }
}

static void timer_cancel(ril_cli_thread_info_t* cli_thread) {
    if (timer_delete(cli_thread->timer_id) == -1) {
        LOGE("timer_delete() fail.");
    } else {
        LOGD("Timer cancel : %s, msg_index - %d", cli_thread->name, cli_thread->msg_index);
    }
}

static void ril_thread_wait(ril_cli_thread_info_t* cli_thread)
{
    if(!cli_thread->is_waitting) {
        LOGD("Thread(%s) waitting...", cli_thread->name);
        cli_thread->is_waitting = TRUE;
        pthread_mutex_lock(&cli_thread->mutex);
        pthread_cond_wait(&cli_thread->cond, &cli_thread->mutex);
        pthread_mutex_unlock(&cli_thread->mutex);
        cli_thread->is_waitting = FALSE;
        LOGD("Thread(%s) running...", cli_thread->name);
    } else {
        LOGD("RSP befor SEND, not waitting...");
        cli_thread->is_waitting = FALSE;
    }
}


static void ril_thread_continue(ril_cli_thread_info_t* cli_thread)
{
    if(cli_thread) { // Found cli thread in list.
        if(cli_thread->is_waitting) {
            LOGD("Thread(%s) will continue...", cli_thread->name);
            pthread_mutex_lock(&cli_thread->mutex);
            pthread_cond_signal(&cli_thread->cond);
            pthread_mutex_unlock(&cli_thread->mutex);

            cli_thread->msg_index = RIL_MSG_INDEX_INVALID;
        } else {
            LOGD("Thread(%s) not continue...", cli_thread->name);
            cli_thread->is_waitting = TRUE;
        }
    } else {
        LOGE("No found cli thread[Cannot occur].");
    }
}

static void timer_handler(int signo, siginfo_t *info, void *context) {
    pthread_t *pid = (pthread_t *)info->si_value.sival_ptr;

//     LOGD("%d timer expired.", *pid);
    ril_cli_thread_info_t* cli_thread = NULL;
    list_first(ril_cli.cli_thread_list);
    while ((cli_thread = (ril_cli_thread_info_t*) list_next(ril_cli.cli_thread_list)))
    {
        if(*pid == cli_thread->pid) {
            break;
        }
    }

    if(cli_thread) { // Found cli thread in list.
        LOGD("Timer handler : %s, msg_index - %d", cli_thread->name, cli_thread->msg_index);

        if(cli_thread->msg_index == RIL_MSG_INDEX_INVALID) {
            LOGE("Invalid message index.");
            return;
        }

        pthread_mutex_lock(&ril_cli.send_mutex);
        ril_msg_info_t* msg = NULL;
        list_first(ril_cli.msg_list);
        while ((msg = (ril_msg_info_t*) list_next(ril_cli.msg_list)))
        {
            if(cli_thread->msg_index == msg->pack->msg_index) {
                break;
            }
        }

        if(NULL == msg) { // No found message in msg list.
            LOGE("Unknown msg.[Cannot occur]");
        } else if(!msg->is_async) {
            LOGD("Msg : Index - %d, ID - %s", cli_thread->msg_index,
                id2str(msg->pack->msg_id));

            *(msg->rsp_data_len) = 0;
            *(msg->rsp_err) = MBTK_RIL_ERR_TIMEOUT;

            ril_thread_continue(cli_thread);

            if(list_remove(ril_cli.msg_list, msg)) {
                ril_msg_pack_free(msg->pack);
                free(msg);
            }
        } else {
            LOGE("is_async[Cannot occur]");
        }
        pthread_mutex_unlock(&ril_cli.send_mutex);
    } else {
        LOGE("No found cli thread[Cannot occur].");
    }
}

static int ril_ind_process(ril_msg_pack_info_t* pack)
{
    LOGD("IND - %s", id2str(pack->msg_id));
    if(pack->msg_id > RIL_MSG_ID_IND_BEGIN && pack->msg_id < RIL_MSG_ID_IND_END) {
        if(ril_cli.cb[pack->msg_id - RIL_MSG_ID_IND_BEGIN - 1])
            ril_cli.cb[pack->msg_id - RIL_MSG_ID_IND_BEGIN - 1](pack->data, pack->data_len);
    }

    switch(pack->msg_id) {
        case RIL_MSG_ID_IND_SER_STATE_CHANGE: // rild service ready.
        {
            if(pack->data && pack->data_len > 0) {
                mbtk_ril_ser_state_enum state = (mbtk_ril_ser_state_enum)pack->data[0];
                if(state == MBTK_RIL_SER_STATE_READY) {
                    ril_cli.ril_ready = TRUE;
                }
            }
            return 0;
        }
        case RIL_MSG_ID_IND_NET_REG_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_CALL_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_SMS_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_RADIO_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_SIM_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_PDP_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_SIGNAL_STATE_CHANGE:
        {
            return 0;
        }
        default:
        {
            LOGE("Unknown IND : %d", pack->msg_id);
            return -1;
        }
    }
}

static int ril_rsp_process(ril_msg_pack_info_t* pack)
{
    if(pack->msg_index == RIL_MSG_INDEX_INVALID) {
        LOGE("Invalid message index.");
        return -1;
    }

    pthread_mutex_lock(&ril_cli.send_mutex);
    ril_msg_info_t* msg = NULL;
    list_first(ril_cli.msg_list);
    while ((msg = (ril_msg_info_t*) list_next(ril_cli.msg_list)))
    {
        if(pack->msg_index == msg->pack->msg_index) {
            break;
        }
    }

    if(NULL == msg) { // No found message in msg list.(message error or has deleted from list because of timeout)
        LOGW("Unknown msg or timeout : Index - %d, Type - %s, ID - %s, Result - %s ,Length - %d",
            pack->msg_index, type2str(pack->msg_type), id2str(pack->msg_id), err2str(pack->err),
            pack->data_len);
        pthread_mutex_unlock(&ril_cli.send_mutex);
        return -1;
    } else {
        int result = 0;
        LOGD("Msg : Index - %d, Type - %s, ID - %s, Result - %s ,Length - %d", pack->msg_index,
            type2str(pack->msg_type),
            id2str(pack->msg_id),
            err2str(pack->err),
            pack->data_len);
        if(msg->pack->msg_id != pack->msg_id) {
            LOGE("msg id not match : %d , %d", msg->pack->msg_id ,pack->msg_id);
            result = -1;
            *(msg->rsp_err) = MBTK_RIL_ERR_FORMAT;
            goto rm_msg_from_list;
        }

        if(msg->rsp_data && pack->data && pack->data_len > 0) {
            memcpy(msg->rsp_data, pack->data, pack->data_len);
        }

        *(msg->rsp_data_len) = pack->data_len;
        *(msg->rsp_err) = pack->err;

        if(!msg->is_async) {
            ril_cli_thread_info_t* cli_thread = NULL;
            list_first(ril_cli.cli_thread_list);
            while ((cli_thread = (ril_cli_thread_info_t*) list_next(ril_cli.cli_thread_list)))
            {
                if(msg->pid == cli_thread->pid) {
                    break;
                }
            }
            // LOGD("msg_index - %d", cli_thread->msg_index);
            if(cli_thread->msg_index > 0) {
                timer_cancel(cli_thread);
            }
            ril_thread_continue(cli_thread);
        }

rm_msg_from_list:
        if(list_remove(ril_cli.msg_list, msg)) {
            ril_msg_pack_free(msg->pack);
            free(msg);
        }
        pthread_mutex_unlock(&ril_cli.send_mutex);
        return result;
    }
}

static int ril_pack_process(ril_msg_pack_info_t* pack)
{
    if(pack->msg_type == RIL_MSG_TYPE_IND) { // IND message.
        return ril_ind_process(pack);
    } else if(pack->msg_type == RIL_MSG_TYPE_RSP) {// Response message.
        return ril_rsp_process(pack);
    } else {
        LOGE("Pack must be RSP or IND.");
        return -1;
    }
}

static void* ril_read_run(void* arg)
{
    int epoll_fd = epoll_create(5);
    if(epoll_fd < 0)
    {
        LOGE("epoll_create() fail[%d].", errno);
        return NULL;
    }

    uint32 event = EPOLLIN | EPOLLET;
    struct epoll_event ev_cli, ev_exit;
    ev_cli.data.fd = ril_cli.cli_fd;
    ev_cli.events = event; //EPOLLIN | EPOLLERR | EPOLLET;
    epoll_ctl(epoll_fd,EPOLL_CTL_ADD, ril_cli.cli_fd, &ev_cli);

    ev_exit.data.fd = ril_cli.exit_fd[0];
    ev_exit.events = event; //EPOLLIN | EPOLLERR | EPOLLET;
    epoll_ctl(epoll_fd,EPOLL_CTL_ADD, ril_cli.exit_fd[0], &ev_exit);

    int nready = -1;
    struct epoll_event epoll_events[EPOLL_LISTEN_MAX];
    while(1)
    {
        nready = epoll_wait(epoll_fd, epoll_events, EPOLL_LISTEN_MAX, -1);
        if(nready > 0)
        {
            int i;
            for(i = 0; i < nready; i++)
            {
                LOGD("fd[%d] event = %x",epoll_events[i].data.fd, epoll_events[i].events);
                if(epoll_events[i].events & EPOLLHUP)   // Closed by server.
                {
                    LOGD("Closed by server.");
                    if(ril_cli.cb[RIL_MSG_ID_IND_SER_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1]) {
                        mbtk_ril_ser_state_enum state = MBTK_RIL_SER_STATE_EXIT;
                        ril_cli.cb[RIL_MSG_ID_IND_SER_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1](&state, 1);
                    }
                    goto read_thread_exit;
                }
                else if(epoll_events[i].events & EPOLLIN)
                {
                    if(ril_cli.cli_fd == epoll_events[i].data.fd)  // Server data arrive.
                    {
                        // Read and process every message.
                        mbtk_ril_err_enum err = MBTK_RIL_ERR_SUCCESS;
                        ril_msg_pack_info_t **pack = ril_pack_recv(ril_cli.cli_fd, false, &err);

                        // Parse packet error,send error response to client.
                        if(pack == NULL)
                        {
                            if(err != MBTK_RIL_ERR_SUCCESS)
                            {
                                LOGE("RSP packet error[%s].", err2str(err));
                            }
                        }
                        else
                        {
                            ril_msg_pack_info_t** pack_ptr = pack;
                            while(*pack_ptr)
                            {
                                ril_pack_process(*pack_ptr);
                                ril_msg_pack_free(*pack_ptr);
                                pack_ptr++;
                            }
                            free(pack);
                        }
                    }
                    else if(ril_cli.exit_fd[0] == epoll_events[i].data.fd) //
                    {
                        char buff[100] = {0};
                        int len = read(ril_cli.exit_fd[0], buff, 100);
                        if(len > 0)
                        {
                            LOGI("CMD : %s", buff);
                            if(strcmp(buff, "EXIT") == 0)
                            {
                                goto read_thread_exit;
                            }
                            else
                            {
                                LOGE("Unkonw cmd : %s", buff);
                            }
                        }
                        else
                        {
                            LOGE("sock_read() fail.");
                        }
                    }
                    else
                    {
                        LOGE("Unknown socket : %d", epoll_events[i].data.fd);
                    }
                }
                else
                {
                    LOGE("Unknown event : %x", epoll_events[i].events);
                }
            }
        }
        else
        {
            LOGE("epoll_wait() fail[%d].", errno);
        }
    }

read_thread_exit:
    LOGD("info_read thread exit.");
    return NULL;
}

/*
* Return recv data length.
* -1 : fail.
*/
static mbtk_ril_err_enum ril_req_process(mbtk_sim_type_enum sim_id, ATPortType_enum port,
                             ril_msg_id_enum  id,
                             const void         *req,
                             int                req_len,
                             void               *rsp,
                             bool is_async,
                             int timeout)
{
    #ifdef LYNQ_DSDS_SUPPORT
    ril_msg_pack_info_t* pack = ril_msg_pack_creat(sim_id, port, RIL_MSG_TYPE_REQ, id, RIL_MSG_INDEX_AUTO, req, req_len);
    #else
    ril_msg_pack_info_t* pack = ril_msg_pack_creat(MBTK_SIM_1, port, RIL_MSG_TYPE_REQ, id, RIL_MSG_INDEX_AUTO, req, req_len);
    #endif
    if(pack == NULL)
    {
        return MBTK_RIL_ERR_MEMORY;
    }

    ril_msg_info_t *msg_info = (ril_msg_info_t*)malloc(sizeof(ril_msg_info_t));
    if(msg_info == NULL)
    {
        ril_msg_pack_free(pack);
        return MBTK_RIL_ERR_MEMORY;
    }

    int rsp_data_len = 0;
    int rsp_err = MBTK_RIL_ERR_SUCCESS;

    pthread_mutex_lock(&ril_cli.send_mutex);
    // Add to msg list.
    msg_info->pid = pthread_self();
    msg_info->pack = pack;
    msg_info->rsp_data_len = &rsp_data_len;
    msg_info->rsp_err = &rsp_err;
    if(!is_async) { // Waitting for response for sync request.
        msg_info->rsp_data = rsp;   // Save response data.
    } else {
        msg_info->rsp_data = NULL;
    }
    msg_info->is_async = is_async;
    list_add(ril_cli.msg_list, msg_info);
    pthread_mutex_unlock(&ril_cli.send_mutex);

    ril_cli_thread_info_t* cli_thread = thread_find_by_pid(msg_info->pid);
    // Send pack to server success.
    if(ril_pack_send(ril_cli.cli_fd, pack) >= RIL_SOCK_PACK_LEN_MIN) {
        if(!is_async) { // Waitting for response for sync request.
            if(timeout == 0) {
                timeout = RIL_TIMEOUT_DEF;
            }
            if(timeout > 0) {
                cli_thread->msg_index = pack->msg_index;
                timer_start(cli_thread, timeout);
            }

            ril_thread_wait(cli_thread);
        }

        LOGD("REQ(%s) success.", id2str(id));
        return rsp_err;
    } else {
        if(list_remove(ril_cli.msg_list, msg_info)) {
            ril_msg_pack_free(msg_info->pack);
            free(msg_info);
        }
        return MBTK_RIL_ERR_SEND_PACK;
    }
}

static void msg_list_free_cb(void *data)
{
    if(data) {
        ril_msg_info_t *msg = (ril_msg_info_t*)data;
        if(msg->pack) {
            free(msg->pack);
        }
        free(msg);
    }
}

static void cli_thread_list_free_cb(void *data)
{
    if(data) {
        ril_cli_thread_info_t *cli_thread = (ril_cli_thread_info_t*)data;
        pthread_mutex_destroy(&cli_thread->mutex);
        pthread_cond_destroy(&cli_thread->cond);
        free(cli_thread);
    }
}

void mbtk_ril_lib_info_print()
{
    MBTK_SOURCE_INFO_PRINT("mbtk_ril_lib");
}

static bool at_port_check(ATPortType_enum port)
{
    if(port < ATPORTTYPE_0 || port >= ATPORTTYPE_NUM) {
        return FALSE;
    }

    return ril_cli.ports[port].port == port && ril_cli.ports[port].open_count > 0;
}

mbtk_ril_handle* mbtk_ril_open(ATPortType_enum port)
{
    if(port < ATPORTTYPE_0 || port >= ATPORTTYPE_NUM) {
        LOGE("Port(%d) error.", port);
        return NULL;
    }

    if(!ril_cli.ril_ready) {
        ril_cli.cli_fd = socket(AF_LOCAL, SOCK_STREAM, 0);
        if(ril_cli.cli_fd < 0)
        {
            LOGE("socket() fail[%d].", errno);
            return NULL;
        }

        // Set O_NONBLOCK
        int flags = fcntl(ril_cli.cli_fd, F_GETFL, 0);
        if (flags < 0)
        {
            LOGE("Get flags error:%d", errno);
            goto error;
        }
        flags |= O_NONBLOCK;
        if (fcntl(ril_cli.cli_fd, F_SETFL, flags) < 0)
        {
            LOGE("Set flags error:%d", errno);
            goto error;
        }

        struct sockaddr_un cli_addr;
        memset(&cli_addr, 0, sizeof(cli_addr));
        cli_addr.sun_family = AF_LOCAL;
        strcpy(cli_addr.sun_path, RIL_SOCK_NAME);
        if(connect(ril_cli.cli_fd, (struct sockaddr *)&cli_addr, sizeof(cli_addr)))
        {
            LOGE("connect() fail[%d].", errno);
            goto error;
        }

        if(pipe(ril_cli.exit_fd))
        {
            LOGE("pipe() fail[%d].", errno);
            goto error;
        }

#if 0
        pthread_attr_t thread_attr;
        pthread_attr_init(&thread_attr);
        if(pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))
        {
            LOGE("pthread_attr_setdetachstate() fail.");
            goto error;
        }

        if(pthread_create(&ril_cli.read_thread_id, &thread_attr, ril_read_run, NULL))
        {
            LOGE("pthread_create() fail.");
            goto error;
        }
        pthread_attr_destroy(&thread_attr);
#else
        if(pthread_create(&ril_cli.read_thread_id, NULL, ril_read_run, NULL))
        {
            LOGE("pthread_create() fail.");
            goto error;
        }
#endif

        pthread_mutex_init(&ril_cli.send_mutex, NULL);
        ril_cli.msg_list = list_create(msg_list_free_cb);
        if(ril_cli.msg_list == NULL) {
            LOGE("list_create(msg_list) fail.");
            goto error;
        }

        ril_cli.cli_thread_list = list_create(cli_thread_list_free_cb);
        if(ril_cli.cli_thread_list == NULL) {
            LOGE("list_create(cli_thread_list) fail.");
            goto error;
        }

        struct sigaction sa;

        // 配置信号处理
        sa.sa_flags = SA_SIGINFO;
        sa.sa_sigaction = timer_handler;
        sigemptyset(&sa.sa_mask);
        if (sigaction(SIGRTMIN, &sa, NULL) == -1) {
            LOGE("sigaction() fail.");
            goto error;
        }

        ril_cli.ril_ready = FALSE;

        // Waitting mbtk_rild ready...
        while(!ril_cli.ril_ready) {
            usleep(100000);
        }

        LOGD("RIL server ready...");
    }

    // Auto open default port.
    ril_cli.ports[MBTK_AT_PORT_DEF].port = MBTK_AT_PORT_DEF;
    ril_cli.ports[MBTK_AT_PORT_DEF].open_count++;
    if(port != MBTK_AT_PORT_DEF) {
        ril_cli.ports[port].port = port;
        ril_cli.ports[port].open_count++;
    }

    return &(ril_cli.ports[port]);
error:
    if(ril_cli.cli_fd > 0)
    {
        close(ril_cli.cli_fd);
        ril_cli.cli_fd = -1;
    }
    if(ril_cli.exit_fd[0] > 0)
    {
        close(ril_cli.exit_fd[0]);
        ril_cli.exit_fd[0] = -1;
    }
    if(ril_cli.exit_fd[1] > 0)
    {
        close(ril_cli.exit_fd[1]);
        ril_cli.exit_fd[1] = -1;
    }

    return NULL;
}

mbtk_ril_err_enum mbtk_ril_close(ATPortType_enum port)
{
    if(!at_port_check(port)) {
        return MBTK_RIL_ERR_PORT;
    }

    // Not def port,should auto close def port.
    if(port != MBTK_AT_PORT_DEF) {
        ril_cli.ports[MBTK_AT_PORT_DEF].open_count--;
        if(ril_cli.ports[MBTK_AT_PORT_DEF].open_count == 0) {
            ril_cli.ports[MBTK_AT_PORT_DEF].port = ATPORTTYPE_NON;
            LOGD("Close port : %d", MBTK_AT_PORT_DEF);
        }
    }

    ril_cli.ports[port].open_count--;
    if(ril_cli.ports[port].open_count == 0) {
        ril_cli.ports[port].port = ATPORTTYPE_NON;
        LOGD("Close port : %d", port);
    }

    // All port is close ???
    int i = ATPORTTYPE_0;
    for(; i < ATPORTTYPE_NUM; i++) {
        //if(ril_cli.ports[i].enable)
        if(ril_cli.ports[i].open_count > 0)
            break;
    }

    if(i == ATPORTTYPE_NUM) { // All port is close.
        LOGD("Will close socket connect.");
        if(!ril_cli.ril_ready)
        {
            return MBTK_RIL_ERR_NOT_INIT;
        }

        if(ril_cli.exit_fd[1] > 0)
        {
            if(write(ril_cli.exit_fd[1], "EXIT", 4) != 4) {
                return MBTK_RIL_ERR_UNKNOWN;
            }
        }

        // Wait read_thread exit.
        pthread_join(ril_cli.read_thread_id,NULL);

        if(ril_cli.exit_fd[0] > 0)
        {
            close(ril_cli.exit_fd[0]);
            ril_cli.exit_fd[0] = -1;
        }

        if(ril_cli.exit_fd[1] > 0)
        {
            close(ril_cli.exit_fd[1]);
            ril_cli.exit_fd[1] = -1;
        }

        if(ril_cli.cli_fd > 0)
        {
            close(ril_cli.cli_fd);
            ril_cli.cli_fd = -1;
        }

        pthread_mutex_destroy(&ril_cli.send_mutex);
        if(ril_cli.msg_list) {
            list_free(ril_cli.msg_list);
        }
        if(ril_cli.cli_thread_list) {
            list_free(ril_cli.cli_thread_list);
        }

        ril_cli.ril_ready = FALSE;

        return MBTK_RIL_ERR_SUCCESS;
    } else {
        i = ATPORTTYPE_0;
        for(; i < ATPORTTYPE_NUM; i++) {
            if(ril_cli.ports[i].open_count > 0) {
                LOGD("Port[%d] open_count is %d, so can't close.", i, ril_cli.ports[i].open_count);
            }
        }
        return MBTK_RIL_ERR_SUCCESS;
    }
}

/*
* Get platform version.
*/
mbtk_ril_err_enum mbtk_ds_version_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, void *version)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(version == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DEV_VERSION, NULL, 0, version, FALSE, 0);
}

/*
* Get platform version.
*/
mbtk_ril_err_enum mbtk_version_get(mbtk_ril_handle* handle, void *version)
{
    return mbtk_ds_version_get(handle, MBTK_SIM_AUTO, version);
}

/*
* Get platform version.
*/
mbtk_ril_err_enum mbtk_ds_model_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, void *model)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(model == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DEV_MODEL, NULL, 0, model, FALSE, 0);
}

/*
* Get platform version.
*/
mbtk_ril_err_enum mbtk_model_get(mbtk_ril_handle* handle, void *model)
{
    return mbtk_ds_model_get(handle, MBTK_SIM_AUTO, model);
}

/*
* Get platform IMEI.
*/
mbtk_ril_err_enum mbtk_imei_get(mbtk_ril_handle* handle, void *imei)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(imei == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_DEV_IMEI, NULL, 0, imei, FALSE, 0);
}


/*
* Get platform SN.
*/
mbtk_ril_err_enum mbtk_sn_get(mbtk_ril_handle* handle, void *sn)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sn == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_DEV_SN, NULL, 0, sn, FALSE, 0);
}


/*
* Get platform MEID.
*/
mbtk_ril_err_enum mbtk_meid_get(mbtk_ril_handle* handle, void *meid)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(meid == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_DEV_MEID, NULL, 0, meid, FALSE, 0);
}


/*
* Return VoLTE state.
*/
mbtk_ril_err_enum mbtk_volte_state_get(mbtk_ril_handle* handle, int *volte_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(volte_state == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *volte_state = INIT_DEFAULT_0;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_DEV_VOLTE, NULL, 0, volte_state, FALSE, 0);
}

/*
* Set VoLTE state.
*
* volte_state:
* 0 : Close VoLTE.
* 1 : Open VoLTE.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_volte_state_set(mbtk_ril_handle* handle, int volte_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if((volte_state != 0 && volte_state != 1) || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_DEV_VOLTE, &volte_state, 1, NULL, FALSE, 0);
}

/*
* Return radio state.
*/
mbtk_ril_err_enum mbtk_ds_radio_state_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_radio_state_enum *radio_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(radio_state == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *radio_state = MBTK_RADIO_STATE_MINI_FUNC;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DEV_MODEM, NULL, 0, &result, FALSE, 0);
    *radio_state = (mbtk_radio_state_enum)result;
    return err;
}


/*
* Return radio state.
*/
mbtk_ril_err_enum mbtk_radio_state_get(mbtk_ril_handle* handle, mbtk_radio_state_enum *radio_state)
{
    return mbtk_ds_radio_state_get(handle, MBTK_SIM_AUTO, radio_state);
}

/*
* Set radio state.
*
* Refor to : mbtk_radio_state_enum
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_radio_state_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_radio_state_enum radio_state, bool reset_modem)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(radio_state == MBTK_RADIO_STATE_UNKNOWN || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    uint8 buff[2];
    buff[0] = (uint8)radio_state;
    buff[1] = reset_modem ? 1 : 0;

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DEV_MODEM, buff, 2, NULL, FALSE, 0);
}

/*
* Set radio state.
*
* Refor to : mbtk_radio_state_enum
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_radio_state_set(mbtk_ril_handle* handle, mbtk_radio_state_enum radio_state, bool reset_modem)
{
    return mbtk_ds_radio_state_set(handle, MBTK_SIM_AUTO, radio_state, reset_modem);
}


/*
* Get system temperature.
*
* type[IN]:
*   0: Soc temperature.
*   1: RF temperature.
* temp[OUT]:
*   temperature in celsius.
*/
mbtk_ril_err_enum mbtk_temp_get(mbtk_ril_handle* handle, mbtk_temp_type_enum type, int* temp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(temp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *temp = INIT_DEFAULT_0;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    int16 result;
    uint8 temp_temp = (uint8)type;
    mbtk_ril_err_enum err = ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_DEV_TEMP, &temp_temp, 1, &result, FALSE, 0);
    *temp = (int16)result;
    return err;
}


/*
* Get time type.
* "23/05/24,06:09:32+32" -> "23/05/24 06:09:32 +32 00"
*/
mbtk_ril_err_enum mbtk_ds_cell_time_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char* time_str)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(time_str == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    char buff[RIL_SOCK_MSG_LEN_MAX] = {0};
    mbtk_ril_err_enum err = ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DEV_CELL_TIME, NULL, 0, &buff, FALSE, 0);
    if(MBTK_RIL_ERR_SUCCESS == err)
    {
        memcpy(time_str,buff,strlen(buff));

        char *temp = strstr(time_str, ",");
        if(temp)
        {
            *temp = ' '; // ',' -> ' '

            temp = strstr(time_str, "+");
            if(temp == NULL)
            {
                temp = strstr(time_str, "-");
            }

            if(temp)
            {
                // Copy +XX or -XX
                char *last_ptr = temp + strlen(temp) + 1;
                while(last_ptr > temp)
                {
                    *last_ptr = *(last_ptr - 1);
                    last_ptr--;
                }

                *last_ptr = ' ';

                memcpy(temp + strlen(temp), " 00", 3);

                LOGD("%s -> %s", buff, time_str);
                return MBTK_RIL_ERR_SUCCESS;
            }
            else
            {
                LOGE("Time error:%s",buff);
                return MBTK_RIL_ERR_TIME_FORMAT;
            }
        }
        else
        {
            LOGE("Time error:%s",buff);
            return MBTK_RIL_ERR_TIME_FORMAT;
        }
    }
    else
    {
        return err;
    }
}

/*
* Get time type.
* "23/05/24,06:09:32+32" -> "23/05/24 06:09:32 +32 00"
*/
mbtk_ril_err_enum mbtk_cell_time_get(mbtk_ril_handle* handle, char* time_str)
{
    return mbtk_ds_cell_time_get(handle, MBTK_SIM_AUTO, time_str);
}

/*
* Get sim state.
*/
mbtk_ril_err_enum mbtk_ds_sim_state_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_sim_state_enum *sim_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sim_state == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *sim_state = MBTK_SIM_STATE_ABSENT;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_STATE, NULL, 0, &result, FALSE, 0);
    *sim_state = (mbtk_sim_state_enum)result;
    return err;
}

/*
* Get sim state.
*/
mbtk_ril_err_enum mbtk_sim_state_get(mbtk_ril_handle* handle, mbtk_sim_state_enum *sim_state)
{
    return mbtk_ds_sim_state_get(handle, MBTK_SIM_AUTO, sim_state);
}


/*
* Get sim card type.
*/
mbtk_ril_err_enum mbtk_ds_sim_type_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_sim_card_type_enum *sim_card_type)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sim_card_type == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *sim_card_type = MBTK_SIM;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_TYPE, NULL, 0, &result, FALSE, 0);
    *sim_card_type = (mbtk_sim_card_type_enum)result;
    return err;
}

/*
* Get sim card type.
*/
mbtk_ril_err_enum mbtk_sim_type_get(mbtk_ril_handle* handle, mbtk_sim_card_type_enum *sim_card_type)
{
    return mbtk_ds_sim_type_get(handle, MBTK_SIM_AUTO, sim_card_type);
}

/*
* Get platform IMSI.
*/
mbtk_ril_err_enum mbtk_ds_imsi_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, void *imsi)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(imsi == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_IMSI, NULL, 0, imsi, FALSE, 0);
}

/*
* Get platform IMSI.
*/
mbtk_ril_err_enum mbtk_imsi_get(mbtk_ril_handle* handle, void *imsi)
{
    return mbtk_ds_imsi_get(handle, MBTK_SIM_AUTO, imsi);
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_ds_iccid_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, void *iccid)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(iccid == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_ICCID, NULL, 0, iccid, FALSE, 0);
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_iccid_get(mbtk_ril_handle* handle, void *iccid)
{
    return mbtk_ds_iccid_get(handle, MBTK_SIM_AUTO, iccid);
}

/*
* Get current phone number.
*/
mbtk_ril_err_enum mbtk_ds_phone_number_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, void *phone_number)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(phone_number == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_PN, NULL, 0, phone_number, FALSE, 0);
}

/*
* Get current phone number.
*/
mbtk_ril_err_enum mbtk_phone_number_get(mbtk_ril_handle* handle, void *phone_number)
{
    return mbtk_ds_phone_number_get(handle, MBTK_SIM_AUTO, phone_number);
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_ds_sim_lock_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int *lock_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(lock_state == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *lock_state = INIT_DEFAULT_0;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_LOCK, NULL, 0, &result, FALSE, 0);
    *lock_state = result;
    return err;
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_sim_lock_get(mbtk_ril_handle* handle, int *lock_state)
{
    return mbtk_ds_sim_lock_get(handle, MBTK_SIM_AUTO, lock_state);
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_ds_sim_lock_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_sim_lock_info_t *lock_info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(lock_info == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_LOCK, lock_info, sizeof(mbtk_sim_lock_info_t), NULL, FALSE, 0);
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_sim_lock_set(mbtk_ril_handle* handle, mbtk_sim_lock_info_t *lock_info)
{
    return mbtk_ds_sim_lock_set(handle, MBTK_SIM_AUTO, lock_info);
}

/*
* Get PIN’s number of remaining retry
*/
mbtk_ril_err_enum mbtk_ds_sim_lock_retry_times_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_pin_puk_last_times_t *retry_times)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(retry_times == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(retry_times, 0x00, sizeof(mbtk_pin_puk_last_times_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_PINPUK_TIMES, NULL, 0, retry_times, FALSE, 0);
}

/*
* Get PIN’s number of remaining retry
*/
mbtk_ril_err_enum mbtk_sim_lock_retry_times_get(mbtk_ril_handle* handle, mbtk_pin_puk_last_times_t *retry_times)
{
    return mbtk_ds_sim_lock_retry_times_get(handle, MBTK_SIM_AUTO, retry_times);
}

/*
* Get plmn list
*/
mbtk_ril_err_enum mbtk_ds_plmn_list_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_plmn_info *plmn_list)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(plmn_list == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(plmn_list, 0x00, sizeof(mbtk_plmn_info));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SIM_PLMN, NULL, 0, plmn_list, FALSE, 0);
}

/*
* Get plmn list
*/
mbtk_ril_err_enum mbtk_plmn_list_get(mbtk_ril_handle* handle, mbtk_plmn_info *plmn_list)
{
    return mbtk_ds_plmn_list_get(handle, MBTK_SIM_AUTO, plmn_list);
}

/*
* Get available network.
*/
mbtk_ril_err_enum mbtk_ds_available_net_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_net_info_array_t *net_array)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(net_array == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(net_array, 0x00, sizeof(mbtk_net_info_array_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_AVAILABLE, NULL, 0, net_array, FALSE, -1);
}

/*
* Get available network.
*/
mbtk_ril_err_enum mbtk_available_net_get(mbtk_ril_handle* handle, mbtk_net_info_array_t *net_array)
{
    return mbtk_ds_available_net_get(handle, MBTK_SIM_AUTO, net_array);
}

/*
* Set network select mode. (+COPS=...)
*/
mbtk_ril_err_enum mbtk_ds_net_sel_mode_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, const mbtk_net_info_t *net)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(net == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_SEL_MODE, net, sizeof(mbtk_net_info_t), NULL, FALSE, 0);
}

/*
* Set network select mode. (+COPS=...)
*/
mbtk_ril_err_enum mbtk_net_sel_mode_set(mbtk_ril_handle* handle, const mbtk_net_info_t *net)
{
    return mbtk_ds_net_sel_mode_set(handle, MBTK_SIM_AUTO, net);
}

/*
* Get network select mode. (+COPS?)
*/
mbtk_ril_err_enum mbtk_ds_net_sel_mode_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_net_info_t *net)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(net == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(net, 0x00, sizeof(mbtk_net_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_SEL_MODE, NULL, 0, net, FALSE, 0);
}

/*
* Get network select mode. (+COPS?)
*/
mbtk_ril_err_enum mbtk_net_sel_mode_get(mbtk_ril_handle* handle, mbtk_net_info_t *net)
{
    return mbtk_ds_net_sel_mode_get(handle, MBTK_SIM_AUTO, net);
}

/*
* Get platform support bands.
*/
mbtk_ril_err_enum mbtk_support_band_get(mbtk_ril_handle* handle, mbtk_band_info_t *band)
{
    uint8 type = 0; // Get support bands.
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(band == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(band, 0x00, sizeof(mbtk_band_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_NET_BAND, &type, sizeof(uint8), band, FALSE, 0);
}

/*
* Get platform current bands.
*/
mbtk_ril_err_enum mbtk_ds_current_band_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_band_info_t *band)
{
    uint8 type = 1; // Get current bands.
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(band == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(band, 0x00, sizeof(mbtk_band_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_BAND, &type, sizeof(uint8), band, FALSE, 0);
}


/*
* Get platform current bands.
*/
mbtk_ril_err_enum mbtk_current_band_get(mbtk_ril_handle* handle, mbtk_band_info_t *band)
{
    return mbtk_ds_current_band_get(handle, MBTK_SIM_AUTO, band);
}

/*
* Set platform current bands.
*/
mbtk_ril_err_enum mbtk_ds_current_band_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, const mbtk_band_info_t *band)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(band == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_BAND, band, sizeof(mbtk_band_info_t), NULL, FALSE, 0);
}

/*
* Set platform current bands.
*/
mbtk_ril_err_enum mbtk_current_band_set(mbtk_ril_handle* handle, const mbtk_band_info_t *band)
{
    return mbtk_ds_current_band_set(handle, MBTK_SIM_AUTO, band);
}

/*
* Get current cell infomation.
*/
mbtk_ril_err_enum mbtk_ds_cell_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_cell_info_array_t *cell_array)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cell_array == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(cell_array, 0x00, sizeof(mbtk_cell_info_array_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_CELL, NULL, 0, cell_array, FALSE, 0);
}

/*
* Get current cell infomation.
*/
mbtk_ril_err_enum mbtk_cell_get(mbtk_ril_handle* handle, mbtk_cell_info_array_t *cell_array)
{
    return mbtk_ds_cell_get(handle, MBTK_SIM_AUTO, cell_array);
}

/*
* Set cell info.
*
* at*CELL=<mode>,<act>,< band>,<freq>,<cellId>
* at*cell=2,3,,40936,429   //
* at*cell=0  //
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_cell_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char *info, char *response)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(info == NULL || response == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_CELL, info, strlen(info)+1, response, FALSE, 0);   // strlen(info)+1 is wrote by hq at 2025/08/26 for bug 2187
}

/*
* Set cell info.
*
* at*CELL=<mode>,<act>,< band>,<freq>,<cellId>
* at*cell=2,3,,40936,429   //
* at*cell=0  //
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_cell_set(mbtk_ril_handle* handle, char *info, char *response)
{
    return mbtk_ds_cell_set(handle, MBTK_SIM_AUTO, info, response);
}

/*
* Get current network signal.
*/
mbtk_ril_err_enum mbtk_ds_net_signal_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_signal_info_t *signal)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(signal == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(signal, 0x00, sizeof(mbtk_signal_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_SIGNAL, NULL, 0, signal, FALSE, 0);
}

/*
* Get current network signal.
*/
mbtk_ril_err_enum mbtk_net_signal_get(mbtk_ril_handle* handle, mbtk_signal_info_t *signal)
{
    return mbtk_ds_net_signal_get(handle, MBTK_SIM_AUTO, signal);
}

/*
* Get current network register information.
*/
mbtk_ril_err_enum mbtk_ds_net_reg_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_net_reg_info_t *reg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(reg == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(reg, 0x00, sizeof(mbtk_net_reg_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_REG, NULL, 0, reg, FALSE, 0);
}

/*
* Get current network register information.
*/
mbtk_ril_err_enum mbtk_net_reg_get(mbtk_ril_handle* handle, mbtk_net_reg_info_t *reg)
{
    return mbtk_ds_net_reg_get(handle, MBTK_SIM_AUTO, reg);
}

/*
* oos get.
*/
mbtk_ril_err_enum mbtk_ds_oos_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_ril_oos_info_t *oos_info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(oos_info == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(oos_info, 0x00, sizeof(mbtk_ril_oos_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_OOS, NULL, 0, oos_info, FALSE, 0);
}

/*
* oos get.
*/
mbtk_ril_err_enum mbtk_oos_get(mbtk_ril_handle* handle, mbtk_ril_oos_info_t *oos_info)
{
    return mbtk_ds_oos_get(handle, MBTK_SIM_AUTO, oos_info);
}

/*
* oos set .
*/
mbtk_ril_err_enum mbtk_ds_oos_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, const mbtk_ril_oos_info_t *oos_info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(oos_info == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_NET_OOS, oos_info, sizeof(mbtk_ril_oos_info_t), NULL, FALSE, 0);
}

/*
* oos set .
*/
mbtk_ril_err_enum mbtk_oos_set(mbtk_ril_handle* handle, const mbtk_ril_oos_info_t *oos_info)
{
    return mbtk_ds_oos_set(handle, MBTK_SIM_AUTO, oos_info);
}

/*
* Set wakeup state.
*
* wakeup_state:(0~31)
* 0 : means resume all
* 1~31 means suspend
* Control the active reporting of some platform modems to reduce wakeup
*/
mbtk_ril_err_enum mbtk_ds_wakeup_state_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int wakeup_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(wakeup_state < 0 || wakeup_state > 31)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DEV_POWERIND, &wakeup_state, sizeof(uint8), NULL, FALSE, 0);
}

/*
* Set wakeup state.
*
* wakeup_state:(0~31)
* 0 : means resume all
* 1~31 means suspend
* Control the active reporting of some platform modems to reduce wakeup
*/
mbtk_ril_err_enum mbtk_wakeup_state_set(mbtk_ril_handle* handle, int wakeup_state)
{
    return mbtk_ds_wakeup_state_set(handle, MBTK_SIM_AUTO, wakeup_state);
}

/*
* Get all APN informations.
*/
mbtk_ril_err_enum mbtk_ds_apn_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_apn_info_array_t *apns)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(apns == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(apns, 0x0, sizeof(mbtk_apn_info_array_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DATA_CALL_APN, NULL, 0, apns, FALSE, 0);
}

/*
* Get all APN informations.
*/
mbtk_ril_err_enum mbtk_apn_get(mbtk_ril_handle* handle, mbtk_apn_info_array_t *apns)
{
    return mbtk_ds_apn_get(handle, MBTK_SIM_AUTO, apns);
}

/*
* Set current APN informations.
*/
mbtk_ril_err_enum mbtk_ds_apn_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_apn_info_t *apn)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(apn == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    mbtk_ril_err_enum err = MBTK_RIL_ERR_SUCCESS;
    mbtk_ril_cid_enum return_cid = MBTK_RIL_CID_NUL;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    if(!apn->b_initial_attach_apn)
    {
        if(str_empty(apn->apn)) { // Delete APN
            if(!ril_cid_check(apn->cid)) {
                return MBTK_RIL_ERR_CID;
            }
        } else { // Set APN
            if(apn->cid != MBTK_RIL_CID_NUL && !ril_cid_check(apn->cid)) {
                return MBTK_RIL_ERR_CID;
            }
        }
    }
    err = ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DATA_CALL_APN, apn, sizeof(mbtk_apn_info_t), &return_cid, FALSE, 0);
    apn->cid = return_cid;
    return err;
}

/*
* Set current APN informations.
*/
mbtk_ril_err_enum mbtk_apn_set(mbtk_ril_handle* handle, mbtk_apn_info_t *apn)
{
    return mbtk_ds_apn_set(handle, MBTK_SIM_AUTO, apn);
}

/*
* Start data call.
*/
mbtk_ril_err_enum mbtk_ds_data_call_start(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_ril_cid_enum cid, mbtk_data_call_item_state_enum auto_boot_call,
            mbtk_data_call_item_state_enum def_route, mbtk_data_call_item_state_enum as_dns, int *retry_interval, int retry_interval_num,
            int timeout, mbtk_ip_info_t *rsp_info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL) {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    if(!ril_cid_check(cid)) {
        return MBTK_RIL_ERR_CID;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    mbtk_data_call_info_t info;
    memset(&info, 0, sizeof(mbtk_data_call_info_t));
    info.type = MBTK_DATA_CALL_START;
    info.cid = cid;
    info.auto_boot_call = auto_boot_call;
    info.def_route = def_route;
    info.as_dns = as_dns;
    int i = 0;
    while(i < retry_interval_num && i < RIL_DATA_CALL_RETRY_MAX) {
        info.retry_interval[i] = (uint16)retry_interval[i];
        i++;
    }
    if(timeout > 0) {
        info.timeout = (uint16)timeout;
    } else {
        info.timeout = (uint16)10;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DATA_CALL_OPT, &info, sizeof(mbtk_data_call_info_t), rsp_info, FALSE, 0);
}

/*
* Start data call.
*/
mbtk_ril_err_enum mbtk_data_call_start(mbtk_ril_handle* handle, mbtk_ril_cid_enum cid, mbtk_data_call_item_state_enum auto_boot_call,
            mbtk_data_call_item_state_enum def_route, mbtk_data_call_item_state_enum as_dns, int *retry_interval, int retry_interval_num,
            int timeout, mbtk_ip_info_t *rsp_info)
{
    return mbtk_ds_data_call_start(handle, MBTK_SIM_AUTO, cid, auto_boot_call,
                def_route, as_dns, retry_interval, retry_interval_num, timeout, rsp_info);
}

/*
* Stop data call.
*/
mbtk_ril_err_enum mbtk_ds_data_call_stop(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_ril_cid_enum cid, int timeout)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL) {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    if(!ril_cid_check(cid)) {
        return MBTK_RIL_ERR_CID;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    mbtk_data_call_info_t info;
    memset(&info, 0, sizeof(mbtk_data_call_info_t));
    info.type = MBTK_DATA_CALL_STOP;
    info.cid = cid;
    if(timeout > 0) {
        info.timeout = (uint16)timeout;
    }
    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DATA_CALL_OPT, &info, sizeof(mbtk_data_call_info_t), NULL, FALSE, 0);
}

/*
* Stop data call.
*/
mbtk_ril_err_enum mbtk_data_call_stop(mbtk_ril_handle* handle, mbtk_ril_cid_enum cid, int timeout)
{
    return mbtk_ds_data_call_stop(handle, MBTK_SIM_AUTO, cid, timeout);
}

/*
* Get data call state.
*/
mbtk_ril_err_enum mbtk_ds_data_call_state_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_ril_cid_enum cid, mbtk_ip_info_t *ip)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(ip == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    if(!ril_cid_check(cid)) {
        return MBTK_RIL_ERR_CID;
    }

    memset(ip, 0x00, sizeof(mbtk_ip_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    mbtk_data_call_info_t info;
    memset(&info, 0, sizeof(mbtk_data_call_info_t));
    info.type = MBTK_DATA_CALL_STATE;
    info.cid = cid;

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_DATA_CALL_OPT, &info, sizeof(mbtk_data_call_info_t), ip, FALSE, 0);
}

/*
* Get data call state.
*/
mbtk_ril_err_enum mbtk_data_call_state_get(mbtk_ril_handle* handle, mbtk_ril_cid_enum cid, mbtk_ip_info_t *ip)
{
    return mbtk_ds_data_call_state_get(handle, MBTK_SIM_AUTO, cid, ip);
}

/*
* Return sms cmgf.
*/
mbtk_ril_err_enum mbtk_ds_sms_cmgf_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int *sms_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sms_state == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *sms_state = INIT_DEFAULT_0;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    uint8 state;
    mbtk_ril_err_enum err = ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGF, NULL, 0, &state, FALSE, 0);
    *sms_state = state;
    return err;
}

/*
* Return sms cmgf.
*/
mbtk_ril_err_enum mbtk_sms_cmgf_get(mbtk_ril_handle* handle, int *sms_state)
{
    return mbtk_ds_sms_cmgf_get(handle, MBTK_SIM_AUTO, sms_state);
}

/*
* Set sms cmgf.
*
* volte_state:
* 0 : PDU mode.
* 1 :  text mode.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cmgf_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int mode)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGF, &mode, sizeof(uint8), NULL, FALSE, 0);
}

/*
* Set sms cmgf.
*
* volte_state:
* 0 : PDU mode.
* 1 :  text mode.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgf_set(mbtk_ril_handle* handle, int mode)
{
    return mbtk_ds_sms_cmgf_set(handle, MBTK_SIM_AUTO, mode);
}

/*
* Set sms cmgs.
*
if PDU mode (+CMGF=0):
+CMGS=<length><CR>
PDU is given<ctrl-Z/ESC>

if text mode (+CMGF=1):
+CMGS=<da>[,<toda>]<CR>
text is entered<ctrl-Z/ESC>

* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cmgs_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * cmgs, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmgs == NULL || resp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGS, cmgs, strlen(cmgs), resp, FALSE, 0);
}

/*
* Set sms cmgs.
*
if PDU mode (+CMGF=0):
+CMGS=<length><CR>
PDU is given<ctrl-Z/ESC>

if text mode (+CMGF=1):
+CMGS=<da>[,<toda>]<CR>
text is entered<ctrl-Z/ESC>

* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgs_set(mbtk_ril_handle* handle, char * cmgs, char *resp)
{
    return mbtk_ds_sms_cmgs_set(handle, MBTK_SIM_AUTO, cmgs, resp);
}

/*
* Set sms cmgw.
*
if text mode (+CMGF=1):
+CMGW=<oa/da>[,<tooa/toda>[,<stat>]]
<CR>
text is entered<ctrl-Z/ESC>
if PDU mode (+CMGF=0):
+CMGW=<length>[,<stat>]<CR>PDU is
given<ctrl-Z/ESC>

*/
mbtk_ril_err_enum mbtk_ds_sms_cmgw_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * cmgw, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmgw == NULL || resp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGW, cmgw, strlen(cmgw), resp, FALSE, 0);
}

/*
* Set sms cmgw.
*
if text mode (+CMGF=1):
+CMGW=<oa/da>[,<tooa/toda>[,<stat>]]
<CR>
text is entered<ctrl-Z/ESC>
if PDU mode (+CMGF=0):
+CMGW=<length>[,<stat>]<CR>PDU is
given<ctrl-Z/ESC>

*/
mbtk_ril_err_enum mbtk_sms_cmgw_set(mbtk_ril_handle* handle, char * cmgw, char *resp)
{
    return mbtk_ds_sms_cmgw_set(handle, MBTK_SIM_AUTO, cmgw, resp);
}


/*
* Set sms cmgd.
*
* +CMGD=<index>[,<delflag>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cmgd_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * cmdg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmdg == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGD, cmdg, strlen(cmdg), NULL, FALSE, 0);
}

/*
* Set sms cmgd.
*
* +CMGD=<index>[,<delflag>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgd_set(mbtk_ril_handle* handle, char * cmdg)
{
    return mbtk_ds_sms_cmgd_set(handle, MBTK_SIM_AUTO, cmdg);
}

/*
* Get sms cmgd.
*
* +CMGD: (XXX,XXX)(0-4)
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cmgd_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * cmdg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmdg == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGD, NULL, 0, cmdg, FALSE, 0);
}

/*
* Get sms cmgd.
*
* +CMGD: (XXX,XXX)(0-4)
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgd_get(mbtk_ril_handle* handle, char * cmdg)
{
    return mbtk_ds_sms_cmgd_get(handle, MBTK_SIM_AUTO, cmdg);
}

/*
* Set sms cmgl.
*
* AT+CMGL[=<stat>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cmgl_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * cmgl, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmgl == NULL || resp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGL, cmgl, strlen(cmgl), resp, FALSE, 0);
}

/*
* Set sms cmgl.
*
* AT+CMGL[=<stat>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgl_set(mbtk_ril_handle* handle, char * cmgl, char *resp)
{
    return mbtk_ds_sms_cmgl_set(handle, MBTK_SIM_AUTO, cmgl, resp);
}


/*
* Return sms csca.
*/
mbtk_ril_err_enum mbtk_ds_sms_csca_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char *buf)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(buf == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CSCA, NULL, 0, buf, FALSE, 0);
}

/*
* Return sms csca.
*/
mbtk_ril_err_enum mbtk_sms_csca_get(mbtk_ril_handle* handle, char *buf)
{
    return mbtk_ds_sms_csca_get(handle, MBTK_SIM_AUTO, buf);
}

/*
* Set sms csca.
*
* AT+CSCA=<number> [,<type>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_csca_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * csca)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(csca == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CSCA, csca, strlen(csca), NULL, FALSE, 0);
}

/*
* Set sms csca.
*
* AT+CSCA=<number> [,<type>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_csca_set(mbtk_ril_handle* handle, char * csca)
{
    return mbtk_ds_sms_csca_set(handle, MBTK_SIM_AUTO, csca);
}


/*
* Set sms csmp.
*
* AT+CSMP=[<fo>[,<vp>[,<pid>[,<dcs>]]]]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_csmp_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * csmp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(csmp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CSMP, csmp, strlen(csmp), NULL, FALSE, 0);
}

/*
* Set sms csmp.
*
* AT+CSMP=[<fo>[,<vp>[,<pid>[,<dcs>]]]]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_csmp_set(mbtk_ril_handle* handle, char * csmp)
{
    return mbtk_ds_sms_csmp_set(handle, MBTK_SIM_AUTO, csmp);
}

/*
* Set sms cscb.
*
* AT+CSCB=<[<mode>[,<mids>[,<dcss>]]]>
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cscb_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * cscb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cscb == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CSCB, cscb, strlen(cscb), NULL, FALSE, 0);
}

/*
* Set sms cscb.
*
* AT+CSCB=<[<mode>[,<mids>[,<dcss>]]]>
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cscb_set(mbtk_ril_handle* handle, char * cscb)
{
    return mbtk_ds_sms_cscb_set(handle, MBTK_SIM_AUTO, cscb);
}

/*
* Set sms cnmi.
*
at+cnmi=1,2

OK
if sending fails:
+CMS ERROR: <err>
*/
mbtk_ril_err_enum mbtk_ds_sms_cnmi_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CNMI, NULL, 0, NULL, FALSE, 0);
}

/*
* Set sms cnmi.
*
at+cnmi=1,2

OK
if sending fails:
+CMS ERROR: <err>
*/
mbtk_ril_err_enum mbtk_sms_cnmi_set(mbtk_ril_handle* handle)
{
    return mbtk_ds_sms_cnmi_set(handle, MBTK_SIM_AUTO);
}


/*
* Set sms cmss.
*
+CMSS=<index>[,<da>[,<toda>]]

if sending successful:
+CMSS: <mr>
OK
if sending fails:
+CMS ERROR: <err>
*/
mbtk_ril_err_enum mbtk_ds_sms_cmss_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * cmss, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmss == NULL || resp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMSS, cmss, strlen(cmss), resp, FALSE, 0);
}

/*
* Set sms cmss.
*
+CMSS=<index>[,<da>[,<toda>]]

if sending successful:
+CMSS: <mr>
OK
if sending fails:
+CMS ERROR: <err>
*/
mbtk_ril_err_enum mbtk_sms_cmss_set(mbtk_ril_handle* handle, char * cmss, char *resp)
{
    return mbtk_ds_sms_cmss_set(handle, MBTK_SIM_AUTO, cmss, resp);
}

/*
* Return sms cmgf.
*/
mbtk_ril_err_enum mbtk_ds_sms_cpms_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * mem)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(mem == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CPMS, NULL, 0, mem, FALSE, 0);
}

/*
* Return sms cmgf.
*/
mbtk_ril_err_enum mbtk_sms_cpms_get(mbtk_ril_handle* handle, char *mem)
{
    return mbtk_ds_sms_cpms_get(handle, MBTK_SIM_AUTO, mem);
}

/*
* Set sms cpms.
*
* AT+CPMS=<mem1>[,<mem2>[,<mem3>]]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cpms_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char * mem, char* resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(mem == NULL || resp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CPMS, mem, strlen(mem), resp, FALSE, 0);
}

/*
* Set sms cpms.
*
* AT+CPMS=<mem1>[,<mem2>[,<mem3>]]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cpms_set(mbtk_ril_handle* handle, char * mem, char* resp)
{
    return mbtk_ds_sms_cpms_set(handle, MBTK_SIM_AUTO, mem, resp);
}


/*
* Set sms cm.
*
* +CMGR=<index>

if PDU mode (+CMGF=0) ��command successful:
+CMGR: <stat>,[<alpha>],<length><CR><LF><pdu>
OK
if text mode (+CMGF=1), command successful and SMS-DELIVER:
+CMGR:<stat>,<oa>,[<alpha>],<scts>[,<tooa>,<fo>,<pid>,<dcs
>, <sca>,<tosca>,<length>]<CR><LF><data>
OK
if text mode (+CMGF=1), command successful and SMS-SUBMIT:
+CMGR:
<stat>,<da>,[<alpha>][,<toda>,<fo>,<pid>,<dcs>,[<vp>],
<sca>,<tosca>,<length>]<CR><LF><data>
OK
otherwise:
+CMS ERROR: <err>
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_sms_cmgr_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int index, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(resp == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_SMS_CMGR, &index, sizeof(uint8), resp, FALSE, 0);
}

/*
* Set sms cm.
*
* +CMGR=<index>

if PDU mode (+CMGF=0) ��command successful:
+CMGR: <stat>,[<alpha>],<length><CR><LF><pdu>
OK
if text mode (+CMGF=1), command successful and SMS-DELIVER:
+CMGR:<stat>,<oa>,[<alpha>],<scts>[,<tooa>,<fo>,<pid>,<dcs
>, <sca>,<tosca>,<length>]<CR><LF><data>
OK
if text mode (+CMGF=1), command successful and SMS-SUBMIT:
+CMGR:
<stat>,<da>,[<alpha>][,<toda>,<fo>,<pid>,<dcs>,[<vp>],
<sca>,<tosca>,<length>]<CR><LF><data>
OK
otherwise:
+CMS ERROR: <err>
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgr_set(mbtk_ril_handle* handle, int index, char *resp)
{
    return mbtk_ds_sms_cmgr_set(handle, MBTK_SIM_AUTO, index, resp);
}


#if 0

/*
* Set sim power state.
* power:
* 0: Sim power off.
* 1: Sim power on.
*/
int mbtk_sim_power_set(int power)
{
    if(power != 0 && power != 1)
    {
        return -1;
    }

    //  /sys/devices/virtual/usim_event/usim0/send_event
    char cmd[100] = {0};
    sprintf(cmd, "echo %d > /sys/devices/virtual/usim_event/usim0/send_event", power ? 0 : 1);
    system(cmd);

    return 0;
}

/*
* System power.
* type:
* 0: Reboot system.
* 1: Poweroff system.
* 2: Halt system.
*/
int mbtk_system_reboot(int type)
{
    if(type != 0 && type != 1 && type != 2)
    {
        return -1;
    }

    switch(type)
    {
        case 0:
        {
            system("reboot");
            break;
        }
        case 1:
        {
            system("poweroff");
            break;
        }
        case 2:
        {
            system("halt");
            break;
        }
        default:
        {
            break;
        }
    }

    return 0;
}

/*
* Get time type.
*/
int mbtk_time_get(mbtk_info_handle_t* handle, int *time_type)
{
    uint8 state;
    if(handle == NULL || time_type == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_DEV_TIME_REQ, NULL, 0, &state) > 0)
    {
        LOG("Time type : %d", state);
        *time_type = state;
        return 0;
    }
    else
    {
        return handle->info_err;
    }
}

/*
*Get Absolute time
*"23/05/24 06:09:32 +32 00"
*/
int mbtk_get_abs_time(char *time_str, time_t *time_out)
{
    struct tm tm_;

    char *ptr = strstr(time_str + 10, " ");
    *ptr = '\0';

    LOGD("time : \"%s\"", time_str);
#if 1
    if(strptime(time_str, "%y/%m/%d %T", &tm_) == NULL)
    {
        LOGE("strptime() fail.");
        return -1;
    }
#else
    int year, month, day, hour, minute,second,time_zone;
    if(strstr(time_str, "+"))
    {
        sscanf(time_str, "%d/%d/%d %d:%d:%d +%d",&year,&month,&day,&hour,&minute,&second,&time_zone);
    }
    else if(strstr(time_str, "-"))
    {
        sscanf(time_str, "%d/%d/%d %d:%d:%d -%d",&year,&month,&day,&hour,&minute,&second,&time_zone);
    }
    else
    {
        LOGE("Time format error:%s", time_str);
        return -1;
    }

    // 1970+
    if(year < 70)   // 20xx
    {
        tm_.tm_year  = 2000 + year;
    }
    else     // 19xx
    {
        tm_.tm_year  = 1900 + year;
    }
    tm_.tm_mon   = month - 1;
    tm_.tm_mday  = day;
    tm_.tm_hour  = hour;
    tm_.tm_min   = minute;
    tm_.tm_sec   = second;
    tm_.tm_isdst = 0;
#endif

    time_t timeStamp = mktime(&tm_);
    LOGD("tm_.tm_year = %d,tm_.tm_mon = %d,tm_.tm_mday = %d,tm_.tm_hour = %d,tm_.tm_min = %d,tm_.tm_sec = %d,tm_.tm_isdst = %d",tm_.tm_year,tm_.tm_mon,tm_.tm_mday,tm_.tm_hour,tm_.tm_min,tm_.tm_sec,tm_.tm_isdst);
    LOGD("time = %ld,%x", timeStamp,timeStamp);
    *time_out = timeStamp;

    return 0;
}

/*
* Set time.
*
* time_type:
* 0: Cell time
* 1: NTP time
* 2: User time
* time_str: "YYYY-MM-DD HH:MM:SS"
*/
int mbtk_time_set(mbtk_info_handle_t* handle, mbtk_time_type_enum time_type, char* time_str)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    uint8 buffer[100] = {0};
    buffer[0] = (uint8)time_type;
    if(time_type == MBTK_TIME_TYPE_USER)
    {
        if(!str_empty(time_str))
        {
            memcpy(buffer + sizeof(uint8), time_str, strlen(time_str));
            return info_item_process(handle, MBTK_INFO_ID_DEV_TIME_REQ,
                                     buffer, sizeof(uint8) + strlen(time_str), NULL) ? handle->info_err : 0;
        }
        else
        {
            return -1;
        }
    }
    else
    {
        return info_item_process(handle, MBTK_INFO_ID_DEV_TIME_REQ,
                                 buffer, sizeof(uint8), NULL) ? handle->info_err : 0;
    }
}


#endif

/*
* call_start
*
*/
mbtk_ril_err_enum mbtk_ds_call_start(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, char* phone_number)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(phone_number == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_START, phone_number, strlen(phone_number)+1, NULL, FALSE, 0);   // strlen(phone_number)+1 is wrote by hq at 2025/08/26 for bug 2187
}

/*
* call_start
*
*/
mbtk_ril_err_enum mbtk_call_start(mbtk_ril_handle* handle, char* phone_number)
{


    return mbtk_ds_call_start(handle, MBTK_SIM_AUTO, phone_number);
}

/*
* Answer the phone call.
*
*/
mbtk_ril_err_enum mbtk_ds_call_answer(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_ANSWER, NULL, 0, NULL, FALSE, 0);
}

/*
* Answer the phone call.
*
*/
mbtk_ril_err_enum mbtk_call_answer(mbtk_ril_handle* handle)
{
    return mbtk_ds_call_answer(handle, MBTK_SIM_AUTO);
}


/*
* Hang up all call.
*
*/
mbtk_ril_err_enum mbtk_ds_call_hang(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_HANGUP, NULL, 0, NULL, FALSE, 0);
}

/*
* Hang up all call.
*
*/
mbtk_ril_err_enum mbtk_call_hang(mbtk_ril_handle* handle)
{
    return mbtk_ds_call_hang(handle, MBTK_SIM_AUTO);
}

/*
* Hang up a call.
*
*/
mbtk_ril_err_enum mbtk_ds_a_call_hang(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int phone_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_HANGUP_A, &phone_id, sizeof(uint8), NULL, FALSE, 0);
}

/*
* Hang up a call.
*
*/
mbtk_ril_err_enum mbtk_a_call_hang(mbtk_ril_handle* handle, int phone_id)
{
    return mbtk_ds_a_call_hang(handle, MBTK_SIM_AUTO, phone_id);
}

/*
* Hang up waiting or background call.
*
*/
mbtk_ril_err_enum mbtk_ds_waiting_or_background_call_hang(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_HANGUP_B, NULL, 0, NULL, FALSE, 0);
}

/*
* Hang up waiting or background call.
*
*/
mbtk_ril_err_enum mbtk_waiting_or_background_call_hang(mbtk_ril_handle* handle)
{
    return mbtk_ds_waiting_or_background_call_hang(handle, MBTK_SIM_AUTO);
}

/*
* Hang up foreground resume background call.
*
*/
mbtk_ril_err_enum mbtk_ds_foreground_resume_background_call_hang(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_HANGUP_C, NULL, 0, NULL, FALSE, 0);
}

/*
* Hang up foreground resume background call.
*
*/
mbtk_ril_err_enum mbtk_foreground_resume_background_call_hang(mbtk_ril_handle* handle)
{
    return mbtk_ds_foreground_resume_background_call_hang(handle, MBTK_SIM_AUTO);
}

/*
* Get current call phone number.
*/
mbtk_ril_err_enum mbtk_ds_call_reg_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_call_info_t *reg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(reg == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(reg, 0x00, sizeof(mbtk_call_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_WAITIN, NULL, 0, reg, FALSE, 0);
}

/*
* Get current call phone number.
*/
mbtk_ril_err_enum mbtk_call_reg_get(mbtk_ril_handle* handle, mbtk_call_info_t *reg)
{
    return mbtk_ds_call_reg_get(handle, MBTK_SIM_AUTO, reg);
}


/*
* Return mute state.
*/
mbtk_ril_err_enum mbtk_ds_mute_state_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int *mute_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(mute_state == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *mute_state = INIT_DEFAULT_0;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_MUTE, NULL, 0, mute_state, FALSE, 0);
}

/*
* Return mute state.
*/
mbtk_ril_err_enum mbtk_mute_state_get(mbtk_ril_handle* handle, int *mute_state)
{
    return mbtk_ds_mute_state_get(handle, MBTK_SIM_AUTO, mute_state);
}

/*
* Set mute state.
*
* mute_state:
* 0 : of mute.
* 1 : on mute.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_ds_mute_state_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int mute_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_MUTE, &mute_state, sizeof(uint8), NULL, FALSE, 0);
}

/*
* Set mute state.
*
* mute_state:
* 0 : of mute.
* 1 : on mute.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_mute_state_set(mbtk_ril_handle* handle, int mute_state)
{
    return mbtk_ds_mute_state_set(handle, MBTK_SIM_AUTO, mute_state);
}

/*
* Set DTMF character.
*
*/
mbtk_ril_err_enum mbtk_ds_dtmf_send(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, mbtk_call_dtmf_info_t *dtmf_character)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(dtmf_character == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_DTMF, dtmf_character, sizeof(mbtk_call_dtmf_info_t), NULL, FALSE, 0);
}

/*
* Set DTMF character.
*
*/
mbtk_ril_err_enum mbtk_dtmf_send(mbtk_ril_handle* handle, mbtk_call_dtmf_info_t *dtmf_character)
{
    return mbtk_ds_dtmf_send(handle, MBTK_SIM_AUTO, dtmf_character);
}

mbtk_ril_err_enum mbtk_ds_centric_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int centric)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if((0 != centric && 1 != centric) || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_CENTRIC, &centric, 1, NULL, FALSE, 0);
}

mbtk_ril_err_enum mbtk_centric_set(mbtk_ril_handle* handle, int centric)
{
    return mbtk_ds_centric_set(handle, MBTK_SIM_AUTO, centric);
}

mbtk_ril_err_enum mbtk_ds_centric_get(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id, int *centric)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(centric == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *centric = INIT_DEFAULT_0;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(sim_id, port_info->port, RIL_MSG_ID_CALL_CENTRIC, NULL, 0, centric, FALSE, 0);
}

mbtk_ril_err_enum mbtk_centric_get(mbtk_ril_handle* handle, int *centric)
{
    return mbtk_ds_centric_get(handle, MBTK_SIM_AUTO, centric);
}

/*
* Set msd item.
*
*/
mbtk_ril_err_enum mbtk_ecall_msd_item_set(mbtk_ril_handle* handle, const mbtk_ecall_msd_cfg_info_t *msd_cfg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(msd_cfg == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_MSDCFG, msd_cfg, sizeof(mbtk_ecall_msd_cfg_info_t), NULL, FALSE, 0);
}

/*
* Generate msd after msd item set.
*
*/
mbtk_ril_err_enum mbtk_ecall_msd_gen(mbtk_ril_handle* handle)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_MSDGEN, NULL, 0, NULL, FALSE, 0);
}

/*
* Set ecall msd.
*
*/
mbtk_ril_err_enum mbtk_ecall_msd_set(mbtk_ril_handle* handle, const void* msd)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    if(str_empty(msd))
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_MSD, msd, strlen(msd)+1, NULL, FALSE, 0); // strlen(msd)+1 is wrote by hq at 2025/08/26 for bug 2187
}

/*
* Get ecall msd.
*
*/
mbtk_ril_err_enum mbtk_ecall_msd_get(mbtk_ril_handle* handle, void* msd)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(msd == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_MSD, NULL, 0, msd, FALSE, 0);
}


/*
* Set ecall msd item.
*
*/
mbtk_ril_err_enum mbtk_ecall_push(mbtk_ril_handle* handle)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_PUSH, NULL, 0, NULL, FALSE, 0);
}

/*
* Set ecall only configs.
*
*/
mbtk_ril_err_enum mbtk_ecall_only_set(mbtk_ril_handle* handle, const mbtk_ecall_only_info_t* info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(info == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_ONLY, info, sizeof(mbtk_ecall_only_info_t), NULL, FALSE, 0);
}

/*
* Get ecall only configs.
*
*/
mbtk_ril_err_enum mbtk_ecall_only_get(mbtk_ril_handle* handle, mbtk_ecall_only_info_t* info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(info == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(info, 0x00, sizeof(mbtk_ecall_only_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_ONLY, NULL, 0, info, FALSE, 0);
}

/*
* Set ecall network reg.
*
*/
mbtk_ril_err_enum mbtk_ecall_reg_set(mbtk_ril_handle* handle, int reg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if((reg != 0 && reg != 1) || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_REG, &reg, 1, NULL, FALSE, 0);
}

/*
* Start ecall dial start.
*
*/
mbtk_ril_err_enum mbtk_ecall_dial_start(mbtk_ril_handle* handle, mbtk_ecall_dial_type_enum type)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_DIAL, &type, 1, NULL, FALSE, 0);
}

/*
* Get ecall dial state.
*
*/
mbtk_ril_err_enum mbtk_ecall_dial_state_get(mbtk_ril_handle* handle, mbtk_ecall_dial_type_enum* type)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(type == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *type = MBTK_ECALL_DIAL_TYPE_TEST;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    memset(type, 0, sizeof(mbtk_ecall_dial_type_enum));

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_DIAL, NULL, 0, type, FALSE, 0);
}

/*
* Set ecall mode.
*
*/
mbtk_ril_err_enum mbtk_ecall_mode_set(mbtk_ril_handle* handle, mbtk_ecall_mode_type_enum mode)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_MODE, &mode, 1, NULL, FALSE, 0);
}

/*
* Get ecall mode.
*
*/
mbtk_ril_err_enum mbtk_ecall_mode_get(mbtk_ril_handle* handle, mbtk_ecall_mode_type_enum *mode)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(mode == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    *mode = MBTK_ECALL_MODE_TYPE_EU;
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    memset(mode, 0, sizeof(mbtk_ecall_mode_type_enum));

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_MODE, NULL, 0, mode, FALSE, 0);
}

/*
* Set ecall configs.
*
*/
mbtk_ril_err_enum mbtk_ecall_cfg_set(mbtk_ril_handle* handle, const mbtk_ecall_cfg_info_t *cfg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cfg == NULL || cfg->type == 0 || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_CFG, cfg, sizeof(mbtk_ecall_cfg_info_t), NULL, FALSE, 0);
}

/*
* Get ecall configs.
*
*/
mbtk_ril_err_enum mbtk_ecall_cfg_get(mbtk_ril_handle* handle, mbtk_ecall_cfg_info_t* cfg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cfg == NULL || cfg->type == 0 || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(cfg, 0x00, sizeof(mbtk_ecall_cfg_info_t));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    uint32 type = cfg->type;

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_CFG, &type, sizeof(uint32), cfg, FALSE, 0);
}

/*
* Set ecall sms number.
*
*/
mbtk_ril_err_enum mbtk_ecall_sms_number_set(mbtk_ril_handle* handle, const void *number)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(str_empty(number) || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_SMS_NUM, number, strlen(number)+1, NULL, FALSE, 0); // strlen(num)+1 is wrote by hq at 2025/08/26 for bug 2187
}

/*
* Get ecall sms number.
*
*/
mbtk_ril_err_enum mbtk_ecall_sms_number_get(mbtk_ril_handle* handle, void *number)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(number == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_SMS_NUM, NULL, 0, number, FALSE, 0);
}

/*
* Set ecall mute speaker.
*
*/
mbtk_ril_err_enum mbtk_ecall_mute_spk_set(mbtk_ril_handle* handle, int mute)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if((mute != 0 && mute != 1) || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_MUTESPK, &mute, 1, NULL, FALSE, 0);
}

/*
* Set ecall dsp gain.
*
*/
mbtk_ril_err_enum mbtk_ecall_dsp_gain_set(mbtk_ril_handle* handle, const mbtk_ecall_gain_info_t *gain_info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(gain_info == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_ECALL_DSP_GAIN, gain_info, sizeof(mbtk_ecall_gain_info_t), NULL, FALSE, 0);
}

int mbtk_get_modem_version(mbtk_ril_handle* handle, void *modem_version)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(handle == NULL || modem_version == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_DEV_MD_VERSION_REQ, NULL, 0, modem_version,FALSE, 0);

}

/*
* Set dual sim switch.
*
*/
mbtk_ril_err_enum mbtk_sim_switch_set(mbtk_ril_handle* handle, mbtk_sim_type_enum sim_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sim_id != MBTK_SIM_1 && sim_id != MBTK_SIM_2)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_SIM_SWITCH, &sim_id, sizeof(mbtk_sim_type_enum), NULL, FALSE, 0);
}

/*
* Get dual sim switch.
*
*/
mbtk_ril_err_enum mbtk_sim_switch_get(mbtk_ril_handle* handle, mbtk_sim_type_enum *sim_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sim_id == NULL || handle == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    memset(sim_id, 0x00, sizeof(mbtk_sim_type_enum));
    ril_at_port_info_t *port_info = (ril_at_port_info_t*)handle;
    if(!at_port_check(port_info->port)) {
        return MBTK_RIL_ERR_PORT;
    }

    return ril_req_process(MBTK_SIM_1, port_info->port, RIL_MSG_ID_SIM_SWITCH, NULL, 0, sim_id, FALSE, 0);
}


#if 0
/*
* Set wakeup state.
*
* wakeup_state:(0~31)
* 0 : means resume all
* 1~31 means suspend
* Control the active reporting of some platform modems to reduce wakeup
*/

int mbtk_wakeup_state_set(mbtk_info_handle_t* handle, uint32 wakeup_state)
{
    return info_item_process(handle, MBTK_INFO_ID_WAKEUP_STA_REQ, (uint32*)&wakeup_state, sizeof(uint32), NULL) ? handle->info_err : 0;
}

/*
* oos get.
*/
int mbtk_oos_get(mbtk_info_handle_t* handle, mbtk_oos_info *oos_info)
{
    if(info_item_process(handle, MBTK_INFO_ID_OOS_STA_REQ, NULL, 0, oos_info) > 0)
    {
        return 0;
    }
    else
    {
        return handle->info_err;
    }
}

/*
* oos set .
*/
int mbtk_oos_set(mbtk_info_handle_t* handle, mbtk_oos_info *oos_info)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }

    return info_item_process(handle, MBTK_INFO_ID_OOS_STA_REQ, oos_info, sizeof(mbtk_oos_info), NULL) ? handle->info_err : 0;
}


#endif

/*
* Set ril server state change callback function.
*/
mbtk_ril_err_enum mbtk_ril_ser_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(MBTK_SIM_1, ATPORTTYPE_0, RIL_MSG_ID_IND_SER_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_SER_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set net reg state change callback function.
*/
mbtk_ril_err_enum mbtk_ds_net_reg_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_NET_REG_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_NET_REG_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set net reg state change callback function.
*/
mbtk_ril_err_enum mbtk_net_reg_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_net_reg_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}

/*
* Set call state change callback function.
*/
mbtk_ril_err_enum mbtk_ds_call_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_CALL_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_CALL_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set call state change callback function.
*/
mbtk_ril_err_enum mbtk_call_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_call_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}

/*
* Set sms state change callback function.
*/
mbtk_ril_err_enum mbtk_ds_sms_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_SMS_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_SMS_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set sms state change callback function.
*/
mbtk_ril_err_enum mbtk_sms_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_sms_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}

/*
* Set radio state change callback function.
*/
mbtk_ril_err_enum mbtk_ds_radio_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_RADIO_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_RADIO_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set radio state change callback function.
*/
mbtk_ril_err_enum mbtk_radio_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_radio_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}

/*
* Set sim state change callback function.
*/
mbtk_ril_err_enum mbtk_ds_sim_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_SIM_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_SIM_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set sim state change callback function.
*/
mbtk_ril_err_enum mbtk_sim_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_sim_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}

/*
* Set pdp state change callback function.
*/
mbtk_ril_err_enum mbtk_ds_pdp_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_PDP_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_PDP_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set pdp state change callback function.
*/
mbtk_ril_err_enum mbtk_pdp_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_pdp_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}

/*
* Set signal state change callback function.
*/
mbtk_ril_err_enum mbtk_ds_signal_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_SIGNAL_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_SIGNAL_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

/*
* Set signal state change callback function.
*/
mbtk_ril_err_enum mbtk_signal_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_signal_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}

mbtk_ril_err_enum mbtk_ds_ecall_state_change_cb_reg(mbtk_sim_type_enum sim_id, mbtk_ril_callback_func cb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!at_port_check(ATPORTTYPE_0)) {
        return MBTK_RIL_ERR_PORT;
    }

    int ret = ril_req_process(sim_id, ATPORTTYPE_0, RIL_MSG_ID_IND_ECALL_STATE_CHANGE, NULL, 0, NULL, FALSE, -1);
    if(MBTK_RIL_ERR_SUCCESS == ret)
    {
        ril_cli.cb[RIL_MSG_ID_IND_ECALL_STATE_CHANGE - RIL_MSG_ID_IND_BEGIN - 1] = cb;
    }
    return ret;
}

mbtk_ril_err_enum mbtk_ecall_state_change_cb_reg(mbtk_ril_callback_func cb)
{
    return mbtk_ds_ecall_state_change_cb_reg(MBTK_SIM_AUTO, cb);
}


