/*
*    mbtk_rtp.c
*
*    MBTK RTP(VOIP) API source. This source depend on mbtk_rtpd server.
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN        WHO       WHAT,WHERE,WHY
--------    --------    -------------------------------------------------------
2024/12/2     LiuBin      Initial version

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <sys/epoll.h>
#include <time.h>
#include <arpa/inet.h>

#include "mbtk_rtp_internal.h"
#include "mbtk_log.h"
#include "mbtk_str.h"

static int rtp_cli_fd = -1;

#define RTP_INIT_CHECK() \
   if(rtp_cli_fd < 0) { \
        LOGW("RTP client not inited."); \
        if(mbtk_rtp_init()!=0) \
        { \
            LOGE("RTP init failed"); \
            return -2; \
        } \
    } 

static char* rtp_cmd_exec(const char* cmd, char *rsp, int rsp_len)
{
    int len = write(rtp_cli_fd, cmd, strlen(cmd));
    if(len != strlen(cmd)) {
        LOGE("write() fail(%d) : %d/%d", errno, len, strlen(cmd));
        return NULL;
    }
    LOGD("Write cmd[%s] success.", cmd);

    memset(rsp, 0, rsp_len);
    len = read(rtp_cli_fd, rsp, rsp_len);
    if(len <= 0) {
        LOGE("read() fail(%d) : len = %d", errno, len);
        return NULL;
    }
    LOGD("Read rsp[%s] success.", rsp);

    if(rsp[0] == MBTK_IND_START_FLAG && rsp[len - 1] == MBTK_IND_END_FLAG) {
        rsp[len - 1] = '\0';
        return rsp + 1;
    } else {
        log_hex("RSP_ERR", rsp, len);
        return NULL;
    }
}

/*hq modify at 20250725 begin*/
int mbtk_rtp_connect()
{
    struct sockaddr_un cli_addr;
    memset(&cli_addr, 0, sizeof(cli_addr));
    cli_addr.sun_family = AF_LOCAL;
    strcpy(cli_addr.sun_path, RTP_IPC_SOCK_PATH);

    int cnt=0;
    while(cnt<40) // max 20 s
    {
        if(connect(rtp_cli_fd, (struct sockaddr *)&cli_addr, sizeof(cli_addr)))
        {
            LOGE("connect() cnt %d fail[%d].",cnt,errno);           
        }
        else
        {    
            return 0;
        }
        usleep(500 * 1000);//500ms
        cnt++;
    }
    return -1;
}
/*hq modify at 20250725 end*/


int mbtk_rtp_init()
{
    if(rtp_cli_fd > 0) {
        LOGW("RTP client has inited.");
        return 0;
    }

    rtp_cli_fd = socket(AF_LOCAL, SOCK_STREAM, 0);
    if(rtp_cli_fd < 0)
    {
        LOGE("socket() fail[%d].", errno);
        goto error;
    }
    
    if(mbtk_rtp_connect())  // hq modify at 20250725
    {
        goto error;
    }

    return 0;
error:
    if(rtp_cli_fd > 0) {
        close(rtp_cli_fd);
        rtp_cli_fd = -1;
    }

    return -1;
}

int mbtk_rtp_deinit()
{
    if(rtp_cli_fd < 0) {
        LOGW("RTP client not inited.");
        return -1;
    }

    close(rtp_cli_fd);
    rtp_cli_fd = -1;

    return 0;
}

int mbtk_rtp_enable(bool enable)
{
    RTP_INIT_CHECK();

    // gnss_init:x
    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "rtp_mode %d", enable ? 1 : 0); // rtp_mode <0/1>
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // rtp_mode:<err>
    if(strcmp(result, "rtp_mode:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}

int mbtk_rtp_volume_set(int volume)
{
    RTP_INIT_CHECK();
       
    // gnss_init:x
    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "volume %d", volume); // volume <0-7>
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // volume:<err>
    if(strcmp(result, "volume:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}

int mbtk_rtp_remote_ip_set(const char *ipv4)
{
    uint32 IPAddr;
    if(str_empty(ipv4) || inet_pton(AF_INET, ipv4, &IPAddr) < 0) {
        LOGE("inet_pton() fail.");
        return -1;
    }
  
    RTP_INIT_CHECK();

    // gnss_init:x
    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "remote_ip %s", ipv4); // remote_ip xxx.xxx.xxx.xxx
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // rtp_mode:<err>
    if(strcmp(result, "remote_ip:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}

int mbtk_rtp_vlan_set(const char *vlan)
{
    if(str_empty(vlan)) {
        LOGE("vlan is empty.");
        return -1;
    }

    RTP_INIT_CHECK();

    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "vlan %s", vlan); // vlan <dev>
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // vlan:<err>
    if(strcmp(result, "vlan:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}


int mbtk_rtp_server_port_set(int port)
{
    RTP_INIT_CHECK();

#if 0
    if(53248 != port) {
        LOGE("Only support 53248 port[For GXX].");
        return -1;
    }
#endif

    // gnss_init:x
    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "server_port %d", port); // server_port <port>
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // volume:<err>
    if(strcmp(result, "server_port:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}

int mbtk_rtp_client_port_set(int port)
{
    RTP_INIT_CHECK();

    // gnss_init:x
    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "client_port %d", port); // client_port <port>
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // volume:<err>
    if(strcmp(result, "client_port:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}

int mbtk_rtp_sample_rate_set(int sample_rate)
{
    if(sample_rate != 8000 && sample_rate != 16000) {
        LOGE("Only support 8000/16000.");
        return -1;
    }

    RTP_INIT_CHECK();

    // gnss_init:x
    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "sample_rate %d", sample_rate); // sample_rate <sample_rate>
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // volume:<err>
    if(strcmp(result, "sample_rate:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}

int mbtk_rtp_channel_set(int channel)
{
    if(channel != 1) {
        LOGE("Only support 1 channel.");
        return -1;
    }

    RTP_INIT_CHECK();

    // gnss_init:x
    char cmd[100] = {0};
    char rsp[100] = {0};
    snprintf(cmd, sizeof(cmd), "channel %d", channel); // channel <channel>
    char *result = rtp_cmd_exec(cmd, rsp, sizeof(rsp));
    if(!result) {
        return -1;
    }

    // volume:<err>
    if(strcmp(result, "channel:0") == 0) {
        return 0;
    } else {
        LOGE("CMD exec error:%s", result);
        return -1;
    }
}


