#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>

#include "mbtk_log.h"
#include "mbtk_utils.h"
#include "ql/ql_gpio.h"

typedef struct {
    int pin;
    int gpio;
} pin_gpio_mmap_s;

#ifdef MBTK_PROJECT_L508
static pin_gpio_mmap_s pin_gpio_mmap[] = {
    {PINNAME_GPIO1, -1},            /*PIN-1*/
    {PINNAME_GPIO2, -1},             /*PIN-2*/
    {PINNAME_GPIO3, -1},             /*PIN-3*/
    {PINNAME_GPIO4, -1},             /*PIN-4*/
    {PINNAME_GPIO5, -1},             /*PIN-5*/
    {PINNAME_NET_STATUS, -1},        /*PIN-6*/
    {PINNAME_DBG_RXD, -1},          /*PIN-11*/
    {PINNAME_DBG_TXD, -1},          /*PIN-12*/
    {PINNAME_USIM_PRESENCE, 19},    /*PIN-13*/
    {PINNAME_SD_INT_DET, -1},       /*PIN-23*/
    {PINNAME_PCM_IN, 28},           /*PIN-24*/
    {PINNAME_PCM_OUT, 27},          /*PIN-25*/
    {PINNAME_PCM_SYNC, -1},         /*PIN-26*/
    {PINNAME_PCM_CLK, -1},          /*PIN-27*/
    {PINNAME_SDC2_DATA3, 21},       /*PIN-28*/
    {PINNAME_SDC2_DATA2, -1},       /*PIN-29*/
    {PINNAME_SDC2_DATA1, 4},       /*PIN-30*/
    {PINNAME_SDC2_DATA0, -1},       /*PIN-31*/
    {PINNAME_SDC2_CLK, -1},         /*PIN-32*/
    {PINNAME_SDC2_CMD, -1},         /*PIN-33*/
    {PINNAME_SPI_CS_N, -1},         /*PIN-37*/
    {PINNAME_SPI_MOSI, -1},         /*PIN-38*/
    {PINNAME_SPI_MISO, -1},         /*PIN-39*/
    {PINNAME_SPI_CLK, -1},          /*PIN-40*/
    {PINNAME_I2C_SCL, -1},          /*PIN-41*/
    {PINNAME_I2C_SDA, -1},          /*PIN-42*/
    {PINNAME_GPIO20, 20},           /*PIN-45*/
    {PINNAME_STATUS, 13},           /*PIN-49*/
    {PINNAME_NETLIGHT, 14},         /*PIN-51*/
    {PINNAME_GPIO6, -1},            /*PIN-62*/
    {PINNAME_DCD, -1},              /*PIN-63*/
    {PINNAME_DTR, -1},              /*PIN-66*/
    {PINNAME_MAIN_CTS, -1},         /*PIN-64*/
    {PINNAME_MAIN_RTS, -1},         /*PIN-65*/
    {PINNAME_MAIN_TXD, -1},         /*PIN-67*/
    {PINNAME_MAIN_RXD, -1},         /*PIN-68*/
    {PINNAME_RMII_RXD1, -1},        /*PIN-73*/
    {PINNAME_RMII_RXCL, -1},        /*PIN-74*/
    {PINNAME_RMII_CLK, 3},         /*PIN-75*/
    {PINNAME_RMII_RXD0, 1},        /*PIN-76*/
    {PINNAME_RMII_TXD0, -1},        /*PIN-77*/
    {PINNAME_RMII_TXD1, -1},        /*PIN-78*/
    {PINNAME_RMII_RXD2, -1},        /*PIN-79*/
    {PINNAME_RMII_TXD2, -1},        /*PIN-80*/
    {PINNAME_RMII_TX_CTRL, -1},     /*PIN-81*/
    {PINNAME_RMII_RXD3, -1},        /*PIN-82*/
    {PINNAME_RMII_TXCL, -1},        /*PIN-83*/
    {PINNAME_RMII_TXD3, 5},        /*PIN-84*/
    {PINNAME_WLAN_SLP_CLK, -1},    /*PIN-118*/
    {PINNAME_RMII_RST, 20},        /*PIN-119*/
    {PINNAME_RMII_INT, -1},        /*PIN-120*/
    {PINNAME_RMII_MDIO, 17},       /*PIN-121*/
    {PINNAME_RMII_MDC, 16},        /*PIN-122*/
    {PINNAME_PRI_TDI, 117},        /*PIN-123*/
    {PINNAME_WLAN_PER_EN, 24},     /*PIN-127*/
    {PINNAME_WLAN_WAKE, 21},       /*PIN-135*/
    {PINNAME_WLAN_EN, 22},         /*PIN-136*/
    {PINNAME_GPIO8, -1},            /*PIN-139*/
};
#else
static pin_gpio_mmap_s pin_gpio_mmap[] = {
    {PINNAME_GPIO1, -1},            /*PIN-1*/
    {PINNAME_GPIO2, -1},             /*PIN-2*/
    {PINNAME_GPIO3, -1},             /*PIN-3*/
    {PINNAME_GPIO4, -1},             /*PIN-4*/
    {PINNAME_GPIO5, -1},             /*PIN-5*/
    {PINNAME_NET_STATUS, -1},        /*PIN-6*/
    {PINNAME_DBG_RXD, -1},          /*PIN-11*/
    {PINNAME_DBG_TXD, -1},          /*PIN-12*/
    {PINNAME_USIM_PRESENCE, 19},    /*PIN-13*/
    {PINNAME_SD_INT_DET, -1},       /*PIN-23*/
    {PINNAME_PCM_IN, 28},           /*PIN-24*/
    {PINNAME_PCM_OUT, 27},          /*PIN-25*/
    {PINNAME_PCM_SYNC, -1},         /*PIN-26*/
    {PINNAME_PCM_CLK, -1},          /*PIN-27*/
    {PINNAME_SDC2_DATA3, 21},       /*PIN-28*/
    {PINNAME_SDC2_DATA2, -1},       /*PIN-29*/
    {PINNAME_SDC2_DATA1, 4},       /*PIN-30*/
    {PINNAME_SDC2_DATA0, -1},       /*PIN-31*/
    {PINNAME_SDC2_CLK, -1},         /*PIN-32*/
    {PINNAME_SDC2_CMD, -1},         /*PIN-33*/
    {PINNAME_SPI_CS_N, -1},         /*PIN-37*/
    {PINNAME_SPI_MOSI, -1},         /*PIN-38*/
    {PINNAME_SPI_MISO, -1},         /*PIN-39*/
    {PINNAME_SPI_CLK, -1},          /*PIN-40*/
    {PINNAME_I2C_SCL, -1},          /*PIN-41*/
    {PINNAME_I2C_SDA, -1},          /*PIN-42*/
    {PINNAME_GPIO20, 20},           /*PIN-45*/
    {PINNAME_STATUS, 13},           /*PIN-49*/
    {PINNAME_NETLIGHT, 14},         /*PIN-51*/
    {PINNAME_GPIO6, -1},            /*PIN-62*/
    {PINNAME_DCD, -1},              /*PIN-63*/
    {PINNAME_DTR, -1},              /*PIN-66*/
    {PINNAME_MAIN_CTS, -1},         /*PIN-64*/
    {PINNAME_MAIN_RTS, -1},         /*PIN-65*/
    {PINNAME_MAIN_TXD, -1},         /*PIN-67*/
    {PINNAME_MAIN_RXD, -1},         /*PIN-68*/
    {PINNAME_RMII_RXD1, -1},        /*PIN-73*/
    {PINNAME_RMII_RXCL, -1},        /*PIN-74*/
    {PINNAME_RMII_CLK, 3},         /*PIN-75*/
    {PINNAME_RMII_RXD0, 1},        /*PIN-76*/
    {PINNAME_RMII_TXD0, -1},        /*PIN-77*/
    {PINNAME_RMII_TXD1, -1},        /*PIN-78*/
    {PINNAME_RMII_RXD2, -1},        /*PIN-79*/
    {PINNAME_RMII_TXD2, -1},        /*PIN-80*/
    {PINNAME_RMII_TX_CTRL, -1},     /*PIN-81*/
    {PINNAME_RMII_RXD3, -1},        /*PIN-82*/
    {PINNAME_RMII_TXCL, -1},        /*PIN-83*/
    {PINNAME_RMII_TXD3, 5},        /*PIN-84*/
    {PINNAME_WLAN_SLP_CLK, -1},    /*PIN-118*/
    {PINNAME_RMII_RST, 20},        /*PIN-119*/
    {PINNAME_RMII_INT, -1},        /*PIN-120*/
    {PINNAME_RMII_MDIO, 17},       /*PIN-121*/
    {PINNAME_RMII_MDC, 16},        /*PIN-122*/
    {PINNAME_PRI_TDI, 117},        /*PIN-123*/
    {PINNAME_WLAN_PER_EN, 24},     /*PIN-127*/
    {PINNAME_WLAN_WAKE, 21},       /*PIN-135*/
    {PINNAME_WLAN_EN, 22},         /*PIN-136*/
    {PINNAME_GPIO8, -1},            /*PIN-139*/
};

#endif

static int gpio_export(int gpio)
{
    int index=0;
    int file=-1;
    int result =-1;
    char pin_index_buffer[5]= {0};

    char buffer[50];
    memset(buffer,0,50);
    sprintf(buffer,"/sys/class/gpio/gpio%d/direction", gpio);
    if(access(buffer , F_OK) == 0)
    {
        LOGD("%d has export.", gpio);
        return 0;
    }

    file = open("/sys/class/gpio/export",O_WRONLY);
    if(file == -1)
    {
        LOGE("Open gpio export file fail.");
        return -1;
    }

    memset(pin_index_buffer,0,5);
    sprintf(pin_index_buffer,"%d", gpio);
    result = write(file,pin_index_buffer,strlen(pin_index_buffer));
    if(result < 0)
    {
        LOGE("Gpio[%d] export fail.", gpio);
        close(file);
        return -1;
    }
    close(file);

    return 0;
}

static int gpio_unexport(int gpio)
{
    int index=0;
    int file=-1;
    int result =-1;
    char pin_index_buffer[5]= {0};
    char buffer[50];
    memset(buffer,0,50);
    sprintf(buffer,"/sys/class/gpio/gpio%d/direction", gpio);
    if(access(buffer , F_OK) == -1)
    {
        LOGD("%d not export.", gpio);
        return 0;
    }

    file = open("/sys/class/gpio/unexport",O_WRONLY);
    if(file == -1)
    {
        LOGE("Open gpio unexport file fail.");
        return -1;
    }

    memset(pin_index_buffer,0,5);
    sprintf(pin_index_buffer,"%d", gpio);
    result=write(file,pin_index_buffer,strlen(pin_index_buffer));
    if(result < 0)
    {
        close(file);
        LOGE("Gpio[%d] unexport fail.", gpio);
        return -1;
    }
    close(file);

    return 0;
}

static int gpio_direct_get(int gpio, char *value, int value_size)
{
    char buffer[50]= {0};
    int file =-1;
    int result =-1;

    memset(buffer,0,50);
    sprintf(buffer,"/sys/class/gpio/gpio%d/direction", gpio);
    file = open(buffer, O_RDONLY);
    if(file == -1)
    {
        LOGE("Open gpio[%d] direct fail.", gpio);
        return -1;
    }

    memset(value, 0x0, value_size);
    result = read(file,value,value_size);
    if(result <= 0)
    {
        LOGE("Get gpio[%d] direct fail.", gpio);
        close(file);
        return -1;
    }
    close(file);

    return 0;
}


static int gpio_direct_set(int gpio, char *value)
{
    char buffer[50]= {0};
    int file =-1;
    int result =-1;

    memset(buffer,0,50);
    sprintf(buffer,"/sys/class/gpio/gpio%d/direction", gpio);
    file = open(buffer, O_WRONLY);
    if(file == -1)
    {
        LOGE("Open gpio[%d] direct fail.", gpio);
        return -1;
    }

    result = write(file,value,strlen(value));
    if(result != strlen(value))
    {
        LOGE("Set gpio[%d] direct fail.", gpio);
        close(file);
        return -1;
    }
    close(file);

    return 0;
}

static int gpio_value_get(int gpio)
{
    char buffer[50];
    char path[10];
    int file =-1;
    int result =-1;
    int value;

    memset(path,0,50);
    memset(buffer,0,10);
    sprintf(path,"/sys/class/gpio/gpio%d/value", gpio);
    file = open(path,O_RDONLY);
    if(file == -1)
    {
        LOGE("Open gpio[%d] fail.", gpio);
        return -1;
    }
    result = read(file,buffer,5);
    if(result <= 0)
    {
        LOGE("Get gpio[%d] value fail", gpio);
        close(file);
        return -1;
    }
    close(file);
    value = atoi(buffer);
    return value;
}

static int gpio_value_set(int gpio, int value)
{
    char buffer[50]= {0};
    int file =-1;
    int result =-1;

    memset(buffer,0,50);
    sprintf(buffer,"/sys/class/gpio/gpio%d/value", gpio);
    file = open(buffer,O_WRONLY);
    if(file == -1)
    {
        LOGE("Open gpio[%d] value fail.", gpio);
        return -1;
    }
    if(value == 0) {
        result = write(file,"0",1);
    } else {
        result = write(file,"1",1);
    }
    if(result != 1)
    {
        LOGE("Set gpio[%d] value fail.", gpio);
        close(file);
        return -1;
    }
    close(file);

    return 0;
}

static int pin_2_gpio(Enum_PinName       pin_name)
{
#if 0
    switch(pin_name){
        case PINNAME_USIM_PRESENCE:    /*PIN-13*/
            return 19;
        case PINNAME_PCM_IN:           /*PIN-24*/
            return 28;
        case PINNAME_PCM_OUT:          /*PIN-25*/
            return 27;
        case PINNAME_RMII_CLK:         /*PIN-75*/
            return 3;
        case PINNAME_RMII_RXD0:        /*PIN-76*/
            return 1;
        case PINNAME_RMII_RST:          /*PIN-119*/
            return 20;
        case PINNAME_RMII_MDIO:         /*PIN-121*/
            return 17;
        case PINNAME_RMII_MDC:          /*PIN-122*/
            return 16;
        case PINNAME_WLAN_PER_EN:       /*PIN-127*/
            return 24;
        case PINNAME_WLAN_WAKE:         /*PIN-135*/
            return 21;
        case PINNAME_WLAN_EN:           /*PIN-136*/
            return 22;

        // Unknown PIN.
        case PINNAME_GPIO1:             /*PIN-1*/
        case PINNAME_GPIO2:             /*PIN-2*/
        case PINNAME_GPIO3:             /*PIN-3*/
        case PINNAME_GPIO4:             /*PIN-4*/
        case PINNAME_GPIO5:             /*PIN-5*/
        case PINNAME_NET_STATUS:        /*PIN-6*/
        case PINNAME_DBG_RXD:           /*PIN-11*/
        case PINNAME_DBG_TXD:          /*PIN-12*/
        case PINNAME_SD_INT_DET:       /*PIN-23*/
        case PINNAME_PCM_SYNC:         /*PIN-26*/
        case PINNAME_PCM_CLK:          /*PIN-27*/
        case PINNAME_SDC2_DATA3:       /*PIN-28*/
        case PINNAME_SDC2_DATA2:       /*PIN-29*/
        case PINNAME_SDC2_DATA1:       /*PIN-30*/
        case PINNAME_SDC2_DATA0:       /*PIN-31*/
        case PINNAME_SDC2_CLK:         /*PIN-32*/
        case PINNAME_SDC2_CMD:         /*PIN-33*/
        case PINNAME_SPI_CS_N:         /*PIN-37*/
        case PINNAME_SPI_MOSI:         /*PIN-38*/
        case PINNAME_SPI_MISO:         /*PIN-39*/
        case PINNAME_SPI_CLK:          /*PIN-40*/
        case PINNAME_I2C_SCL:          /*PIN-41*/
        case PINNAME_I2C_SDA:          /*PIN-42*/
        case PINNAME_GPIO6:            /*PIN-62*/
        case PINNAME_DCD:              /*PIN-63*/
        case PINNAME_DTR:              /*PIN-66*/
        case PINNAME_MAIN_CTS:         /*PIN-64*/
        case PINNAME_MAIN_RTS:         /*PIN-65*/
        case PINNAME_MAIN_TXD:         /*PIN-67*/
        case PINNAME_MAIN_RXD:         /*PIN-68*/
        case PINNAME_RMII_RXD1:        /*PIN-73*/
        case PINNAME_RMII_RXCL:        /*PIN-74*/
        case PINNAME_RMII_TXD0:        /*PIN-77*/
        case PINNAME_RMII_TXD1:        /*PIN-78*/
        case PINNAME_RMII_RXD2:        /*PIN-79*/
        case PINNAME_RMII_TXD2:        /*PIN-80*/
        case PINNAME_RMII_TX_CTRL:     /*PIN-81*/
        case PINNAME_RMII_RXD3:        /*PIN-82*/
        case PINNAME_RMII_TXCL:        /*PIN-83*/
        case PINNAME_RMII_TXD3:        /*PIN-84*/
        case PINNAME_WLAN_SLP_CLK:      /*PIN-118*/
        case PINNAME_RMII_INT:          /*PIN-120*/
        case PINNAME_GPIO8:             /*PIN-139*/
        default:
            LOGE("Unknown PIN : %d", pin_name);
            return -1;
    }
#else
    int i = 0;
    while(i < ARRAY_SIZE(pin_gpio_mmap)) {
        if(pin_name == pin_gpio_mmap[i].pin) {
            return pin_gpio_mmap[i].gpio;
        }
        i++;
    }

    LOGE("No found PIN : %d", pin_name);
    return -1;
#endif
}


/*****************************************************************
* Function:     Ql_GPIO_Init
*
* Description:
*               This function enables the GPIO function of the specified pin,
*               and initialize the configurations, including direction,
*               level and pull selection.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
*               dir:
*                   The initial direction of GPIO, one value of Enum_PinDirection.
*               level:
*                   The initial level of GPIO, one value of Enum_PinLevel.
*               pull_sel:
*                   Pull selection, one value of Enum_PinPullSel.
* Return:
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int Ql_GPIO_Init(Enum_PinName       pin_name,
                 Enum_PinDirection  dir,
                 Enum_PinLevel      level,
                 Enum_PinPullSel    pull_sel
                 )
{
    int gpio = pin_2_gpio(pin_name);
    if(gpio == -1) {
        return RES_IO_ERROR;
    }
    LOGD("PIN-%d => GPIO-%d", pin_name, gpio);

    if(gpio_export(gpio))
    {
        LOGE("gpio_export() fail.");
        return RES_IO_ERROR;
    }

    if(gpio_direct_set(gpio, dir == PINDIRECTION_IN ? "in" : "out"))
    {
        LOGE("gpio_direct_set() fail.");
        return RES_IO_ERROR;
    }

    if(dir == PINDIRECTION_OUT){
        if(gpio_value_set(gpio, level))
        {
            LOGE("gpio_value_set() fail.");
            return RES_IO_ERROR;
        }
    }
    // No support pull mode now.

    return RES_OK;
}

/*****************************************************************
* Function:     Ql_GPIO_Base_Init
*
* Description:
*               This function enables the GPIO function of the specified pin.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
*
* Return:
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*               RES_IO_ERR, the function failed
*****************************************************************/
int Ql_GPIO_Base_Init(Enum_PinName pin_name );

/*****************************************************************
* Function:     Ql_GPIO_SetLevel
*
* Description:
*               This function sets the level of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
*               level:
*                   The initial level of GPIO, one value of Enum_PinLevel.
* Return:
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int Ql_GPIO_SetLevel(Enum_PinName pin_name, Enum_PinLevel level)
{
    int gpio = pin_2_gpio(pin_name);
    if(gpio == -1) {
        return RES_IO_ERROR;
    }
    LOGD("PIN-%d => GPIO-%d", pin_name, gpio);

    if(gpio_value_set(gpio, level)) {
        LOGE("gpio_value_set() fail.");
        return RES_IO_ERROR;
    } else {
        return RES_OK;
    }
}

/*****************************************************************
* Function:     Ql_GPIO_GetLevel
*
* Description:
*               This function gets the level of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
* Return:
*               The level value of the specified GPIO, which is
*               nonnegative integer.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*****************************************************************/
int Ql_GPIO_GetLevel(Enum_PinName pin_name)
{
    int gpio = pin_2_gpio(pin_name);
    if(gpio == -1) {
        return RES_IO_ERROR;
    }
    LOGD("PIN-%d => GPIO-%d", pin_name, gpio);

    return gpio_value_get(gpio);
}

/*****************************************************************
* Function:     Ql_GPIO_SetDirection
*
* Description:
*               This function sets the direction of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
*               dir:
*                   The initial direction of GPIO, one value of Enum_PinDirection.
* Return:
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int Ql_GPIO_SetDirection(Enum_PinName pin_name, Enum_PinDirection dir)
{
    int gpio = pin_2_gpio(pin_name);
    if(gpio == -1) {
        return RES_IO_ERROR;
    }
    LOGD("PIN-%d => GPIO-%d", pin_name, gpio);

    if(gpio_direct_set(gpio, dir == PINDIRECTION_IN ? "in" : "out")) {
        LOGE("gpio_direct_set() fail.");
        return RES_IO_ERROR;
    } else {
        return RES_OK;
    }
}

/*****************************************************************
* Function:     Ql_GPIO_GetDirection
*
* Description:
*               This function gets the direction of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
* Return:
*               0  INPUT
*               1  OUTPUT
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int Ql_GPIO_GetDirection(Enum_PinName pin_name)
{
    char buff[10];
    int gpio = pin_2_gpio(pin_name);
    if(gpio == -1) {
        return RES_IO_ERROR;
    }
    LOGD("PIN-%d => GPIO-%d", pin_name, gpio);

    if(gpio_direct_get(gpio, buff, 10)) {
        LOGE("gpio_direct_get() fail.");
        return RES_IO_NOT_SUPPORT;
    } else {
        if(strncmp(buff, "in",2) == 0) {
            return PINDIRECTION_IN;
        } else if(strncmp(buff, "out",3) == 0) {
            return PINDIRECTION_OUT;
        } else {
            return RES_IO_NOT_SUPPORT;
        }
    }
}

/*****************************************************************
* Function:     Ql_GPIO_SetPullSelection
*
* Description:
*               This function sets the pull selection of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
*               Enum_PinPullSel:
*                   Pull selection, one value of Enum_PinPullSel.
* Return:
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int Ql_GPIO_SetPullSelection(Enum_PinName pin_name, Enum_PinPullSel pull_sel);

/*****************************************************************
* Function:     ql_gpio_get_pull_selection
*
* Description:
*               This function gets the pull selection of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
* Return:
*               0<<13   no pull
*               5<<13   pull down
*               6<<13   pull up
*****************************************************************/
int Ql_GPIO_GetPullSelection(Enum_PinName pin_name);


/*****************************************************************
* Function:     Ql_GPIO_Uninit
*
* Description:
*               This function releases the specified GPIO that was
*               initialized by calling Ql_GPIO_Init() previously.
*               After releasing, the GPIO can be used for other purpose.
* Parameters:
*               pin_name:
*                   Pin name, one value of Enum_PinName.
* Return:
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int Ql_GPIO_Uninit(Enum_PinName pin_name)
{
    int gpio = pin_2_gpio(pin_name);
    if(gpio == -1) {
        return RES_IO_ERROR;
    }
    LOGD("PIN-%d => GPIO-%d", pin_name, gpio);

    if(gpio_unexport(gpio))
    {
        LOGE("gpio_unexport() fail.");
        return RES_IO_ERROR;
    }

    return RES_OK;
}

//------------------------------------------------------------------------------
/*
* Function:     Ql_EINT_Enable
*
* Description:
*               Set the interrupt sense mode, and enable interrupt.
*
* Parameters:
*               eint_pin_name:
*                   EINT pin name, one value of Enum_PinName that has
*                   the interrupt function.
*
*               eint_type:
*                   Interrupt type, level-triggered or edge-triggered.
*                   Now, only edge-triggered interrupt is supported.
*
*               eint_callback:
*                   call back function
*
* Return:
*               RES_OK, this function succeeds.
*               else failed to execute the function.
*/
//------------------------------------------------------------------------------
int Ql_EINT_Enable(Enum_PinName eint_pin_name, Enum_EintType eint_type, Ql_EINT_Callback eint_callback);


//------------------------------------------------------------------------------
/*
* Function:     Ql_EINT_Disable
*
* Description:
*               Disable the interrupt sense.
*
* Parameters:
*               eint_pin_name:
*                   EINT pin name, one value of Enum_PinName that has
*                   the interrupt function.
*
* Return:
*               RES_OK, this function succeeds.
*               else failed to execute the function.
*/
//------------------------------------------------------------------------------
int Ql_EINT_Disable(Enum_PinName eint_pin_name);

