#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <pthread.h>
#include <sys/epoll.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>

#include "mbtk_ril.h"
#include "mbtk_list.h"
#include "mbtk_utils.h"
#include "mbtk_log.h"
#include "mbtk_str.h"

typedef struct {
    int cli_fd;
    bool ril_ready;
    int exit_fd[2];
    pthread_t read_thread_id;
    pthread_mutex_t send_mutex;
    list_node_t* msg_list;           // ril_msg_info_t
    list_node_t* cli_thread_list;     // ril_cli_thread_info_t
} ril_cli_info_t;

#define EPOLL_LISTEN_MAX 100

static ril_cli_info_t ril_cli;
static int pthread_id_index = 1;

static bool ril_cid_check(mbtk_ril_cid_enum cid)
{
    if(cid < MBTK_APN_CID_MIN || cid > MBTK_APN_CID_MAX)
        return FALSE;

    return TRUE;
}

static ril_cli_thread_info_t* thread_find_by_pid(pthread_t pid)
{
    ril_cli_thread_info_t* cli_thread = NULL;
    list_first(ril_cli.cli_thread_list);
    while ((cli_thread = (ril_cli_thread_info_t*)list_next(ril_cli.cli_thread_list)))
    {
        if(pid == cli_thread->pid) {
            break;
        }
    }

    if(!cli_thread) { // No found cli thread in list.
        cli_thread = (ril_cli_thread_info_t*)malloc(sizeof(ril_cli_thread_info_t));
        memset(cli_thread, 0, sizeof(ril_cli_thread_info_t));
        cli_thread->pid = pid;
        sprintf(cli_thread->name, "PID-%d", pthread_id_index++);
        pthread_mutex_init(&cli_thread->mutex, NULL);
        pthread_cond_init(&cli_thread->cond, NULL);
        cli_thread->is_waitting = FALSE;
        list_add(ril_cli.cli_thread_list, cli_thread);
    }

    return cli_thread;
}

static void ril_thread_wait(ril_cli_thread_info_t* cli_thread)
{
    if(!cli_thread->is_waitting) {
        LOGD("Thread(%s) waitting...", cli_thread->name);
        cli_thread->is_waitting = TRUE;
        pthread_mutex_lock(&cli_thread->mutex);
        pthread_cond_wait(&cli_thread->cond, &cli_thread->mutex);
        pthread_mutex_unlock(&cli_thread->mutex);
        cli_thread->is_waitting = FALSE;
        LOGD("Thread(%s) running...", cli_thread->name);
    } else {
        LOGD("RSP befor SEND, not waitting...");
        cli_thread->is_waitting = FALSE;
    }
}


static void ril_thread_continue(pthread_t pid)
{
    ril_cli_thread_info_t* cli_thread = NULL;
    list_first(ril_cli.cli_thread_list);
    while ((cli_thread = (ril_cli_thread_info_t*) list_next(ril_cli.cli_thread_list)))
    {
        if(pid == cli_thread->pid) {
            break;
        }
    }

    if(cli_thread) { // Found cli thread in list.
        if(cli_thread->is_waitting) {
            LOGD("Thread(%s) will continue...", cli_thread->name);
            pthread_mutex_lock(&cli_thread->mutex);
            pthread_cond_signal(&cli_thread->cond);
            pthread_mutex_unlock(&cli_thread->mutex);
        } else {
            LOGD("Thread(%s) not continue...", cli_thread->name);
            cli_thread->is_waitting = TRUE;
        }
    } else {
        LOGE("No found cli thread[Cannot occur].");
    }
}

static int ril_ind_process(ril_msg_pack_info_t* pack)
{
    switch(pack->msg_id) {
        case RIL_MSG_ID_IND_SER_READY: // rild service ready.
        {
            ril_cli.ril_ready = TRUE;
            return 0;
        }
        case RIL_MSG_ID_IND_NET_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_CALL_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_SMS_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_RADIO_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_SIM_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_PDP_STATE_CHANGE:
        {
            return 0;
        }
        case RIL_MSG_ID_IND_SIGNAL_STATE_CHANGE:
        {
            return 0;
        }
        default:
        {
            LOGE("Unknown IND : %d", pack->msg_id);
            return -1;
        }
    }
}

static int ril_rsp_process(ril_msg_pack_info_t* pack)
{
    if(pack->msg_index == RIL_MSG_INDEX_INVALID) {
        LOGE("Invalid message index.");
        return -1;
    }

    pthread_mutex_lock(&ril_cli.send_mutex);
    ril_msg_info_t* msg = NULL;
    list_first(ril_cli.msg_list);
    while ((msg = (ril_msg_info_t*) list_next(ril_cli.msg_list)))
    {
        if(pack->msg_index == msg->pack->msg_index) {
            break;
        }
    }

    if(NULL == msg) { // No found message in msg list.
        LOGW("Unknown msg : Index - %d, Type - %s, ID - %s, Result - %s ,Length - %d",
            pack->msg_index, type2str(pack->msg_type), id2str(pack->msg_id), err2str(pack->err),
            pack->data_len);
        pthread_mutex_unlock(&ril_cli.send_mutex);
        return -1;
    } else {
        int result = 0;
        LOGD("Msg : Index - %d, Type - %s, ID - %s, Result - %s ,Length - %d", pack->msg_index,
            type2str(pack->msg_type),
            id2str(pack->msg_id),
            err2str(pack->err),
            pack->data_len);
        if(msg->pack->msg_id != pack->msg_id) {
            LOGE("msg id not match : %d , %d", msg->pack->msg_id ,pack->msg_id);
            result = -1;
            *(msg->rsp_err) = MBTK_RIL_ERR_FORMAT;
            goto rm_msg_from_list;
        }

        if(msg->rsp_data && pack->data && pack->data_len > 0) {
            memcpy(msg->rsp_data, pack->data, pack->data_len);
        }

        *(msg->rsp_data_len) = pack->data_len;
        *(msg->rsp_err) = pack->err;

        if(!msg->is_async) {
            ril_thread_continue(msg->pid);
        }

rm_msg_from_list:
        if(list_remove(ril_cli.msg_list, msg)) {
            ril_msg_pack_free(msg->pack);
            free(msg);
        }
        pthread_mutex_unlock(&ril_cli.send_mutex);
        return result;
    }
}

static int ril_pack_process(ril_msg_pack_info_t* pack)
{
    if(pack->msg_type == RIL_MSG_TYPE_IND) { // IND message.
        return ril_ind_process(pack);
    } else if(pack->msg_type == RIL_MSG_TYPE_RSP) {// Response message.
        return ril_rsp_process(pack);
    } else {
        LOGE("Pack must be RSP or IND.");
        return -1;
    }
}

static void* ril_read_run(void* arg)
{
    int epoll_fd = epoll_create(5);
    if(epoll_fd < 0)
    {
        LOGE("epoll_create() fail[%d].", errno);
        return NULL;
    }

    uint32 event = EPOLLIN | EPOLLET;
    struct epoll_event ev_cli, ev_exit;
    ev_cli.data.fd = ril_cli.cli_fd;
    ev_cli.events = event; //EPOLLIN | EPOLLERR | EPOLLET;
    epoll_ctl(epoll_fd,EPOLL_CTL_ADD, ril_cli.cli_fd, &ev_cli);

    ev_exit.data.fd = ril_cli.exit_fd[0];
    ev_exit.events = event; //EPOLLIN | EPOLLERR | EPOLLET;
    epoll_ctl(epoll_fd,EPOLL_CTL_ADD, ril_cli.exit_fd[0], &ev_exit);

    int nready = -1;
    struct epoll_event epoll_events[EPOLL_LISTEN_MAX];
    while(1)
    {
        nready = epoll_wait(epoll_fd, epoll_events, EPOLL_LISTEN_MAX, -1);
        if(nready > 0)
        {
            int i;
            for(i = 0; i < nready; i++)
            {
                LOGD("fd[%d] event = %x",epoll_events[i].data.fd, epoll_events[i].events);
                if(epoll_events[i].events & EPOLLHUP)   // Closed by server.
                {
                    LOGD("Closed by server.");
                    goto read_thread_exit;
                }
                else if(epoll_events[i].events & EPOLLIN)
                {
                    if(ril_cli.cli_fd == epoll_events[i].data.fd)  // Server data arrive.
                    {
                        // Read and process every message.
                        mbtk_ril_err_enum err = MBTK_RIL_ERR_SUCCESS;
                        ril_msg_pack_info_t **pack = ril_pack_recv(ril_cli.cli_fd, false, &err);

                        // Parse packet error,send error response to client.
                        if(pack == NULL)
                        {
                            if(err != MBTK_RIL_ERR_SUCCESS)
                            {
                                LOGE("RSP packet error[%s].", err2str(err));
                            }
                        }
                        else
                        {
                            ril_msg_pack_info_t** pack_ptr = pack;
                            while(*pack_ptr)
                            {
                                ril_pack_process(*pack_ptr);
                                ril_msg_pack_free(*pack_ptr);
                                pack_ptr++;
                            }
                            free(pack);
                        }
                    }
                    else if(ril_cli.exit_fd[0] == epoll_events[i].data.fd) //
                    {
                        char buff[100] = {0};
                        int len = read(ril_cli.exit_fd[0], buff, 100);
                        if(len > 0)
                        {
                            LOGI("CMD : %s", buff);
                            if(strcmp(buff, "EXIT") == 0)
                            {
                                goto read_thread_exit;
                            }
                            else
                            {
                                LOGE("Unkonw cmd : %s", buff);
                            }
                        }
                        else
                        {
                            LOGE("sock_read() fail.");
                        }
                    }
                    else
                    {
                        LOGE("Unknown socket : %d", epoll_events[i].data.fd);
                    }
                }
                else
                {
                    LOGE("Unknown event : %x", epoll_events[i].events);
                }
            }
        }
        else
        {
            LOGE("epoll_wait() fail[%d].", errno);
        }
    }

read_thread_exit:
    LOGD("info_read thread exit.");
    return NULL;
}

/*
* Return recv data length.
* -1 : fail.
*/
static mbtk_ril_err_enum ril_req_process(ril_msg_id_enum  id,
                             const void         *req,
                             int                req_len,
                             void               *rsp,
                             bool is_async)
{
    ril_msg_pack_info_t* pack = ril_msg_pack_creat(RIL_MSG_TYPE_REQ, id, RIL_MSG_INDEX_AUTO, req, req_len);
    if(pack == NULL)
    {
        return MBTK_RIL_ERR_MEMORY;
    }

    ril_msg_info_t *msg_info = (ril_msg_info_t*)malloc(sizeof(ril_msg_info_t));
    if(msg_info == NULL)
    {
        ril_msg_pack_free(pack);
        return MBTK_RIL_ERR_MEMORY;
    }

    int rsp_data_len = 0;
    int rsp_err = MBTK_RIL_ERR_SUCCESS;

    pthread_mutex_lock(&ril_cli.send_mutex);
    // Add to msg list.
    msg_info->pid = pthread_self();
    msg_info->pack = pack;
    msg_info->rsp_data_len = &rsp_data_len;
    msg_info->rsp_err = &rsp_err;
    if(!is_async) { // Waitting for response for sync request.
        msg_info->rsp_data = rsp;   // Save response data.
    } else {
        msg_info->rsp_data = NULL;
    }
    msg_info->is_async = is_async;
    list_add(ril_cli.msg_list, msg_info);
    pthread_mutex_unlock(&ril_cli.send_mutex);

    ril_cli_thread_info_t* cli_thread = thread_find_by_pid(msg_info->pid);
    // Send pack to server success.
    if(ril_pack_send(ril_cli.cli_fd, pack) >= RIL_SOCK_PACK_LEN_MIN) {
        if(!is_async) { // Waitting for response for sync request.
            ril_thread_wait(cli_thread);
        }

        LOGD("REQ(%s) success.", id2str(id));
        return rsp_err;
    } else {
        if(list_remove(ril_cli.msg_list, msg_info)) {
            ril_msg_pack_free(msg_info->pack);
            free(msg_info);
        }
        return MBTK_RIL_ERR_SEND_PACK;
    }
}

static void msg_list_free_cb(void *data)
{
    if(data) {
        ril_msg_info_t *msg = (ril_msg_info_t*)data;
        if(msg->pack) {
            free(msg->pack);
        }
        free(msg);
    }
}

static void cli_thread_list_free_cb(void *data)
{
    if(data) {
        ril_cli_thread_info_t *cli_thread = (ril_cli_thread_info_t*)data;
        pthread_mutex_destroy(&cli_thread->mutex);
        pthread_cond_destroy(&cli_thread->cond);
        free(cli_thread);
    }
}

void mbtk_ril_lib_info_print()
{
    MBTK_SOURCE_INFO_PRINT("mbtk_ril_lib");
}

mbtk_ril_err_enum mbtk_ril_init()
{
    if(ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_INITED;
    }

    ril_cli.cli_fd = socket(AF_LOCAL, SOCK_STREAM, 0);
    if(ril_cli.cli_fd < 0)
    {
        LOGE("socket() fail[%d].", errno);
        return MBTK_RIL_ERR_UNKNOWN;
    }

    // Set O_NONBLOCK
    int flags = fcntl(ril_cli.cli_fd, F_GETFL, 0);
    if (flags < 0)
    {
        LOGE("Get flags error:%d", errno);
        goto error;
    }
    flags |= O_NONBLOCK;
    if (fcntl(ril_cli.cli_fd, F_SETFL, flags) < 0)
    {
        LOGE("Set flags error:%d", errno);
        goto error;
    }

    struct sockaddr_un cli_addr;
    memset(&cli_addr, 0, sizeof(cli_addr));
    cli_addr.sun_family = AF_LOCAL;
    strcpy(cli_addr.sun_path, RIL_SOCK_NAME);
    if(connect(ril_cli.cli_fd, (struct sockaddr *)&cli_addr, sizeof(cli_addr)))
    {
        LOGE("connect() fail[%d].", errno);
        goto error;
    }

    if(pipe(ril_cli.exit_fd))
    {
        LOGE("pipe() fail[%d].", errno);
        goto error;
    }

#if 1
    pthread_attr_t thread_attr;
    pthread_attr_init(&thread_attr);
    if(pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))
    {
        LOGE("pthread_attr_setdetachstate() fail.");
        goto error;
    }

    if(pthread_create(&ril_cli.read_thread_id, &thread_attr, ril_read_run, NULL))
    {
        LOGE("pthread_create() fail.");
        goto error;
    }
    pthread_attr_destroy(&thread_attr);
#else
    if(pthread_create(&read_thread_id, NULL, ril_read_run, NULL))
    {
        LOGE("pthread_create() fail.");
        goto error;
    }
#endif

    pthread_mutex_init(&ril_cli.send_mutex, NULL);
    ril_cli.msg_list = list_create(msg_list_free_cb);
    if(ril_cli.msg_list == NULL) {
        LOGE("list_create(msg_list) fail.");
        goto error;
    }

    ril_cli.cli_thread_list = list_create(cli_thread_list_free_cb);
    if(ril_cli.cli_thread_list == NULL) {
        LOGE("list_create(cli_thread_list) fail.");
        goto error;
    }

    ril_cli.ril_ready = FALSE;

    // Waitting mbtk_rild ready...
    while(!ril_cli.ril_ready) {
        usleep(100000);
    }

    LOGD("RIL server ready...");

    return MBTK_RIL_ERR_SUCCESS;
error:
    if(ril_cli.cli_fd > 0)
    {
        close(ril_cli.cli_fd);
        ril_cli.cli_fd = -1;
    }
    if(ril_cli.exit_fd[0] > 0)
    {
        close(ril_cli.exit_fd[0]);
        ril_cli.exit_fd[0] = -1;
    }
    if(ril_cli.exit_fd[1] > 0)
    {
        close(ril_cli.exit_fd[1]);
        ril_cli.exit_fd[1] = -1;
    }

    return MBTK_RIL_ERR_UNKNOWN;
}

mbtk_ril_err_enum mbtk_ril_deinit()
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(ril_cli.exit_fd[1] > 0)
    {
        if(write(ril_cli.exit_fd[1], "EXIT", 4) != 4) {
            return MBTK_RIL_ERR_UNKNOWN;
        }
    }

    // Wait read_thread exit.
    pthread_join(ril_cli.read_thread_id,NULL);

    if(ril_cli.exit_fd[0] > 0)
    {
        close(ril_cli.exit_fd[0]);
        ril_cli.exit_fd[0] = -1;
    }

    if(ril_cli.exit_fd[1] > 0)
    {
        close(ril_cli.exit_fd[1]);
        ril_cli.exit_fd[1] = -1;
    }

    if(ril_cli.cli_fd > 0)
    {
        close(ril_cli.cli_fd);
        ril_cli.cli_fd = -1;
    }

    pthread_mutex_destroy(&ril_cli.send_mutex);
    if(ril_cli.msg_list) {
        list_free(ril_cli.msg_list);
    }
    if(ril_cli.cli_thread_list) {
        list_free(ril_cli.cli_thread_list);
    }

    ril_cli.ril_ready = FALSE;

    return MBTK_RIL_ERR_SUCCESS;
}

/*
* Get platform version.
*/
mbtk_ril_err_enum mbtk_version_get(void *version)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(version == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_DEV_VERSION, NULL, 0, version, FALSE);
}

/*
* Get platform version.
*/
mbtk_ril_err_enum mbtk_model_get(void *model)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(model == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_DEV_MODEL, NULL, 0, model, FALSE);
}

/*
* Get platform IMEI.
*/
mbtk_ril_err_enum mbtk_imei_get(void *imei)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(imei == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_DEV_IMEI, NULL, 0, imei, FALSE);
}


/*
* Get platform SN.
*/
mbtk_ril_err_enum mbtk_sn_get(void *sn)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sn == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_DEV_SN, NULL, 0, sn, FALSE);
}


/*
* Get platform MEID.
*/
mbtk_ril_err_enum mbtk_meid_get(void *meid)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(meid == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_DEV_MEID, NULL, 0, meid, FALSE);
}


/*
* Return VoLTE state.
*/
mbtk_ril_err_enum mbtk_volte_state_get(int *volte_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(volte_state == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_DEV_VOLTE, NULL, 0, volte_state, FALSE);
}

/*
* Set VoLTE state.
*
* volte_state:
* 0 : Close VoLTE.
* 1 : Open VoLTE.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_volte_state_set(int volte_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(volte_state != 0 && volte_state != 1)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_DEV_VOLTE, &volte_state, 1, NULL, FALSE);
}

/*
* Return radio state.
*/
mbtk_ril_err_enum mbtk_radio_state_get(mbtk_radio_state_enum *radio_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(radio_state == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(RIL_MSG_ID_DEV_MODEM, NULL, 0, &result, FALSE);
    *radio_state = (mbtk_radio_state_enum)result;
    return err;
}

/*
* Set radio state.
*
* Refor to : mbtk_radio_state_enum
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_radio_state_set(mbtk_radio_state_enum radio_state, bool reset_modem)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(radio_state == MBTK_RADIO_STATE_UNKNOWN)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    uint8 buff[2];
    buff[0] = (uint8)radio_state;
    buff[1] = reset_modem ? 1 : 0;

    return ril_req_process(RIL_MSG_ID_DEV_MODEM, buff, 2, NULL, FALSE);
}

/*
* Get system temperature.
*
* type[IN]:
*   0: Soc temperature.
*   1: RF temperature.
* temp[OUT]:
*   temperature in celsius.
*/
mbtk_ril_err_enum mbtk_temp_get(mbtk_temp_type_enum type, int* temp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(temp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    int16 result;
    uint8 temp_temp = (uint8)type;
    mbtk_ril_err_enum err = ril_req_process(RIL_MSG_ID_DEV_TEMP, &temp_temp, 1, &result, FALSE);
    *temp = (int16)result;
    return err;
}


/*
* Get time type.
* "23/05/24,06:09:32+32" -> "23/05/24 06:09:32 +32 00"
*/
mbtk_ril_err_enum mbtk_cell_time_get(char* time_str)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(time_str == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    char buff[RIL_SOCK_MSG_LEN_MAX] = {0};
    mbtk_ril_err_enum err = ril_req_process(RIL_MSG_ID_DEV_CELL_TIME, NULL, 0, &buff, FALSE);
    if(MBTK_RIL_ERR_SUCCESS == err)
    {
        memcpy(time_str,buff,strlen(buff));

        char *temp = strstr(time_str, ",");
        if(temp)
        {
            *temp = ' '; // ',' -> ' '

            temp = strstr(time_str, "+");
            if(temp == NULL)
            {
                temp = strstr(time_str, "-");
            }

            if(temp)
            {
                // Copy +XX or -XX
                char *last_ptr = temp + strlen(temp) + 1;
                while(last_ptr > temp)
                {
                    *last_ptr = *(last_ptr - 1);
                    last_ptr--;
                }

                *last_ptr = ' ';

                memcpy(temp + strlen(temp), " 00", 3);

                LOGD("%s -> %s", buff, time_str);
                return MBTK_RIL_ERR_SUCCESS;
            }
            else
            {
                LOGE("Time error:%s",buff);
                return MBTK_RIL_ERR_TIME_FORMAT;
            }
        }
        else
        {
            LOGE("Time error:%s",buff);
            return MBTK_RIL_ERR_TIME_FORMAT;
        }
    }
    else
    {
        return err;
    }
}

/*
* Get sim state.
*/
mbtk_ril_err_enum mbtk_sim_state_get(mbtk_sim_state_enum *sim_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sim_state == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(RIL_MSG_ID_SIM_STATE, NULL, 0, &result, FALSE);
    *sim_state = (mbtk_sim_state_enum)result;
    return err;
}

/*
* Get sim card type.
*/
mbtk_ril_err_enum mbtk_sim_type_get(mbtk_sim_card_type_enum *sim_card_type)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sim_card_type == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(RIL_MSG_ID_SIM_TYPE, NULL, 0, &result, FALSE);
    *sim_card_type = (mbtk_sim_card_type_enum)result;
    return err;
}

/*
* Get platform IMSI.
*/
mbtk_ril_err_enum mbtk_imsi_get(void *imsi)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(imsi == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SIM_IMSI, NULL, 0, imsi, FALSE);
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_iccid_get(void *iccid)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(iccid == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SIM_ICCID, NULL, 0, iccid, FALSE);
}

/*
* Get current phone number.
*/
mbtk_ril_err_enum mbtk_phone_number_get(void *phone_number)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(phone_number == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SIM_PN, NULL, 0, phone_number, FALSE);
}

/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_sim_lock_get(int *lock_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(lock_state == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    uint8 result;
    mbtk_ril_err_enum err = ril_req_process(RIL_MSG_ID_SIM_LOCK, NULL, 0, &result, FALSE);
    *lock_state = result;
    return err;
}


/*
* Get platform ICCID.
*/
mbtk_ril_err_enum mbtk_sim_lock_set(mbtk_sim_lock_info_t *lock_info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(lock_info == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SIM_LOCK, lock_info, sizeof(mbtk_sim_lock_info_t), NULL, FALSE);
}

/*
* Get PIN’s number of remaining retry
*/
mbtk_ril_err_enum mbtk_sim_lock_retry_times_get(mbtk_pin_puk_last_times_t *retry_times)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(retry_times == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SIM_PINPUK_TIMES, NULL, 0, retry_times, FALSE);
}

/*
* Get plmn list
*/
mbtk_ril_err_enum mbtk_plmn_list_get(mbtk_plmn_info *plmn_list)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(plmn_list == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SIM_PLMN, NULL, 0, plmn_list, FALSE);
}

/*
* Get available network.
*/
mbtk_ril_err_enum mbtk_available_net_get(mbtk_net_info_array_t *net_array)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(net_array == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_AVAILABLE, NULL, 0, net_array, FALSE);
}

/*
* Set network select mode. (+COPS=...)
*/
mbtk_ril_err_enum mbtk_net_sel_mode_set(const mbtk_net_info_t *net)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(net == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_SEL_MODE, net, sizeof(mbtk_net_info_t), NULL, FALSE);
}

/*
* Get network select mode. (+COPS?)
*/
mbtk_ril_err_enum mbtk_net_sel_mode_get(mbtk_net_info_t *net)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(net == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_SEL_MODE, NULL, 0, net, FALSE);
}

/*
* Get platform support bands.
*/
mbtk_ril_err_enum mbtk_support_band_get(mbtk_band_info_t *band)
{
    uint8 type = 0; // Get support bands.
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(band == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_BAND, &type, sizeof(uint8), band, FALSE);
}

/*
* Get platform current bands.
*/
mbtk_ril_err_enum mbtk_current_band_get(mbtk_band_info_t *band)
{
    uint8 type = 1; // Get current bands.
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(band == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_BAND, &type, sizeof(uint8), band, FALSE);
}

/*
* Set platform current bands.
*/
mbtk_ril_err_enum mbtk_current_band_set(const mbtk_band_info_t *band)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(band == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_BAND, band, sizeof(mbtk_band_info_t), NULL, FALSE);
}

/*
* Get current cell infomation.
*/
mbtk_ril_err_enum mbtk_cell_get(mbtk_cell_info_array_t *cell_array)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cell_array == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_CELL, NULL, 0, cell_array, FALSE);
}

/*
* Set cell info.
*
* at*CELL=<mode>,<act>,< band>,<freq>,<cellId>
* at*cell=2,3,,40936,429   //
* at*cell=0  //
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_cell_set(char *info, char *response)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(info == NULL || response == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_CELL, info, strlen(info), response, FALSE);
}

/*
* Get current network signal.
*/
mbtk_ril_err_enum mbtk_net_signal_get(mbtk_signal_info_t *signal)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(signal == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_SIGNAL, NULL, 0, signal, FALSE);
}

/*
* Get current network register information.
*/
mbtk_ril_err_enum mbtk_net_reg_get(mbtk_net_reg_info_t *reg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(reg == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_REG, NULL, 0, reg, FALSE);
}

/*
* Get all APN informations.
*/
mbtk_ril_err_enum mbtk_apn_get(mbtk_apn_info_array_t *apns)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(apns == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_NET_APN, NULL, 0, apns, FALSE);
}


/*
* Set current APN informations.
*/
mbtk_ril_err_enum mbtk_apn_set(const mbtk_apn_info_t *apn)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(apn == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    if(str_empty(apn->apn)) { // Delete APN
        if(!ril_cid_check(apn->cid)) {
            return MBTK_RIL_ERR_CID;
        }
    } else { // Set APN
        if(apn->cid != MBTK_RIL_CID_NUL && !ril_cid_check(apn->cid)) {
            return MBTK_RIL_ERR_CID;
        }
    }

    return ril_req_process(RIL_MSG_ID_NET_APN, apn, sizeof(mbtk_apn_info_t), NULL, FALSE);
}

/*
* Start data call.
*/
mbtk_ril_err_enum mbtk_data_call_start(mbtk_ril_cid_enum cid, bool auto_boot_call,
            bool def_route, bool as_dns, int retry_interval, int timeout, mbtk_ip_info_t *rsp_info)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!ril_cid_check(cid)) {
        return MBTK_RIL_ERR_CID;
    }

    mbtk_data_call_info_t info;
    memset(&info, 0, sizeof(mbtk_data_call_info_t));
    info.type = MBTK_DATA_CALL_START;
    info.cid = cid;
    info.auto_boot_call = (uint8)auto_boot_call;
    info.def_route = (uint8)def_route;
    info.as_dns = (uint8)as_dns;
    info.retry_interval = (uint16)retry_interval;
    if(timeout > 0) {
        info.timeout = (uint16)timeout;
    }

    return ril_req_process(RIL_MSG_ID_NET_DATA_CALL, &info, sizeof(mbtk_data_call_info_t), rsp_info, FALSE);
}

/*
* Stop data call.
*/
mbtk_ril_err_enum mbtk_data_call_stop(mbtk_ril_cid_enum cid, int timeout)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(!ril_cid_check(cid)) {
        return MBTK_RIL_ERR_CID;
    }

    mbtk_data_call_info_t info;
    memset(&info, 0, sizeof(mbtk_data_call_info_t));
    info.type = MBTK_DATA_CALL_STOP;
    info.cid = cid;
    if(timeout > 0) {
        info.timeout = (uint16)timeout;
    }
    return ril_req_process(RIL_MSG_ID_NET_DATA_CALL, &info, sizeof(mbtk_data_call_info_t), NULL, FALSE);
}

/*
* Get data call state.
*/
mbtk_ril_err_enum mbtk_data_call_state_get(mbtk_ril_cid_enum cid, mbtk_ip_info_t *ip)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(ip == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    if(!ril_cid_check(cid)) {
        return MBTK_RIL_ERR_CID;
    }

    mbtk_data_call_info_t info;
    memset(&info, 0, sizeof(mbtk_data_call_info_t));
    info.type = MBTK_DATA_CALL_STATE;
    info.cid = cid;

    return ril_req_process(RIL_MSG_ID_NET_DATA_CALL, &info, sizeof(mbtk_data_call_info_t), ip, FALSE);
}

/*
* Return sms cmgf.
*/
mbtk_ril_err_enum mbtk_sms_cmgf_get(int *sms_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(sms_state == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    uint8 state;
    mbtk_ril_err_enum err = ril_req_process(RIL_MSG_ID_SMS_CMGF, NULL, 0, &state, FALSE);
    *sms_state = state;
    return err;
}

/*
* Set sms cmgf.
*
* volte_state:
* 0 : PDU mode.
* 1 :  text mode.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgf_set(int mode)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMGF, &mode, sizeof(uint8), NULL, FALSE);
}

/*
* Set sms cmgs.
*
if PDU mode (+CMGF=0):
+CMGS=<length><CR>
PDU is given<ctrl-Z/ESC>

if text mode (+CMGF=1):
+CMGS=<da>[,<toda>]<CR>
text is entered<ctrl-Z/ESC>

* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgs_set(char * cmgs, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmgs == NULL || resp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMGS, cmgs, strlen(cmgs), resp, FALSE);
}

/*
* Set sms cmgw.
*
if text mode (+CMGF=1):
+CMGW=<oa/da>[,<tooa/toda>[,<stat>]]
<CR>
text is entered<ctrl-Z/ESC>
if PDU mode (+CMGF=0):
+CMGW=<length>[,<stat>]<CR>PDU is
given<ctrl-Z/ESC>

*/
mbtk_ril_err_enum mbtk_sms_cmgw_set(char * cmgw, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmgw == NULL || resp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMGW, cmgw, strlen(cmgw), resp, FALSE);
}

/*
* Set sms cmgd.
*
* +CMGD=<index>[,<delflag>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgd_set(char * cmdg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmdg == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMGD, cmdg, strlen(cmdg), NULL, FALSE);
}

/*
* Get sms cmgd.
*
* +CMGD: (XXX,XXX)(0-4)
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgd_get(char * cmdg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmdg == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMGD, NULL, 0, cmdg, FALSE);
}


/*
* Set sms cmgl.
*
* AT+CMGL[=<stat>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgl_set(char * cmgl, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmgl == NULL || resp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMGL, cmgl, strlen(cmgl), resp, FALSE);
}

/*
* Return sms csca.
*/
mbtk_ril_err_enum mbtk_sms_csca_get(char *buf)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(buf == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CSCA, NULL, 0, buf, FALSE);
}

/*
* Set sms csca.
*
* AT+CSCA=<number> [,<type>]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_csca_set(char * csca)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(csca == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CSCA, csca, strlen(csca), NULL, FALSE);
}

/*
* Set sms csmp.
*
* AT+CSMP=[<fo>[,<vp>[,<pid>[,<dcs>]]]]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_csmp_set(char * csmp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(csmp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CSMP, csmp, strlen(csmp), NULL, FALSE);
}

/*
* Set sms cscb.
*
* AT+CSCB=<[<mode>[,<mids>[,<dcss>]]]>
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cscb_set(char * cscb)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cscb == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CSCB, cscb, strlen(cscb), NULL, FALSE);
}

/*
* Set sms cnmi.
*
at+cnmi=1,2

OK
if sending fails:
+CMS ERROR: <err>
*/
mbtk_ril_err_enum mbtk_sms_cnmi_set()
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CNMI, NULL, 0, NULL, FALSE);
}

/*
* Set sms cmss.
*
+CMSS=<index>[,<da>[,<toda>]]

if sending successful:
+CMSS: <mr>
OK
if sending fails:
+CMS ERROR: <err>
*/
mbtk_ril_err_enum mbtk_sms_cmss_set(char * cmss, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(cmss == NULL || resp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMSS, cmss, strlen(cmss), resp, FALSE);
}

/*
* Return sms cmgf.
*/
mbtk_ril_err_enum mbtk_sms_cpms_get(char * mem)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(mem == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CPMS, NULL, 0, mem, FALSE);
}


/*
* Set sms cpms.
*
* AT+CPMS=<mem1>[,<mem2>[,<mem3>]]
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cpms_set(char * mem, char* resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(mem == NULL || resp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CPMS, mem, strlen(mem), resp, FALSE);
}

/*
* Set sms cm.
*
* +CMGR=<index>

if PDU mode (+CMGF=0) ��command successful:
+CMGR: <stat>,[<alpha>],<length><CR><LF><pdu>
OK
if text mode (+CMGF=1), command successful and SMS-DELIVER:
+CMGR:<stat>,<oa>,[<alpha>],<scts>[,<tooa>,<fo>,<pid>,<dcs
>, <sca>,<tosca>,<length>]<CR><LF><data>
OK
if text mode (+CMGF=1), command successful and SMS-SUBMIT:
+CMGR:
<stat>,<da>,[<alpha>][,<toda>,<fo>,<pid>,<dcs>,[<vp>],
<sca>,<tosca>,<length>]<CR><LF><data>
OK
otherwise:
+CMS ERROR: <err>
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_sms_cmgr_set(int index, char *resp)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(resp == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_SMS_CMGR, &index, sizeof(uint8), resp, FALSE);
}


#if 0

/*
* Set sim power state.
* power:
* 0: Sim power off.
* 1: Sim power on.
*/
int mbtk_sim_power_set(int power)
{
    if(power != 0 && power != 1)
    {
        return -1;
    }

    //  /sys/devices/virtual/usim_event/usim0/send_event
    char cmd[100] = {0};
    sprintf(cmd, "echo %d > /sys/devices/virtual/usim_event/usim0/send_event", power ? 0 : 1);
    system(cmd);

    return 0;
}

/*
* System power.
* type:
* 0: Reboot system.
* 1: Poweroff system.
* 2: Halt system.
*/
int mbtk_system_reboot(int type)
{
    if(type != 0 && type != 1 && type != 2)
    {
        return -1;
    }

    switch(type)
    {
        case 0:
        {
            system("reboot");
            break;
        }
        case 1:
        {
            system("poweroff");
            break;
        }
        case 2:
        {
            system("halt");
            break;
        }
        default:
        {
            break;
        }
    }

    return 0;
}

/*
* Get time type.
*/
int mbtk_time_get(mbtk_info_handle_t* handle, int *time_type)
{
    uint8 state;
    if(handle == NULL || time_type == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_DEV_TIME_REQ, NULL, 0, &state) > 0)
    {
        LOG("Time type : %d", state);
        *time_type = state;
        return 0;
    }
    else
    {
        return handle->info_err;
    }
}

/*
*Get Absolute time
*"23/05/24 06:09:32 +32 00"
*/
int mbtk_get_abs_time(char *time_str, time_t *time_out)
{
    struct tm tm_;

    char *ptr = strstr(time_str + 10, " ");
    *ptr = '\0';

    LOGD("time : \"%s\"", time_str);
#if 1
    if(strptime(time_str, "%y/%m/%d %T", &tm_) == NULL)
    {
        LOGE("strptime() fail.");
        return -1;
    }
#else
    int year, month, day, hour, minute,second,time_zone;
    if(strstr(time_str, "+"))
    {
        sscanf(time_str, "%d/%d/%d %d:%d:%d +%d",&year,&month,&day,&hour,&minute,&second,&time_zone);
    }
    else if(strstr(time_str, "-"))
    {
        sscanf(time_str, "%d/%d/%d %d:%d:%d -%d",&year,&month,&day,&hour,&minute,&second,&time_zone);
    }
    else
    {
        LOGE("Time format error:%s", time_str);
        return -1;
    }

    // 1970+
    if(year < 70)   // 20xx
    {
        tm_.tm_year  = 2000 + year;
    }
    else     // 19xx
    {
        tm_.tm_year  = 1900 + year;
    }
    tm_.tm_mon   = month - 1;
    tm_.tm_mday  = day;
    tm_.tm_hour  = hour;
    tm_.tm_min   = minute;
    tm_.tm_sec   = second;
    tm_.tm_isdst = 0;
#endif

    time_t timeStamp = mktime(&tm_);
    LOGD("tm_.tm_year = %d,tm_.tm_mon = %d,tm_.tm_mday = %d,tm_.tm_hour = %d,tm_.tm_min = %d,tm_.tm_sec = %d,tm_.tm_isdst = %d",tm_.tm_year,tm_.tm_mon,tm_.tm_mday,tm_.tm_hour,tm_.tm_min,tm_.tm_sec,tm_.tm_isdst);
    LOGD("time = %ld,%x", timeStamp,timeStamp);
    *time_out = timeStamp;

    return 0;
}

/*
* Set time.
*
* time_type:
* 0: Cell time
* 1: NTP time
* 2: User time
* time_str: "YYYY-MM-DD HH:MM:SS"
*/
int mbtk_time_set(mbtk_info_handle_t* handle, mbtk_time_type_enum time_type, char* time_str)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    uint8 buffer[100] = {0};
    buffer[0] = (uint8)time_type;
    if(time_type == MBTK_TIME_TYPE_USER)
    {
        if(!str_empty(time_str))
        {
            memcpy(buffer + sizeof(uint8), time_str, strlen(time_str));
            return info_item_process(handle, MBTK_INFO_ID_DEV_TIME_REQ,
                                     buffer, sizeof(uint8) + strlen(time_str), NULL) ? handle->info_err : 0;
        }
        else
        {
            return -1;
        }
    }
    else
    {
        return info_item_process(handle, MBTK_INFO_ID_DEV_TIME_REQ,
                                 buffer, sizeof(uint8), NULL) ? handle->info_err : 0;
    }
}


#endif

/*
* call_start
*
*/
mbtk_ril_err_enum mbtk_call_start(char* phone_number)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(phone_number == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_CALL_START, phone_number, strlen(phone_number), NULL, FALSE);
}

/*
* Answer the phone call.
*
*/
mbtk_ril_err_enum mbtk_call_answer()
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_CALL_ANSWER, NULL, 0, NULL, FALSE);
}

/*
* Hang up all call.
*
*/
mbtk_ril_err_enum mbtk_call_hang()
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_CALL_HANGUP, NULL, 0, NULL, FALSE);
}

/*
* Hang up a call.
*
*/
mbtk_ril_err_enum mbtk_a_call_hang(int phone_id)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_CALL_HANGUP_A, &phone_id, sizeof(uint8), NULL, FALSE);
}

/*
* Hang up waiting or background call.
*
*/
mbtk_ril_err_enum mbtk_waiting_or_background_call_hang()
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_CALL_HANGUP_B, NULL, 0, NULL, FALSE);
}

/*
* Hang up foreground resume background call.
*
*/
mbtk_ril_err_enum mbtk_foreground_resume_background_call_hang()
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_CALL_HANGUP_C, NULL, 0, NULL, FALSE);
}

/*
* Get current call phone number.
*/
mbtk_ril_err_enum mbtk_call_reg_get(mbtk_call_info_t *reg)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(reg == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_CALL_WAITIN, NULL, 0, reg, FALSE);
}

/*
* Return mute state.
*/
mbtk_ril_err_enum mbtk_mute_state_get(int *mute_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(mute_state == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_CALL_MUTE, NULL, 0, mute_state, FALSE);
}

/*
* Set mute state.
*
* mute_state:
* 0 : of mute.
* 1 : on mute.
*
* Restarting takes effect after execution.
*/
mbtk_ril_err_enum mbtk_mute_state_set(int mute_state)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    return ril_req_process(RIL_MSG_ID_CALL_MUTE, &mute_state, sizeof(uint8), NULL, FALSE);
}

/*
* Set DTMF character.
*
*/
mbtk_ril_err_enum mbtk_dtmf_send(mbtk_call_dtmf_info_t *dtmf_character)
{
    if(!ril_cli.ril_ready)
    {
        return MBTK_RIL_ERR_NOT_INIT;
    }

    if(dtmf_character == NULL)
    {
        LOGE("ARG error.");
        return MBTK_RIL_ERR_PARAMETER;
    }

    return ril_req_process(RIL_MSG_ID_CALL_DTMF, dtmf_character, sizeof(mbtk_call_dtmf_info_t), NULL, FALSE);
}


#if 0
/*
* Set wakeup state.
*
* wakeup_state:(0~31)
* 0 : means resume all
* 1~31 means suspend
* Control the active reporting of some platform modems to reduce wakeup
*/

int mbtk_wakeup_state_set(mbtk_info_handle_t* handle, uint32 wakeup_state)
{
    return info_item_process(handle, MBTK_INFO_ID_WAKEUP_STA_REQ, (uint32*)&wakeup_state, sizeof(uint32), NULL) ? handle->info_err : 0;
}

/*
* oos get.
*/
int mbtk_oos_get(mbtk_info_handle_t* handle, mbtk_oos_info *oos_info)
{
    if(info_item_process(handle, MBTK_INFO_ID_OOS_STA_REQ, NULL, 0, oos_info) > 0)
    {
        return 0;
    }
    else
    {
        return handle->info_err;
    }
}

/*
* oos set .
*/
int mbtk_oos_set(mbtk_info_handle_t* handle, mbtk_oos_info *oos_info)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }

    return info_item_process(handle, MBTK_INFO_ID_OOS_STA_REQ, oos_info, sizeof(mbtk_oos_info), NULL) ? handle->info_err : 0;
}


#endif

#if 0
/*
* Set pdp state change callback function.
*/
int mbtk_pdp_state_change_cb_reg(mbtk_info_handle_t* handle, mbtk_info_callback_func cb)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_IND_PDP_STATE_CHANGE, NULL, 0, NULL) < 0)
    {
        return handle->info_err;
    }
    else
    {
        handle->pdp_state_cb = cb;
        return 0;
    }
}

/*
* Set network state change callback function.
*/
int mbtk_net_state_change_cb_reg(mbtk_info_handle_t* handle, mbtk_info_callback_func cb)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_IND_NET_STATE_CHANGE, NULL, 0, NULL) < 0)
    {
        return handle->info_err;
    }
    else
    {
        handle->net_state_cb = cb;
        return 0;
    }
}

/*
* Set call state change callback function.
*/
int mbtk_call_state_change_cb_reg(mbtk_info_handle_t* handle, mbtk_info_callback_func cb)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_IND_CALL_STATE_CHANGE, NULL, 0, NULL) < 0)
    {
        return handle->info_err;
    }
    else
    {
        handle->call_state_cb = cb;
        return 0;
    }
}

/*
* Set sms state change callback function.
*/
int mbtk_sms_state_change_cb_reg(mbtk_info_handle_t* handle, mbtk_info_callback_func cb)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_IND_SMS_STATE_CHANGE, NULL, 0, NULL) < 0)
    {
        return handle->info_err;
    }
    else
    {
        handle->sms_state_cb = cb;
        return 0;
    }
}

/*
* Set radio state change callback function.
*/
int mbtk_radio_state_change_cb_reg(mbtk_info_handle_t* handle, mbtk_info_callback_func cb)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_IND_RADIO_STATE_CHANGE, NULL, 0, NULL) < 0)
    {
        return handle->info_err;
    }
    else
    {
        handle->radio_state_cb = cb;
        return 0;
    }
}

/*
* Set sim state change callback function.
*/
int mbtk_sim_state_change_cb_reg(mbtk_info_handle_t* handle, mbtk_info_callback_func cb)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_IND_SIM_STATE_CHANGE, NULL, 0, NULL) < 0)
    {
        return handle->info_err;
    }
    else
    {
        handle->sim_state_cb = cb;
        return 0;
    }
}

/*
* Set signal state change callback function.
*/
int mbtk_signal_state_change_cb_reg(mbtk_info_handle_t* handle, mbtk_info_callback_func cb)
{
    if(handle == NULL)
    {
        LOGE("ARG error.");
        return -1;
    }
    if(info_item_process(handle, MBTK_INFO_ID_IND_SIGNAL_STATE_CHANGE, NULL, 0, NULL) < 0)
    {
        return handle->info_err;
    }
    else
    {
        handle->signal_state_cb = cb;
        return 0;
    }
}
#endif
