#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <pthread.h>
#include <sys/epoll.h>
#include <fcntl.h>
#include <signal.h>

#include "mbtk_type.h"
#include "mbtk_ril.h"
#include "atchannel.h"
#include "at_tok.h"
#include "mbtk_utils.h"
#include "ril_info.h"

void ril_rsp_pack_send(int fd, int ril_id, int msg_index, const void* data, int data_len);

/** Returns SIM_NOT_READY on error */
mbtk_sim_state_enum ril_sim_state_get()
{
    ATResponse *p_response = NULL;
    int err;
    mbtk_sim_state_enum ret = MBTK_SIM_STATE_UNKNOWN;
    char *cpinLine;
    char *cpinResult;

    err = at_send_command_singleline("AT+CPIN?", "+CPIN:", &p_response);

    if (err < 0 || p_response->success == 0 || !p_response->p_intermediates)
    {
        ret = MBTK_SIM_STATE_NOT_READY;
        goto done;
    }

    switch (at_get_cme_error(p_response))
    {
        case CME_SUCCESS:
            break;

        case CME_SIM_NOT_INSERTED:
            ret = MBTK_SIM_STATE_ABSENT;
            goto done;

        default:
            ret = MBTK_SIM_STATE_NOT_READY;
            goto done;
    }

    /* CPIN? has succeeded, now look at the result */

    cpinLine = p_response->p_intermediates->line;
    err = at_tok_start (&cpinLine);

    if (err < 0)
    {
        ret = MBTK_SIM_STATE_NOT_READY;
        goto done;
    }

    err = at_tok_nextstr(&cpinLine, &cpinResult);

    if (err < 0)
    {
        ret = MBTK_SIM_STATE_NOT_READY;
        goto done;
    }

    if (0 == strcmp (cpinResult, "SIM PIN"))
    {
        ret = MBTK_SIM_STATE_SIM_PIN;
    }
    else if (0 == strcmp (cpinResult, "SIM PUK"))
    {
        ret = MBTK_SIM_STATE_SIM_PUK;
    }
    else if (0 == strcmp (cpinResult, "PH-SIMLOCK PIN"))
    {
        ret = MBTK_SIM_STATE_PH_SIMLOCK_PIN;
    }
    else if (0 == strcmp (cpinResult, "PH-SIMLOCK PIN"))
    {
        ret = MBTK_SIM_STATE_PH_SIMLOCK_PUK;
    }
    else if (0 == strcmp (cpinResult, "PH-FSIM PIN"))
    {
        ret = MBTK_SIM_STATE_PH_FSIM_PIN;
    }
    else if (0 == strcmp (cpinResult, "PH-FSIM PUK"))
    {
        ret = MBTK_SIM_STATE_PH_FSIM_PUK;
    }
    else if (0 == strcmp (cpinResult, "SIM PIN2"))
    {
        ret = MBTK_SIM_STATE_SIM_PIN2;
    }
    else if (0 == strcmp (cpinResult, "SIM PUK2"))
    {
        ret = MBTK_SIM_STATE_SIM_PUK2;
    }
    else if (0 == strcmp (cpinResult, "PH-NET PIN"))
    {
        ret = MBTK_SIM_STATE_PH_NET_PIN;
    }
    else if (0 == strcmp (cpinResult, "PH-NET PUK"))
    {
        ret = MBTK_SIM_STATE_PH_NET_PUK;
    }
    else if (0 == strcmp (cpinResult, "PH-NETSUB PIN"))
    {
        ret = MBTK_SIM_STATE_PH_NETSUB_PIN;
    }
    else if (0 == strcmp (cpinResult, "PH-NETSUB PUK"))
    {
        ret = MBTK_SIM_STATE_PH_NETSUB_PUK;
    }
    else if (0 == strcmp (cpinResult, "PH-SP PIN"))
    {
        ret = MBTK_SIM_STATE_PH_SP_PIN;
    }
    else if (0 == strcmp (cpinResult, "PH-SP PUK"))
    {
        ret = MBTK_SIM_STATE_PH_SP_PUK;
    }
    else if (0 == strcmp (cpinResult, "PH-CORP PIN"))
    {
        ret = MBTK_SIM_STATE_PH_CORP_PIN;
    }
    else if (0 == strcmp (cpinResult, "PH-CORP PUK"))
    {
        ret = MBTK_SIM_STATE_PH_CORP_PUK;
    }
    else if (0 == strcmp (cpinResult, "READY"))
    {
        ret = MBTK_SIM_STATE_READY;
    } else {
        ret = MBTK_SIM_STATE_ABSENT;
    }

done:
    at_response_free(p_response);
    p_response = NULL;
    cpinResult = NULL;
    ril_info.sim_state = ret;
    return ret;
}

/*
AT*EUICC?
*EUICC: 1

OK
*/
static int req_sim_type_get(uint8 *type, int *cme_err)
{
    ATResponse *response = NULL;
    char *tmp_ptr = NULL;
    int err = at_send_command_singleline("AT*EUICC?", "*EUICC:", &response);

    if (err < 0 || response->success == 0 || !response->p_intermediates){
        *cme_err = at_get_cme_error(response);
        goto exit;
    }

    char *line = response->p_intermediates->line;
    err = at_tok_start(&line);
    if (err < 0)
    {
        goto exit;
    }
    int sim_card_type = -1;
    err = at_tok_nextint(&line, &sim_card_type);
    if (err < 0)
    {
        goto exit;
    }
    if(sim_card_type != -1)
        *type = sim_card_type;
    goto exit;
exit:
    at_response_free(response);
    return err;
}

/*
AT+ICCID
+ICCID: 89860621330065648041

OK
*/
static int req_iccid_get(void *data, int *cme_err)
{
    ATResponse *response = NULL;
    char *tmp_ptr = NULL;
    int err = at_send_command_singleline("AT+ICCID", "+ICCID:", &response);

    if (err < 0 || response->success == 0 || !response->p_intermediates){
        *cme_err = at_get_cme_error(response);
        goto exit;
    }

    char *line = response->p_intermediates->line;
    err = at_tok_start(&line);
    if (err < 0)
    {
        goto exit;
    }

    err = at_tok_nextstr(&line, &tmp_ptr);
    if (err < 0)
    {
        goto exit;
    }

    memcpy(data, tmp_ptr, strlen(tmp_ptr));
exit:
    at_response_free(response);
    return err;
}

/*
AT+CNUM?
+CNUM: "","13980414101",129

OK

*/
static int req_phone_number_get(void *data, int *cme_err)
{
    ATResponse *response = NULL;
    char *tmp_ptr = NULL;
    int err = at_send_command_singleline("AT+CNUM?", "+CNUM:", &response);
    if (err < 0 || response == NULL || response->success == 0 || !response->p_intermediates){
        if(response) {
            *cme_err = at_get_cme_error(response);
        }
        LOGD("AT+CNUM? fail.");
        goto exit;
    }

    char *line = response->p_intermediates->line;
    if(line == NULL) {
        LOGD("line is NULL");
        goto exit;
    }
    err = at_tok_start(&line);
    if (err < 0)
    {
        goto exit;
    }

    err = at_tok_nextstr(&line, &tmp_ptr);
    if (err < 0)
    {
        goto exit;
    }

    err = at_tok_nextstr(&line, &tmp_ptr);
    if (err < 0)
    {
        goto exit;
    }

    memcpy(data, tmp_ptr, strlen(tmp_ptr));
exit:
    at_response_free(response);
    return err;
}


/*
AT+CIMI
460068103383304

OK

*/
static int req_imsi_get(void *data, int *cme_err)
{
    ATResponse *response = NULL;
    int err = at_send_command_numeric("AT+CIMI", &response);

    if (err < 0 || response->success == 0 || !response->p_intermediates){
        *cme_err = at_get_cme_error(response);
        goto exit;
    }

    memcpy(data, response->p_intermediates->line, strlen(response->p_intermediates->line));
exit:
    at_response_free(response);
    return err;
}

static int req_sim_lock_state_get(int *state, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[64]={0};

    int err = at_send_command_singleline("AT+CLCK=SC,2", "+CLCK:", &response);
    if (err < 0 || response == NULL || response->success == 0 || !response->p_intermediates){
        if(response) {
            *cme_err = at_get_cme_error(response);
        }
        LOGD("AT+CLCK fail.");
        goto exit;
    }

    char *line = response->p_intermediates->line;
    if(line == NULL) {
        LOGD("line is NULL");
        goto exit;
    }
    err = at_tok_start(&line);
    if (err < 0)
    {
        goto exit;
    }
    err = at_tok_nextint(&line, state);
    if (err < 0)
    {
        goto exit;
    }

exit:
    at_response_free(response);
    return err;
}

/*
*
AT+CLCK=SC,0,1234
+CLCK: 0

OK

AT+CLCK=SC,1,1234
+CLCK: 1

OK

*
*/
static int req_sim_lock_switch(bool enable, uint8* pin, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[64]={0};
    snprintf(cmd, sizeof(cmd), "AT+CLCK=SC,%d,%s", enable ? 1 : 0, pin);
    int err = at_send_command_singleline(cmd, "+CLCK:", &response);
    if (err < 0 || response == NULL || response->success == 0 || !response->p_intermediates){
        if(response) {
            *cme_err = at_get_cme_error(response);
        }
        LOGD("AT+CLCK fail.");
        goto exit;
    }

    int ret;
    char *line = response->p_intermediates->line;
    if(line == NULL) {
        LOGD("line is NULL");
        goto exit;
    }
    err = at_tok_start(&line);
    if (err < 0)
    {
        goto exit;
    }
    err = at_tok_nextint(&line, &ret);
    if (err < 0)
    {
        goto exit;
    }

    if(enable) {
        if(ret != 1) {
            LOGW("+CLCK not 1.");
            err = -1;
            goto exit;
        }
    } else {
        if(ret != 0) {
            LOGW("+CLCK not 0.");
            err = -1;
            goto exit;
        }
    }

exit:
    at_response_free(response);
    return err;
}

static int req_sim_pin_change(uint8* old_pin, uint8 *new_pin, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[128]={0};
    int err = 0;
    int state;
    if(req_sim_lock_state_get(&state, cme_err) || *cme_err != MBTK_RIL_ERR_CME_NON)
    {
        return -1;
    }

    if(state == 0) { // Sim lock not open.
        return req_sim_lock_switch(TRUE, new_pin, cme_err);
    } else {// Change password.
        snprintf(cmd, sizeof(cmd), "AT+CPWD=SC,%s,%s", old_pin, new_pin);
        err = at_send_command(cmd, &response);
        if (err < 0 || response->success == 0){
            if(cme_err) {
                *cme_err = at_get_cme_error(response);
            }
            goto exit;
        }
    }

exit:
    at_response_free(response);
    return err;
}

static int req_sim_pin_verify(uint8* pin, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[64]={0};
    sprintf(cmd, "AT+CPIN=%s", pin);
    int err = at_send_command(cmd, &response);
    if (err < 0 || response->success == 0){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

exit:
    at_response_free(response);
    return err;
}

static int req_sim_puk_verify(uint8* puk, uint8* pin, int *cme_err)
{
    ATResponse *response = NULL;
    char cmd[64]={0};
    sprintf(cmd, "AT+CPIN=%s,%s", puk, pin);
    int err = at_send_command(cmd, &response);
    if (err < 0 || response->success == 0){
        if(cme_err) {
            *cme_err = at_get_cme_error(response);
        }
        goto exit;
    }

exit:
    at_response_free(response);
    return err;
}

static int req_sim_lock_set(mbtk_sim_lock_info_t *lock_info, int *cme_err)
{
    switch(lock_info->type)
    {
        case MBTK_SIM_LOCK_TYPE_DISABLE:
        {
            return req_sim_lock_switch(FALSE, lock_info->pin1, cme_err);
        }
        case MBTK_SIM_LOCK_TYPE_ENABLE:
        {
            return req_sim_lock_switch(TRUE, lock_info->pin1, cme_err);
        }
        case MBTK_SIM_LOCK_TYPE_CHANGE:
        {
            return req_sim_pin_change(lock_info->pin1, lock_info->pin2, cme_err);
        }
        case MBTK_SIM_LOCK_TYPE_VERIFY_PIN:
        {
            return req_sim_pin_verify(lock_info->pin1, cme_err);
        }
        case MBTK_SIM_LOCK_TYPE_VERIFY_PUK:
        {
            return req_sim_puk_verify(lock_info->puk, lock_info->pin1, cme_err);
        }
        default:
        {
            LOGE("Unknown type : %d", lock_info->type);
            break;
        }
    }

    return -1;
}

/*
AT+EPIN?
+EPIN: 3,0,10,0

OK
*/
static int req_pin_puk_last_times_get(mbtk_pin_puk_last_times_t *times, int *cme_err)
{
    ATResponse *response = NULL;
    char *tmp_ptr = NULL;
    int err = at_send_command_singleline("AT+EPIN?", "+EPIN:", &response);

    if (err < 0 || response->success == 0 || !response->p_intermediates){
        *cme_err = at_get_cme_error(response);
        goto exit;
    }

    char *line = response->p_intermediates->line;
    err = at_tok_start(&line);
    if (err < 0)
    {
        goto exit;
    }
    mbtk_pin_puk_last_times_t last_times={0};
    err = at_tok_nextint(&line, &(last_times.p1_retry));
    if (err < 0)
    {
        goto exit;
    }
    times->p1_retry = last_times.p1_retry;
    err = at_tok_nextint(&line, &(last_times.p2_retry));
    if (err < 0)
    {
        goto exit;
    }
    times->p2_retry = last_times.p2_retry;
    err = at_tok_nextint(&line, &(last_times.puk1_retry));
    if (err < 0)
    {
        goto exit;
    }
    times->puk1_retry = last_times.puk1_retry;
    err = at_tok_nextint(&line, &(last_times.puk2_retry));
    if (err < 0)
    {
        goto exit;
    }
    times->puk2_retry = last_times.puk2_retry;

exit:
    at_response_free(response);
    return err;
}

/*
AT+CPOL?
*EUICC: 1

OK
*/
static int req_plmn_get(mbtk_plmn_info *info, int *cme_err)
{
    ATResponse *response = NULL;
    char *tmp_ptr = NULL;
    int err = at_send_command_multiline("AT+CPOL?", "+CPOL:", &response);

    if (err < 0 || response->success == 0 || !response->p_intermediates){
        *cme_err = at_get_cme_error(response);
        goto exit;
    }

    int mccmnc_type = -1;
    int count = -1;
    char *mccmnc_name = NULL;
    ATLine* lines_ptr = response->p_intermediates;
    char *line = NULL;
    while(lines_ptr)
    {
        line = lines_ptr->line;
        //if(strStartsWith(line, "+CPOL:"))
        {
            err = at_tok_start(&line);
            if (err < 0)
            {
                goto exit;
            }
            err = at_tok_nextint(&line, &count);
            if (err < 0)
            {
                goto exit;
            }
            info->count = count;

            err = at_tok_nextint(&line, &mccmnc_type);
            if (err < 0)
            {
                goto exit;
            }
            info->plmn[count-1].format = mccmnc_type;

            err = at_tok_nextstr(&line, &mccmnc_name);
            if (err < 0)
            {
                goto exit;
            }
            memcpy(info->plmn[count-1].name, mccmnc_name, strlen(mccmnc_name));
            mccmnc_name = NULL;
        }
        lines_ptr = lines_ptr->p_next;
    }

exit:
    at_response_free(response);
    return err;
}

//void net_list_free(void *data);
// Return MBTK_INFO_ERR_SUCCESS,will call pack_error_send() to send RSP.
// Otherwise, do not call pack_error_send().
mbtk_ril_err_enum sim_pack_req_process(sock_cli_info_t* cli_info, ril_msg_pack_info_t* pack)
{
    mbtk_ril_err_enum err = MBTK_RIL_ERR_SUCCESS;
    int cme_err = MBTK_RIL_ERR_CME_NON;
    switch(pack->msg_id)
    {
        case RIL_MSG_ID_SIM_STATE:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                mbtk_sim_state_enum state = ril_sim_state_get();
                if(state == MBTK_SIM_STATE_UNKNOWN)
                {
                    err = MBTK_RIL_ERR_UNKNOWN;
                    LOG("Get sim state fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &state, sizeof(uint8));
                }
            }
            else     // Set
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
                LOG("Unsupport set sim state.");
            }
            break;
        }
        case RIL_MSG_ID_SIM_TYPE:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                uint8 sim_card_type;
                if(req_sim_type_get(&sim_card_type, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Get IMSI fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &sim_card_type, sizeof(uint8));
                }
            }
            else     // Set
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
                LOG("Unsupport set sim type.");
            }
            break;
        }
        case RIL_MSG_ID_SIM_IMSI:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                char imsi[20] = {0};
                if(req_imsi_get(imsi, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Get IMSI fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, imsi, strlen(imsi));
                }
            }
            else     // Set
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
                LOG("Unsupport set IMSI.");
            }
            break;
        }
        case RIL_MSG_ID_SIM_PN:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                char pn[50] = {0};
                if(req_phone_number_get(pn, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Get PN fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, pn, strlen(pn));
                }
            }
            else     // Set
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
                LOG("Unsupport set PN.");
            }
            break;
        }
        case RIL_MSG_ID_SIM_ICCID:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                char iccid[50] = {0};
                if(req_iccid_get(iccid, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Get ICCID fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, iccid, strlen(iccid));
                }
            }
            else     // Set
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
                LOG("Unsupport set ICCID.");
            }
            break;
        }
        case RIL_MSG_ID_SIM_LOCK:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                int state;
                if(req_sim_lock_state_get(&state, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Get sim lock state fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &state, sizeof(uint8));
                }
            }
            else     // Set
            {
                mbtk_sim_lock_info_t *lock_info = (mbtk_sim_lock_info_t*)pack->data;
                if(req_sim_lock_set(lock_info, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Set sim lock fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, NULL, 0);
                }
            }
            break;
        }
        case RIL_MSG_ID_SIM_PINPUK_TIMES:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                mbtk_pin_puk_last_times_t last_times;
                if(req_pin_puk_last_times_get(&last_times, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Get PIN/PUK Last times fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &last_times, sizeof(mbtk_pin_puk_last_times_t));
                }
            }
            else     // Set
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
                LOG("Unsupport set PIN/PUK Last times.");
            }
            break;
        }
        case RIL_MSG_ID_SIM_PLMN:
        {
            if(pack->data_len == 0 || pack->data == NULL)
            {
                mbtk_plmn_info info;
                memset(&info, 0, sizeof(mbtk_plmn_info));
                if(req_plmn_get(&info, &cme_err) || cme_err != MBTK_RIL_ERR_CME_NON)
                {
                    if(cme_err != MBTK_RIL_ERR_CME_NON) {
                        err = MBTK_RIL_ERR_CME + cme_err;
                    } else {
                        err = MBTK_RIL_ERR_UNKNOWN;
                    }
                    LOG("Get PLMN list fail.");
                }
                else
                {
                    ril_rsp_pack_send(cli_info->fd, pack->msg_id, pack->msg_index, &info, sizeof(mbtk_plmn_info));
                }
            }
            else     // Set
            {
                err = MBTK_RIL_ERR_UNSUPPORTED;
                LOG("Unsupport set PLMN list.");
            }
            break;
        }
        default:
        {
            err = MBTK_RIL_ERR_REQ_UNKNOWN;
            LOG("Unknown request : %s", id2str(pack->msg_id));
            break;
        }
    }

    return err;
}

