/*
*    gnss_hd8122.c
*
*    HD8122 GNSS source.
*
*/
/******************************************************************************

                          EDIT HISTORY FOR FILE

  WHEN        WHO       WHAT,WHERE,WHY
--------    --------    -------------------------------------------------------
2024/6/14     LiuBin      Initial version

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>

#include "mbtk_log.h"
#include "mbtk_type.h"
#include "mbtk_gpio.h"
#include "mbtk_utils.h"
#include "gnss_utils.h"
#include "gnss_hd8122.h"
#include "hd8122_dl/hd8040_upgrade.h"
#include "agps/agnss_http_download.h"
#include "agps/8122_agnss.h"
#include "mbtk_utils.h"
// #include "hd8122_dl/port.h"


#define UART_BITRATE_NMEA_DEF_FW    115200   // Default bitrate.
#define GNSS_POWER_GPIO 43
#define GNSS_SET_TIMEOUT           3000     // 3s
#define GNSS_PACK_BUFF_SIZE           1024
#define GNSS_MSG_NUM_MAX 30
#define READ_LEN_MAX 1024 //BOOT_UPGRADE_BUFF_MAX_1
#define GNSS_FW_GQALS_PATH "/lib/firmware/gp_gl_ga_fw.bin"
#define GNSS_FW_GAQBS_PATH "/lib/firmware/gp_bd_ga_fw.bin"

//#define AGNSS_TEST_URL               "http://aclientt.allystar.com:80/ephemeris/HD_GPS_BDS.hdb"
#define AGNSS_URL                    "http://uagnss.allystar.com:80/ephemeris/%s?compid=yikecs1&token=Z38w5urAuawubTxi"
#define AGNSS_INDIVIDUATION_URL_HEAD "http://cagnss.allystar.com/api/v1/eph/rt?"
#define AGNSS_ALLSTAR_URL_HEAD       "http://cagnss.allystar.com/api/v1/eph/rta?"
#define AGNSS_EPH_FILE_PATH          "/lib/firmware/eph_data.txt"

#define AGNSS_EPH_GPS                "HD_GPS.hdb"
#define AGNSS_EPH_BDS                "HD_BDS.hdb"
#define AGNSS_EPH_GLO                "HD_GLO.hdb"
#define AGNSS_EPH_GPS_BDS            "HD_GPS_BDS.hdb"
#define AGNSS_EPH_GPS_GLO            "HD_GPS_GLO.hdb"

static pthread_cond_t read_cond;
static pthread_mutex_t read_mutex;
static bool setting_waitting = FALSE;
static bool setting_busy = FALSE;
static void *gnss_set_rsp_ptr = NULL;
static gnss_err_enum gnss_set_result = GNSS_ERR_OK;
static hd8122_msg_id_t msg_array[GNSS_MSG_NUM_MAX];
//static char gnss_ctrl_path[128] = "/sys/devices/platform/mbtk-gnss/ctrl";
static gnss_agps_info_t agps_info;
static char *gnss_set_cmd_rsp = NULL;
static int gnss_cmd_rsp_len = 0;

int gnss_write(int fd, const void *data, int data_len);
int OpenUart(const char* UART_DEV);
int uart_close(int fd);
int fw_update_boot(int uart_fd, uint8_t *data, uint32_t len);
int fw_update_boot_8122(int uart_fd, uint8_t *data, uint32_t len);


static uint16 fletcher16(const uint8_t* data, int data_len)
{
    uint32_t sum1 = 0;
    uint32_t sum2 = 0;
    int index;

    for (index = 0; index < data_len; ++index )
    {
        sum1 += data[index];
        sum2 += sum1;
    }

    return ((0xFF & sum2) << 8) | (0xFF & sum1);
}

static void gnss_set_timer_cb(int signo)
{
    if(setting_busy)
    {
        pthread_mutex_lock(&read_mutex);
        pthread_cond_signal(&read_cond);
        pthread_mutex_unlock(&read_mutex);
        gnss_set_result = GNSS_ERR_TIMEOUT;
    }
    return;
}

static void msg_init()
{
    unsigned int i = 0;
    while(i < ARRAY_SIZE(msg_array))
    {
        msg_array[i].enable = FALSE;
        i++;
    }
}

static int msg_insert(uint8 gid, uint8 sid)
{
    unsigned int i = 0;
    while(i < ARRAY_SIZE(msg_array))
    {
        if(!msg_array[i].enable)
            break;
        i++;
    }

    if(i == ARRAY_SIZE(msg_array))
    {
        LOGE("Msg full : %d", i);
        return -1;
    }
    else
    {
        msg_array[i].enable = TRUE;
        msg_array[i].gid = gid;
        msg_array[i].sid = sid;
        return 0;
    }
}

static int msg_find(uint8 gid, uint8 sid)
{
    unsigned int i = 0;
    while(i < ARRAY_SIZE(msg_array))
    {
        if(msg_array[i].enable && gid == msg_array[i].gid && sid == msg_array[i].sid)
            break;
        i++;
    }

    if(i == ARRAY_SIZE(msg_array))
    {
        LOGE("No found %d - %d", gid, sid);
        return -1;
    }
    else
    {
        return i;
    }
}

#if 0
static int msg_remove(uint8 gid, uint8 sid)
{
    int i = msg_find(gid, sid);
    if(i >= 0)
    {
        msg_array[i].enable = FALSE;
        msg_array[i].gid = 0;
        msg_array[i].sid = 0;
        return 0;
    }
    else
    {
        return -1;
    }
}
#endif

static int msg_count()
{
    unsigned int i = 0;
    int count = 0;
    while(i < ARRAY_SIZE(msg_array))
    {
        if(msg_array[i].enable)
            count++;
        i++;
    }
    return count;
}


static uint32_t read_bin_file(uint8_t *path, uint8_t *buff)
{
    int fp = -1;
    int ret = 0;
    int i = 0;
    int size = 0;

    if (NULL == path || NULL == buff)
    {
        LOGE("ARG error");
        return 0;
    }

    fp = open((char *)path, O_RDONLY);
    if(fp < 0)
    {
        LOGE( "open file failed ! errno is %d", errno);
        return 0;
    }

    size = lseek(fp, 0x00, SEEK_END);
    if(size <= 0)
    {
        LOGE( "file is empty.");
        return 0;
    }

    LOGD( "file size is:%d", size);
    lseek(fp, 0x00, SEEK_SET);
    while(1)
    {
        ret = read(fp, buff, READ_LEN_MAX);
        i += ret;
        if(ret == READ_LEN_MAX)
        {
            buff += READ_LEN_MAX;
        }
        else
        {
            break;
        }
    }

    LOGD("file size is:%d,i:%d", size, i);
    close(fp);
    if(size != i)
    {
        return 0;
    }
    return size;
}


static int pack_create(hd8122_id_type_enum id_type, uint8 id, uint16 data_len, const uint8 *data, uint8 *pack, int pack_len)
{
    if(pack == NULL || pack_len < HD8122_PACK_LEN_MIN)
    {
        return -1;
    }
    memset(pack, 0, pack_len);
    uint8 *data_ptr = pack;
    data_ptr += uint16_2_byte(HD8122_PACK_HEAD, data_ptr, false);
    *data_ptr++ = (uint8)id_type;
    *data_ptr++ = id;
    data_ptr += uint16_2_byte(data_len, data_ptr, false);
    if(data_len > 0)
    {
        memcpy(data_ptr, data, data_len);
        data_ptr += data_len;
    }
    data_ptr += uint16_2_byte(fletcher16(pack + 2, 4 + data_len), data_ptr, false);
    return (data_ptr - pack);
}

// f1 d9 05 01 02 00 06 01 0f 38
// or
// f1 d9 05 00 02 00 06 01 0f 38
static int msg_array_change(const uint8 *pack, int pack_len, hd8122_id_ack_enum *ack_nak)
{
    if(pack_len == 0)
    {
        LOGE("pack_len(%d) error.", pack_len);
        return -1;
    }

    //info result
    if(gnss_set_cmd_rsp != NULL)
    {
        if(pack_len > 4 && pack[2] == 0x0a && pack[3] == 0x04)  //VER INFO
        {
            char sw_ver[32] = {0};
            char hw_ver[32] = {0};
            memcpy(sw_ver, pack + 6, 16);
            memcpy(hw_ver, pack + 22, 16);
            snprintf(gnss_set_cmd_rsp, gnss_cmd_rsp_len, "%s,%s", sw_ver, hw_ver);
            LOGE("fw ver:%s", gnss_set_cmd_rsp);
            int index = msg_find(pack[2], pack[3]);
            if(index >= 0)
            {
                msg_array[index].enable = FALSE;
            }
            else
            {
                LOGE("Unknown gid - %d, sid - %d", pack[2], pack[3]);
                return -1;
            }
            return 0;
        }
    }

    //set result
    if(pack_len % 10)
    {
        LOGE("pack_len(%d) error.", pack_len);
        return -1;
    }
    int count = pack_len / 10;
    int i = 0;
    while(i < count)
    {
        const uint8 *ptr = pack + i * 10;
        if(ptr[0] != 0xf1 || ptr[1] != 0xd9)
        {
            LOGE("Pack head error : %02x %02x", ptr[0], ptr[1]);
            return -1;
        }

        if(ptr[2] != 0x05)
        {
            LOGE("Type not 0x05 : %02x", ptr[2]);
            return -1;
        }

        int index = msg_find(ptr[6], ptr[7]);
        if(index >= 0)
        {
            if(ptr[3] == 0x01)
            {
                msg_array[index].ack_nak = HD8122_ID_ACK_ACK;
            }
            else if(ptr[3] == 0x00)
            {
                msg_array[index].ack_nak = HD8122_ID_ACK_NAK;

                // There is a nak as a failure.
                *ack_nak = HD8122_ID_ACK_NAK;
            }
            else
            {
                LOGE("ID not 0x00 or 0x01 : %02x", ptr[3]);
                return -1;
            }

            msg_array[index].enable = FALSE;
        }
        else
        {
            LOGE("Unknown gid - %d, sid - %d", ptr[6], ptr[7]);
            return -1;
        }
        i++;
    }

    return 0;
}

static void gnss_cmd_rsp_process(const void *data, int data_len)
{
//    const char *ptr = (const char*)data;
    log_hex("RSP", data, data_len);

    hd8122_id_ack_enum ack_nak = HD8122_ID_ACK_ACK;
    if(!msg_array_change((const uint8*)data, data_len, &ack_nak))
    {
        if(setting_waitting && msg_count() == 0)
        {
            if(ack_nak == HD8122_ID_ACK_ACK)
            {
                gnss_set_result = GNSS_ERR_OK;
            }
            else
            {
                gnss_set_result = GNSS_ERR_UNKNOWN;
            }

            mbtk_timer_clear();

            pthread_mutex_lock(&read_mutex);
            pthread_cond_signal(&read_cond);
            pthread_mutex_unlock(&read_mutex);
            setting_waitting = FALSE;
        }

    }
    else
    {
        LOGW("Unknown rsp data.");
    }
}

static gnss_err_enum gnss_8122_reset(int fd, uint8 reset)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("RESET");
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_SIMPLERST, 1, (uint8*)(&reset), buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    return GNSS_ERR_OK;
}

static gnss_err_enum gnss_8122_syscfg(int fd, uint32 mode)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("SYSCFG");
    //uint8 mode_str[4];
    //uint32_2_byte(mode, mode_str, TRUE);
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_NAVSAT, 4, (uint8*)(&mode), buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_NAVSAT);
    return GNSS_ERR_OK;
}

static gnss_err_enum gnss_8122_freqcfg(int fd, uint8 mode)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("FREQCFG");
    uint8 data[20];
    memset(data, 0x00, 20);
    data[1] = mode;
    data[2] = 0x66;
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_PWRCTL, 20, data, buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_PWRCTL);
    return GNSS_ERR_OK;
}

static gnss_err_enum gnss_8122_cfg_save(int fd)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("CFG SAVE");
    uint8 data[8];
    memset(data, 0x00, 8);
    data[4] = 0x0f;
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_CFG, 8, data, buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_CFG);
    return GNSS_ERR_OK;
}

static gnss_err_enum gnss_8122_msgcfg(int fd, uint8 type, uint8 id, uint8 period)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("MSGCFG");
    uint8 data[3];
    data[0] = type;
    data[1] = id;
    data[2] = period;
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_MSG, 3, data, buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_MSG);
    return GNSS_ERR_OK;
}

static gnss_err_enum gnss_8122_ver(int fd)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("VER");
    int len = pack_create(HD8122_ID_TYPE_MON, HD8122_ID_MON_VER, 0, NULL, buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_MON, HD8122_ID_MON_VER);
    return GNSS_ERR_OK;
}


static gnss_err_enum gnss_8122_minel(int fd, uint8 *elev)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("ELEV");
    //uint8 elev_buff[4];
    //uint32_2_byte((uint32)elev, elev_buff, TRUE);
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_ELEV, 8, elev, buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_ELEV);
    return GNSS_ERR_OK;
}

static gnss_err_enum gnss_8122_PZ90CONV(int fd, uint8 *coord)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("PZ90CONV");
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_COORD, 44, coord, buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_COORD);
    return GNSS_ERR_OK;
}

#if 0
static gnss_err_enum gnss_8122_nmeaver(int fd, uint8 ver)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("NMEA-VER");
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_NMEAVER, 1, (uint8*)(&ver), buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    return GNSS_ERR_OK;
}
#endif

static gnss_err_enum gnss_8122_ephsave(int fd, uint8 status)
{
    uint8 buff[GNSS_PACK_BUFF_SIZE];
    LOGD("EPHSAVE");
    int len = pack_create(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_EPHSAVE, 1, (uint8*)(&status), buff, sizeof(buff));
    if(len <= 0)
    {
        LOGE("pack_create() fail.");
        return GNSS_ERR_ARG;
    }
    log_hex("PACK", buff, len);
    gnss_write(fd, buff, len);
    msg_insert(HD8122_ID_TYPE_CFG, HD8122_ID_CFG_EPHSAVE);
    return GNSS_ERR_OK;
}


int gnss_8122_dev_open()
{
    //return mbtk_gpio_value_set(GNSS_POWER_GPIO, MBTK_GPIO_DIRECT_OUT, 1);
    mbtk_system("i2cset -y -f 2 0x31 0x15 0x86");
    return 0;
}

int gnss_8122_dev_close(int fd)
{
    //return mbtk_gpio_value_set(GNSS_POWER_GPIO, MBTK_GPIO_DIRECT_OUT, 0);
    mbtk_system("i2cset -y -f 2 0x31 0x15 0x00");
    return 0;
}

int gnss_8122_open(const char *dev)
{
    pthread_mutex_init(&read_mutex, NULL);
    pthread_cond_init(&read_cond, NULL);
    return gnss_port_open(dev, O_RDWR | O_NONBLOCK | O_NOCTTY, UART_BITRATE_NMEA_DEF_FW, TRUE);
}

int gnss_8122_close(int fd)
{
    pthread_mutex_destroy(&read_mutex);
    pthread_cond_destroy(&read_cond);
    return gnss_port_close(fd);
}

int gnss_8122_init_set(int fd)
{
    sleep(1); //wait 8122 ready...
    gnss_8122_ephsave(fd, (uint8)HD8122_EPHSAVE_STATUS_DISABLE);
    gnss_8122_cfg_save(fd);
    return GNSS_ERR_OK;
}

int gnss_8122_fw_dl(int fd, const char *fw_name, const char *dev)
{
    //PORT_NAME234
    int uart_fd = open(dev , O_RDWR|O_NOCTTY);
    if (uart_fd < 0)
    {
        LOGE("open uart failed %d[%d]", uart_fd, errno);
        return GNSS_ERR_OPEN_DEV;
    }

    uint8_t fw_path[256] = {0};
    memset(fw_path, 0x0, 256);
    if(memcmp(fw_name, "gp_gl_ga", 8) == 0)
    {
        memcpy(fw_path, GNSS_FW_GQALS_PATH, strlen(GNSS_FW_GQALS_PATH));
    }
    else if(memcmp(fw_name, "gp_bd_ga", 8) == 0)
    {
        memcpy(fw_path, GNSS_FW_GAQBS_PATH, strlen(GNSS_FW_GAQBS_PATH));
    }
    else if(fw_name != NULL)
    {
        memcpy(fw_path, fw_name, strlen(fw_name));
        LOGE("fw_path: %s",fw_path);
    }

   
    else
    {
        LOGE("fw dl param error");
        return GNSS_ERR_ARG;
    }

    uint8_t *g_bin_buff = (uint8_t*)malloc(500*1024);
    if(g_bin_buff == NULL) {
        LOGE("malloc() fail : %d", errno);
        return GNSS_ERR_UNKNOWN;
    }
    memset(g_bin_buff, 0, 500*1024);
    uint32_t len = read_bin_file(fw_path, g_bin_buff);
    if (len <= 0)
    {
        LOGE("Read file failed ,len = %d", len);
        goto error;
    }

    int ret;
    // branch1：gp_gl_ga & gp_bd_ga
    if (memcmp(fw_name, "gp_gl_ga", 8) == 0 || memcmp(fw_name, "gp_bd_ga", 8) == 0) {
        if(gnss_8122_dev_open())
        {
            LOGE("open gnss device fail:%d", errno);
            goto error;
        }

        mbtk_system("echo 63 > /sys/class/gpio/export");
        mbtk_system("echo 62 > /sys/class/gpio/export");
        mbtk_system("echo out > /sys/class/gpio/gpio63/direction");
        mbtk_system("echo out > /sys/class/gpio/gpio62/direction");
        mbtk_system("echo 0 > /sys/class/gpio/gpio63/value");
        usleep(100000);//100ms
        mbtk_system("echo 1 > /sys/class/gpio/gpio62/value");//reset
        usleep(1000000);//1s
        mbtk_system("echo 0 > /sys/class/gpio/gpio62/value");
        usleep(100000);//100ms
        mbtk_system("echo 1 > /sys/class/gpio/gpio63/value");//boot
        mbtk_system("echo 62 > /sys/class/gpio/unexport");
        mbtk_system("echo 63 > /sys/class/gpio/unexport");


        ret = fw_update_boot_8122(uart_fd, g_bin_buff, len);
        if (ret < 0)
        {
            LOGE("fw_update_boot() fail : %d", ret);
            goto error;
        }

        if(gnss_8122_dev_close(0))
        {
            LOGE("close gnss device fail:%d", errno);
            goto error;
        }

        if(ret == HDBD_UPG_SUCESS)
        {
            LOGD("upgrade sucess!");
        }
        else
        {
            LOGD("upgrade FAIL, fail style:%d", ret);
            goto error;
        }

        free(g_bin_buff);
        uart_close(uart_fd);
        return GNSS_ERR_OK;
    }
    // branch2: else
    else {
        mbtk_system("echo 0 > /sys/class/gpio/gpio41/value");//41 gnss_power_en
        usleep(500000);//500ms

        mbtk_system("echo 1 > /sys/class/gpio/gpio120/value");//120 boot
        usleep(500000);//500ms
        mbtk_system("echo 1 > /sys/class/gpio/gpio41/value");//41 gnss_power_en
        usleep(500000);//500ms


        ret = fw_update_boot(uart_fd, g_bin_buff, len);
        if (ret < 0)
        {
            LOGE("fw_update_boot() fail : %d", ret);
            goto error;
        }

        if(ret == HDBD_UPG_SUCESS)
        {
            LOGD("upgrade sucess!");
        }
        else
        {
            LOGD("upgrade FAIL, fail style:%d", ret);
            goto error;
        }

        free(g_bin_buff);
        uart_close(uart_fd);
        return GNSS_ERR_OK;
    }

error:
    if(g_bin_buff)
    {
        free(g_bin_buff);
        g_bin_buff = NULL;
    }
    if(uart_fd > 0)
    {
        uart_close(uart_fd);
        uart_fd = -1;
    }
    return GNSS_ERR_DL_FW;
}

gnss_err_enum gnss_8122_agnss_get_eph(const char *param)
{
    if(param == NULL)
    {
        LOGD("gnss_8122_agnss_get_eph param is NULL");
        return GNSS_ERR_ARG;
    }

    int eph_type = 0;
    int alam_flag = 0;
    int ret = -1;
    char url[1024] = {0};
    if(2 == sscanf(param, "%d,%d", &eph_type, &alam_flag))
    {
        if((gnss_eph_data_enum)eph_type == GNSS_EPH_GPS)
        {
            snprintf(url, sizeof(url),AGNSS_URL, AGNSS_EPH_GPS);
        }
        else if((gnss_eph_data_enum)eph_type == GNSS_EPH_BDS)
        {
            snprintf(url, sizeof(url),AGNSS_URL, AGNSS_EPH_BDS);
        }
        else if((gnss_eph_data_enum)eph_type == GNSS_EPH_GLO)
        {
            snprintf(url, sizeof(url),AGNSS_URL, AGNSS_EPH_GLO);
        }
        else if((gnss_eph_data_enum)eph_type == GNSS_EPH_GPS_BDS)
        {
            snprintf(url, sizeof(url),AGNSS_URL, AGNSS_EPH_GPS_BDS);
        }
        else if((gnss_eph_data_enum)eph_type == GNSS_EPH_GPS_GLO)
        {
            snprintf(url, sizeof(url),AGNSS_URL, AGNSS_EPH_GPS_GLO);
        }
        else if((gnss_eph_data_enum)eph_type == GNSS_EPH_CFG)
        {
            snprintf(url, sizeof(url),"%s?compid=%s&token=%s", strlen(agps_info.host) > 0 ? agps_info.host : "http://uagnss.allystar.com:80/ephemeris/HD_GPS.hdb",
                                                        strlen(agps_info.id) > 0 ? agps_info.id : "yikecs1",
                                                        strlen(agps_info.passwd) > 0 ? agps_info.passwd : "Z38w5urAuawubTxi");
        }
        else
        {
            return GNSS_ERR_UNSUPPORT;
        }

        ret = eph_data_from_http_get(url, AGNSS_EPH_FILE_PATH);
        if (ret < 0)
        {
            LOGD("eph_data_from_http_get fail");
            return GNSS_ERR_EPH_GET_FAIL;
        }
        LOGD("get_eph_data_from_http success");
    }
    else
    {
        LOGD("param num error");
        return GNSS_ERR_ARG;
    }
    return GNSS_ERR_OK;
}


gnss_err_enum gnss_8122_agnss_inject(int fd)
{
    int ret = 0;
    hd_set_gnss_dev_fd(fd);
    ret = hd_agnss_inject(AGNSS_EPH_FILE_PATH, 0, 0, 0, 0, NULL);
    hd_set_gnss_dev_fd(-1);
    if(ret < 0)
    {
        LOGD("hd_agnss_inject fail");
        return GNSS_ERR_EPH_INJECT_FAIL;
    }
    return GNSS_ERR_OK;
}

void gnss_8122_set_cb(const void *data, int data_len)
{
//    const char *buff = (const char*)data;
    if(setting_busy)   // Has setting cmd process.
    {
        gnss_cmd_rsp_process(data, data_len);
    }

#if 0
    if(hd_get_eph_inject_status() == HD_EPH_INJECT_STATUS_WAIT_RETURN)
    {
        log_hex("EPH_RSP", (const char*)data, data_len);
        int ret = hd_eph_inject_result_check(data, data_len);
        if(ret < 0)
        {
            hd_set_eph_inject_status(HD_EPH_INJECT_STATUS_FAIL);
        }
        else
        {
            hd_set_eph_inject_status(HD_EPH_INJECT_STATUS_SUCCESS);
        }
    }
#endif
}

gnss_err_enum gnss_8122_set(int fd, const char *cmd, void *cmd_rsp, int cmd_rsp_len)
{
    if(setting_busy)
    {
        return GNSS_ERR_SET_BUSY;
    }
    else
    {
        bool should_wait_rsp = TRUE;
        setting_busy = TRUE;
        gnss_set_rsp_ptr = cmd_rsp;
        gnss_set_result = GNSS_ERR_OK;
        msg_init();
        mbtk_timer_set(gnss_set_timer_cb, GNSS_SET_TIMEOUT);

        if(memcmp(cmd, "$RESET", 6) == 0)   // $RESET,<mode>
        {
            gnss_reset_type_enum mode = (gnss_reset_type_enum)atoi(cmd + 7);
            LOGD("set reset: %d", mode);
            if(mode == GNSS_RESET_TYPE_HOT)
            {
                gnss_set_result = gnss_8122_reset(fd, 3);
            }
            else if(mode == GNSS_RESET_TYPE_WARM)
            {
                gnss_set_result = gnss_8122_reset(fd, 2);
            }
            else if(mode == GNSS_RESET_TYPE_COLD)
            {
                gnss_set_result = gnss_8122_reset(fd, 1);
            }
            else
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }
            if(gnss_set_result != GNSS_ERR_OK)
            {
                goto set_fail;
            }
            should_wait_rsp = FALSE;
        }
        else if(memcmp(cmd, "$SYSCFG", 7) == 0)     // $SYSCFG,<mode>
        {
            uint32 mode = 0;
            mode = (uint32)atoi(cmd + 8);
            uint32 new_mode = 0;
            if(((GNSS_SET_SYSCFG_GPS | GNSS_SET_SYSCFG_BDS | GNSS_SET_SYSCFG_GLO | GNSS_SET_SYSCFG_GAL | GNSS_SET_SYSCFG_SBAS | GNSS_SET_SYSCFG_QZSS) & mode) != mode)
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }

            if(mode & GNSS_SET_SYSCFG_GPS)   // GPS
            {
                new_mode |= 0x00000001;
            }
            if(mode & GNSS_SET_SYSCFG_BDS)   // BDS
            {
                new_mode |= 0x00000004;
            }
            if(mode & GNSS_SET_SYSCFG_GLO)   // GLO
            {
                new_mode |= 0x00000002;
            }
            if(mode & GNSS_SET_SYSCFG_GAL)   // GAL
            {
                new_mode |= 0x00000010;
            }
            if(mode & GNSS_SET_SYSCFG_QZSS)   // QZSS
            {
                new_mode |= 0x00000020;
            }
            if(mode & GNSS_SET_SYSCFG_SBAS)   // SBAS
            {
                new_mode |= 0x00000040;
            }

            gnss_set_result = gnss_8122_syscfg(fd, new_mode);
            if(gnss_set_result != GNSS_ERR_OK)
            {
                goto set_fail;
            }
            should_wait_rsp = TRUE;
        }
        else if(memcmp(cmd, "$FREQCFG", 8) == 0)        //$FREQCFG,<freq>
        {
            uint32 freq = 0;
            freq = (uint32)atoi(cmd + 9);
            LOGD("set freq: %d", freq);
            if((GNSS_SET_FREQCFG_1 != freq) && (GNSS_SET_FREQCFG_2 != freq) && (GNSS_SET_FREQCFG_5 != freq))
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }
            gnss_set_result = gnss_8122_freqcfg(fd, (uint8)freq);
            if(gnss_set_result != GNSS_ERR_OK)
            {
                goto set_fail;
            }

            gnss_set_result = gnss_8122_cfg_save(fd);
            should_wait_rsp = TRUE;
        }
        else if(memcmp(cmd, "$MSGCFG", 7) == 0)     // $MSGCFG,<mode>,<rate>
        {
            uint32 mode;
            int rate;
            if(2 == sscanf(cmd, "$MSGCFG,%d,%d", &mode, &rate))
            {
                LOGD("set msgcfg: %d, %d", mode, rate);
                int time = rate / 1000; // s
                if(time < 0)
                {
                    gnss_set_result = GNSS_ERR_ARG;
                    goto set_fail;
                }

                if(((GNSS_SET_MSGCFG_RMC | GNSS_SET_MSGCFG_VTG | GNSS_SET_MSGCFG_GGA | GNSS_SET_MSGCFG_GSA
                     | GNSS_SET_MSGCFG_GRS | GNSS_SET_MSGCFG_GSV | GNSS_SET_MSGCFG_GLL | GNSS_SET_MSGCFG_ZDA
                     | GNSS_SET_MSGCFG_GST | GNSS_SET_MSGCFG_TXT) & mode) != mode)
                {
                    LOGD("msgcfg not support mode");
                    gnss_set_result = GNSS_ERR_ARG;
                    goto set_fail;
                }

                if(mode & GNSS_SET_MSGCFG_RMC)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x05, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_VTG)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x06, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_GGA)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x00, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_GSA)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x02, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_GRS)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x03, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_GSV)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x04, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_GLL)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x01, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_ZDA)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x07, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_GST)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x08, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }

                if(mode & GNSS_SET_MSGCFG_TXT)
                {
                    gnss_set_result = gnss_8122_msgcfg(fd, 0xF0, 0x20, time);
                    if(gnss_set_result != GNSS_ERR_OK)
                    {
                        goto set_fail;
                    }
                }
            }
            else
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }

            should_wait_rsp = TRUE;
        }
        else if(memcmp(cmd, "$AGPSCFG", 8) == 0)     // $AGPSCFG,<host>,<id>,<passwd>
        {
            char host[GNSS_AGPS_LEN_MAX] = {0};
            char id[GNSS_AGPS_LEN_MAX] = {0};
            char passwd[GNSS_AGPS_LEN_MAX] = {0};
            if(3 == sscanf(cmd, "$AGPSCFG,%[^,],%[^,],%s", host, id, passwd))
            {
                LOGD("agps: %s, %s, %s", host, id, passwd);
                memset(agps_info.host, 0x0, GNSS_AGPS_LEN_MAX);
                if(memcmp(host, "NULL", 4))
                {
                    memcpy(agps_info.host, host, strlen(host));
                }

                memset(agps_info.id, 0x0, GNSS_AGPS_LEN_MAX);
                if(memcmp(id, "NULL", 4))
                {
                    memcpy(agps_info.id, id, strlen(id));
                }

                memset(agps_info.passwd, 0x0, GNSS_AGPS_LEN_MAX);
                if(memcmp(passwd, "NULL", 4))
                {
                    memcpy(agps_info.passwd, passwd, strlen(passwd));
                }
            }
            else
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }

            should_wait_rsp = FALSE;
        }
        else if(memcmp(cmd, "$VER", 4) == 0)     // $VER
        {
            gnss_set_cmd_rsp = (char *)cmd_rsp;
            gnss_cmd_rsp_len = cmd_rsp_len;
            gnss_set_result = gnss_8122_ver(fd);
            if(gnss_set_result != GNSS_ERR_OK)
            {
                goto set_fail;
            }
        }
        else if(memcmp(cmd, "$MINEL", 6) == 0)     // $MINEL,<elev>
        {
            uint32 elev = 0;
            elev = (uint32)atoi(cmd + 7);
            LOGD("set minel: %d", elev);
            if(GNSS_SET_EVEL_5 == elev)
            {
                uint8 elev_buff[8] = {0x35, 0xFA, 0x8E, 0x3C, 0xC2, 0xB8, 0xB2, 0x3D};
                gnss_set_result = gnss_8122_minel(fd, elev_buff);
                if(gnss_set_result != GNSS_ERR_OK)
                {
                    goto set_fail;
                }
            }
            else if(GNSS_SET_EVEL_15 == elev)
            {
                uint8 elev_buff[8] = {0x35, 0xFA, 0x8E, 0x3C, 0x92, 0x0A, 0x86, 0x3E};
                gnss_set_result = gnss_8122_minel(fd, elev_buff);
                if(gnss_set_result != GNSS_ERR_OK)
                {
                    goto set_fail;
                }
            }
            else
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }
        }
        else if(memcmp(cmd, "$PZ90CONV", 9) == 0)     // $PZ90CONV,local,ref
        {
            int local = 0;
            int ref = 0;
            uint8 buff[44] = {0x00, 0x00, 0x00, 0x00, 0xA6, 0x54, 0x58, 0x41,
                              0xCD, 0x8C, 0x8F, 0xCC, 0x56, 0x77, 0x6B, 0x3F,
                              0x8F, 0xC2, 0xF5, 0xBC, 0x0A, 0xD7, 0x23, 0xBC,
                              0x00, 0x00, 0x00, 0x00, 0x30, 0x62, 0x9F, 0x37,
                              0x28, 0x29, 0x30, 0xB8, 0xBD, 0x37, 0x06, 0x36,
                              0x00, 0x00, 0x00, 0x00};

            if(2 == sscanf(cmd, "$PZ90CONV,%d,%d", &local, &ref))
            {
                LOGD("set pz90conv: %d, %d", local, ref);


                if(local == 1 && ref == 0) //PZ90
                {
                    //
                }
                else if(local == 0 && ref == 0)
                {
                    memset(buff, 0x00, 44);
                }
                else
                {
                    gnss_set_result = GNSS_ERR_UNSUPPORT;
                    goto set_fail;
                }
            }
            else
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }

            gnss_set_result = gnss_8122_PZ90CONV(fd, buff);
            if(gnss_set_result != GNSS_ERR_OK)
            {
                goto set_fail;
            }
        }
        else if(memcmp(cmd, "$NMEACFG", 8) == 0)     // $NMEACFG,<ver>
        {
#if 0
            gnss_memaver_type_enum version = (gnss_memaver_type_enum)atoi(cmd + 9);
            if(version == GNSS_MEMAVER_TYPE_3_0)
            {
                gnss_set_result = gnss_8122_nmeaver(fd, 1);
            }
            else if(version == GNSS_MEMAVER_TYPE_4_0)
            {
                gnss_set_result = gnss_8122_nmeaver(fd, 2);
            }
            else if(version == GNSS_MEMAVER_TYPE_4_1)
            {
                gnss_set_result = gnss_8122_nmeaver(fd, 3);
            }
            else
            {
                gnss_set_result = GNSS_ERR_ARG;
                goto set_fail;
            }
            if(gnss_set_result != GNSS_ERR_OK)
            {
                goto set_fail;
            }
            should_wait_rsp = FALSE;
#else
            gnss_set_result = GNSS_ERR_UNSUPPORT;
            goto set_fail;
#endif
        }
        else
        {
            LOGW("Unknown cmd:%s", cmd);
            gnss_set_result = GNSS_ERR_UNSUPPORT;
            goto set_fail;
        }

//    set_success:
        if(should_wait_rsp)
        {
            setting_waitting = TRUE;
            pthread_mutex_lock(&read_mutex);
            pthread_cond_wait(&read_cond, &read_mutex);
            pthread_mutex_unlock(&read_mutex);
        }
        else
        {
            mbtk_timer_clear();
        }

        if(gnss_set_cmd_rsp != NULL)
        {
            gnss_set_cmd_rsp = NULL;
            gnss_cmd_rsp_len = 0;
        }
        setting_busy = FALSE;
        return gnss_set_result;
    set_fail:
        if(gnss_set_cmd_rsp != NULL)
        {
            gnss_set_cmd_rsp = NULL;
            gnss_cmd_rsp_len = 0;
        }
        setting_busy = FALSE;
        mbtk_timer_clear();
        return gnss_set_result;
    }
}


