/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <string.h>
#include <mtk_wrapper.h>
#include <uart_cust.h>
#include "uart_sw.h"
#include "uart_hw.h"

static const unsigned long uart_paddr[DRV_SUPPORT_UART_PORTS] = {
    UART0_BASE, UART1_BASE, UART2_BASE,
    UART3_BASE, UART4_BASE, UART5_BASE,
};

static const uint32_t uart_irq[DRV_SUPPORT_UART_PORTS] = {
    UART0_IRQ_ID, UART1_IRQ_ID, UART2_IRQ_ID,
    UART3_IRQ_ID, UART4_IRQ_ID, UART5_IRQ_ID,
};

static struct uart_info uart_port_info[DRV_SUPPORT_UART_PORTS];

void __uart_port_init(uint32_t port_line)
{
    struct uart_info *uart_port;

    uart_port = &uart_port_info[port_line];

    uart_port->irq = uart_irq[port_line];
    uart_port->line = port_line;
    uart_port->dropped = 0;
    uart_port->status = UART_INIT;
    uart_port->base_addr = plat_wrap_paddr_to_kvaddr(uart_paddr[port_line]);
}

int __uart_xmit_init(struct uart_info *uart_port, char *tx_buff,
                   uint32_t tx_buf_len, char *rx_buff, uint32_t rx_buf_len)
{
    struct circ_buf *xmit;
    dprintf(ULOG_INFO,"%s with tx_buff:%p, rx_buff:%p\n", __func__, tx_buff, rx_buff);

    if (tx_buff == NULL) {
        dprintf(ULOG_INFO, "tx_buff is NULL!\n");
        return EFAULT_POINT;
    }

    if (rx_buff == NULL) {
        dprintf(ULOG_INFO, "rx_buff is NULL!\n");
        return EFAULT_POINT;
    }

    memset(tx_buff, 0x0 , tx_buf_len);
    memset(rx_buff, 0x0 , rx_buf_len);

    /* init tx circle */
    xmit = &uart_port->tx_circ;
    xmit->buf = tx_buff;
    xmit->read = 0;
    xmit->write = 0;
    xmit->size = tx_buf_len;

    /* init rx circle */
    xmit = &uart_port->rx_circ;
    xmit->buf = rx_buff;
    xmit->read = 0;
    xmit->write = 0;
    xmit->size = rx_buf_len;

    return 0;
}

#if UART_DEBUG
void __uart_circle_debug(struct circ_buf *xmit)
{
    dprintf(ULOG_ALWAYS, "write is %d ", xmit->write);
    dprintf(ULOG_ALWAYS, "read is %d ", xmit->read);
    dprintf(ULOG_ALWAYS, "size is %d ", xmit->size);
    dprintf(ULOG_ALWAYS, "CIRC_AVAIL is %d ", CIRC_AVAIL(xmit));
    dprintf(ULOG_ALWAYS, "CIRC_LEFT is %d ", CIRC_LEFT(xmit));
    dprintf(ULOG_ALWAYS, "CIRC_IS_EMPTY is %d ", CIRC_IS_EMPTY(xmit));
    dprintf(ULOG_ALWAYS, "CIRC_IS_FULL is %d \n", CIRC_IS_FULL(xmit));

    dprintf(ULOG_ALWAYS, "write data is %d ", *(xmit->buf + xmit->write - 1));
    dprintf(ULOG_ALWAYS, "read data is %d \n", *(xmit->buf + xmit->read));
}
#endif

void __uart_ls_handler(struct uart_info *uart_port)
{
    struct uart_info *info;
    uint32_t lsr = 0;
    uint32_t port_line = 0;
    uint32_t *uart_addr = NULL;

    info = uart_port;
    port_line = info->line;
    uart_addr = info->base_addr;

    lsr = UART_READ32(UART_LSR(uart_addr));
    if (lsr & 0x2)
        dprintf(ULOG_ALWAYS, "UART%d OE+\n", port_line);
}

void __uart_receive_handler(struct uart_info *uart_port, uint32_t lsr)
{
    struct uart_info *info;
    struct circ_buf *xmit = NULL;
    uint32_t room_left = 0;
    uint8_t rx_data = 0;
    uint32_t dropped = 0;
    uint32_t *uart_addr = NULL;

    info = uart_port;
    uart_addr = info->base_addr;
    xmit = &info->rx_circ;

    dprintf(ULOG_INFO, "%s CIRC_IS_FULL:%d,lsr:0x%x\n", __func__, CIRC_IS_FULL(xmit), lsr);

    /* Need drop data */
    if (CIRC_IS_FULL(xmit)) {
        while (1) {
            dropped += 1;
            UART_READ32(UART_RBR(uart_addr));
            lsr = UART_READ32(UART_LSR(uart_addr));
            if (lsr & UART_LSR_DR)
                dropped++;
            else {
                info->dropped += dropped;
                dprintf(ULOG_ALWAYS, "BUFF_FULL[%d] total drop:%d\n", xmit->size, info->dropped);
                return;
            }
        }
    }

    room_left = CIRC_LEFT(xmit);
    while (room_left) {
        if (lsr & UART_LSR_OE)
            dprintf(ULOG_ALWAYS, "OE\n");

        if (lsr & UART_LSR_DR) {
            rx_data = UART_READ8(UART_RBR(uart_addr));
            CIRC_PUSH(xmit, rx_data);
            room_left = CIRC_LEFT(xmit);
            lsr = UART_READ32(UART_LSR(uart_addr));
            dprintf(ULOG_INFO, "%s, RX:0x%x,room_left:%d, lsr:0x%x\n",
                    __func__, rx_data, room_left, lsr);
        } else
            break;
    }
}

void __uart_transmit_handler(struct uart_info *uart_port)
{
    struct uart_info *info;
    struct circ_buf *xmit = NULL;
    uint32_t *uart_addr = NULL;
    uint32_t fifo_size, real_count, index;
    uint8_t tx_data;

    info = uart_port;
    uart_addr = info->base_addr;
    xmit = &info->tx_circ;

    fifo_size = UART_TX_FIFO_LENGTH;
    real_count = CIRC_AVAIL(xmit);

    dprintf(ULOG_INFO, "%s real_count:%d\n", __func__, real_count);
    real_count = min(real_count, fifo_size);

    for (index = 0; index < real_count; index++) {
        CIRC_POP(xmit, &tx_data);
        UART_WRITE8(UART_THR(uart_addr), tx_data);
    }

    /* Switch off TX interrupt if TX buffer empty */
    if (CIRC_IS_EMPTY(xmit) == true)
        UART_WRITE32(UART_IER(uart_addr), IER_HW_ONTY_RX);
}

void __uart_ms_handler(struct uart_info *uart_port)
{
    struct uart_info *info;
    uint32_t msr = 0;
    uint32_t port_line = 0;
    uint32_t *uart_addr = NULL;

    info = uart_port;
    port_line = info->line;
    uart_addr = info->base_addr;

    msr = UART_READ32(UART_MSR(uart_addr));
    dprintf(ULOG_INFO, "UART%d msr:0x%x\n", port_line, msr);
}

static enum wrap_handler_return __uart_irq_handle(void *arg)
{
    struct uart_info *info = arg;
    uint32_t irq = info->irq;
    uint32_t port_line = info->line;
    void *uart_addr = info->base_addr;
    uint32_t iir = 0, lsr = 0;

    plat_wrap_mask_interrupt(irq);

    iir = UART_READ32(UART_IIR(uart_addr));

    /* need read LSR here, otherwise will have rx read timeout error! */
    lsr = UART_READ32(UART_LSR(uart_addr));

    dprintf(ULOG_INFO, "IRQ handle, UART%d irq:%d, iir:0x%x, lsr:0x%x, IER:0x%x\n",
            port_line, irq, iir, lsr, UART_READ32(UART_IER(uart_addr)));
    if (iir & UART_IIR_INT_INVALID) {
        plat_wrap_unmask_interrupt(irq);
        return WRAP_INT_NO_RESCHEDULE;
    }

    if (lsr & (UART_LSR_DR | UART_LSR_BI)) {
        dprintf(ULOG_DEBUG, "IIR_RDA\n");
        __uart_receive_handler(info, lsr);
    }

    if (iir & UART_IIR_MS) {
        dprintf(ULOG_INFO, "MS\n");
        __uart_ms_handler(info);
    }

    if (lsr & UART_LSR_THRE) {
        dprintf(ULOG_INFO, "THRE\n");
        __uart_transmit_handler(info);
    }

    plat_wrap_unmask_interrupt(irq);

    return WRAP_INT_RESCHEDULE;
}

void __uart_register_interrupt(struct uart_info *uart_port)
{
    uint32_t irq = 0;

    irq = uart_port->irq;

    plat_wrap_irq_set_sensitive(irq, WRAP_LEVEL_SENSITIVE);
    plat_wrap_irq_set_polarity(irq, WRAP_POLARITY_LOW);
    plat_wrap_register_int_handler(irq, &__uart_irq_handle, uart_port);
    plat_wrap_unmask_interrupt(irq);
}

void __uart_baudrate_config(struct uart_info *uart_port)
{
    void *addr;
    unsigned int baudrate;

    int highspeed, quot;
    int sample, sample_count, sample_point;
    int dll, dlh;
    unsigned int uart_clk = UART_CLOCK_RATE;

    addr = uart_port->base_addr;
    baudrate = uart_port->baudrate;

    if (baudrate <= 115200) {
        highspeed = 0;
        quot = DIV_ROUND_CLOSEST(uart_clk, 16 * baudrate);
    } else if (baudrate <= 576000) {
        highspeed = 2;
        if ((baudrate == 500000) || (baudrate == 576000))
            baudrate = 460800;
        quot = DIV_ROUND_UP(uart_clk, 4 * baudrate);
    } else {
        highspeed = 3;
        quot = DIV_ROUND_UP(uart_clk, 256 * baudrate);
    }

    sample = DIV_ROUND_CLOSEST(uart_clk, quot * baudrate);
    sample_count = sample - 1;
    sample_point = (sample - 2) >> 1;
    dll = quot & 0xff;
    dlh = quot >> 8;

    dprintf(ULOG_INFO, "baudrate:0x%x, highspeed:0x%x, sample_count:0x%x, sample_point:0x%x, dll:0x%x, dlh:0x%x\n",
            baudrate, highspeed, sample_count, sample_point, dll, dlh);

    UART_WRITE32(UART_HIGHSPEED(addr), highspeed);

    UART_WRITE32(UART_LCR(addr), UART_LCR_DLAB); // set LCR to DLAB to set DLL,DLH
    UART_WRITE32(UART_DLL(addr), dll);
    UART_WRITE32(UART_DLH(addr), dlh);
    UART_WRITE32(UART_LCR(addr), UART_WLS_8); //word length 8
    UART_WRITE32(UART_SAMPLE_COUNT(addr), sample_count);
    UART_WRITE32(UART_SAMPLE_POINT(addr), sample_point);

    dprintf(ULOG_INFO, "addr:%p, UART_LCR:0x%x, highspeed:0x%x, sample_count:0x%x, sample_point:0x%x\n",
            addr, UART_READ32(UART_LCR(addr)), UART_READ32(UART_HIGHSPEED(addr)),
            UART_READ32(UART_SAMPLE_COUNT(addr)), UART_READ32(UART_SAMPLE_POINT(addr)));
}

#if UART_USING_DMA
#include "uart_dma_sw.h"
#include "uart_dma_hw.h"
void __uart_enable_dma(struct uart_info *uart_port, char *dma_buff,
                   uint32_t dma_len)
{
    void *uart_addr = uart_port->base_addr;

    UART_WRITE32(UART_DMA_EN(uart_addr),
    	UART_RX_DMA_EN|UART_TX_DMA_EN|UART_TIMEROUT_AUTOSET);
    __uart_dma_start(uart_port, dma_buff, dma_len);

    dprintf(ULOG_INFO, "UART%d enable DMA finish, DMA_EN:0x%x.\n",
            uart_port->line, UART_READ32(UART_DMA_EN(uart_addr)));
}
#endif

int uart_cust_init(UART_PORT_e port_line, char *tx_buff, uint32_t tx_buf_len,
                   char *rx_buff, uint32_t rx_buf_len)
{
    int ret = 0;
    struct uart_info *uart_port;

    if (port_line >= HW_SUPPORT_UART_PORTS) {
        dprintf(ULOG_INFO, "UART%d not exist!\n", port_line);
        return EINVAL_PORT_NUM;
    }

    memset(uart_port_info, 0x0, sizeof(uart_port_info));
    uart_port = &uart_port_info[port_line];
    __uart_port_init(port_line);

    ret = __uart_xmit_init(uart_port, tx_buff, tx_buf_len, rx_buff, rx_buf_len);
    if (ret != 0)
        uart_port->status = UART_UNINIT;

    return ret;
}

int uart_cust_config(UART_PORT_e port_line, uint32_t baudrate,
                   char *dma_buff, uint32_t dma_len)
{
    void *uart_addr = NULL;
    struct uart_info *uart_port;

    if (port_line >= HW_SUPPORT_UART_PORTS) {
        dprintf(ULOG_INFO, "UART%d not exist!\n", port_line);
        return EINVAL_PORT_NUM;
    }

    uart_port = &uart_port_info[port_line];
    if (uart_port->status != UART_INIT) {
        dprintf(ULOG_INFO, "UART%d not configured!\n", port_line);
        return EUNINIT_PORT;
    }

    uart_port->baudrate = baudrate;
    uart_addr = uart_port->base_addr;
    dprintf(ULOG_INFO, "UART%d addr %p\n", port_line, uart_addr);
    __uart_baudrate_config(uart_port);

    UART_WRITE32(UART_MCR(uart_addr), UART_MCR_Normal); //for hw flow control
    UART_WRITE32(UART_IER(uart_addr), UART_IER_ALLOFF); //disable all INT
    UART_WRITE32(UART_FCR(uart_addr), UART_FCR_NORMAL_TRIG); //enable FIFO

    /* register interrupt and only enable RX interrupt */
    __uart_register_interrupt(uart_port);
    UART_WRITE32(UART_IER(uart_addr), IER_HW_ONTY_RX);

#if UART_USING_DMA
    if (dma_buff == NULL) {
        dprintf(ULOG_INFO, "dma_buff is NULL!\n");
        return EFAULT_POINT;
    }

    if ((dma_len < UART_DMA_VFF_LEN_MIN) || (dma_len > UART_DMA_VFF_LEN_MAX)) {
        dprintf(ULOG_INFO, "dma_len[%d] should between [%d,%d]!\n",
                dma_len, UART_DMA_VFF_LEN_MIN, UART_DMA_VFF_LEN_MAX);
        return EOUT_RANGE;
    }

    __uart_enable_dma(uart_port, dma_buff, dma_len);
#endif

    return 0;
}

int uart_cust_tx(UART_PORT_e port_line, const uint8_t *pbuf, uint32_t size)
{
    struct uart_info *uart_port;
    struct circ_buf *xmit = NULL;
    void *uart_addr = NULL;
    int tmp = 0;
    uint32_t left = 0, write_cnt = 0;

    if (port_line >= HW_SUPPORT_UART_PORTS) {
        dprintf(ULOG_INFO, "UART%d not exist!\n", port_line);
        return EINVAL_PORT_NUM;
    }

    if (pbuf == NULL) {
        dprintf(ULOG_INFO, "pbuf is NULL!\n");
        return EFAULT_POINT;
    }

    uart_port = &uart_port_info[port_line];
    uart_addr = uart_port->base_addr;
    xmit = &uart_port->tx_circ;

    if (uart_port->baudrate == 0) {
        dprintf(ULOG_INFO, "UART%d not configured!\n", port_line);
        return EUNCONFIGURE;
    }

    left = CIRC_LEFT(xmit);
    write_cnt = size > left ? left : size;

    /* push data to xmit dircetly, then trigger GIC INT */
    if (xmit->write + write_cnt < xmit->size) {
        dprintf(ULOG_INFO, "UART%d write point:%d, write_cnt:%d\n",
                port_line, xmit->write, write_cnt);
        memcpy(xmit->buf + xmit->write, pbuf, write_cnt);
        xmit->write += write_cnt;
    } else {
        tmp = xmit->size - xmit->write;
        dprintf(ULOG_INFO, "UART%d write point:%d, write_cnt:%d, tmp:%d\n",
                port_line, xmit->write, write_cnt, tmp);
        memcpy(xmit->buf + xmit->write, pbuf, tmp);
        memcpy(xmit->buf, pbuf + tmp, write_cnt - tmp);

        xmit->write = xmit->write + write_cnt - xmit->size;
        dprintf(ULOG_INFO, "UART%d write point:%d\n", port_line, xmit->write);
    }

#if UART_USING_DMA
    /* enable DMA TX interrupt */
    __uart_dma_enable_tx_INT(port_line);
#else
    /* enable UART TX interrupt */
    UART_WRITE32(UART_IER(uart_addr), IER_HW_NORMALINTS);
#endif

    dprintf(ULOG_INFO, "UART%d [%p] with write count:%d\n",
            port_line, uart_addr, write_cnt);

    return write_cnt;
}

int uart_cust_rx(UART_PORT_e port_line, uint8_t *pbuf, uint32_t read_size)
{
    struct uart_info *uart_port = NULL;
    struct circ_buf *xmit = NULL;
    uint32_t read_cnt, avail = 0, cnt;

    if (port_line >= HW_SUPPORT_UART_PORTS) {
        dprintf(ULOG_INFO, "UART%d not exist!\n", port_line);
        return EINVAL_PORT_NUM;
    }
    if (pbuf == NULL) {
        dprintf(ULOG_INFO, "buf is NULL!\n");
        return EFAULT_POINT;
    }

    uart_port = &uart_port_info[port_line];
    xmit = &uart_port->rx_circ;

    if (uart_port->baudrate == 0) {
        dprintf(ULOG_INFO, "UART%d not configured!\n", port_line);
        return EUNCONFIGURE;
    }

    avail = CIRC_AVAIL(xmit);
    cnt = read_cnt = avail < read_size ? avail : read_size;
    dprintf(ULOG_INFO, "%s return count is %d\n", __func__, cnt);

    while (cnt--) {
        CIRC_POP(xmit, pbuf);
        pbuf++;
    }

    dprintf(ULOG_INFO, "%s read size:%d, avail:%d, read_cnt:%d\n",
            __func__, read_size, avail, read_cnt);

    return read_cnt;
}
