/*
 * Copyright (c) 2018 MediaTek Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <string.h>
#include <mtk_wrapper.h>
#include "uart_dma_sw.h"
#include "uart_dma_hw.h"

#if UART_USING_DMA

static const unsigned long uart_dma_paddr[DRV_SUPPORT_UART_PORTS][2] = {
    {VFF_BASE_CH_0, VFF_BASE_CH_1},
    {VFF_BASE_CH_2, VFF_BASE_CH_3},
    {VFF_BASE_CH_4, VFF_BASE_CH_5},
    {VFF_BASE_CH_6, VFF_BASE_CH_7},
    {VFF_BASE_CH_8, VFF_BASE_CH_9},
    {VFF_BASE_CH_10, VFF_BASE_CH_11},
};

static const uint32_t uart_dma_irq[DRV_SUPPORT_UART_PORTS][2] = {
    {VFF_UART0_TX_IRQ_ID, VFF_UART0_RX_IRQ_ID},
    {VFF_UART1_TX_IRQ_ID, VFF_UART1_RX_IRQ_ID},
    {VFF_UART2_TX_IRQ_ID, VFF_UART2_RX_IRQ_ID},
    {VFF_UART3_TX_IRQ_ID, VFF_UART3_RX_IRQ_ID},
    {VFF_UART4_TX_IRQ_ID, VFF_UART4_RX_IRQ_ID},
    {VFF_UART5_TX_IRQ_ID, VFF_UART5_RX_IRQ_ID},
};

static struct channel_info dma_channel_info[DRV_SUPPORT_VFF_CHANNEL];

struct channel_info *__uart_dma_init(uint32_t port_line, uint32_t dir)
{
    struct channel_info *channel;

    channel = &dma_channel_info[port_line*2 + dir];
    if (dir == DMA_DIR_TX) {
        channel->irq = uart_dma_irq[port_line][0];
        channel->line = port_line;
        channel->status = DMA_INIT;
        channel->direction = dir;
        channel->base_addr = plat_wrap_paddr_to_kvaddr(uart_dma_paddr[port_line][0]);
    } else if (dir == DMA_DIR_RX) {
        channel->irq = uart_dma_irq[port_line][1];
        channel->line = port_line;
        channel->status = DMA_INIT;
        channel->direction = dir;
        channel->base_addr = plat_wrap_paddr_to_kvaddr(uart_dma_paddr[port_line][1]);
    } else {
        dprintf(ULOG_INFO, "UART%d with wrong dir[%d]\n", port_line, dir);
        channel = NULL;
    }

    dprintf(ULOG_INFO, "%s with irq:%d, dir:%d, base_addr:%p\n",
            __func__, channel->irq, dir, channel->base_addr);

    return channel;
}

int __uart_dma_config(struct channel_info *channel, char *dma_buff,
                   uint32_t dma_len)
{
    uint32_t dir = 0;
    uint32_t buff_size = 0;
    char *dma_addr = NULL;
    void *chan_addr = NULL;
    struct uart_info *uart_port = NULL;
    struct circ_buf *xmit = NULL;

    if (channel == NULL) {
        dprintf(ULOG_INFO, "channel is NULL!\n");
        return -1;
    }

    uart_port = (struct uart_info *)channel->port_info;
    chan_addr = channel->base_addr;
    dir = channel->direction;

    if (dir == DMA_DIR_TX) {
        /* TX DMA buff share with xmit buff */
        xmit = &uart_port->tx_circ;
        buff_size = xmit->size;
        dma_addr = xmit->buf;
    } else if (dir == DMA_DIR_RX) {
        dma_addr = dma_buff;
        buff_size = dma_len;
    } else {
        return -2;
    }

    if (dma_addr == NULL) {
        dprintf(ULOG_INFO, "dma_addr is NULL!\n");
        return -3;
    }

    dprintf(ULOG_INFO, "%s with buff_size:%d, dir:%d, buff addr:%p, chan_addr:%p\n",
            __func__, buff_size, dir, dma_addr, chan_addr);

    channel->mem_base = (uintptr_t)dma_addr;
    channel->mem_length = buff_size;

    if (dir == DMA_DIR_TX) {
        DMA_WRITE32(VFF_ADDR(chan_addr), (uint32_t)(uintptr_t)dma_addr);
        DMA_WRITE32(VFF_LEN(chan_addr), buff_size);
        DMA_WRITE32(VFF_THRE(chan_addr), VFF_TX_THRE(buff_size));

        DMA_WRITE32(VFF_WPT(chan_addr), 0);
        DMA_WRITE32(VFF_INT_FLAG(chan_addr), VFF_TX_INT_FLAG_CLR_B);
        DMA_WRITE32(VFF_EN(chan_addr), VFF_EN_B);
    } else if (dir == DMA_DIR_RX) {
        DMA_WRITE32(VFF_ADDR(chan_addr), (uint32_t)(uintptr_t)dma_addr);
        DMA_WRITE32(VFF_LEN(chan_addr), buff_size);
        DMA_WRITE32(VFF_THRE(chan_addr), VFF_RX_THRE(buff_size));

        DMA_WRITE32(VFF_RPT(chan_addr), 0);
        DMA_WRITE32(VFF_INT_FLAG(chan_addr), VFF_RX_INT_FLAG_CLR_B);
        DMA_WRITE32(VFF_EN(chan_addr), VFF_EN_B);
        DMA_WRITE32(VFF_INT_EN(chan_addr), VFF_RX_INT_ENABLE);
    } else {
        dprintf(ULOG_ALWAYS, "UART%d with wrong dir[%d]\n", channel->line, dir);
        return -4;
    }

    dprintf(ULOG_INFO, "VFF_EN:0x%x, VFF_INT_FLAG:0x%x, VFF_INT_EN:0x%x\n",
            DMA_READ32(VFF_EN(chan_addr)), DMA_READ32(VFF_INT_FLAG(chan_addr)),
            DMA_READ32(VFF_INT_EN(chan_addr)));

    return 0;
}

void __uart_dma_enable_INT(struct channel_info *channel, uint32_t enable)
{
    void *chan_addr = channel->base_addr;

    if (enable == INT_ENABLE) {
        if (channel->direction == DMA_DIR_TX)
            DMA_WRITE32(VFF_INT_EN(chan_addr), VFF_TX_INT_ENABLE);
        else
            DMA_WRITE32(VFF_INT_EN(chan_addr), VFF_RX_INT_ENABLE);
    } else {
        DMA_WRITE32(VFF_INT_EN(chan_addr), VFF_INT_EN_CLR_B);
    }
}

void __uart_dma_clear_INT(struct channel_info *channel)
{
    void *chan_addr = channel->base_addr;

    if (channel->direction == DMA_DIR_TX)
        DMA_WRITE32(VFF_INT_FLAG(chan_addr), VFF_TX_INT_FLAG_CLR_B);
    else
        DMA_WRITE32(VFF_INT_FLAG(chan_addr), VFF_RX_INT_FLAG_CLR_B);
}

void __uart_dma_pop_byte(struct channel_info *channel, unsigned char *ch)
{
    char c_char = 0;
    void *chan_addr = channel->base_addr;
    uint32_t rpt = DMA_READ32(VFF_RPT(chan_addr));
    uint64_t read_addr = channel->mem_base + (rpt&UART_DMA_RING_SIZE);

    c_char = DMA_READ8(read_addr);
    dprintf(ULOG_DEBUG, "%s 0x%llx, char:%d, RPT:0x%x\n",
            __func__, read_addr, c_char, rpt);

    *ch = c_char;
    if ((rpt&UART_DMA_RING_SIZE) == DMA_READ32(VFF_LEN(chan_addr)) - 1)
        DMA_WRITE32(VFF_RPT(chan_addr), (~rpt)&UART_DMA_RING_WRAP);
    else
        DMA_WRITE32(VFF_RPT(chan_addr), rpt+1);
}

void __uart_dma_start_rx(struct channel_info *channel)
{
    void *chan_addr = channel->base_addr;
    struct uart_info *uart_port = (struct uart_info *)channel->port_info;
    struct circ_buf *xmit = &uart_port->rx_circ;

    uint32_t room_left, dropped=0, count=0;
    unsigned char rx_data;

    room_left = CIRC_LEFT(xmit);
    dprintf(ULOG_INFO, "%s LEFT:%d, AVAIL:%d\n",
            __func__, room_left, VFIFO_GET_AVAIL(chan_addr));

    if (!room_left) {
        while (1) {
            if (VFIFO_GET_AVAIL(chan_addr) > 0) {
                __uart_dma_pop_byte(channel, &rx_data);
                dropped++;
            } else {
                uart_port->dropped += dropped;
                return;
            }
        }
    }

    while (CIRC_LEFT(xmit)) {
        if (VFIFO_GET_AVAIL(chan_addr) > 0) {
            __uart_dma_pop_byte(channel, &rx_data);
            CIRC_PUSH(xmit, rx_data);
            count++;
        } else
            break;
    }

    dprintf(ULOG_INFO, "%s receive data len:%d, LEFT:%d, AVAIL:%d, VFF_WPT:0x%x, VFF_RPT:0x%x\n",
            __func__, count, CIRC_LEFT(xmit), VFIFO_GET_AVAIL(chan_addr),
            DMA_READ32(VFF_WPT(chan_addr)), DMA_READ32(VFF_RPT(chan_addr)));
}

void __uart_dma_start_tx(struct channel_info *channel)
{
    void *chan_addr = channel->base_addr;
    struct uart_info *uart_port = (struct uart_info *)channel->port_info;
    struct circ_buf *xmit = &uart_port->tx_circ;
    uint32_t left, avail, send_count, wrap;

    uint32_t len = channel->mem_length;
    uint32_t wpt = DMA_READ32(VFF_WPT(chan_addr));

    left = VFIFO_GET_LEFT(chan_addr);
    avail = VFIFO_GET_AVAIL(chan_addr);

    /* update the xmit read point as value of DMA RPT */
    xmit->read = DMA_READ32(VFF_RPT(chan_addr));

    send_count = min(CIRC_AVAIL(xmit), left);
    dprintf(ULOG_INFO, "%s left:%d, data count:%d \n", __func__, left, send_count);

    /* only triggel DMA after last data all have out */
    if (!avail && (send_count > 0)) {
        /* update the WPT */
        wrap = (wpt & UART_DMA_RING_WRAP) ? 0 : UART_DMA_RING_WRAP;
        if ((wpt & (len - 1U)) + send_count < len)
            DMA_WRITE32(VFF_WPT(chan_addr), wpt + send_count);
        else
            DMA_WRITE32(VFF_WPT(chan_addr), ((wpt + send_count) & (len - 1U)) | wrap);

        /* Do FLUSH after write finish */
        DMA_WRITE32(VFF_FLUSH(chan_addr), VFF_FLUSH_B);
    }

    /* Switch off TX interrupt if TX buffer empty */
    if (CIRC_IS_EMPTY(xmit) == true)
        __uart_dma_enable_INT(channel, INT_DISABLE);
    else
        __uart_dma_enable_INT(channel, INT_ENABLE);
}

static enum wrap_handler_return __uart_dma_tx_irq_handle(void *arg)
{
    struct channel_info *channel = (struct channel_info *)arg;
    uint32_t irq = channel->irq;

    plat_wrap_mask_interrupt(irq);
    __uart_dma_clear_INT(channel);
    __uart_dma_start_tx(channel);
    plat_wrap_unmask_interrupt(irq);

    return WRAP_INT_RESCHEDULE;
}

static enum wrap_handler_return __uart_dma_rx_irq_handle(void *arg)
{
    struct channel_info *channel = (struct channel_info *)arg;
    uint32_t irq = channel->irq;

    plat_wrap_mask_interrupt(irq);
    __uart_dma_clear_INT(channel);
    __uart_dma_start_rx(channel);
    plat_wrap_unmask_interrupt(irq);

    return WRAP_INT_RESCHEDULE;
}

int __uart_dma_register_interrupt(struct channel_info *channel)
{
    uint32_t irq;
    uint32_t dir;

    if (channel == NULL) {
        dprintf(ULOG_INFO, "channel is NULL!\n");
        return -1;
    }

    irq = channel->irq;
    dir = channel->direction;
    if (dir == DMA_DIR_TX) {
        plat_wrap_irq_set_sensitive(irq, WRAP_LEVEL_SENSITIVE);
        plat_wrap_irq_set_polarity(irq, WRAP_POLARITY_LOW);
        plat_wrap_register_int_handler(irq, &__uart_dma_tx_irq_handle, channel);
        plat_wrap_unmask_interrupt(irq);
    } else if (dir == DMA_DIR_RX) {
        plat_wrap_irq_set_sensitive(irq, WRAP_LEVEL_SENSITIVE);
        plat_wrap_irq_set_polarity(irq, WRAP_POLARITY_LOW);
        plat_wrap_register_int_handler(irq, &__uart_dma_rx_irq_handle, channel);
        plat_wrap_unmask_interrupt(irq);
    } else {
        dprintf(ULOG_ALWAYS, "PORT%d with wrong dir[%d]\n", channel->line, dir);
        return -2;
    }

    dprintf(ULOG_INFO, "PORT%d Register DMA Interrupt finish, irq:%d, dir:%d\n",
            channel->line, irq, dir);

    return 0;
}

void __uart_dma_start(struct uart_info *uart_port, char *dma_buff,
                       uint32_t dma_len)
{
    struct channel_info *channel;
    uint32_t port_line = uart_port->line;

    channel = __uart_dma_init(port_line, DMA_DIR_TX);
    channel->port_info = uart_port;
    dprintf(ULOG_INFO, "UART%d TX start with channel:%p\n", port_line, channel);
    __uart_dma_config(channel, dma_buff, dma_len);
    __uart_dma_register_interrupt(channel);

    channel = __uart_dma_init(port_line, DMA_DIR_RX);
    channel->port_info = uart_port;
    dprintf(ULOG_INFO, "UART%d RX start with channel:%p\n", port_line, channel);
    __uart_dma_config(channel, dma_buff, dma_len);
    __uart_dma_register_interrupt(channel);
}

void __uart_dma_enable_tx_INT(uint32_t port_line)
{
    struct channel_info *channel;

    channel = &dma_channel_info[port_line*2 + DMA_DIR_TX];
    __uart_dma_enable_INT(channel, INT_ENABLE);

    dprintf(ULOG_DEBUG, "UART%d enable tx INT finish!\n", port_line);
}

#endif
