// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2019 MediaTek Inc.
 */

#define pr_fmt(fmt) "<sensorlist> " fmt

#include <linux/module.h>
#include <linux/miscdevice.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include "sensor_list.h"
#include "hf_sensor_type.h"

enum sensorlist {
	accel,
	gyro,
	mag,
	als,
	ps,
	baro,
	sar,
	maxhandle,
};

static struct sensorlist_info_t sensorlist_info[maxhandle];
static struct mag_libinfo_t mag_libinfo;
static DEFINE_SPINLOCK(sensorlist_info_lock);

int sensorlist_find_sensor(int sensor)
{
	int handle = -1;

	switch (sensor) {
	case SENSOR_TYPE_ACCELEROMETER:
		handle = accel;
		break;
	case SENSOR_TYPE_GYROSCOPE:
		handle = gyro;
		break;
	case SENSOR_TYPE_MAGNETIC_FIELD:
		handle = mag;
		break;
	case SENSOR_TYPE_LIGHT:
		handle = als;
		break;
	case SENSOR_TYPE_PROXIMITY:
		handle = ps;
		break;
	case SENSOR_TYPE_PRESSURE:
		handle = baro;
		break;
	case SENSOR_TYPE_SAR:
		handle = sar;
		break;
	}
	return handle;
}

int sensorlist_find_type(int handle)
{
	int type = -1;

	switch (handle) {
	case accel:
		type = SENSOR_TYPE_ACCELEROMETER;
		break;
	case gyro:
		type = SENSOR_TYPE_GYROSCOPE;
		break;
	case mag:
		type = SENSOR_TYPE_MAGNETIC_FIELD;
		break;
	case als:
		type = SENSOR_TYPE_LIGHT;
		break;
	case ps:
		type = SENSOR_TYPE_PROXIMITY;
		break;
	case baro:
		type = SENSOR_TYPE_PRESSURE;
		break;
	case sar:
		type = SENSOR_TYPE_SAR;
		break;
	}
	return type;
}

static void init_sensorlist_info(void)
{
	int handle = -1;

	spin_lock(&sensorlist_info_lock);
	for (handle = accel; handle < maxhandle; ++handle)
		strlcpy(sensorlist_info[handle].name, "NULL",
			sizeof(sensorlist_info[handle].name));
	spin_unlock(&sensorlist_info_lock);
}

int sensorlist_register_devinfo(int sensor,
		struct sensorlist_info_t *devinfo)
{
	int handle = -1;

	handle = sensorlist_find_sensor(sensor);
	if (handle < 0)
		return -1;
	pr_notice("name(%s) type(%d) registered\n", devinfo->name, sensor);
	spin_lock(&sensorlist_info_lock);
	strlcpy(sensorlist_info[handle].name, devinfo->name,
			sizeof(sensorlist_info[handle].name));
	spin_unlock(&sensorlist_info_lock);
	return 0;
}

int sensorlist_register_maginfo(struct mag_libinfo_t *maginfo)
{
	spin_lock(&sensorlist_info_lock);
	memcpy(&mag_libinfo, maginfo, sizeof(struct mag_libinfo_t));
	spin_unlock(&sensorlist_info_lock);
	return 0;
}

static int sensorlist_open(struct inode *inode, struct file *file)
{
	return nonseekable_open(inode, file);
}

static ssize_t
sensorlist_read(struct file *file, char __user *buf,
		size_t count, loff_t *ptr)
{
	struct sensorlist_info_t temp[maxhandle];

	if (count == 0)
		return -EINVAL;
	if (count < sizeof(struct sensorlist_info_t))
		return -EINVAL;
	if (count > maxhandle * sizeof(struct sensorlist_info_t))
		count = maxhandle * sizeof(struct sensorlist_info_t);

	memset(temp, 0, sizeof(temp));
	spin_lock(&sensorlist_info_lock);
	memcpy(temp, sensorlist_info, sizeof(temp));
	spin_unlock(&sensorlist_info_lock);
	if (copy_to_user(buf, temp, count))
		return -EFAULT;
	return count;
}

static long sensorlist_ioctl(struct file *filp,
			unsigned int cmd, unsigned long arg)
{
	unsigned int size = _IOC_SIZE(cmd);
	void __user *ubuf = (void __user *)arg;
	struct mag_libinfo_t temp;

	if (size != sizeof(struct mag_libinfo_t))
		return -EINVAL;

	switch (cmd) {
	case SENSOR_LIST_GET_MAG_LIB_INFO:
		memset(&temp, 0, sizeof(struct mag_libinfo_t));
		spin_lock(&sensorlist_info_lock);
		memcpy(&temp, &mag_libinfo, sizeof(struct mag_libinfo_t));
		spin_unlock(&sensorlist_info_lock);
		if (copy_to_user(ubuf, &temp, sizeof(struct mag_libinfo_t)))
			return -EFAULT;
		break;
	}
	return 0;
}

static const struct file_operations sensorlist_fops = {
	.owner          = THIS_MODULE,
	.open           = sensorlist_open,
	.read           = sensorlist_read,
	.unlocked_ioctl = sensorlist_ioctl,
	.compat_ioctl   = sensorlist_ioctl,
};

static struct miscdevice sensorlist_miscdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "sensorlist",
	.fops = &sensorlist_fops,
};

static int sensor_list_proc_show(struct seq_file *m, void *v)
{
	int handle = -1, type = -1;

	seq_puts(m, "dynamic hardware sensorlist:\n");
	spin_lock(&sensorlist_info_lock);
	for (handle = accel; handle < maxhandle; ++handle) {
		type = sensorlist_find_type(handle);
		if (type < 0)
			continue;
		seq_printf(m, "sensortype:%d chipname:%s\n",
			type, sensorlist_info[handle].name);
	}
	spin_unlock(&sensorlist_info_lock);
	return 0;
}

static int sensor_list_proc_open(struct inode *inode,
		struct file *filp)
{
	return single_open(filp, sensor_list_proc_show, NULL);
}

static const struct file_operations sensor_list_proc_fops = {
	.open           = sensor_list_proc_open,
	.release        = single_release,
	.read           = seq_read,
	.llseek         = seq_lseek,
};

static int __init sensorlist_init(void)
{
	init_sensorlist_info();
	if (misc_register(&sensorlist_miscdev) < 0)
		return -1;
	proc_create("sensorlist", 0644, NULL, &sensor_list_proc_fops);
	return 0;
}

subsys_initcall(sensorlist_init);
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("dynamic sensorlist driver");
MODULE_AUTHOR("hongxu.zhao@mediatek.com");
