// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2019 MediaTek Inc.
 */


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/of.h>
#include <linux/delay.h>
#include <linux/math64.h>

#include "hf_manager.h"
#include "bmp280_baro.h"
#include "sensor_list.h"

#define BMP280_BARO_NAME "bmp280_baro"
//#define DEBUG_DATA 1

#ifdef DEBUG_DATA
#define	BMP_DATA_LOG(fmt, args...)  pr_info(fmt, ##args)
#else
#define	BMP_DATA_LOG(fmt, args...)
#endif

static unsigned char support_sensors[] = {
	SENSOR_TYPE_PRESSURE,
};

/* data type */
#define BMP280_U16_t u16
#define BMP280_S16_t s16
#define BMP280_U32_t u32
#define BMP280_S32_t s32
#define BMP280_U64_t u64
#define BMP280_S64_t s64

#define C_I2C_FIFO_SIZE               8
#define CHECK_CHIP_ID_TIME_MAX        5

/* power mode */
enum BMP_POWERMODE_ENUM {
	BMP280_SLEEP_MODE = 0x00,
	BMP280_FORCED_MODE = 0x01,
	BMP280_NORMAL_MODE = 0x03,

	BMP280_UNDEFINED_POWERMODE = 0xff
};

/* filter */
enum BMP_FILTER_ENUM {
	BMP_FILTER_OFF = 0x0,
	BMP_FILTER_2,
	BMP_FILTER_4,
	BMP_FILTER_8,
	BMP_FILTER_16,

	BMP_UNDEFINED_FILTER = 0xff
};
/* oversampling */
enum BMP_OVERSAMPLING_ENUM {
	BMP_OVERSAMPLING_SKIPPED = 0x0,
	BMP_OVERSAMPLING_1X,
	BMP_OVERSAMPLING_2X,
	BMP_OVERSAMPLING_4X,
	BMP_OVERSAMPLING_8X,
	BMP_OVERSAMPLING_16X,

	BMP_UNDEFINED_OVERSAMPLING = 0xff
};

/* bmp280 calibration */
struct bmp280_calibration_data {
	BMP280_U16_t dig_T1;
	BMP280_S16_t dig_T2;
	BMP280_S16_t dig_T3;
	BMP280_U16_t dig_P1;
	BMP280_S16_t dig_P2;
	BMP280_S16_t dig_P3;
	BMP280_S16_t dig_P4;
	BMP280_S16_t dig_P5;
	BMP280_S16_t dig_P6;
	BMP280_S16_t dig_P7;
	BMP280_S16_t dig_P8;
	BMP280_S16_t dig_P9;
};

#define C_MAX_FIR_LENGTH (32)
#define BMP_DATA_NUM 1

/* s/w filter */
struct data_filter {
	u32 raw[C_MAX_FIR_LENGTH][BMP_DATA_NUM];
	int sum[BMP_DATA_NUM];
	int num;
	int idx;
};

struct baro_hw {
	int i2c_num;
	int direction;
	int power_id;
	int power_vol;
	int firlen;    /*!< the length of low pass filter */
	int (*power)(struct baro_hw *hw, unsigned int on, char *devname);
	//unsigned char i2c_addr[B_CUST_I2C_ADDR_NUM];

	/*!< i2c address list,for chips which has different addresses with
	 * different HW layout
	 */
	bool is_batch_supported;
};

struct bmp280_baro_device {
	struct hf_device hf_dev;
	atomic_t raw_enable;
	enum BMP_POWERMODE_ENUM power_mode;
	enum BMP_FILTER_ENUM filter_num;
	enum BMP_OVERSAMPLING_ENUM  ovsampling_p;
	enum BMP_OVERSAMPLING_ENUM  ovsampling_t;
	unsigned long last_temp_measurement;
	unsigned long temp_measurement_period;
	struct bmp280_calibration_data bmp280_cali;

	/* calculated temperature correction coefficient */
	s32 t_fine;
	atomic_t filter;
	uint8_t placement[3];

#if defined(CONFIG_BMP_LOWPASS)
	struct baro_hw hw;
	atomic_t firlen;
	atomic_t fir_en;
	struct data_filter fir;
#endif

};

/* I2C operation functions */
static int bmp280_baro_i2c_read_block(struct i2c_client *client,
				u8 addr, u8 *data, u8 len)
{
	int err = 0;
	u8 beg = addr;
	struct i2c_msg msgs[2] = {
		{/*.addr = client->addr,*/
		 .flags = 0,
		 .len = 1,
		 .buf = &beg},
		{
			/*.addr = client->addr*/
			.flags = I2C_M_RD,
			.len = len,
			.buf = data,
		} };
	if (!client)
		return -EINVAL;
	msgs[0].addr = client->addr;
	msgs[1].addr = client->addr;

	err = i2c_transfer(client->adapter, msgs, ARRAY_SIZE(msgs));
	if (err != 2) {
		pr_err_ratelimited("bmp280 i2c_trans err:%x %x (%d %p %d) %d\n",
				   msgs[0].addr, client->addr, addr, data, len,
				   err);
		err = -EIO;
	} else {
		err = 0; /*no error*/
	}
	return err;
}

static int bmp280_baro_i2c_write_block(struct i2c_client *client,
			u8 addr, u8 *data, u8 len)
{
	/* because address also occupies one byte,
	 * the maximum length for write is 7 bytes
	 */
	int err = 0, idx = 0, num = 0;
	char buf[32];

	if (!client)
		return -EINVAL;
	else if (len > C_I2C_FIFO_SIZE) {
		pr_err_ratelimited("bmp280 len %d fi %d\n", len,
				   C_I2C_FIFO_SIZE);
		return -EINVAL;
	}
	buf[num++] = addr;
	for (idx = 0; idx < len; idx++)
		buf[num++] = data[idx];

	err = i2c_master_send(client, buf, num);
	if (err < 0) {
		pr_err_ratelimited("bmp280 send command error!!\n");
		return -EFAULT;
	}

	return err;
}

static int bmp280_baro_soft_reset(struct i2c_client *client)
{
	int err = -1;
	uint8_t data = BMP280_BARO_SOFT_RESET_VALUE;

	err = bmp280_baro_i2c_write_block(client,
			BMP280_BARO_RESET_ADDR, &data, 1);

	return err;
}

static int bmp280_baro_check_chip_id(struct i2c_client *client)
{
	int err = -1;
	u8 chip_id = 0;
	u8 read_count = 0;

	while (read_count++ < CHECK_CHIP_ID_TIME_MAX) {
		bmp280_baro_i2c_read_block(client,
				BMP280_BARO_CHIP_ID_ADDR, &chip_id, 1);

		if (chip_id != BMP280_BARO_CHIP_ID_VALUE1 &&
			chip_id != BMP280_BARO_CHIP_ID_VALUE2 &&
			chip_id != BMP280_BARO_CHIP_ID_VALUE3) {
			pr_info("bmp280 get chipId fail(0x%2x).\n", chip_id);
		} else {
			err = 0;
			pr_info("bmp280 get chipId success(0x%2x).\n", chip_id);
			break;
		}
	}
	return err;
}

static int bmp280_get_calibration_data(struct i2c_client *client)
{
	int status = 0;
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);
	u8 a_data_u8r[BMP280_CALIBRATION_DATA_LENGTH] = {0};

	status = bmp280_baro_i2c_read_block(
			client, BMP280_CALIBRATION_DATA_START, a_data_u8r,
			BMP280_CALIBRATION_DATA_LENGTH);
	if (status < 0) {
		pr_info("bmp280 get cali exit: read data fail!");
		return status;
	}

	for (status = 0; status < BMP280_CALIBRATION_DATA_LENGTH; status++)
		BMP_DATA_LOG("bmp280 cali data: data[%d]:0x%x\n", status,
			     a_data_u8r[status]);

	driver_dev->bmp280_cali.dig_T1 = (BMP280_U16_t)(
			(((BMP280_U16_t)((unsigned char)a_data_u8r[1]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[0]);
	driver_dev->bmp280_cali.dig_T2 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[3]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[2]);
	driver_dev->bmp280_cali.dig_T3 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[5]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[4]);
	driver_dev->bmp280_cali.dig_P1 = (BMP280_U16_t)(
			(((BMP280_U16_t)((unsigned char)a_data_u8r[7]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[6]);
	driver_dev->bmp280_cali.dig_P2 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[9]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[8]);
	driver_dev->bmp280_cali.dig_P3 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[11]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[10]);
	driver_dev->bmp280_cali.dig_P4 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[13]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[12]);
	driver_dev->bmp280_cali.dig_P5 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[15]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[14]);
	driver_dev->bmp280_cali.dig_P6 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[17]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[16]);
	driver_dev->bmp280_cali.dig_P7 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[19]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[18]);
	driver_dev->bmp280_cali.dig_P8 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[21]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[20]);
	driver_dev->bmp280_cali.dig_P9 = (BMP280_S16_t)(
			(((BMP280_S16_t)((signed char)a_data_u8r[23]))
			 << SHIFT_LEFT_8_POSITION) |
			a_data_u8r[22]);

	return 0;
}


static int bmp280_baro_set_powermode(struct i2c_client *client,
				     enum BMP_POWERMODE_ENUM power_mode)
{
	int err;
	u8 data = 0;
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);

	if (power_mode == BMP280_UNDEFINED_POWERMODE) {
		pr_info("bmp280 power invalid:%d,(old %d)\n",
			 power_mode, driver_dev->power_mode);
		err = -1;
		return err;
	}

	if (power_mode == driver_dev->power_mode) {
		pr_info("bmp280 power no change:%d,(old %d)\n",
			power_mode, driver_dev->power_mode);
		return 0;
	}

	err = bmp280_baro_i2c_read_block(client, BMP280_CTRLMEAS_REG_MODE__REG,
					 &data, 1);
	data = BMP_SET_BITSLICE(data, BMP280_CTRLMEAS_REG_MODE,
				power_mode);
	err += bmp280_baro_i2c_write_block(client,
					   BMP280_CTRLMEAS_REG_MODE__REG,
					   &data, 1);

	if (err < 0)
		pr_err("bmp280 set power mode failed, err = %d\n", err);
	else
		driver_dev->power_mode = power_mode;

	return err;
}

static int bmp280_baro_set_filter(struct i2c_client *client,
				  enum BMP_FILTER_ENUM filter)
{
	int err;
	u8 data = 0;
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);

	if (filter == driver_dev->filter_num ||
	    filter == BMP_UNDEFINED_FILTER) {
		pr_info("bmp280 filter no change or invalid:%d,(old %d)\n",
			filter, driver_dev->filter_num);
		return 0;
	}

	err = bmp280_baro_i2c_read_block(client, BMP280_CONFIG_REG_FILTER__REG,
					 &data, 1);
	data = BMP_SET_BITSLICE(data, BMP280_CONFIG_REG_FILTER, filter);
	err += bmp280_baro_i2c_write_block(client,
					   BMP280_CONFIG_REG_FILTER__REG,
					   &data, 1);

	if (err < 0)
		pr_err("bmp280 set filter failed, err = %d\n", err);
	else
		driver_dev->filter_num = filter;

	return err;
}

static int bmp280_baro_set_oversampling_p
				(struct i2c_client *client,
				enum BMP_OVERSAMPLING_ENUM oversampling_p)
{
	int err;
	u8 data = 0;
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);

	if (oversampling_p == driver_dev->ovsampling_p ||
		oversampling_p == BMP_UNDEFINED_OVERSAMPLING) {
		pr_info("bmp280 ovsampling_p no change or invlid:%d,(old:%d)\n",
			oversampling_p, driver_dev->ovsampling_p);
		return 0;
	}

	err = bmp280_baro_i2c_read_block(client, BMP280_CTRLMEAS_REG_OSRSP__REG,
					&data, 1);
	data = BMP_SET_BITSLICE(data, BMP280_CTRLMEAS_REG_OSRSP,
				oversampling_p);
	err += bmp280_baro_i2c_write_block(client,
					   BMP280_CTRLMEAS_REG_OSRSP__REG,
					   &data, 1);

	if (err < 0)
		pr_err("bmp280 set oversampling_p failed, err = %d\n", err);
	else
		driver_dev->ovsampling_p = oversampling_p;

	return err;
}

static int bmp280_baro_set_oversampling_t
		(struct i2c_client *client,
		 enum BMP_OVERSAMPLING_ENUM oversampling_t)
{
	int err;
	u8 data = 0;
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);

	if (oversampling_t == driver_dev->ovsampling_t ||
		oversampling_t == BMP_UNDEFINED_OVERSAMPLING) {
		pr_info("bmp280 ovsampling_t no change or invlid:%d,(old:%d)\n",
			oversampling_t, driver_dev->ovsampling_t);
		return 0;
	}

	err = bmp280_baro_i2c_read_block(client, BMP280_CTRLMEAS_REG_OSRST__REG,
					 &data, 1);
	data = BMP_SET_BITSLICE(data, BMP280_CTRLMEAS_REG_OSRST,
				oversampling_t);
	err += bmp280_baro_i2c_write_block(client,
					   BMP280_CTRLMEAS_REG_OSRST__REG,
					   &data, 1);

	if (err < 0)
		pr_err("bmp280 set oversampling_t failed, err = %d\n", err);
	else
		driver_dev->ovsampling_t = oversampling_t;

	return err;
}

static int bmp280_baro_init_device(struct i2c_client *client)
{
	int err = 0;

	err = bmp280_get_calibration_data(client);
	if (err < 0) {
		pr_err("get calibration data failed, err = %d\n", err);
		return err;
	}

	err = bmp280_baro_set_powermode(client, BMP280_SLEEP_MODE);
	if (err < 0) {
		pr_err("bmp280 set power mode failed, err = %d\n", err);
		return err;
	}

	err = bmp280_baro_set_filter(client, BMP_FILTER_8);
	if (err < 0) {
		pr_err("bmp280 set hw filter failed, err = %d\n", err);
		return err;
	}

	err = bmp280_baro_set_oversampling_p(client, BMP_OVERSAMPLING_8X);
	if (err < 0) {
		pr_err("bmp280 set oversampling_p failed, err = %d\n", err);
		return err;
	}

	err = bmp280_baro_set_oversampling_t(client, BMP_OVERSAMPLING_1X);
	if (err < 0) {
		pr_err("bmp280 set oversampling_t failed, err = %d\n", err);
		return err;
	}

	pr_info("bmp280 baro init device success.\n");

	return err;
}

static int bmp_read_raw_temperature(struct i2c_client *client,
				    s32 *temperature)
{
	struct bmp280_baro_device *obj;
	s32 err = 0;
	unsigned char a_data_u8r[3] = {0};

	if (client == NULL) {
		err = -EINVAL;
		return err;
	}

	obj = i2c_get_clientdata(client);

	err = bmp280_baro_i2c_read_block(client, BMP280_TEMPERATURE_MSB_REG,
					a_data_u8r, 3);
	if (err < 0) {
		pr_err("bmp280 read raw temperature failed, err = %d\n", err);
		return err;
	}
	*temperature = (BMP280_S32_t)((((BMP280_U32_t)(a_data_u8r[0]))
				       << SHIFT_LEFT_12_POSITION) |
				      (((BMP280_U32_t)(a_data_u8r[1]))
				       << SHIFT_LEFT_4_POSITION) |
				      ((BMP280_U32_t)a_data_u8r[2] >>
				       SHIFT_RIGHT_4_POSITION));

	obj->last_temp_measurement = jiffies;

	return err;
}

static int bmp_read_raw_pressure(struct i2c_client *client, s32 *pressure)
{
	struct bmp280_baro_device *priv;
	s32 err = 0;
	unsigned char a_data_u8r[3] = {0};

	if (client == NULL) {
		err = -EINVAL;
		return err;
	}

	priv = i2c_get_clientdata(client);

	err = bmp280_baro_i2c_read_block(client, BMP280_PRESSURE_MSB_REG,
					 a_data_u8r, 3);
	if (err < 0) {
		pr_err("bmp280 read raw pressure failed, err = %d\n", err);
		return err;
	}
	*pressure = (BMP280_S32_t)((((BMP280_U32_t)(a_data_u8r[0]))
				    << SHIFT_LEFT_12_POSITION) |
				   (((BMP280_U32_t)(a_data_u8r[1]))
				    << SHIFT_LEFT_4_POSITION) |
				   ((BMP280_U32_t)a_data_u8r[2] >>
				    SHIFT_RIGHT_4_POSITION));

#ifdef CONFIG_BMP_LOWPASS
	/*
	 *Example: firlen = 16, filter buffer = [0] ... [15],
	 *when 17th data come, replace [0] with this new data.
	 *Then, average this filter buffer and report average value to upper
	 *layer.
	 */
	if (atomic_read(&priv->filter)) {
		if (atomic_read(&priv->fir_en) &&
		    !atomic_read(&priv->suspend)) {
			int idx, firlen = atomic_read(&priv->firlen);

			if (priv->fir.num < firlen) {
				priv->fir.raw[priv->fir.num][BMP_PRESSURE] =
					*pressure;
				priv->fir.sum[BMP_PRESSURE] += *pressure;
				if (atomic_read(&priv->trace) &
				    BAR_TRC_FILTER) {
					pr_debug("add [%2d] [%5d] => [%5d]\n",
						priv->fir.num,
						priv->fir.raw[priv->fir.num]
							     [BMP_PRESSURE],
						priv->fir.sum[BMP_PRESSURE]);
				}
				priv->fir.num++;
				priv->fir.idx++;
			} else {
				idx = priv->fir.idx % firlen;
				priv->fir.sum[BMP_PRESSURE] -=
					priv->fir.raw[idx][BMP_PRESSURE];
				priv->fir.raw[idx][BMP_PRESSURE] = *pressure;
				priv->fir.sum[BMP_PRESSURE] += *pressure;
				priv->fir.idx++;
				*pressure =
					priv->fir.sum[BMP_PRESSURE] / firlen;
				if (atomic_read(&priv->trace) &
				    BAR_TRC_FILTER) {
					pr_debug("add[%2d][%5d]=>[%5d]:[%5d]\n",
						idx,
						priv->fir
							.raw[idx][BMP_PRESSURE],
						priv->fir.sum[BMP_PRESSURE],
						*pressure);
				}
			}
		}
	}
#endif

	return err;
}

/*
 *get compensated temperature
 *unit:10 degrees centigrade
 */
static int bmp_get_temperature(struct i2c_client *client, s32 *t_buf)
{
	struct bmp280_baro_device *driver_dev;
	int status;
	s32 utemp = 0; /* uncompensated temperature */
	s32 temperature = 0;
	BMP280_S32_t v_x1_u32r = 0;
	BMP280_S32_t v_x2_u32r = 0;

	if (client == NULL || t_buf == NULL) {
		pr_err("bmp280 get temperature fail, invalid parameter\n");
		return -1;
	}

	driver_dev = i2c_get_clientdata(client);

	status = bmp_read_raw_temperature(client, &utemp);
	if (status != 0) {
		pr_info("bmp280 read raw temperature fail,status:%d\n", status);
		return status;
	}
	v_x1_u32r = ((((utemp >> 3) -
		    ((BMP280_S32_t)driver_dev->bmp280_cali.dig_T1 << 1))) *
		    ((BMP280_S32_t)driver_dev->bmp280_cali.dig_T2)) >> 11;
	v_x2_u32r = (((((utemp >> 4) -
		    ((BMP280_S32_t)driver_dev->bmp280_cali.dig_T1)) *
		    ((utemp >> 4) -
		    ((BMP280_S32_t)driver_dev->bmp280_cali.dig_T1))) >> 12) *
		    ((BMP280_S32_t)driver_dev->bmp280_cali.dig_T3)) >> 14;

	driver_dev->t_fine = v_x1_u32r + v_x2_u32r;

	temperature = (driver_dev->t_fine * 5 + 128) >> 8;

	*t_buf = temperature;

	BMP_DATA_LOG("bmp280 temperature: %d\n", temperature);
	BMP_DATA_LOG("bmp280 temperature/100: %d\n", temperature / 100);

	return status;
}

/*
 *get compensated pressure
 *unit: hectopascal(hPa)
 */
 #define BMP_BUFSIZE	128
static int bmp_get_pressure(struct i2c_client *client, s32 *p_buf)
{
	struct bmp280_baro_device *driver_dev;
	int status;
	s32 temperature = 0, upressure = 0, pressure = 0;
	BMP280_S64_t v_x1_u32r = 0;
	BMP280_S64_t v_x2_u32r = 0;
	BMP280_S64_t p = 0;

	BMP_DATA_LOG("bmp280 get pressure\n");

	if (client == NULL || p_buf == NULL) {
		pr_err("bmp280 get pressure fail, invalid parameter\n");
		return -1;
	}

	driver_dev = i2c_get_clientdata(client);
	/* update the ambient temperature according to the given meas. period */
	/* below method will have false problem when jiffies wrap around.
	 *so replace.
	 */

	if (time_before_eq((unsigned long)(driver_dev->last_temp_measurement +
			   driver_dev->temp_measurement_period), jiffies)) {

		status = bmp_get_temperature(client, &temperature);
		if (status != 0) {
			pr_info("bmp280 get temperature:%d, exit get prssure\n",
				status);
			goto exit;
		}
	}

	status = bmp_read_raw_pressure(client, &upressure);
	if (status != 0) {
		pr_info("bmp280 get raw pressure status:%d, exit get prssure\n",
			status);
		goto exit;
	}

	v_x1_u32r = ((BMP280_S64_t)driver_dev->t_fine) - 128000;
	v_x2_u32r = v_x1_u32r * v_x1_u32r *
			(BMP280_S64_t)driver_dev->bmp280_cali.dig_P6;
	v_x2_u32r = v_x2_u32r +
		    ((v_x1_u32r * (BMP280_S64_t)driver_dev->bmp280_cali.dig_P5)
		     << 17);
	v_x2_u32r = v_x2_u32r +
			(((BMP280_S64_t)driver_dev->bmp280_cali.dig_P4) << 35);
	v_x1_u32r = ((v_x1_u32r * v_x1_u32r *
		     (BMP280_S64_t)driver_dev->bmp280_cali.dig_P3) >> 8) +
		    ((v_x1_u32r * (BMP280_S64_t)driver_dev->bmp280_cali.dig_P2)
		    << 12);
	v_x1_u32r = (((((BMP280_S64_t)1) << 47) + v_x1_u32r)) *
		    ((BMP280_S64_t)driver_dev->bmp280_cali.dig_P1) >> 33;
	if (v_x1_u32r == 0) {
		/* Avoid exception caused by division by zero */
		pr_info("bmp280 get_pressure v_x1_32=0, exit get pressure\n");
		return -1;
	}

	p = 1048576 - upressure;
	p = div64_s64(((p << 31) - v_x2_u32r) * 3125, v_x1_u32r);
	v_x1_u32r = (((BMP280_S64_t)driver_dev->bmp280_cali.dig_P9) *
			(p >> 13) * (p >> 13)) >> 25;
	v_x2_u32r = (((BMP280_S64_t)driver_dev->bmp280_cali.dig_P8) * p) >> 19;
	p = ((p + v_x1_u32r + v_x2_u32r) >> 8) +
	    (((BMP280_S64_t)driver_dev->bmp280_cali.dig_P7) << 4);
	pressure = (BMP280_U32_t)p / 256;

	*p_buf = pressure;
	BMP_DATA_LOG("bmp280 pressure: %d\n", pressure);
	BMP_DATA_LOG("bmp280 pressure/100: %d\n", pressure / 100);

exit:
	return status;
}

int bmp280_baro_sample(struct hf_device *hfdev)
{
	struct i2c_client *client;
	struct bmp280_baro_device *driver_dev;
	struct hf_manager *manager;
	struct hf_manager_event event;
	int64_t current_time;
	int value = 0, temperature = 0;
	int err = 0;

	BMP_DATA_LOG("Sample bmp280\n");

	if (!hfdev) {
		pr_err("bmp280 sample failed:invalid hfdev\n");
		return -1;
	}
	client = hf_device_get_private_data(hfdev);
	driver_dev = i2c_get_clientdata(client);
	manager = driver_dev->hf_dev.manager;

	err = bmp_get_pressure(client, &value);
	if (err) {
		pr_err("bmp280 get compensated pressure failed, err = %d\n",
			err);
		return -1;
	}

	err = bmp_get_temperature(client, &temperature);
	if (err != 0) {
		pr_err("bmp280 get temperature:%d, exit get prssure\n",
			err);
		return -1;
	}

	current_time = ktime_get_boot_ns();
	if (atomic_read(&driver_dev->raw_enable)) {
		memset(&event, 0, sizeof(struct hf_manager_event));
		event.timestamp = current_time;
		event.sensor_type = SENSOR_TYPE_PRESSURE;
		event.accurancy = SENSOR_ACCURANCY_HIGH;
		event.action = RAW_ACTION;
		event.word[0] = value;
		event.word[1] = temperature;
		manager->report(manager, &event);
	}

	memset(&event, 0, sizeof(struct hf_manager_event));
	event.timestamp = current_time;
	event.sensor_type = SENSOR_TYPE_PRESSURE;
	event.accurancy = SENSOR_ACCURANCY_HIGH;
	event.action = DATA_ACTION;
	event.word[0] = value;
	event.word[1] = temperature;

	manager->report(manager, &event);
	manager->complete(manager);

	return 0;
}

int bmp280_baro_enable(struct hf_device *hfdev, int sensor_type, int en)
{
	struct i2c_client *client;
	struct bmp280_baro_device *driver_dev;
	int ret = 0;
	int retry = 0;
	enum BMP_POWERMODE_ENUM power;

	pr_info("%s bmp280\n", en?"En":"Dis");
	if (!hfdev) {
		pr_err("bmp280 enable failed:invalid hfdev\n");
		return -1;
	}
	client = hf_device_get_private_data(hfdev);
	driver_dev = i2c_get_clientdata(client);

	if (en)
		power = BMP280_NORMAL_MODE;
	else if (!en)
		power = BMP280_SLEEP_MODE;

	for (retry = 0; retry < 3; retry++) {
		ret = bmp280_baro_set_powermode(client, power);
		if (ret >= 0) {
			pr_info("bmp280 set pwer mode(%d) done(retry:%d)\n",
				power, retry);
			break;
		}
	}
	driver_dev->last_temp_measurement =
		jiffies - driver_dev->temp_measurement_period;

	if (ret < 0) {
		pr_err("bmp280 set power mode(%d) fail!\n", power);
		return -1;
	}

	pr_info("bmp280 (%s) success\n", power?"enable":"disable");
	return 0;
}

static int bmp_set_delay(u64 ns)
{
	return 0;
}

int bmp280_baro_batch(struct hf_device *hfdev,
		      int sensor_type, int64_t delay, int64_t latency)
{
	struct i2c_client *client = hf_device_get_private_data(hfdev);
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);
	struct hf_manager *manager = driver_dev->hf_dev.manager;
	struct hf_manager_event event;
	int64_t current_time;

	memset(&event, 0, sizeof(struct hf_manager_event));
	current_time = ktime_get_boot_ns();
	event.timestamp = current_time;
	event.sensor_type = SENSOR_TYPE_ADDITIONAL_INFO;
	event.accurancy = SENSOR_ACCURANCY_HIGH;
	event.reserved = SENSOR_TYPE_PRESSURE;
	event.action = DATA_ACTION;
	event.byte[12] = 1;
	event.byte[3] = driver_dev->placement[0];
	event.byte[7] = driver_dev->placement[1];
	event.byte[11] = driver_dev->placement[2];

	manager->report(manager, &event);
	manager->complete(manager);

	return bmp_set_delay(delay);
}

static int bmp280_baro_flush(struct hf_device *hfdev, int sensor_type)
{
	struct i2c_client *client = hf_device_get_private_data(hfdev);
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);
	struct hf_manager *manager = driver_dev->hf_dev.manager;
	struct hf_manager_event event;
	int64_t current_time;

	memset(&event, 0, sizeof(struct hf_manager_event));
	current_time = ktime_get_boot_ns();
	event.sensor_type = sensor_type;
	event.timestamp = current_time;
	event.action = FLUSH_ACTION;

	manager->report(manager, &event);
	manager->complete(manager);

	return 0;
}

int bmp280_baro_raw_enable(struct hf_device *hfdev, int sensor_type, int en)
{
	struct i2c_client *client = hf_device_get_private_data(hfdev);
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);

	atomic_set(&driver_dev->raw_enable, en);
	return 0;
}

static int bmp280_baro_remove(struct i2c_client *client)
{
	struct bmp280_baro_device *driver_dev = i2c_get_clientdata(client);

	hf_manager_destroy(driver_dev->hf_dev.manager);
	kfree(driver_dev);
	return 0;
}

static int bmp280_baro_probe(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	int err = 0;
	struct bmp280_baro_device *driver_dev;
	struct sensorlist_info_t listinfo;

	pr_info("%s enter Slaver 0x76.\n", __func__);

	err = bmp280_baro_soft_reset(client);
	if (err < 0) {
		pr_err("bmp280 soft reset fail,exit probe!\n");
		err = -EINVAL;
		goto findHW_fail;
	}

	/* check chip id */
	err = bmp280_baro_check_chip_id(client);
	if (err < 0) {
		pr_err("bmp280 chip id mismatch,exit probe!\n");
		err = -EINVAL;
		goto findHW_fail;
	}
	mdelay(1);

	driver_dev = kzalloc(sizeof(*driver_dev), GFP_KERNEL);
	if (!driver_dev) {
		err = -ENOMEM;
		goto malloc_fail;
	}

	if (of_property_read_u8_array(client->dev.of_node, "placement",
		driver_dev->placement, ARRAY_SIZE(driver_dev->placement))) {
		pr_err("%s get placement dts fail\n", __func__);
		err = -EFAULT;
		goto init_fail;
	}

	atomic_set(&driver_dev->raw_enable, 0);

	driver_dev->hf_dev.dev_name = BMP280_BARO_NAME;
	driver_dev->hf_dev.device_poll = HF_DEVICE_IO_POLLING;
	driver_dev->hf_dev.device_bus = HF_DEVICE_IO_SYNC;
	driver_dev->hf_dev.support_list = support_sensors;
	driver_dev->hf_dev.support_size = ARRAY_SIZE(support_sensors);
	driver_dev->hf_dev.sample = bmp280_baro_sample;
	driver_dev->hf_dev.enable = bmp280_baro_enable;
	driver_dev->hf_dev.batch = bmp280_baro_batch;
	driver_dev->hf_dev.flush = bmp280_baro_flush;
	driver_dev->hf_dev.rawdata = bmp280_baro_raw_enable;

#ifdef CONFIG_BMP_LOWPASS
	if (driver_dev->hw.firlen > C_MAX_FIR_LENGTH)
		atomic_set(&driver_dev->firlen, C_MAX_FIR_LENGTH);
	else
		atomic_set(&driver_dev->firlen, driver_dev->hw.firlen);

	if (atomic_read(&driver_dev->firlen) > 0)
		atomic_set(&driver_dev->fir_en, 1);
#endif

	err = hf_manager_create(&driver_dev->hf_dev);
	if (err < 0) {
		pr_err("bmp280 hf_manager_create fail,exit probe!\n");
		err = -1;
		goto create_manager_fail;
	}

	i2c_set_clientdata(client, driver_dev);
	hf_device_set_private_data(&driver_dev->hf_dev, client);

	driver_dev->power_mode = BMP280_UNDEFINED_POWERMODE;
	driver_dev->filter_num = BMP_UNDEFINED_FILTER;
	driver_dev->ovsampling_p = BMP_UNDEFINED_OVERSAMPLING;
	driver_dev->ovsampling_t = BMP_UNDEFINED_OVERSAMPLING;
	driver_dev->last_temp_measurement = 0;
	driver_dev->temp_measurement_period = 1 * HZ;

	err = bmp280_baro_init_device(client);
	if (err < 0) {
		pr_err("bmp280 init device fail,exit probe!\n");
		goto init_fail;
	}

	memset(&listinfo, 0, sizeof(struct sensorlist_info_t));
	strlcpy(listinfo.name, BMP280_BARO_NAME, sizeof(listinfo.name));
	err = sensorlist_register_devinfo(SENSOR_TYPE_PRESSURE, &listinfo);
	if (err < 0) {
		pr_err("bmp280 register sensorlist fail,exit probe!\n");
		goto init_fail;
	}

	pr_info("%s success!\n", __func__);
	return 0;

init_fail:
	hf_manager_destroy(driver_dev->hf_dev.manager);
create_manager_fail:
	kfree(driver_dev);
malloc_fail:
findHW_fail:
	return err;
}

static const struct of_device_id bmp280_baro_of_match[] = {
	{.compatible = "mediatek,bmp280_baro"},
	{},
};

static const struct i2c_device_id bmp280_baro_id[] = {
					{BMP280_BARO_NAME, 0}, {} };

static struct i2c_driver bmp280_baro_driver = {
	.driver = {
		.name = BMP280_BARO_NAME,
		.bus = &i2c_bus_type,
		.owner = THIS_MODULE,
		.of_match_table = bmp280_baro_of_match,
	},
	.probe = bmp280_baro_probe,
	.remove = bmp280_baro_remove,
	.id_table =  bmp280_baro_id,
};

module_i2c_driver(bmp280_baro_driver);

MODULE_AUTHOR("Mediatek");
MODULE_DESCRIPTION("bmp280 baro i2c driver");
MODULE_LICENSE("GPL");
