/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Copyright (C) 2019 MediaTek Inc.
 */

#ifndef _HF_SENSOR_MANAGER_H_
#define _HF_SENSOR_MANAGER_H_

#include <linux/hrtimer.h>
#include <linux/interrupt.h>
#include <linux/kthread.h>
#include <linux/ioctl.h>
#include <linux/types.h>

#include "hf_sensor_type.h"
#include "hf_sensor_io.h"

#define HF_MANAGER_IO_IN_PROGRESS 0
#define HF_MANAGER_IO_READY       1

#define HF_DEVICE_IO_SYNC  0
#define HF_DEVICE_IO_ASYNC 1

#define HF_DEVICE_IO_INTERRUPT 0
#define HF_DEVICE_IO_POLLING   1

#define HF_CLIENT_FIFO_SIZE 128

struct coordinate {
	int8_t sign[3];
	uint8_t map[3];
};

struct sensor_state {
	bool enable;
	bool bias;
	bool cali;
	bool temp;
	bool test;
	bool raw;
	int64_t delay;
	int64_t latency;
	atomic_t flush;
	atomic64_t start_time;
};

struct hf_device {
	int (*sample)(struct hf_device *hfdev);
	int (*enable)(struct hf_device *hfdev, int sensor_type, int en);
	int (*batch)(struct hf_device *hfdev, int sensor_type,
		int64_t delay, int64_t latency);
	int (*flush)(struct hf_device *hfdev, int sensor_type);
	int (*calibration)(struct hf_device *hfdev, int sensor_type);
	int (*config_cali)(struct hf_device *hfdev,
		int sensor_type, int32_t *data);
	int (*selftest)(struct hf_device *hfdev, int sensor_type);
	int (*rawdata)(struct hf_device *hfdev, int sensor_type, int en);

	char *dev_name;
	unsigned char device_poll;
	unsigned char device_bus;

	unsigned char *support_list;
	unsigned int support_size;

	struct hf_manager *manager;
	void *private_data;
};

struct hf_client_fifo {
	spinlock_t buffer_lock;
	unsigned int head;
	unsigned int tail;
	unsigned int bufsize;
	unsigned int buffull;
	int64_t last_time_stamp[SENSOR_TYPE_SENSOR_MAX];
	struct hf_manager_event *buffer;
	wait_queue_head_t wait;
};

struct hf_manager {
	struct list_head list;
	struct tasklet_struct io_work_tasklet;
	struct kthread_work io_kthread_work;
	struct hrtimer io_poll_timer;
	atomic64_t io_poll_interval;
	atomic_t io_enabled;
	unsigned long flags;
	struct hf_device *hf_dev;

	int (*report)(struct hf_manager *manager,
		struct hf_manager_event *event);
	void (*complete)(struct hf_manager *manager);
	void (*interrupt)(struct hf_manager *manager);
};

struct hf_client {
	struct list_head list;
	struct hf_client_fifo hf_fifo;
	struct sensor_state request[SENSOR_TYPE_SENSOR_MAX];
	spinlock_t request_lock;

	/* record process info */
	char proc_comm[TASK_COMM_LEN];
	pid_t leader_pid;
	pid_t pid;
};

static inline void hf_device_set_private_data(struct hf_device *device,
		void *data)
{
	device->private_data = data;
}

static inline void *hf_device_get_private_data(struct hf_device *device)
{
	return device->private_data;
}

int hf_manager_create(struct hf_device *device);
int hf_manager_destroy(struct hf_manager *manager);
void coordinate_map(unsigned char direction, int32_t *data);
void get_placement_info(unsigned char direction, int8_t *data);

#endif
