/*
 * Copyright (C) 2016 MediaTek Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See http://www.gnu.org/licenses/gpl-2.0.html for more details.
 */


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/of.h>
#include <linux/delay.h>
#include <linux/math64.h>
#include <linux/atomic.h>

#include "hf_manager.h"
#include "kxtj3_acc_i2c.h"
#include "sensor_list.h"

#define KXTJ3_ACC_I2C_NAME "kxtj3_acc_i2c"
#define KXTJ3_ACC_AXES_NUM          3
#define C_I2C_FIFO_SIZE              8

//#define GRAVITY_EARTH_SCALAR       9.807f
//#define ACCELEROMETER_INCREASE_NUM_AP	1000

#define KXTJ3_ACC_SOFT_RESET_VALUE  0x80
#define KXTJ3_ACC_CHIP_ID_VALUE     (0x35)
#define CHECK_CHIP_ID_TIME_MAX       5



#define KXTJ3_ACC_ODR_7HZ 0x0b
#define KXTJ3_ACC_ODR_12HZ 0x00
#define KXTJ3_ACC_ODR_25HZ 0x01
#define KXTJ3_ACC_ODR_50HZ 0x02
#define KXTJ3_ACC_ODR_100HZ 0x03

#define KXTJ3_ACC_ODR_200HZ 0x04
#define KXTJ3_ACC_ODR_400HZ 0x05
#define KXTJ3_ACC_ODR_800HZ 0x06
#define KXTJ3_ACC_ODR_1600HZ 0x0F


#define KXTJ3_ACC_RANGE_2G 0x40
#define KXTJ3_ACC_RANGE_4G 0x48
#define KXTJ3_ACC_RANGE_8G_1 0x50
#define KXTJ3_ACC_RANGE_8G_2 0x58
#define KXTJ3_ACC_RANGE_16G_1 0x44
#define KXTJ3_ACC_RANGE_16G_2 0x4C
#define KXTJ3_ACC_RANGE_16G_3 0x54
#define KXTJ3_ACC_RANGE_16G_4 0x5C





//static struct sensor_info support_sensors[] = {
//	{
//	.sensor_type = SENSOR_TYPE_ACCELEROMETER,
//	.gain = 100,
//	},
//};
static unsigned char support_sensors[] = {
	SENSOR_TYPE_ACCELEROMETER,
//	SENSOR_TYPE_GYROSCOPE,
//	SENSOR_TYPE_GYRO_TEMPERATURE,
};
struct kxtj3_acc_device {
	struct hf_device hf_dev;
	struct i2c_client *acc_client;
	uint32_t direction;
	uint8_t placement[3];
	atomic_t raw_enable;
	atomic_t acc_enable;
};

/* I2C operation functions */
static int kxtj3_acc_i2c_read_block(struct i2c_client *client,
			u8 addr, u8 *data, u8 len)
{
	int err = 0;
	u8 beg = addr;
	struct i2c_msg msgs[2] = {
		{/*.addr = client->addr,*/
		 .flags = 0,
		 .len = 1,
		 .buf = &beg},
		{
			/*.addr = client->addr*/
			.flags = I2C_M_RD,
			.len = len,
			.buf = data,
		} };
	if (!client)
		return -EINVAL;
	msgs[0].addr = client->addr;
	msgs[1].addr = client->addr;
pr_info("kxtj3_acc_i2c_read_block.\n");
	err = i2c_transfer(client->adapter, msgs, ARRAY_SIZE(msgs));
	if (err != 2) {
		pr_err_ratelimited("i2c_trans err: %x %x (%d %p %d) %d\n",
		       msgs[0].addr, client->addr, addr, data, len, err);
		err = -EIO;
	} else {
		err = 0; /*no error*/
	}
	return err;
}

static int kxtj3_acc_i2c_write_block(struct i2c_client *client,
			u8 addr, u8 *data, u8 len)
{
	/* because address also occupies one byte,
	 * the maximum length for write is 7 bytes
	 */
	int err = 0, idx = 0, num = 0;
	char buf[32];
	pr_info("kxtj3_acc_i2c_write_block.\n");
	if (!client){
		return -EINVAL;
	}
	else if (len > C_I2C_FIFO_SIZE) {
		pr_err_ratelimited("len %d fi %d\n", len, C_I2C_FIFO_SIZE);
		return -EINVAL;
	}
	buf[num++] = addr;
	for (idx = 0; idx < len; idx++)
		buf[num++] = data[idx];

	err = i2c_master_send(client, buf, num);
	if (err < 0) {
		pr_err_ratelimited("send command error!!\n");
		return -EFAULT;
	}

	return err;
}


static int kxtj3_acc_init_device(struct i2c_client *client)
{
	int err = 0;
	uint8_t data = 0;
pr_info("kxtj3_acc_init_device.\n");
	data = KXTJ3_ACC_ODR_100HZ;
	err = kxtj3_acc_i2c_write_block(client,
			KXTJ3_ACC_BW_ADDR, &data, 1);
	if (err < 0)
		return err;
#if 0
	data = 0x00; //filter and lock enable
	err = kxtj3_acc_i2c_write_block(client,
			KXTJ3_ACC_RATED_HBW_ADDR, &data, 1);
	if (err < 0)
		return err;
#endif
	data = KXTJ3_ACC_RANGE_4G;
	err = kxtj3_acc_i2c_write_block(client,
			KXTJ3_ACC_RANGE_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //disable int
	err = kxtj3_acc_i2c_write_block(client,
			KXTJ3_ACC_INT_ENABLE1_ADDR, &data, 1);
	if (err < 0)
		return err;
#if 0
	data = 0x80; //map int
	err = kxtj3_acc_i2c_write_block(client,
			KXTJ3_ACC_INT_MAP_1_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //push-pull, active low
	err = kxtj3_acc_i2c_write_block(client,
			KXTJ3_ACC_INT_OUT_CTRL_ADDR, &data, 1);
	if (err < 0)
		return err;
#endif
	pr_info("KXTJ3_ACC init OK.\n");

	return err;
}

static int kxtj3_acc_set_soft_reset(struct i2c_client *client)
{
	int err = 0;
/*	uint8_t data = KXTJ3_ACC_SOFT_RESET_VALUE;
pr_info("kxtj3_acc_set_soft_reset.\n");
	err = kxtj3_acc_i2c_write_block(client,
			KXTJ3_ACC_BGW_SOFTRESET_ADDR, &data, 1);*/
	return err;
}

static int kxtj3_acc_check_chip_id(struct i2c_client *client)
{
	int err = -1;
	u8 chip_id = 0;
	u8 read_count = 0;
pr_info("kxtj3_acc_check_chip_id\n");
	while (read_count++ < CHECK_CHIP_ID_TIME_MAX) {
		kxtj3_acc_i2c_read_block(client,
				KXTJ3_ACC_CHIP_ID_ADDR, &chip_id, 1);

		if ((chip_id & 0xff) != KXTJ3_ACC_CHIP_ID_VALUE) {
			continue;
		} else {
			err = 0;
			pr_info("kxtj3_acc_check_chip_id err = 0\n");
			break;
		}
	}
	return err;
}









int kxtj3_acc_sample(struct hf_device *hfdev)
{

	int err = 0;
	uint8_t buf[KXTJ3_ACC_AXES_NUM * 2] = {0};
	int32_t data[KXTJ3_ACC_AXES_NUM] = {0};
	struct hf_manager_event event;
	int64_t current_time;
//	struct i2c_client *client = hf_device_get_private_data(hfdev);
	struct kxtj3_acc_device *driver_dev = hf_device_get_private_data(hfdev);
	struct hf_manager *manager = driver_dev->hf_dev.manager;
pr_info("kxtj3_acc_sample\n");
	current_time = ktime_get_boot_ns();
	if (atomic_read(&driver_dev->acc_enable)) {
	err = kxtj3_acc_i2c_read_block(driver_dev->acc_client,
			KXTJ3_ACC_RATE_X_LSB_ADDR,
			&buf[0], KXTJ3_ACC_AXES_NUM * 2);
	if (err < 0) {
		pr_err_ratelimited("read fail\n");
		return err;
	}

	data[0] = ((int16_t)(buf[0] | (buf[1] << 8))) >> 4;
	data[1] = ((int16_t)(buf[2] | (buf[3] << 8))) >> 4;
	data[2] = ((int16_t)(buf[4] | (buf[5] << 8))) >> 4;

	 pr_err("kxtj3 register :%x, %x, %x %x, %x, %x data: %x, %x, %x\n",
	  buf[0], buf[1], buf[2],
	  buf[3], buf[4], buf[5],
	  data[0], data[1], data[2]);
	 
	coordinate_map(driver_dev->direction, data);
	current_time = ktime_get_boot_ns();

//	if (atomic_read(&driver_dev->raw_enable)) {
		memset(&event, 0, sizeof(struct hf_manager_event));
		event.timestamp = current_time;
		event.sensor_type = SENSOR_TYPE_ACCELEROMETER;
		event.accurancy = SENSOR_ACCURANCY_HIGH;
		event.action = RAW_ACTION;
		event.word[0] =
			(int32_t)div64_s64((int64_t)data[0] * 9807, 256);
		event.word[1] =
			(int32_t)div64_s64((int64_t)data[1] * 9807, 256);
		event.word[2] =
			(int32_t)div64_s64((int64_t)data[2] * 9807, 256);
		/* pr_err("kxtj3 raw %d, %d, %d,  m/s %d, %d, %d\n",
		 * data[0], data[1], data[2],
		 * event.word[0], event.word[1], event.word[2]);
		 */
		manager->report(manager, &event);
	}

	memset(&event, 0, sizeof(struct hf_manager_event));
	event.timestamp = current_time;
	event.sensor_type = SENSOR_TYPE_ACCELEROMETER;
	event.accurancy = SENSOR_ACCURANCY_HIGH;
	event.action = DATA_ACTION;

	event.word[0] = (int32_t)div64_s64((int64_t)data[0] * 1000, 512);
	event.word[1] = (int32_t)div64_s64((int64_t)data[1] * 1000, 512);
	event.word[2] = (int32_t)div64_s64((int64_t)data[2] * 1000, 512);
	
	/* pr_err("kxtj3 raw %d, %d, %d,  m/s %d, %d, %d\n",
	 * data[0], data[1], data[2],
	 * event.word[0], event.word[1], event.word[2]);
	 */
		
	manager->report(manager, &event);

	manager->complete(manager);

	return 0;
}

int kxtj3_acc_raw_enable(struct hf_device *hfdev, int sensor_type, int en)
{
	struct kxtj3_acc_device  *driver_dev = hf_device_get_private_data(hfdev);
//	struct kxtj3_acc_device *driver_dev = i2c_get_clientdata(client);
pr_info("kxtj3_acc_raw_enable\n");
	atomic_set(&driver_dev->raw_enable, en);
	return 0;
}

int kxtj3_acc_enable(struct hf_device *hfdev, int sensor_type, int en)
{
	int err = 0;
	uint8_t data = 0;
	struct kxtj3_acc_device *driver_dev = hf_device_get_private_data(hfdev);

	pr_info("%s id:%d en:%d\n", __func__, sensor_type, en);

	if (en) {
		
		kxtj3_acc_i2c_read_block(driver_dev->acc_client,KXTJ3_ACC_RANGE_ADDR, &data, 1);
		data |= 1<<7;
		err = kxtj3_acc_i2c_write_block(driver_dev->acc_client, KXTJ3_ACC_RANGE_ADDR, &data, 1);
		if (err < 0) {
			pr_err_ratelimited("write power mode failed.\n");
			return -1;
		}
		mdelay(1);
		atomic_set(&driver_dev->acc_enable, en);
	} else {
		data = 0x00;
		err = kxtj3_acc_i2c_write_block(driver_dev->acc_client,KXTJ3_ACC_RANGE_ADDR, &data, 1);
		if (err < 0) {
			pr_err_ratelimited("write power mode failed.\n");
			return -1;
		}
			atomic_set(&driver_dev->acc_enable, en);
		mdelay(1);
	}
	return 0;

}

int kxtj3_acc_batch(struct hf_device *hfdev, int sensor_type,
	int64_t delay, int64_t latency)
{

	int err = 0;
	uint8_t data = 0;
	int value = 0;
	struct kxtj3_acc_device *driver_dev = hf_device_get_private_data(hfdev);
	struct hf_manager *manager = driver_dev->hf_dev.manager;
	struct hf_manager_event event;
	int64_t current_time;

	pr_info("%s id:%d rate:%lld latency:%lld\n",
		__func__, sensor_type, delay, latency);

	value = div64_s64(1000000000LL, delay);

	if (value <= 7)
		data = KXTJ3_ACC_ODR_7HZ;
	else if (value <= 12)
		data = KXTJ3_ACC_ODR_12HZ;
	else if (value <= 25)
		data = KXTJ3_ACC_ODR_25HZ;
	else if (value <= 100)
		data = KXTJ3_ACC_ODR_100HZ;
	else if (value <= 200)
		data = KXTJ3_ACC_ODR_200HZ;
	else if (value <= 400)
		data = KXTJ3_ACC_ODR_400HZ;
	else if (value <= 800)
		data = KXTJ3_ACC_ODR_800HZ;
	else
		data = KXTJ3_ACC_ODR_1600HZ;

	err = kxtj3_acc_i2c_write_block(driver_dev->acc_client, KXTJ3_ACC_BW_ADDR, &data, 1);
	if (err < 0) {
		pr_err_ratelimited("write rate failed.\n");
		return -1;
	}
		memset(&event, 0, sizeof(struct hf_manager_event));
		current_time = ktime_get_boot_ns();
		event.timestamp = current_time;
		event.sensor_type = SENSOR_TYPE_ADDITIONAL_INFO;
		event.accurancy = SENSOR_ACCURANCY_HIGH;
		event.reserved = SENSOR_TYPE_ACCELEROMETER;
		event.action = DATA_ACTION;
		get_placement_info(driver_dev->direction, event.byte);
		event.byte[12] = 1;
		event.byte[3] = driver_dev->placement[0];
		event.byte[7] = driver_dev->placement[1];
		event.byte[11] = driver_dev->placement[2];

		manager->report(manager, &event);
	mdelay(1);
	manager->complete(manager);
	return 0;

}

static int kxtj3_acc_flush(struct hf_device *hfdev, int sensor_type)
{
	struct kxtj3_acc_device *driver_dev = hf_device_get_private_data(hfdev);
	struct hf_manager *manager = driver_dev->hf_dev.manager;
	struct hf_manager_event event;
	int64_t current_time;

	memset(&event, 0, sizeof(struct hf_manager_event));
	current_time = ktime_get_boot_ns();
	event.sensor_type = sensor_type;
	event.timestamp = current_time;
	event.action = FLUSH_ACTION;

	manager->report(manager, &event);
	manager->complete(manager);

	return 0;
}

static int kxtj3_acc_i2c_remove(struct i2c_client *client)
{
	struct kxtj3_acc_device *driver_dev = i2c_get_clientdata(client);
pr_info("kxtj3_acc_i2c_remove\n");
	hf_manager_destroy(driver_dev->hf_dev.manager);
	kfree(driver_dev);
	return 0;
}

static int kxtj3_acc_i2c_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{

	int err = 0;
	struct kxtj3_acc_device *driver_dev;
	struct sensorlist_info_t listinfo;

	pr_info("%s\n", __func__);

	driver_dev = devm_kzalloc(&client->dev, sizeof(*driver_dev),
								GFP_KERNEL);
	if (!driver_dev)
		return -ENOMEM;

//	driver_dev->acc_client = client;
	/* check chip id */
	err = kxtj3_acc_check_chip_id(client);
	if (err < 0) {
		pr_err("Bosch Sensortec Device not found, chip id mismatch\n");
		err = -EINVAL;
		goto init_fail;
	}

	err = kxtj3_acc_set_soft_reset(client);
	if (err < 0) {
		pr_err("erro soft reset!\n");
		err = -EINVAL;
		goto init_fail;
	}
	mdelay(3);

	err = kxtj3_acc_init_device(client);
	if (err < 0) {
		pr_err("%s init device fail\n", __func__);
		goto init_fail;
	}

	driver_dev = kzalloc(sizeof(*driver_dev), GFP_KERNEL);
	if (!driver_dev) {
		err = -ENOMEM;
		goto malloc_fail;
	}

	driver_dev->acc_client = client;

	if (of_property_read_u32(client->dev.of_node,
		"direction", &driver_dev->direction)) {
		pr_err("%s get direction dts fail\n", __func__);
		err = -EFAULT;
		goto dts_fail;
	}
if(0){
	if (of_property_read_u8_array(client->dev.of_node, "placement",
		driver_dev->placement, ARRAY_SIZE(driver_dev->placement))) {
		pr_err("%s get placement dts fail\n", __func__);
		err = -EFAULT;
		goto dts_fail;
	}
}
	atomic_set(&driver_dev->raw_enable, 0);

	driver_dev->hf_dev.dev_name = KXTJ3_ACC_I2C_NAME;
	driver_dev->hf_dev.device_poll = HF_DEVICE_IO_POLLING;
	driver_dev->hf_dev.device_bus = HF_DEVICE_IO_SYNC;
	driver_dev->hf_dev.support_list = support_sensors;
	driver_dev->hf_dev.support_size = ARRAY_SIZE(support_sensors);
	driver_dev->hf_dev.enable = kxtj3_acc_enable;
	driver_dev->hf_dev.batch = kxtj3_acc_batch;
	driver_dev->hf_dev.flush = kxtj3_acc_flush;
	driver_dev->hf_dev.sample = kxtj3_acc_sample;
	driver_dev->hf_dev.rawdata = kxtj3_acc_raw_enable;

	err = hf_manager_create(&driver_dev->hf_dev);
	if (err < 0) {
		pr_err("%s hf_manager_create fail\n", __func__);
		err = -1;
		goto create_manager_fail;
	}

	i2c_set_clientdata(client, driver_dev);
	hf_device_set_private_data(&driver_dev->hf_dev, driver_dev);

	//memset(&listinfo, 0, sizeof(struct sensorlist_info_t));
	//strlcpy(listinfo.name, KXTJ3_ACC_I2C_NAME, sizeof(listinfo.name));
	//sensorlist_register_devinfo(SENSOR_TYPE_ACCELEROMETER, &listinfo);
	memset(&listinfo, 0, sizeof(struct sensorlist_info_t));
	strlcpy(listinfo.name, KXTJ3_ACC_I2C_NAME, sizeof(listinfo.name));
	sensorlist_register_devinfo(SENSOR_TYPE_ACCELEROMETER, &listinfo);

	pr_info("%s success!\n", __func__);
	return 0;

create_manager_fail:
dts_fail:
	kfree(driver_dev);
malloc_fail:
init_fail:
	pr_err("%s fail!\n", __func__);
	return err;
}

static const struct of_device_id kxtj3_acc_of_match[] = {
	{.compatible = "mediatek,kxtj3_gsensor"},
	{},
};
static const struct i2c_device_id kxtj3_acc_i2c_id[] = {
					{KXTJ3_ACC_I2C_NAME, 0}, {} };

static struct i2c_driver kxtj3_acc_i2c_driver = {
	.driver = {
		.name = KXTJ3_ACC_I2C_NAME,
		.bus = &i2c_bus_type,
		.owner = THIS_MODULE,
		.of_match_table = kxtj3_acc_of_match,
	},
	.probe = kxtj3_acc_i2c_probe,
	.remove = kxtj3_acc_i2c_remove,
	.id_table =  kxtj3_acc_i2c_id,
};

module_i2c_driver(kxtj3_acc_i2c_driver);

MODULE_AUTHOR("Mediatek");
MODULE_DESCRIPTION("kxtj3 acc i2c driver");
MODULE_LICENSE("GPL");
