// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2019 MediaTek Inc.
 */


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/of.h>
#include <linux/delay.h>
#include <linux/math64.h>
#include <linux/atomic.h>

#include "hf_manager.h"
#include "smi130_i2c.h"
#include "sensor_list.h"

#define GYRO_TEMP_AMP    100
#define SMI130_I2C_NAME "smi130_i2c"
#define GYRO_ADDR               0x68
#define SMI130_AXES_NUM          3
#define C_I2C_FIFO_SIZE              8

#define SMI130_SOFT_RESET_VALUE  0xB6
#define SMI130_ACC_CHIP_ID_VALUE     (0xfa)
#define SMI130_GYRO_CHIP_ID_VALUE     (0x0f)
#define CHECK_CHIP_ID_TIME_MAX       5

#define SMI130_ACC_ODR_7HZ 0x08
#define SMI130_ACC_ODR_15HZ 0x09
#define SMI130_ACC_ODR_31HZ 0x0A
#define SMI130_ACC_ODR_62HZ 0x0B
#define SMI130_ACC_ODR_125HZ 0x0C
#define SMI130_ACC_ODR_250HZ 0x0D
#define SMI130_ACC_ODR_500HZ 0x0E
#define SMI130_ACC_ODR_1000HZ 0x0F
#define SMI130_ACC_RANGE_4G 0x03
#define SMI130_ACC_RANGE_8G 0x05
#define SMI130_ACC_RANGE_16G 0x08
#define SMI130_ACC_RANGE_32G 0x0C

#define SMI130_GYRO_ODR_32HZ 0x07
#define SMI130_GYRO_ODR_64HZ 0x06
#define SMI130_GYRO_ODR_12HZ 0x05
#define SMI130_GYRO_ODR_23HZ 0x04
#define SMI130_GYRO_ODR_47HZ 0x03
#define SMI130_GYRO_ODR_116HZ 0x02
#define SMI130_GYRO_ODR_230HZ 0x01
#define SMI130_GYRO_ODR_523HZ 0x00
#define SMI130_GYRO_RANGE_2000 0x00
#define SMI130_GYRO_RANGE_1000 0x01
#define SMI130_GYRO_RANGE_500 0x02
#define SMI130_GYRO_RANGE_250 0x03
#define SMI130_GYRO_RANGE_125 0x04

static unsigned char support_sensors[] = {
	SENSOR_TYPE_ACCELEROMETER,
	SENSOR_TYPE_GYROSCOPE,
	SENSOR_TYPE_GYRO_TEMPERATURE,
};

struct smi130_device {
	struct hf_device hf_dev;
	struct i2c_client *acc_client;
	struct i2c_client *gyro_client;
	uint32_t direction;
	uint8_t placement[3];
	atomic_t raw_acc_enable;
	atomic_t raw_gyro_enable;
	atomic_t acc_enable;
	atomic_t gyro_enable;
	atomic_t gyro_temp_enable;
};

/* I2C operation functions */
static int smi130_i2c_read_block(struct i2c_client *client,
			u8 addr, u8 *data, u8 len)
{
	int err = 0;
	u8 beg = addr;
	struct i2c_msg msgs[2] = {
		{/*.addr = client->addr,*/
		 .flags = 0,
		 .len = 1,
		 .buf = &beg},
		{
			/*.addr = client->addr*/
			.flags = I2C_M_RD,
			.len = len,
			.buf = data,
		} };
	if (!client)
		return -EINVAL;
	msgs[0].addr = client->addr;
	msgs[1].addr = client->addr;

	err = i2c_transfer(client->adapter, msgs, ARRAY_SIZE(msgs));
	if (err != 2) {
		pr_err_ratelimited("i2c_trans err: %x %x (%d %p %d) %d\n",
		       msgs[0].addr, client->addr, addr, data, len, err);
		err = -EIO;
	} else {
		err = 0; /*no error*/
	}
	return err;
}

static int smi130_i2c_write_block(struct i2c_client *client,
			u8 addr, u8 *data, u8 len)
{
	/* because address also occupies one byte,
	 * the maximum length for write is 7 bytes
	 */
	int err = 0, idx = 0, num = 0;
	char buf[32];

	if (!client)
		return -EINVAL;
	else if (len > C_I2C_FIFO_SIZE) {
		pr_err_ratelimited("len %d fi %d\n", len, C_I2C_FIFO_SIZE);
		return -EINVAL;
	}
	buf[num++] = addr;
	for (idx = 0; idx < len; idx++)
		buf[num++] = data[idx];

	err = i2c_master_send(client, buf, num);
	if (err < 0) {
		pr_err_ratelimited("send command error!!\n");
		return -EFAULT;
	}

	return err;
}


static int smi130_acc_init_device(struct i2c_client *client)
{
	int err = 0;
	uint8_t data = 0;

	data = SMI130_ACC_ODR_7HZ;
	err = smi130_i2c_write_block(client,
			SMI130_ACC_BW_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //filter and lock enable
	err = smi130_i2c_write_block(client,
			SMI130_ACC_RATED_HBW_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = SMI130_ACC_RANGE_16G;
	err = smi130_i2c_write_block(client,
			SMI130_ACC_RANGE_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //disable int
	err = smi130_i2c_write_block(client,
			SMI130_ACC_INT_ENABLE1_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x80; //map int
	err = smi130_i2c_write_block(client,
			SMI130_ACC_INT_MAP_1_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //push-pull, active low
	err = smi130_i2c_write_block(client,
			SMI130_ACC_INT_OUT_CTRL_ADDR, &data, 1);
	if (err < 0)
		return err;

	pr_debug("SMI130_ACC init OK.\n");

	return err;
}

static int smi130_acc_set_soft_reset(struct i2c_client *client)
{
	int err = 0;
	uint8_t data = SMI130_SOFT_RESET_VALUE;

	err = smi130_i2c_write_block(client,
			SMI130_ACC_BGW_SOFTRESET_ADDR, &data, 1);
	return err;
}

static int smi130_acc_check_chip_id(struct i2c_client *client)
{
	int err = -1;
	u8 chip_id = 0;
	u8 read_count = 0;

	while (read_count++ < CHECK_CHIP_ID_TIME_MAX) {
		smi130_i2c_read_block(client,
				SMI130_ACC_CHIP_ID_ADDR, &chip_id, 1);

		if ((chip_id & 0xff) != SMI130_ACC_CHIP_ID_VALUE) {
			continue;
		} else {
			err = 0;
			break;
		}
	}
	return err;
}

static int smi130_gyro_init_device(struct i2c_client *client)
{
	int err = 0;
	uint8_t data = 0;

	data = SMI130_GYRO_ODR_32HZ;
	err = smi130_i2c_write_block(client,
			SMI130_GYRO_BW_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //filter and lock enable
	err = smi130_i2c_write_block(client,
			SMI130_GYRO_RATED_HBW_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = SMI130_GYRO_RANGE_2000;
	err = smi130_i2c_write_block(client,
			SMI130_GYRO_RANGE_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //disable int
	err = smi130_i2c_write_block(client,
			SMI130_GYRO_INT_ENABLE0_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x01; //map int
	err = smi130_i2c_write_block(client,
			SMI130_GYRO_INT_MAP_1_ADDR, &data, 1);
	if (err < 0)
		return err;

	data = 0x00; //push-pull, active low
	err = smi130_i2c_write_block(client,
			SMI130_GYRO_INT_ENABLE1_ADDR, &data, 1);
	if (err < 0)
		return err;

	pr_debug("SMI130 gyro init OK.\n");

	return err;
}

static int smi130_gyro_check_chip_id(struct i2c_client *client)
{
	int err = -1;
	u8 chip_id = 0;
	u8 read_count = 0;

	while (read_count++ < CHECK_CHIP_ID_TIME_MAX) {
		smi130_i2c_read_block(client,
				SMI130_GYRO_CHIP_ID_ADDR, &chip_id, 1);

		if ((chip_id & 0xff) != SMI130_GYRO_CHIP_ID_VALUE) {
			continue;
		} else {
			err = 0;
			break;
		}
	}
	return err;
}

int smi130_sample(struct hf_device *hfdev)
{

	int err = 0;
	uint8_t buf[SMI130_AXES_NUM * 2] = {0};
	int32_t data[SMI130_AXES_NUM] = {0};
	int8_t gyro_temp[1] = {0};
	struct hf_manager_event event;
	int64_t current_time;
	struct smi130_device *driver_dev = hf_device_get_private_data(hfdev);
	struct hf_manager *manager = driver_dev->hf_dev.manager;

	current_time = ktime_get_boot_ns();
	if (atomic_read(&driver_dev->acc_enable)) {
		err = smi130_i2c_read_block(driver_dev->acc_client,
				SMI130_ACC_RATE_X_LSB_ADDR,
				&buf[0], SMI130_AXES_NUM * 2);
		if (err < 0) {
			pr_err_ratelimited("read fail\n");
			return err;
		}

		data[0] = ((int16_t)(buf[0] | (buf[1] << 8))) >> 4;
		data[1] = ((int16_t)(buf[2] | (buf[3] << 8))) >> 4;
		data[2] = ((int16_t)(buf[4] | (buf[5] << 8))) >> 4;

		coordinate_map(driver_dev->direction, data);

		if (atomic_read(&driver_dev->raw_acc_enable)) {
			memset(&event, 0, sizeof(struct hf_manager_event));
			event.timestamp = current_time;
			event.sensor_type = SENSOR_TYPE_ACCELEROMETER;
			event.accurancy = SENSOR_ACCURANCY_HIGH;
			event.action = RAW_ACTION;
			event.word[0] =
				(int32_t)div64_s64((int64_t)data[0] * 9807,
									256);
			event.word[1] =
				(int32_t)div64_s64((int64_t)data[1] * 9807,
									256);
			event.word[2] =
				(int32_t)div64_s64((int64_t)data[2] * 9807,
									256);
			manager->report(manager, &event);
		}

		memset(&event, 0, sizeof(struct hf_manager_event));
		event.timestamp = current_time;
		event.sensor_type = SENSOR_TYPE_ACCELEROMETER;
		event.accurancy = SENSOR_ACCURANCY_HIGH;
		event.action = DATA_ACTION;
		event.word[0] = (int32_t)div64_s64((int64_t)data[0] * 9807,
									256);
		event.word[1] = (int32_t)div64_s64((int64_t)data[1] * 9807,
									256);
		event.word[2] = (int32_t)div64_s64((int64_t)data[2] * 9807,
									256);
		manager->report(manager, &event);
	}

	if (atomic_read(&driver_dev->gyro_enable)) {
		err = smi130_i2c_read_block(driver_dev->gyro_client,
			SMI130_GYRO_RATE_X_LSB_ADDR, &buf[0],
						SMI130_AXES_NUM * 2);
		if (err < 0) {
			pr_err_ratelimited("read fail\n");
			return err;
		}

		data[0] = (int16_t)(buf[0] | (buf[1] << 8));
		data[1] = (int16_t)(buf[2] | (buf[3] << 8));
		data[2] = (int16_t)(buf[4] | (buf[5] << 8));

		coordinate_map(driver_dev->direction, data);

		/* read gyro temp */
		if (atomic_read(&driver_dev->gyro_temp_enable)) {
			err = smi130_i2c_read_block(driver_dev->gyro_client,
				SMI130_GYRO_TEMP_ADDR, &gyro_temp[0], 1);
			if (err < 0) {
				pr_err_ratelimited("read gyro temp fail\n");
				return err;
			}
		}
		if (atomic_read(&driver_dev->raw_gyro_enable)) {
			memset(&event, 0, sizeof(struct hf_manager_event));
			event.timestamp = current_time;
			event.sensor_type = SENSOR_TYPE_GYROSCOPE;
			event.accurancy = SENSOR_ACCURANCY_HIGH;
			event.action = RAW_ACTION;
			event.word[0] =
				(int32_t)div64_s64((int64_t)data[0] * 1310000,
									164);
			event.word[1] =
				(int32_t)div64_s64((int64_t)data[1] * 1310000,
									164);
			event.word[2] =
				(int32_t)div64_s64((int64_t)data[2] * 1310000,
									164);
			manager->report(manager, &event);
		}

		memset(&event, 0, sizeof(struct hf_manager_event));
		event.timestamp = current_time;
		event.sensor_type = SENSOR_TYPE_GYROSCOPE;
		event.accurancy = SENSOR_ACCURANCY_HIGH;
		event.action = DATA_ACTION;
		event.word[0] = (int32_t)div64_s64((int64_t)data[0] * 1310000,
									164);
		event.word[1] = (int32_t)div64_s64((int64_t)data[1] * 1310000,
									164);
		event.word[2] = (int32_t)div64_s64((int64_t)data[2] * 1310000,
									164);
		manager->report(manager, &event);

		if (atomic_read(&driver_dev->gyro_temp_enable)) {
			memset(&event, 0, sizeof(struct hf_manager_event));
			event.timestamp = current_time;
			event.sensor_type = SENSOR_TYPE_GYRO_TEMPERATURE;
			event.accurancy = SENSOR_ACCURANCY_HIGH;
			event.action = DATA_ACTION;
			event.word[0] = (int32_t)gyro_temp[0] * GYRO_TEMP_AMP /
						2 + GYRO_TEMP_AMP * 24;
			manager->report(manager, &event);
		}
	}

	manager->complete(manager);

	return 0;
}

int smi130_raw_enable(struct hf_device *hfdev, int sensor_type, int en)
{
	struct smi130_device *driver_dev = hf_device_get_private_data(hfdev);

	if (sensor_type == SENSOR_TYPE_ACCELEROMETER)
		atomic_set(&driver_dev->raw_acc_enable, en);
	else if (sensor_type == SENSOR_TYPE_GYROSCOPE)
		atomic_set(&driver_dev->raw_gyro_enable, en);
	return 0;
}

int smi130_enable(struct hf_device *hfdev, int sensor_type, int en)
{
	int err = 0;
	uint8_t data = 0;
	struct smi130_device *driver_dev = hf_device_get_private_data(hfdev);

	pr_debug("%s id:%d en:%d\n", __func__, sensor_type, en);

	if (sensor_type == SENSOR_TYPE_ACCELEROMETER) {
		if (en) {
			data = SMI130_ACC_DATA_INT_EN;
			err = smi130_i2c_write_block(driver_dev->acc_client,
					SMI130_ACC_INT_ENABLE1_ADDR, &data, 1);
			if (err < 0) {
				pr_err_ratelimited("write power mode failed\n");
				return -1;
			}
			atomic_set(&driver_dev->acc_enable, en);
		} else {
			data = 0x00;
			err = smi130_i2c_write_block(driver_dev->acc_client,
					SMI130_ACC_INT_ENABLE1_ADDR, &data, 1);
			if (err < 0) {
				pr_err_ratelimited("write power mode failed\n");
				return -1;
			}
			atomic_set(&driver_dev->acc_enable, en);
		}
	} else if (sensor_type == SENSOR_TYPE_GYROSCOPE) {
		if (en) {
			data = SMI130_GYRO_DATA_INT_EN;
			err = smi130_i2c_write_block(driver_dev->gyro_client,
					SMI130_GYRO_INT_ENABLE0_ADDR, &data, 1);
			if (err < 0) {
				pr_err_ratelimited("write power mode failed\n");
				return -1;
			}
			atomic_set(&driver_dev->gyro_enable, en);
		} else {
			data = 0x00;
			err = smi130_i2c_write_block(driver_dev->gyro_client,
					SMI130_GYRO_INT_ENABLE0_ADDR, &data, 1);
			if (err < 0) {
				pr_err_ratelimited("write power mode failed\n");
				return -1;
			}
			atomic_set(&driver_dev->gyro_enable, en);
		}
	} else if (sensor_type == SENSOR_TYPE_GYRO_TEMPERATURE)
		atomic_set(&driver_dev->gyro_temp_enable, en);

	return 0;

}

int smi130_batch(struct hf_device *hfdev, int sensor_type,
	int64_t delay, int64_t latency)
{

	int err = 0;
	uint8_t data = 0;
	int value = 0;
	struct smi130_device *driver_dev = hf_device_get_private_data(hfdev);
	struct hf_manager *manager = driver_dev->hf_dev.manager;
	struct hf_manager_event event;
	int64_t current_time;

	pr_debug("%s id:%d rate:%lld latency:%lld\n",
		__func__, sensor_type, delay, latency);

	value = div64_s64(1000000000LL, delay);

	if (sensor_type == SENSOR_TYPE_ACCELEROMETER) {
		if (value <= 7)
			data = SMI130_ACC_ODR_7HZ;
		else if (value <= 15)
			data = SMI130_ACC_ODR_15HZ;
		else if (value <= 31)
			data = SMI130_ACC_ODR_31HZ;
		else if (value <= 62)
			data = SMI130_ACC_ODR_62HZ;
		else if (value <= 125)
			data = SMI130_ACC_ODR_125HZ;
		else if (value <= 250)
			data = SMI130_ACC_ODR_250HZ;
		else if (value <= 500)
			data = SMI130_ACC_ODR_500HZ;
		else
			data = SMI130_ACC_ODR_1000HZ;

		err = smi130_i2c_write_block(driver_dev->acc_client,
					SMI130_ACC_BW_ADDR, &data, 1);
		if (err < 0) {
			pr_err_ratelimited("write rate failed.\n");
			return -1;
		}
		memset(&event, 0, sizeof(struct hf_manager_event));
		current_time = ktime_get_boot_ns();
		event.timestamp = current_time;
		event.sensor_type = SENSOR_TYPE_ADDITIONAL_INFO;
		event.accurancy = SENSOR_ACCURANCY_HIGH;
		event.reserved = SENSOR_TYPE_ACCELEROMETER;
		event.action = DATA_ACTION;
		get_placement_info(driver_dev->direction, event.byte);
		event.byte[12] = 1;
		event.byte[3] = driver_dev->placement[0];
		event.byte[7] = driver_dev->placement[1];
		event.byte[11] = driver_dev->placement[2];

		manager->report(manager, &event);
	} else if (sensor_type == SENSOR_TYPE_GYROSCOPE) {
		if (value <= 12)
			data = SMI130_GYRO_ODR_12HZ;
		else if (value <= 23)
			data = SMI130_GYRO_ODR_23HZ;
		else if (value <= 32)
			data = SMI130_GYRO_ODR_32HZ;
		else if (value <= 47)
			data = SMI130_GYRO_ODR_47HZ;
		else if (value <= 64)
			data = SMI130_GYRO_ODR_64HZ;
		else if (value <= 116)
			data = SMI130_GYRO_ODR_116HZ;
		else if (value <= 230)
			data = SMI130_GYRO_ODR_230HZ;
		else
			data = SMI130_GYRO_ODR_523HZ;

		err = smi130_i2c_write_block(driver_dev->gyro_client,
				SMI130_GYRO_BW_ADDR, &data, 1);
		if (err < 0) {
			pr_err_ratelimited("write rate failed.\n");
			return -1;
		}
		memset(&event, 0, sizeof(struct hf_manager_event));
		current_time = ktime_get_boot_ns();
		event.timestamp = current_time;
		event.sensor_type = SENSOR_TYPE_ADDITIONAL_INFO;
		event.accurancy = SENSOR_ACCURANCY_HIGH;
		event.reserved = SENSOR_TYPE_GYROSCOPE;
		event.action = DATA_ACTION;
		get_placement_info(driver_dev->direction, event.byte);
		event.byte[12] = 1;
		event.byte[3] = driver_dev->placement[0];
		event.byte[7] = driver_dev->placement[1];
		event.byte[11] = driver_dev->placement[2];

		manager->report(manager, &event);
	}

	manager->complete(manager);

	return 0;

}

static int smi130_flush(struct hf_device *hfdev, int sensor_type)
{
	struct smi130_device *driver_dev = hf_device_get_private_data(hfdev);
	struct hf_manager *manager = driver_dev->hf_dev.manager;
	struct hf_manager_event event;
	int64_t current_time;

	memset(&event, 0, sizeof(struct hf_manager_event));
	current_time = ktime_get_boot_ns();
	event.sensor_type = sensor_type;
	event.timestamp = current_time;
	event.action = FLUSH_ACTION;

	manager->report(manager, &event);
	manager->complete(manager);

	return 0;
}

static int smi130_i2c_remove(struct i2c_client *client)
{
	struct smi130_device *driver_dev = i2c_get_clientdata(client);

	hf_manager_destroy(driver_dev->hf_dev.manager);
	kfree(driver_dev);
	return 0;
}

static int smi130_i2c_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{

	int err = 0;
	struct smi130_device *driver_dev;
	struct sensorlist_info_t listinfo;

	pr_info("%s\n", __func__);

	driver_dev = devm_kzalloc(&client->dev, sizeof(*driver_dev),
								GFP_KERNEL);
	if (!driver_dev)
		return -ENOMEM;

	driver_dev->acc_client = client;
	driver_dev->gyro_client = i2c_new_dummy(client->adapter, GYRO_ADDR);
	if (!driver_dev->gyro_client) {
		pr_err("Failed to allocate i2c device for gyro\n");
		return -ENODEV;
	}

	/* acc init */
	err = smi130_acc_check_chip_id(client);
	if (err < 0) {
		pr_err("smi130 acc chip id mismatch\n");
		err = -EINVAL;
		goto init_fail;
	}

	err = smi130_acc_set_soft_reset(client);
	if (err < 0) {
		pr_err("erro soft reset!\n");
		err = -EINVAL;
		goto init_fail;
	}

	err = smi130_acc_init_device(client);
	if (err < 0) {
		pr_err("%s init device fail\n", __func__);
		goto init_fail;
	}

	/* gyro init */
	err = smi130_gyro_check_chip_id(driver_dev->gyro_client);
	if (err < 0) {
		pr_err("smi130 gyro chip id mismatch\n");
		err = -EINVAL;
		goto init_fail;
	}

	err = smi130_gyro_init_device(driver_dev->gyro_client);
	if (err < 0) {
		pr_err("%s init device fail\n", __func__);
		goto init_fail;
	}

	if (of_property_read_u32(client->dev.of_node,
		"direction", &driver_dev->direction)) {
		pr_err("%s get direction dts fail\n", __func__);
		err = -EFAULT;
		goto dts_fail;
	}

	if (of_property_read_u8_array(client->dev.of_node, "placement",
		driver_dev->placement, ARRAY_SIZE(driver_dev->placement))) {
		pr_err("%s get placement dts fail\n", __func__);
		err = -EFAULT;
		goto dts_fail;
	}

	atomic_set(&driver_dev->raw_acc_enable, 0);
	atomic_set(&driver_dev->raw_gyro_enable, 0);

	driver_dev->hf_dev.dev_name = SMI130_I2C_NAME;
	driver_dev->hf_dev.device_poll = HF_DEVICE_IO_POLLING;
	driver_dev->hf_dev.device_bus = HF_DEVICE_IO_SYNC;
	driver_dev->hf_dev.support_list = support_sensors;
	driver_dev->hf_dev.support_size = ARRAY_SIZE(support_sensors);
	driver_dev->hf_dev.enable = smi130_enable;
	driver_dev->hf_dev.batch = smi130_batch;
	driver_dev->hf_dev.flush = smi130_flush;
	driver_dev->hf_dev.sample = smi130_sample;
	driver_dev->hf_dev.rawdata = smi130_raw_enable;

	err = hf_manager_create(&driver_dev->hf_dev);
	if (err < 0) {
		pr_err("%s hf_manager_create fail\n", __func__);
		err = -1;
		goto create_manager_fail;
	}

	i2c_set_clientdata(client, driver_dev);
	hf_device_set_private_data(&driver_dev->hf_dev, driver_dev);

	memset(&listinfo, 0, sizeof(struct sensorlist_info_t));
	strlcpy(listinfo.name, SMI130_I2C_NAME, sizeof(listinfo.name));
	sensorlist_register_devinfo(SENSOR_TYPE_ACCELEROMETER, &listinfo);
	sensorlist_register_devinfo(SENSOR_TYPE_GYROSCOPE, &listinfo);

	pr_info("%s success!\n", __func__);
	return 0;

create_manager_fail:
dts_fail:
init_fail:
	i2c_unregister_device(driver_dev->gyro_client);
	return err;
}

static const struct of_device_id smi130_acc_of_match[] = {
	{.compatible = "mediatek,smi130_sensor"},
	{},
};
static const struct i2c_device_id smi130_acc_i2c_id[] = {
					{SMI130_I2C_NAME, 0}, {} };

static struct i2c_driver smi130_acc_i2c_driver = {
	.driver = {
		.name = SMI130_I2C_NAME,
		.bus = &i2c_bus_type,
		.owner = THIS_MODULE,
		.of_match_table = smi130_acc_of_match,
	},
	.probe = smi130_i2c_probe,
	.remove = smi130_i2c_remove,
	.id_table =  smi130_acc_i2c_id,
};

module_i2c_driver(smi130_acc_i2c_driver);

MODULE_AUTHOR("Mediatek");
MODULE_DESCRIPTION("smi130 acc i2c driver");
MODULE_LICENSE("GPL");
