#!/usr/bin/env python3

import ecc_bch
import sys
import os
import struct
import argparse
import pp_tbl

def calc_bcc(buf, buf_len):
	raw = bytearray(buf)
	bcc = 0

	for i in range(buf_len):
		bcc ^= raw[i]

	return bcc

def gen_header(nand_name):
	print ("NAND: %s" % (nand_name))
	with open(os.getcwd() + "/tools/nand-utils/nand_device_list_v4.txt", "r") as f:
		found = 0
		info_valid = 0
		nfi_valid = 0
		mag_valid = 0
		pp_valid = 0
		for l in f:
			s = l.split()
			if s[0] == nand_name:
				found = 1
				nfi_start_idx = 0
				mag_start_idx = 0

				enable_header = int(s[1])
				if enable_header == 0:
					break
				info_valid = int(s[2])
				nfi_valid = int(s[3])
				mag_valid = int(s[4])
				pp_valid = int(s[5])
				pp_type = s[6]
				if info_valid == 1:
					nand_type = int(s[7])
					io_width = int(s[8])
					addr_cycle = int(s[9])
					target_num = int(s[10])
					lun_num = int(s[11])
					plane_num = int(s[12])
					program_order_type = int(s[13])
					address_table_type = int(s[14])
					bad_block_type = int(s[15])
					read_retry_type = int(s[16])
					interface_type = int(s[17])
					sdr_timing_type = int(s[18])
					ddr_timing_type = int(s[19])
					block_num = int(s[20])
					block_size = int(s[21])
					page_size = int(s[22])
					spare_size = int(s[23])
					mode_type = int(s[24])
					slc_pe_cycle = int(s[25])
					slc_ecc_required = int(s[26])
					slc_bitflip_thres = int(s[27])
					xlc_pe_cycle = int(s[28])
					xlc_ecc_required = int(s[29])
					xlc_bitflip_thres = int(s[30])
					nfi_start_idx = 31
					mag_start_idx = 31
					if nand_type == 0:
						nfc_format = 1
						rand_en = 0
					else:
						nfc_format = 0
						rand_en = 1
				if nfi_valid == 1:
					dqs_dly_sel = int(s[nfi_start_idx])
					dqs_dly_mux = int(s[nfi_start_idx + 1])
					rd_dqs_ddr_timing = int(s[nfi_start_idx + 2])
					mag_start_idx = nfi_start_idx + 3
				if mag_valid == 1:
					bm_type = int(s[mag_start_idx])
					dma_en = int(s[mag_start_idx + 1])
					ddr_en = int(s[mag_start_idx + 2])
					nfc_format = int(s[mag_start_idx + 3])
				break #found

		if not found:
			raise KeyError("not support " + nand_name)

	# nand header is total 440B include ecc parity 140B #
	header_size = 300
	info_size = 64
	setting_size = 16
	mag_size = 8
	pptbl_size = 128
	dummy_size = header_size - 4 - info_size - setting_size - mag_size - pptbl_size - 3

	header = b"NAND"

	# struct nand_info #
	if info_valid == 0:
		header += '\x00' * info_size
		bcc_chip = '\xff'
	else:
		info_buf = b"INFO"
		info_buf += struct.pack("B", nand_type)
		info_buf += struct.pack("B", io_width)
		info_buf += struct.pack("B", addr_cycle)
		info_buf += struct.pack("B", target_num)
		info_buf += struct.pack("B", lun_num)
		info_buf += struct.pack("B", plane_num)
		info_buf += struct.pack("B", program_order_type)
		info_buf += struct.pack("B", address_table_type)
		info_buf += struct.pack("B", bad_block_type)
		info_buf += struct.pack("B", read_retry_type)
		info_buf += struct.pack("B", interface_type)
		info_buf += struct.pack("B", sdr_timing_type)
		info_buf += struct.pack("B", ddr_timing_type)
		info_buf += b'\xff' * 3
		info_buf += struct.pack("I", block_num)
		info_buf += struct.pack("I", block_size)
		info_buf += struct.pack("I", page_size)
		info_buf += struct.pack("I", spare_size)
		info_buf += struct.pack("I", mode_type)
		info_buf += struct.pack("I", slc_pe_cycle)
		info_buf += struct.pack("I", slc_ecc_required)
		info_buf += struct.pack("I", slc_bitflip_thres)
		info_buf += struct.pack("I", xlc_pe_cycle)
		info_buf += struct.pack("I", xlc_ecc_required)
		info_buf += struct.pack("I", xlc_bitflip_thres)
		header += info_buf
		bcc_chip = calc_bcc(info_buf, info_size)
		bcc_chip = struct.pack("B", bcc_chip)

	# struct nfi_setting #
	if nfi_valid == 0:
		header += b'\x00' * setting_size
		bcc_setting = b'\xff'
	else:
		setting_buf = "NFIS"
		setting_buf += struct.pack("I", dqs_dly_sel)
		setting_buf += struct.pack("I", dqs_dly_mux)
		setting_buf += struct.pack("I", rd_dqs_ddr_timing)
		header += setting_buf
		bcc_setting = calc_bcc(setting_buf, setting_size)
		bcc_setting = struct.pack("B", bcc_setting)

	# struct nand_manage #
	if mag_valid == 0:
		header += b'\x00' * mag_size
		bcc_mag = b'\xff'
	else:
		mag_buf = "MANG"
		mag_buf += struct.pack("B", bm_type)
		mag_buf += struct.pack("B", dma_en)
		mag_buf += struct.pack("B", ddr_en)
		mag_buf += struct.pack("B", nfc_format)
		header += mag_buf
		bcc_mag = calc_bcc(mag_buf, mag_size)
		bcc_mag = struct.pack("B", bcc_mag)

	# bcc code #
	header += bcc_chip
	header += bcc_setting
	header += bcc_mag

	# pad dummy byte #
	header += b'\xff' * dummy_size

	if pp_valid == 1:
		# add pptable #
		if pp_type == 'TSB':
			header += str(bytearray(pp_tbl.TSB_PP))
		else:
			raise KeyError(" no %s pptable " % vendor)
	else:
		header += b'\xff' * pptbl_size

	# calculate header ecc parity #
	ecc_level = 80
	ecc = ecc_bch.bch_enc_14(header, header_size, ecc_level)
	header += ecc

	header_file = nand_name + '_header.bin'
	with open(header_file, "wb") as f:
		f.write(header)

	if info_valid == 0:
		print ("NAND info invalid, return!")
		return

	# feedback ecc & randomizer generateion info
	fdm_size = spare_size / (page_size / 1024) - (xlc_ecc_required * 14 + 7) / 8
	if fdm_size > 8:
		fdm_size = 8
	if nfc_format == 0:
		fdmecc_size = fdm_size
		bad_swap = 0
	else:
		fdmecc_size = 1
		bad_swap = 1
	ppb = block_size / page_size

	return (pp_type, fdm_size, fdmecc_size, xlc_ecc_required, spare_size, page_size, pp_valid, ppb, bad_swap, rand_en)

def main(argv):
	parser = argparse.ArgumentParser()
	parser.add_argument('nand_name', help = 'nand device name')
	parser.add_argument('gpt_image', help = 'pmbr + gpt entry image ')
	args = parser.parse_args()

	# generate nand device header #
	gen_header(args.nand_name)

	# pad nand header #
	with open(args.nand_name + '_header.bin', "rb") as f_h:
		h_buf = f_h.read()
	with open(args.gpt_image, "rb+") as f_g:
		f_g.seek(440, 0)
		h_buf = h_buf + f_g.read()
		f_g.seek(0, 0)
		f_g.write(h_buf)

	os.remove(args.nand_name + '_header.bin')

if __name__ == "__main__":
	sys.exit(main(sys.argv))
