/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#ifndef TEL_DEMO_SRC_UTIL_LOG_EXTRA_H_
#define TEL_DEMO_SRC_UTIL_LOG_EXTRA_H_

/*****************************************************************************
 * Include
 *****************************************************************************/

#include <log/log.h>

/*****************************************************************************
 * Example
 *****************************************************************************/
/* Add a debug log with your tag, write it like:
 * LOG_D(tag, "this is a sample");
 *
 * Print a variable, write it like:
 * LOG_D(tag, "this is a sample %d", variable);
 *
 * Print multi-variable, write it like:
 * LOG_D(tag, "this is a sample %d,%d", variable1, variable2);
 *
 * Staple output format
 * %c  char
 * %s  char* string
 * %d  sign decimal
 * %p  pointer
 * %x  hex
 *
 * Add a debug log with your condition and tag, write it like:
 * LOG_D_IF(condition, tag, "this is a sample");
 * When condition is not 0 (this is true), the log will be printed, otherwise, no log printed.
 *
 */

/*****************************************************************************
 * Define
 *****************************************************************************/

/*
 * Simplified macro to send a verbose radio log message using the user given tag - _tag.
 */
#ifndef LOG_V
#define __LOG_V(_tag, ...) \
    ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_VERBOSE, _tag, __VA_ARGS__))
#if LOG_NDEBUG
#define LOG_V(_tag, ...) do { if (0) { __LOG_V(_tag, __VA_ARGS__); } } while (0)
#else
#define LOG_V(_tag, ...) __LOG_V(_tag, __VA_ARGS__)
#endif
#endif

#define CONDITION(cond)     (__builtin_expect((cond) != 0, 0))

#ifndef LOG_V_IF
#if LOG_NDEBUG
#define LOG_V_IF(cond, _tag, ...)   ((void)0)
#else
#define LOG_V_IF(cond, _tag, ...) \
    ( (CONDITION(cond)) \
    ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_VERBOSE, _tag, __VA_ARGS__)) \
    : (void)0 )
#endif
#endif

/*
 * Simplified macro to send a debug radio log message using the user given tag - _tag.
 */
#ifndef LOG_D
#define LOG_D(_tag, ...) \
    ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_DEBUG, _tag, __VA_ARGS__))
#endif

#ifndef LOG_D_IF
#define LOG_D_IF(cond, _tag, ...) \
    ( (CONDITION(cond)) \
    ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_DEBUG, _tag, __VA_ARGS__)) \
    : (void)0 )
#endif

/*
 * Simplified macro to send an info radio log message using the user given tag - _tag.
 */
#ifndef LOG_I
#define LOG_I(_tag, ...) \
    ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_INFO, _tag, __VA_ARGS__))
#endif

#ifndef LOG_I_IF
#define LOG_I_IF(cond, _tag, ...) \
    ( (CONDITION(cond)) \
    ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_INFO, _tag, __VA_ARGS__)) \
    : (void)0 )
#endif

/*
 * Simplified macro to send a warning radio log message using the user given tag - _tag.
 */
#ifndef LOG_W
#define LOG_W(_tag, ...) \
    ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_WARN, _tag, __VA_ARGS__))
#endif

#ifndef LOG_W_IF
#define LOG_W_IF(cond, _tag, ...) \
    ( (CONDITION(cond)) \
    ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_WARN, _tag, __VA_ARGS__)) \
    : (void)0 )
#endif

/*
 * Simplified macro to send an error radio log message using the user given tag - _tag.
 */
#ifndef LOG_E
#define LOG_E(_tag, ...) \
    ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_ERROR, _tag, __VA_ARGS__))
#endif

#ifndef LOG_E_IF
#define LOG_E_IF(cond, _tag, ...) \
    ( (CONDITION(cond)) \
    ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_ERROR, _tag, __VA_ARGS__)) \
    : (void)0 )
#endif

#ifndef ASSERT_EX
#define ASSERT_EX(_expr)                                                         \
    do {                                                                          \
        if (!(_expr)) {                                                           \
            LOG_E("ASSERT_EX", "ASSERT_EX:%s, %d", __FILE__, __LINE__);     \
            LOG_ALWAYS_FATAL();                                        \
        }                                                                         \
    } while(0)
#endif



#endif /* TEL_DEMO_SRC_UTIL_LOG_EXTRA_H_ */
