 /*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef CDMASMSCBPROGRAMDATA_H_
#define CDMASMSCBPROGRAMDATA_H_
#include <string>

class CdmaSmsCbProgramData {
public:
  CdmaSmsCbProgramData(int operation, int category, int language,
      int maxMessages, int alertOption, std::string categoryName);
  virtual ~CdmaSmsCbProgramData();

  /** Delete the specified service category from the list of enabled categories. */
  static constexpr int OPERATION_DELETE_CATEGORY = 0;

  /** Add the specified service category to the list of enabled categories. */
  static constexpr int OPERATION_ADD_CATEGORY = 1;

  /** Clear all service categories from the list of enabled categories. */
  static constexpr int OPERATION_CLEAR_CATEGORIES = 2;

  /** Alert option: no alert. */
  static constexpr int ALERT_OPTION_NO_ALERT = 0;

  /** Alert option: default alert. */
  static constexpr int ALERT_OPTION_DEFAULT_ALERT = 1;

  /** Alert option: vibrate alert once. */
  static constexpr int ALERT_OPTION_VIBRATE_ONCE = 2;

  /** Alert option: vibrate alert - repeat. */
  static constexpr int ALERT_OPTION_VIBRATE_REPEAT = 3;

  /** Alert option: visual alert once. */
  static constexpr int ALERT_OPTION_VISUAL_ONCE = 4;

  /** Alert option: visual alert - repeat. */
  static constexpr int ALERT_OPTION_VISUAL_REPEAT = 5;

  /** Alert option: low-priority alert once. */
  static constexpr int ALERT_OPTION_LOW_PRIORITY_ONCE = 6;

  /** Alert option: low-priority alert - repeat. */
  static constexpr int ALERT_OPTION_LOW_PRIORITY_REPEAT = 7;

  /** Alert option: medium-priority alert once. */
  static constexpr int ALERT_OPTION_MED_PRIORITY_ONCE = 8;

  /** Alert option: medium-priority alert - repeat. */
  static constexpr int ALERT_OPTION_MED_PRIORITY_REPEAT = 9;

  /** Alert option: high-priority alert once. */
  static constexpr int ALERT_OPTION_HIGH_PRIORITY_ONCE = 10;

  /** Alert option: high-priority alert - repeat. */
  static constexpr int ALERT_OPTION_HIGH_PRIORITY_REPEAT = 11;
  /**
   * Returns the service category operation, e.g. {@link #OPERATION_ADD_CATEGORY}.
   * @return one of the {@code OPERATION_*} values
   */
  int getOperation();

  /**
   * Returns the CDMA service category to modify.
   * @return a 16-bit CDMA service category value
   */
  int getCategory();

  /**
   * Returns the CDMA language code for this service category.
   * @return one of the language values defined in BearerData.LANGUAGE_*
   */
  int getLanguage();

  /**
   * Returns the maximum number of messages to store for this service category.
   * @return the maximum number of messages to store for this service category
   */
  int getMaxMessages();

  /**
   * Returns the service category alert option, e.g. {@link #ALERT_OPTION_DEFAULT_ALERT}.
   * @return one of the {@code ALERT_OPTION_*} values
   */
  int getAlertOption();

  /**
   * Returns the service category name, in the language specified by {@link #getLanguage()}.
   * @return an optional service category name
   */
  std::string getCategoryName();

  std::string toString();

  /**
   * Describe the kinds of special objects contained in the marshalled representation.
   * @return a bitmask indicating this Parcelable contains no special objects
   */
  int describeContents();

private:
  /** Service category operation (add/delete/clear). */
  int mOperation;

  /** Service category to modify. */
  int mCategory;

  /** Language used for service category name (defined in BearerData.LANGUAGE_*). */
  int mLanguage;

  /** Maximum number of messages to store for this service category. */
  int mMaxMessages;

  /** Service category alert option. */
  int mAlertOption;

  /** Name of service category. */
  std::string mCategoryName;
};

#endif /* CDMASMSCBPROGRAMDATA_H_ */
