 // SPDX-License-Identifier: MediaTekProprietary
 /*
 * Copyright (C) 2006 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cstdarg>
#include <cstdio>
#include <string>
#include <algorithm>
#include <cutils/properties.h>
#include <errno.h>
#include <inttypes.h>

#include "utils.h"
#include "log_extra.h"

#undef LOG_TAG
#define LOG_TAG "DEMO_UTILS"

constexpr int utils::MODEM_UNKNOWN;
constexpr int utils::MODEM_GSM;
constexpr int utils::MODEM_TDSCDMA;
constexpr int utils::MODEM_WCDMA;
constexpr int utils::MODEM_LTE_TDD;
constexpr int utils::MODEM_LTE_FDD;
constexpr int utils::MODEM_CDMA_EVDO;
constexpr int utils::MODEM_CDMA_1X;

utils::utils() {
    // TODO Auto-generated constructor stub

}

utils::~utils() {
    // TODO Auto-generated destructor stub
}

bool utils::is93ModemAndAbove() {
#if defined(MD_93_SUPPORT) || defined(MD_97_SUPPORT)
    return true;
#else
    return false;
#endif
}

bool utils::is97Modem() {
#ifdef MD_97_SUPPORT
    return true;
#else
    return false;
#endif
}

bool utils::is93Modem() {
#ifdef MD_93_SUPPORT
    return true;
#else
    return false;
#endif
}

bool utils::is90Modem() {
#ifdef MD_90_SUPPORT
    return true;
#else
    return false;
#endif
}

bool utils::isC2KSupport() {
#ifdef C2K_SUPPORT
    return true;
#else
    return false;
#endif
}


bool utils::isMt2635() {
#ifdef TARGET_PLATFORM_MT2635
    return true;
#else
    return false;
#endif
}

bool utils::isMt2731(){
#ifdef TARGET_PLATFORM_MT2731
    return true;
#else
    return false;
#endif
}

bool utils::isMt2735(){
#ifdef TARGET_PLATFORM_MT2735
    return true;
#else
    return false;
#endif
}

bool utils::is_support_dsds(){
#ifdef MODE_DSDS
    return true;
#else
    return false;
#endif
}

bool utils::is_suppport_dsss(){
#ifdef MODE_DSSS
    return true;
#else
    return false;
#endif
}

/*
 * Get property
 */
int utils::mtk_property_get(const char *key, char *value, const char *default_value)
{
    int ali_pro_res = property_get(key, value, default_value);
    LOG_D(LOG_TAG, "get key is %s, value is %s, result: %d", key, value, ali_pro_res);
    return ali_pro_res;
}

/*
 * Set property
 */
int utils::mtk_property_set(const char *key, const char *value)
{
    int ret_val = property_set(key, value);
    LOG_D(LOG_TAG, "set key is %s, value is %s,result: %d", key, value, ret_val);
    return ret_val;
}

bool utils::mtk_property_get_bool(const char *key, bool default_value) {
    if (!key) {
        return default_value;
    }

    bool result = default_value;
    char buf[PROPERTY_VALUE_MAX] = {'\0',};

    int len = property_get(key, buf, "");
    if (len == 1) {
        char ch = buf[0];
        if (ch == '0' || ch == 'n') {
            result = false;
        } else if (ch == '1' || ch == 'y') {
            result = true;
        }
    } else if (len > 1) {
         if (!strcmp(buf, "no") || !strcmp(buf, "false") || !strcmp(buf, "off")) {
            result = false;
        } else if (!strcmp(buf, "yes") || !strcmp(buf, "true") || !strcmp(buf, "on")) {
            result = true;
        }
    }

    return result;
}

intmax_t utils::property_get_imax(const char *key, intmax_t lower_bound, intmax_t upper_bound,
        intmax_t default_value) {
    if (!key) {
        return default_value;
    }

    intmax_t result = default_value;
    char buf[PROPERTY_VALUE_MAX] = {'\0',};
    char *end = NULL;

    int len = property_get(key, buf, "");
    if (len > 0) {
        int tmp = errno;
        errno = 0;

        // Infer base automatically
        result = strtoimax(buf, &end, /*base*/0);
        if ((result == INTMAX_MIN || result == INTMAX_MAX) && errno == ERANGE) {
            // Over or underflow
            result = default_value;
            ALOGV("%s(%s,%" PRIdMAX ") - overflow", __FUNCTION__, key, default_value);
        } else if (result < lower_bound || result > upper_bound) {
            // Out of range of requested bounds
            result = default_value;
            ALOGV("%s(%s,%" PRIdMAX ") - out of range", __FUNCTION__, key, default_value);
        } else if (end == buf) {
            // Numeric conversion failed
            result = default_value;
            ALOGV("%s(%s,%" PRIdMAX ") - numeric conversion failed",
                    __FUNCTION__, key, default_value);
        }

        errno = tmp;
    }

    return result;
}

int64_t utils::mtk_property_get_int64(const char *key, int64_t default_value){
    return (int64_t)property_get_imax(key, INT64_MIN, INT64_MAX, default_value);
}

int32_t utils::mtk_property_get_int32(const char *key, int32_t default_value) {
    return (int32_t)property_get_imax(key, INT32_MIN, INT32_MAX, default_value);
}

int utils::find_index(std::vector<std::string> v, std::string &str) {
    auto is_find = std::find(v.begin(), v.end(), str);
    int index = -1;
    if(is_find != v.end()) {
        index = std::distance(v.begin(), is_find);
        LOG_D(LOG_TAG,"find_index: %d, band: %s", index, str.c_str());
    }
    return index;
}

std::string utils::format(const std::string& format, ...) {
    va_list args;
    va_start (args, format);
    size_t len = std::vsnprintf(NULL, 0, format.c_str(), args);
    va_end(args);
    std::vector<char> vec(len + 1);
    va_start(args, format);
    std::vsnprintf(&vec[0], len + 1, format.c_str(), args);
    va_end(args);
    return &vec[0];
}

void utils::tokenize(std::string const &str, const char delim, std::vector<std::string> &out){
    std::stringstream ss(str);
    std::string s;
    while(std::getline(ss, s ,delim)) {
        out.push_back(s);
    }
}

void utils::tokenize(std::string const &str, const char* delim, std::vector<std::string> &out){
    char* token = strtok(const_cast<char*>(str.c_str()), delim);
    while (token != nullptr) {
        out.push_back(std::string(token));
        token = strtok(nullptr, delim);
    }
}

std::string utils::addZeroForNum(std::string const &str, int strLength) {
    std::string result_str;
    result_str.append(str);

    while (result_str.length() < strLength) {
        result_str.append("0");
    }

    LOG_D(LOG_TAG,"addZeroForNum- result_str=%s", result_str.c_str());

    return result_str;
}

void utils::setMSimProperty(int phoneId, char *pPropertyName, char *pUpdateValue) {
    #define MAX_PHONE_NUM 10
    #define MIN(a,b) ((a)<(b) ? (a) : (b))

    char oldItemValue[PROPERTY_VALUE_MAX] = {0};
    char newPropertyValue[PROPERTY_VALUE_MAX] = {0};
    int i = 0;
    int strLen = 0;

    for (i = 0; i < MAX_PHONE_NUM; i++) {
        if (i == phoneId) {
            // use new value
            strncat(newPropertyValue, pUpdateValue, PROPERTY_VALUE_MAX - strlen(newPropertyValue));
        } else {
            getMSimProperty(i, pPropertyName, oldItemValue);
            strncat(newPropertyValue, oldItemValue, PROPERTY_VALUE_MAX - strlen(newPropertyValue));
        }
        if (i != MAX_PHONE_NUM-1) {
            strncat(newPropertyValue, ",", 1);
        }
        memset(oldItemValue, 0, PROPERTY_VALUE_MAX);
    }
    LOG_D(LOG_TAG,"setMSimProperty phoneId=%d, newPropertyValue=%s", phoneId, newPropertyValue);
    // remove no use ','
    strLen = strlen(newPropertyValue);
    for (i = (strLen-1); i >= 0; i--) {
        if (newPropertyValue[i] == ',') {
            // remove
            newPropertyValue[i] = '\0';
        } else {
            break;
        }
    }
    LOG_D(LOG_TAG,"newPropertyValue %s\n", newPropertyValue);
    mtk_property_set(pPropertyName, newPropertyValue);
}

void utils::getMSimProperty(int phoneId, char *pPropertyName,char *pPropertyValue) {
    char prop[PROPERTY_VALUE_MAX] = {0};
    char value[PROPERTY_VALUE_MAX] = {0};
    int count= 0;
    int propLen = 0;
    int i = 0;
    int j = 0;

    mtk_property_get(pPropertyName, prop, "");
    LOG_D(LOG_TAG,"getMSimProperty pPropertyName=%s, prop=%s", pPropertyName, prop);
    propLen = strlen(prop);
    for (i = 0; i < propLen; i++) {
        if(prop[i] == ',') {
            count++;
            if((count-1) == phoneId) {
                // return current buffer
                LOG_D(LOG_TAG,"getMSimProperty found! phoneId=%d, value =%s", phoneId, value);
                strncpy(pPropertyValue, value, strlen(value));
                return;
            } else {
                // clear current buffer
                j = 0;
                memset(value, 0, sizeof(char) * PROPERTY_VALUE_MAX);
            }
        } else {
            value[j] = prop[i];
            j++;
        }
    }
    if (count == phoneId) {
        strncpy(pPropertyValue, value, strlen(value));
        LOG_D(LOG_TAG,"getMSimProperty found at end! phoneId=%d, value =%s", phoneId, value);
    }
}

bool utils::is_number(const std::string& s) {
    std::string::const_iterator it = s.begin();
    while(it != s.end() && std::isdigit(*it))
    {
        ++it;
    }
    return !s.empty() && it == s.end();
}
