#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <pthread.h>
#include <unistd.h>
#include <tee_client_api.h>
#include <ta_storage.h>
#include <tee_api_defines.h>

#define TEEC_OPERATION_INITIALIZER {}

static TEEC_Result fs_open(TEEC_Session *sess, void *id, uint32_t id_size,
			   uint32_t flags, uint32_t *obj, uint32_t storage_id)
{
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	TEEC_Result res = TEEC_ERROR_GENERIC;
	uint32_t org = 0;

	op.params[0].tmpref.buffer = id;
	op.params[0].tmpref.size = id_size;
	op.params[1].value.a = flags;
	op.params[1].value.b = 0;
	op.params[2].value.a = storage_id;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_INPUT,
					 TEEC_VALUE_INOUT, TEEC_VALUE_INPUT,
					 TEEC_NONE);

	res = TEEC_InvokeCommand(sess, TA_STORAGE_CMD_OPEN, &op, &org);

	if (res == TEEC_SUCCESS)
		*obj = op.params[1].value.b;

	return res;
}

static TEEC_Result fs_create(TEEC_Session *sess, void *id, uint32_t id_size,
			     uint32_t flags, uint32_t attr, void *data,
			     uint32_t data_size, uint32_t *obj,
			     uint32_t storage_id)
{
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	TEEC_Result res = TEEC_ERROR_GENERIC;
	uint32_t org = 0;

	op.params[0].tmpref.buffer = id;
	op.params[0].tmpref.size = id_size;
	op.params[1].value.a = flags;
	op.params[1].value.b = 0;
	op.params[2].value.a = attr;
	op.params[2].value.b = storage_id;
	op.params[3].tmpref.buffer = data;
	op.params[3].tmpref.size = data_size;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_INPUT,
					 TEEC_VALUE_INOUT, TEEC_VALUE_INPUT,
					 TEEC_MEMREF_TEMP_INPUT);

	res = TEEC_InvokeCommand(sess, TA_STORAGE_CMD_CREATE, &op, &org);

	if (res == TEEC_SUCCESS)
		*obj = op.params[1].value.b;

	return res;
}

static TEEC_Result fs_create_overwrite(TEEC_Session *sess, void *id,
				       uint32_t id_size, uint32_t storage_id)
{
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	TEEC_Result res = TEEC_ERROR_GENERIC;
	uint32_t org = 0;

	op.params[0].tmpref.buffer = id;
	op.params[0].tmpref.size = id_size;
	op.params[1].value.a = storage_id;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_INPUT,
					 TEEC_VALUE_INPUT, TEEC_NONE,
					 TEEC_NONE);

	res = TEEC_InvokeCommand(sess, TA_STORAGE_CMD_CREATE_OVERWRITE, &op, &org);

	return res;
}

static TEEC_Result fs_close(TEEC_Session *sess, uint32_t obj)
{
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	uint32_t org = 0;

	op.params[0].value.a = obj;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_VALUE_INPUT, TEEC_NONE,
					 TEEC_NONE, TEEC_NONE);

	return TEEC_InvokeCommand(sess, TA_STORAGE_CMD_CLOSE, &op, &org);
}

static TEEC_Result fs_read(TEEC_Session *sess, uint32_t obj, void *data,
			   uint32_t data_size, uint32_t *count)
{
	TEEC_Result res = TEEC_ERROR_GENERIC;
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	uint32_t org = 0;

	op.params[0].tmpref.buffer = data;
	op.params[0].tmpref.size = data_size;
	op.params[1].value.a = obj;
	op.params[1].value.b = 0;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_OUTPUT,
					 TEEC_VALUE_INOUT, TEEC_NONE,
					 TEEC_NONE);

	res = TEEC_InvokeCommand(sess, TA_STORAGE_CMD_READ, &op, &org);

	if (res == TEEC_SUCCESS)
		*count = op.params[1].value.b;

	return res;
}

static TEEC_Result fs_write(TEEC_Session *sess, uint32_t obj, void *data,
			    uint32_t data_size)
{
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	uint32_t org = 0;

	op.params[0].tmpref.buffer = data;
	op.params[0].tmpref.size = data_size;
	op.params[1].value.a = obj;
	op.params[1].value.b = 0;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_INPUT,
					 TEEC_VALUE_INPUT, TEEC_NONE,
					 TEEC_NONE);

	return TEEC_InvokeCommand(sess, TA_STORAGE_CMD_WRITE, &op, &org);
}

static TEEC_Result fs_seek(TEEC_Session *sess, uint32_t obj, int32_t offset,
			   int32_t whence)
{
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	uint32_t org = 0;

	op.params[0].value.a = obj;
	op.params[0].value.b = *(uint32_t *)&offset;
	op.params[1].value.a = whence;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_VALUE_INPUT, TEEC_VALUE_INOUT,
					 TEEC_NONE, TEEC_NONE);

	return TEEC_InvokeCommand(sess, TA_STORAGE_CMD_SEEK, &op, &org);
}

static TEEC_Result fs_unlink(TEEC_Session *sess, uint32_t obj)
{
	TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
	uint32_t org = 0;

	op.params[0].value.a = obj;

	op.paramTypes = TEEC_PARAM_TYPES(TEEC_VALUE_INPUT, TEEC_NONE,
					 TEEC_NONE, TEEC_NONE);

	return TEEC_InvokeCommand(sess, TA_STORAGE_CMD_UNLINK, &op, &org);
}

int example_read ()
{
    TEEC_Context teec_ctx;
    TEEC_Session sess = {};
    TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
    TEEC_Result res = TEEC_ERROR_GENERIC;
    TEEC_UUID uuid = TA_STORAGE_UUID;
    uint32_t org = 0;
    uint32_t obj_id = 0;
    uint32_t orig = 0;
    char buff[128] = {0};
    uint32_t count = 0;
    
    TEEC_InitializeContext(NULL, &teec_ctx);
    TEEC_OpenSession(&teec_ctx, &sess, &uuid,
                                TEEC_LOGIN_PUBLIC, NULL, NULL, &orig);

    strcpy(buff, "abc");
    op.params[0].tmpref.buffer = buff;
    op.params[0].tmpref.size = 3;
    op.params[1].value.a = TEE_DATA_FLAG_ACCESS_WRITE |
			TEE_DATA_FLAG_ACCESS_READ;
    op.params[1].value.b = 0;
    op.params[2].value.a = TEE_STORAGE_PRIVATE;

    op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_INPUT,
                     TEEC_VALUE_INOUT, TEEC_VALUE_INPUT,
                     TEEC_NONE);

    res = TEEC_InvokeCommand(&sess, TA_STORAGE_CMD_OPEN, &op, &org);

    if (res == TEEC_SUCCESS)
        obj_id = op.params[1].value.b;
    else
    {
    TEEC_CloseSession(&sess);
    TEEC_FinalizeContext(&teec_ctx);
    
          return -1;
    }
    op = TEEC_OPERATION_INITIALIZER;

    op.params[0].tmpref.buffer = buff;
    op.params[0].tmpref.size = sizeof(buff);
    op.params[1].value.a = obj_id;
    op.params[1].value.b = 0;
    
    op.paramTypes = TEEC_PARAM_TYPES(TEEC_MEMREF_TEMP_OUTPUT,
    					 TEEC_VALUE_INOUT, TEEC_NONE,
    					 TEEC_NONE);
    
    res = TEEC_InvokeCommand(&sess, TA_STORAGE_CMD_READ, &op, &org);
    
    if (res == TEEC_SUCCESS)
        count = op.params[1].value.b;
    else
        count = -1;
    
    TEEC_CloseSession(&sess);
    TEEC_FinalizeContext(&teec_ctx);
    return count;
}


void usage(char *program);

void usage(char *program)
{
	printf("Usage: %s <options> [content]\n", program);
	printf("\n");
	printf("options:\n");
	printf("\t-c <command>       Security Storage commands in [create, read, write, delete]\n");
	printf("\t-n <object_name>   the name of the storage object, less than 64\n");
	printf("\t-p <pos>           the pos to seek, valid only when the command is read or write\n");
	printf("\t[content]          the content to be write, used only when the command is create or write\n");
	printf("\n");
}

int main(int argc, char *argv[])
{
	int opt = 0;
	int ret = 0;
	char * command = NULL;
	char * object_name = NULL;
	char * pos_str = NULL;
	char * content = NULL;

    uint32_t orig = 0;
    uint32_t object_id = 0;
    TEEC_Context teec_ctx;
    TEEC_Session sess = {};
    TEEC_Operation op = TEEC_OPERATION_INITIALIZER;
    TEEC_Result res = TEEC_ERROR_GENERIC;
    TEEC_UUID uuid = TA_STORAGE_UUID;

    char buff[512] = {0};
    uint32_t count = 0;

    if (argc <= 1)
    {
        usage(argv[0]);
		return -1;
    }


	while ((opt = getopt(argc, argv, "c:n:p:h")) != -1)
	{
		switch (opt) {
		case 'c':
			command = optarg;
			printf("command :%s\n", command);
			break;
		case 'n':
			object_name = optarg;
			printf("object name:%s\n", object_name);
			break;
		case 'p':
			pos_str = optarg;
			printf("pos:%s\n", pos_str);
			break;
		case 'h':
			usage(argv[0]);
			return 0;
		default:
			usage(argv[0]);
			return -1;
 		}
	}

	if (NULL == command || NULL == object_name) {
		usage(argv[0]);
		return -1;
	}

	if (optind < argc)
	{
		content = argv[optind];
		printf("content: %s\n", content);
	}

    
    res = TEEC_InitializeContext(NULL, &teec_ctx);
	if (TEEC_SUCCESS != res)
	{
		return -1;
	}

    res = TEEC_OpenSession(&teec_ctx, &sess, &uuid,
                                TEEC_LOGIN_PUBLIC, NULL, NULL, &orig);
	if (TEEC_SUCCESS != res)
	{
        TEEC_FinalizeContext(&teec_ctx);
		return -1;
	}

	if (strcmp(command, "create") == 0) 
	{
		res = fs_create(&sess, object_name, strlen(object_name),
				TEE_DATA_FLAG_ACCESS_WRITE | TEE_DATA_FLAG_ACCESS_READ, 0,
				content, NULL == content ? 0 : strlen(content),
			    &object_id, TEE_STORAGE_PRIVATE);
		if (TEEC_SUCCESS != res)
			ret = -1;
	}
	else 
	{
		res = fs_open(&sess, object_name, strlen(object_name), 
				TEE_DATA_FLAG_ACCESS_WRITE | TEE_DATA_FLAG_ACCESS_READ |TEE_DATA_FLAG_ACCESS_WRITE_META,
				&object_id, TEE_STORAGE_PRIVATE);
		if (TEEC_SUCCESS == res)
		{
			if (NULL != pos_str && (strcmp(command, "read") == 0  || strcmp(command, "write") == 0))
			{
				count = atoi(pos_str);
				if (count < 0)
				{
					printf("bad pos [%d]\n", count);
					ret = -1;
				}
				else
				{
					res = fs_seek( &sess, object_id, count, 0);
					if (TEEC_SUCCESS != res)
						ret = -1;
				}
			}
			if (ret == -1)
			{
				printf("seek pos [%d]  fail\n", count);
			}
			else if (strcmp(command, "read") == 0) 
			{
				count = 0;
				res = fs_read(&sess, object_id, buff, 512, &count);
				if (TEEC_SUCCESS == res)
				{
					buff[count] = '\0';
					printf("read %d data: [ %s ]\n", count, buff);
				}
				else
				{
					printf("read %s fail\n", object_name);
					ret = -1;
				}
			}
			else if (strcmp(command, "write") == 0 && NULL != content ) 
			{
				res = fs_write(&sess, object_id, content, strlen(content));
				if (TEEC_SUCCESS == res)
				{
					printf("write data success\n");
				}
				else
				{
					printf("write %s fail [%d]\n", object_name, res);
					ret = -1;
				}
			}
			else if (strcmp(command, "delete") == 0) 
			{
				res = fs_unlink(&sess, object_id);
				if (TEEC_SUCCESS == res)
				{
					printf("delete object %s success\n", object_name);
				}
				else
				{
					printf("delete object %s fail\n", object_name);
					ret = -1;
				}
			}
			else
			{
				printf("unkown command [ %s ]\n", command);
				ret = -1;
			}
		}
		else
		{
		    printf("object [%s] is not exist\n", object_name);
			ret = -1;
		}

	}

	TEEC_CloseSession(&sess);
	TEEC_FinalizeContext(&teec_ctx);

	return ret;
}



