/* SPDX-License-Identifier: MediaTekProprietary */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <cutils/properties.h>
#include "request.h"
#include "prop_debug.h"
#include "prop_test.h"
#include "list.h"

/*******************************************************************************/
/* LIBSNCFG functions for Android Property Mechanism                           */
/*******************************************************************************/
int _property_check(const char *key, const char *value)
{
    if (key == 0) {
		 PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "[PropSet] key==0 return directly\n");
		 return -1;
	}

    if (strlen(key) >= PROPERTY_KEY_MAX) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "[PropSet] key=%s size:%zd >= max:%d\n",
				                            key, strlen(key), PROPERTY_KEY_MAX);
		 return -1;
	}

    if (strlen(value) >= PROPERTY_VALUE_MAX) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "[PropSet] value=%s size:%zd >= max:%d\n",
				                            value, strlen(value), PROPERTY_VALUE_MAX);
		return -1;
	}
	
    return  0;
}

int property_get(const char *key, char *value, const char *default_value)
{
	struct  sncfg_request request;
    int     ret_len = 1;

    /*check validaty*/
    if ((key == 0) || (value == 0)) {
		 PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "[PropSet] key==0 return directly\n");
		 return -1;
	}

	if (strlen(key) >= PROPERTY_KEY_MAX) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "[PropSet] key=%s size:%zd >= max:%d\n",
											key, strlen(key), PROPERTY_KEY_MAX);
		 return -1;
	}

    /*send socket to get the value*/
	bzero(&request, sizeof(struct sncfg_request));
	request.magic = SNCFG_REQUEST_MAGIC;
	request.type = SNCFG_REQUEST_TYPE_PROP_GET;
	request.key = (char *)key;
	if (request_for_sncfg(&request) < 0) {
        ret_len = 0;        
	} else if (NULL == request.value) {
        ret_len = 0;
    }

    /*
    *  1. normal case: use db's value.
    *  2. other case: use default value.
    */
    if (ret_len>0) {
        ret_len = strlen(request.value);
        if (ret_len >= PROPERTY_VALUE_MAX)  {
            ret_len = PROPERTY_VALUE_MAX - 1;
        }
        strncpy(value, request.value, ret_len);
        value[ret_len] = '\0';
    } else {
        if (default_value) {
            ret_len = strlen(default_value);
            if (ret_len >= PROPERTY_VALUE_MAX) {
                ret_len = PROPERTY_VALUE_MAX - 1;
            }
            memcpy(value, default_value, ret_len);
            value[ret_len] = '\0';
        } else {
            value[0] = '\0';
        }
    }

	return  ret_len;	
}

int property_set(const char *key, const char *in_value)
{
	struct 	sncfg_request request;
	char	*value = "";

	if (in_value != NULL)
		value = in_value;
	
    if (_property_check(key, value) < 0) {
        return  -1;
    }

	bzero(&request, sizeof(struct sncfg_request));
	request.magic = SNCFG_REQUEST_MAGIC;
	request.type = SNCFG_REQUEST_TYPE_PROP_SET;
	request.key = (char *)key;
	request.value = (char *)value;
	if (request_for_sncfg(&request) < 0) {
		return -1;
	}
	
	return 0;
}

int property_reload(void)
{
	struct sncfg_request request;
	
	bzero(&request, sizeof(struct sncfg_request));
	
	request.magic = SNCFG_REQUEST_MAGIC;
	request.type = SNCFG_REQUEST_TYPE_PROP_RELOAD;	
	if (request_for_sncfg(&request) < 0) {
		return -1;
	}
	
	return 0;		
}

int property_debug(char *key, char *in_value)
{
	struct sncfg_request request;
	char	*value = "";

	if (in_value != NULL)
		value = in_value;

    if (_property_check(key, value) < 0) {
        return  -1;
    }
    
	bzero(&request, sizeof(struct sncfg_request));
	request.magic = SNCFG_REQUEST_MAGIC;
	request.type = SNCFG_REQUEST_TYPE_PROP_DEBUG;
	request.key = key;
	request.value = value;
	if (request_for_sncfg(&request) < 0) {
		return -1;
	}

	return 0;	
}

int property_test(char *key, char *in_value)
{
	struct sncfg_request request;
	char	*value = "";

	if (in_value != NULL)
		value = in_value;

    if (_property_check(key, value) < 0) {
        return  -1;
    }

    if (strncmp("UT", key, strlen("UT")) == 0) {
        prop_test_func(key, value);
        return  0;
    }
    if (strncmp("NULL", key, strlen("NULL")) == 0) {
        prop_test_func(key, value);
        return  0;
    } else {        
    	bzero(&request, sizeof(struct sncfg_request));
    	request.magic = SNCFG_REQUEST_MAGIC;
    	request.type = SNCFG_REQUEST_TYPE_PROP_TEST;
    	request.key = key;
    	request.value = value;
    	if (request_for_sncfg(&request) < 0) {
    		return -1;
    	}

    	return 0;	
    }
}
