/* SPDX-License-Identifier: MediaTekProprietary */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include <fcntl.h>
#include <dirent.h>
#include <signal.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <arpa/inet.h>
#include <sched.h>
#include <syslog.h>
#include "list.h"
#include "request.h"
#include "utility.h"
#include "property.h"
#include "prop_debug.h"
#include "prop_sncfg.h"

/*******************************************************************************/
/* SNCFGD local definitions                                                    */
/*******************************************************************************/
#ifndef PID_FILE
#define PID_FILE "/tmp/prop/sncfgd.pid"
#endif

#define CONFIG_HASH_SIZE		64

/*******************************************************************************/
/* property deamon local variables                                                      */
/*******************************************************************************/
static struct sockaddr_un RequestSockAddr = { AF_UNIX, SNCFG_REQUEST_UNIX_SOCKET };
static struct list ReqList;
static pthread_mutex_t MutexReqList, MutexConfig;
static pthread_cond_t ConditionReqList;

/*******************************************************************************/
/* property deamon local functions                                                      */
/*******************************************************************************/
static void reapchild(int sig)
{
	int status;

	while (waitpid(-1, &status, WNOHANG) > 0) {
		/* do nothing, just prevent from zombie children */
	}
}

static void term_handler(int sig
#ifdef SA_SIGINFO
	     , siginfo_t *siginfo, void *context
#endif
	    )
{
	printf("[SNCFGD] Received a signal %d, Terminating !!", sig);
	exit(-1);
}

static void trap_signals(void)
{
	unsigned int i, j;
	struct sigaction act, oldact;
	const int term_signals[] = {
#ifdef SIGHUP
	SIGHUP,
#endif
#ifdef SIGINT
	SIGINT,
#endif
#ifdef SIGUSR1
	SIGUSR1,
#endif
#ifdef SIGUSR2
	SIGUSR2,
#endif
#ifdef SIGPIPE
	SIGPIPE,
#endif
#ifdef SIGALRM
	SIGALRM,
#endif
#ifdef SIGTERM
	SIGTERM,
#endif
#ifdef SIGSTKFLT
	SIGSTKFLT,
#endif
#ifdef SIGVTALRM
	SIGVTALRM,
#endif
#ifdef SIGPROF
	SIGPROF,
#endif
#ifdef SIGPOLL
	SIGPOLL,
#endif
#ifdef SIGPWR
	SIGPWR,
#endif
	};
	const int ignore_signals[] = {
#ifdef SIGPIPE
	SIGPIPE,
#endif
	};
	const struct {
		const int *sigs;
		unsigned int nsigs;
		void (*handler)(int signo
#ifdef SA_SIGINFO
		, siginfo_t *info, void *context
#endif
		);
	} sigmap[] = {
		{term_signals, sizeof(term_signals)/sizeof(term_signals[0]), term_handler},
		{ignore_signals, sizeof(ignore_signals)/sizeof(ignore_signals[0]), NULL},
	};

	for (i = 0; i < sizeof(sigmap)/sizeof(sigmap[0]); i++) {
		for (j = 0; j < sigmap[i].nsigs; j++) {
			sigfillset(&act.sa_mask);
			if (sigmap[i].handler == NULL) {
				act.sa_handler = SIG_IGN;
				act.sa_flags = 0;
			}
			else {
#ifdef SA_SIGINFO
				act.sa_sigaction = sigmap[i].handler;
				act.sa_flags = SA_SIGINFO;
#else
				act.sa_handler = sigmap[i].handler;
				act.sa_flags = 0;
#endif
			}
			if (sigaction(sigmap[i].sigs[j], &act, &oldact) < 0) {
				DEBUG_PRINT("Can't set signal handler for signal %d: %s",
						sigmap[i].sigs[j], strerror(errno));
			}
		}
	}
}

static int pidfile_acquire(void)
{
	int pidfd;

	pidfd = open(PID_FILE, O_CREAT | O_WRONLY, 0644);
	if (pidfd < 0) {
		DEBUG_PRINT("Can't open pidfile %s : %s !!", PID_FILE, strerror(errno));
	}
	else {
		lockf(pidfd, F_LOCK, 0);
	}

	return pidfd;
}

static void pidfile_write_release(int pidfd)
{
	FILE *out;

	if (pidfd < 0) {
		return;
	}

	if ((out = fdopen(pidfd, "w")) != NULL) {
		fprintf(out, "%d\n", getpid());
	}

	lockf(pidfd, F_UNLCK, 0);
	if (out) {
		fclose(out);
	}
	else {
		safe_close(pidfd);
	}
}

static void req_enqueue(struct req *req)
{
	pthread_mutex_lock(&MutexReqList);
	LIST_CHECK(&ReqList);
	list_add_tail((struct list *)req, &ReqList);
	pthread_cond_signal(&ConditionReqList);
	pthread_mutex_unlock(&MutexReqList);
}

static struct req *req_dequeue(void)
{
	struct list *p, *n;
	pthread_mutex_lock(&MutexReqList);
	LIST_CHECK(&ReqList);
	if (list_empty(&ReqList)) {
		pthread_cond_wait(&ConditionReqList, &MutexReqList);
	}
	LIST_FOR_EACH_SAFE(p, n, &ReqList) {
		LIST_CHECK(p);
		list_del(p);
		break;
	}
	pthread_mutex_unlock(&MutexReqList);
	return (struct req *)p;
}

static void request_handle(int reqfd)
{
	int size, sock;
	struct req *req;
 	struct sncfg_request *request;

	sock = accept_request(reqfd);
	if (sock < 0) {
		syslog(LOG_ERR, "Can't accept request socket on %d : %s !!", reqfd, strerror(errno));
		return;
	}

	req = malloc(sizeof(struct req));
	if (!req) {
		syslog(LOG_ERR, "Can't malloc for pending request : %s !!", strerror(errno));
		safe_close(sock);
		return;
	}
	req->sock = sock;
	request = &req->request;

	size = safe_read(sock, request, sizeof(struct sncfg_request));
	if (size < 0) {
		syslog(LOG_ERR, "Can't read request on %d : %s !!", sock, strerror(errno));
		free(req);
		safe_close(sock);
		return;
    }

	if (unpack_request(request, size) < 0) {
		syslog(LOG_ERR, "Can't unpack request, size = %d !!", size);
		free(req);
		safe_close(sock);
		return;
    }

	switch (request->type) {
	case SNCFG_REQUEST_TYPE_PROP_SET:
	case SNCFG_REQUEST_TYPE_PROP_GET:
	case SNCFG_REQUEST_TYPE_PROP_DEBUG:
	case SNCFG_REQUEST_TYPE_PROP_RELOAD:
    case SNCFG_REQUEST_TYPE_PROP_TEST:
	    if (prop_sncfg_enqueue(req) < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Can't prop_sncfg_enqueue!!");
		    free(req);
		    safe_close(sock);
        }
		break;

	default:
		syslog(LOG_WARNING, "Unknown request type : %d !!", request->type);
        free(req);
		safe_close(sock);
		break;
	} /* switch (request.type) */
}

static int request_init(void)
{
	int reqfd, res;
	const int reuse = 1;

	if (access(SNCFG_REQUEST_FOLDER, F_OK) != 0) {
        if (mkdir(SNCFG_REQUEST_FOLDER, 0774) == -1)
            printf("mkdir %s error! %s\n", SNCFG_REQUEST_FOLDER, (char*)strerror(errno));
    }

	/* force to re-create */
	unlink(RequestSockAddr.sun_path);

	reqfd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (reqfd == -1) {
		LOG_PRINT(LOG_ERR, "Can't create socket on %s : %s !!", RequestSockAddr.sun_path, strerror(errno));
		return -1;
	}

	if (fcntl(reqfd, F_SETFD, FD_CLOEXEC) == -1) {
		LOG_PRINT(LOG_ERR, "Can't fcntl set FD_CLOEXEC on %s : %s !!", RequestSockAddr.sun_path, strerror(errno));
		safe_close(reqfd);
		return -1;
	}

	if (setsockopt(reqfd, SOL_SOCKET, SO_REUSEADDR, (const void *)&reuse, sizeof(reuse)) < 0) {
		LOG_PRINT(LOG_ERR, "Can't setsockopt SO_REUSEADDR on %s : %s !!", RequestSockAddr.sun_path, strerror(errno));
		safe_close(reqfd);
		return -1;
	}

	if (bind(reqfd, (struct sockaddr *)&RequestSockAddr,
	    offsetof(struct sockaddr_un, sun_path) + strlen(RequestSockAddr.sun_path)) < 0) {
		LOG_PRINT(LOG_ERR, "Can't bind on %s : %s !!", RequestSockAddr.sun_path, strerror(errno));
		safe_close(reqfd);
		return -1;
	}

	if (listen(reqfd, 5) < 0) {
		LOG_PRINT(LOG_ERR, "Can't listen on %s : %s !!", RequestSockAddr.sun_path, strerror(errno));
		safe_close(reqfd);
		return -1;
	}

	/* initialize request list */
	INIT_LIST(&ReqList);

	return reqfd;
}

/*******************************************************************************/
/* Prperty deamon                                                            */
/*******************************************************************************/
int main(int argc, char **argv)
{
	int reqfd, pidfd;
	pid_t pid;
#ifdef RT_SCHED
	struct sched_param sched;

	/* realtime priority */
	sched.sched_priority = 89;
	if (sched_setscheduler(0, SCHED_FIFO, &sched) == -1) {
		LOG_PRINT(LOG_ERR, "Can't set realtime scheduler : %s !!", strerror(errno));
	}
#endif

	openlog("[prop]", LOG_CONS | LOG_ODELAY | LOG_PERROR, LOG_DAEMON);

    LOG_PRINT(LOG_ERR, "Bootup...prop daemon BEGIN.");

	/* initialize request interface */
	if ((reqfd = request_init()) < 0) {
		LOG_PRINT(LOG_ERR, "Request interface initialization error !!");
		exit(-1);
	}

#if 0
	/* it's time to become a daemon */
	trap_signals();
	signal(SIGCHLD, reapchild);
	pidfd = pidfile_acquire();
	pid = fork();
	if (pid < 0) {
		LOG_PRINT(LOG_ERR, "Can't fork to become a daemon : %s  !!", strerror(errno));
		exit(-1);
	}
	if (pid != 0) {
		exit(0);
	}
	pidfile_write_release(pidfd);

    LOG_PRINT(LOG_ERR, "Bootup...prop begin...service done");
#endif

    if (prop_sncfg_init() < 0) {
		safe_close(reqfd);
		exit(-1);
    }
    LOG_PRINT(LOG_ERR, "Bootup...prop begin...property done");

	for (;;) {
		int ndes, maxfd;
		fd_set readfds;
		fd_set writefds;

		maxfd = 0;
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_SET(reqfd, &readfds);
		if (maxfd < reqfd) {
			maxfd = reqfd;
		}
		ndes = select((maxfd + 1), &readfds, &writefds, NULL, NULL);
		if (ndes == -1) {
			/* select failed, may be interrupted, try again */
			continue;
		} else if (ndes == 0) {
			/* select timeout, should not be */
			continue;
		} else {
			if (FD_ISSET(reqfd, &readfds))	{
				request_handle(reqfd);
			}
		}
	}

	return 0;
}
