/* SPDX-License-Identifier: MediaTekProprietary */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <dirent.h>		//for read directories
#include <cutils/properties.h>
#include "prop_debug.h"
#include "prop_core.h"
#include "prop_svc.h"

/*for APSIM, must put in /mnt/sncfg/apsim/*/
#if defined(PARTITION_FOR_APSIM)
#define PROP_TEST_PERSISTENT_PROPERTY_DIR   "/mnt/sncfg/apsim/property"
#else
#define PROP_TEST_PERSISTENT_PROPERTY_DIR   "/data/property"
#endif

#define PROP_TEST_FILE_PREFIX				"persist.test."

int __prop_test_get_time(void)
{
 	struct  timeval current;
	int	cur_usec;

	gettimeofday(&current, NULL);

	cur_usec = (int)(current.tv_sec * 1000000 + current.tv_usec);
	return	cur_usec;
}


void __prop_test_trim_line(char *line)
{
	int		len;
	int		index;

	if (!line)
		return;

	len = strlen(line);
	if (len <= 0)
		return;

	for (index=len-1; index >= 0; index-- ) {
		if ((line[index] == ' ') ||
			(line[index] == '\n')) {
			line[index] = '\0';
		} else {
			break;
		}
	}
	return;
}

int __prop_test_func_def(const char *key, const char*def_value)
{
    int     ret_val = 0;
    char    value_buffer[PROPERTY_VALUE_MAX];

    if (!key || !def_value)
        return  -1;

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin!");

    ret_val = property_get(key, value_buffer, def_value);
    if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL! Key:%s ", key);
        ret_val = -1;
    } else {
        if (strcmp(value_buffer, def_value)) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL! key:%s value:%s def_value:%s", key, value_buffer, def_value);
            ret_val = -1;

        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "End!\n");
    return  ret_val;
}

int __prop_test_func_basic(const char *key, const char*value)
{
    int     ret_val = 0;
    int     total_ret = 0;
    char    value_buffer[PROPERTY_VALUE_MAX];

    if (!key || !value)
        return  -1;

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin! key:%s value:%s ", key, value);

    ret_val = property_set(key, value);
    if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...1 ");
        total_ret = -1;
    }

	if (total_ret >= 0) {
	    ret_val = property_get(key, value_buffer, "123");
	    if (ret_val < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...2 ");
	        total_ret = -1;
	    }  else  {
	        if (strcmp(value_buffer, value)) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...get_value:%s len:%d set_value:%s len:%d ",
						value_buffer, strlen(value_buffer), value, strlen(value));
	            total_ret = -1;
	        }
	    }
	}

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "End");

    return  total_ret;
}

int _prop_test_func_normal(void)
{
    int     ret_val;
    int     total_ret = 0;
    char    *ok_value[] = {
                            "func_test.value_1_01234567",
                            "func_test.value_1_0123456789_123456789_123456789_123456789_123456789_123456789_1234",
                        };
    char    *default_value[] = {
                            "func_test.def_value_1_01",
                            "func_test.def_value_1_012",
                        };
    char    key_buffer[PROPERTY_KEY_MAX];
    int     case_num;
    int     index = 1;

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin!");

    case_num = sizeof(default_value)/sizeof(default_value[0]);
    for (index=0; index<case_num; index++) {
 	    sprintf(key_buffer, "%s.%d.%d", "normal.key", __prop_test_get_time(), index);
        ret_val = __prop_test_func_def(key_buffer, default_value[index]);
        if (ret_val < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...1...key:%s ", key_buffer);
			total_ret = -1;
		}

        ret_val = __prop_test_func_basic(key_buffer, ok_value[index]);
        if (ret_val < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...2...key:%s ", key_buffer);
			total_ret = -1;
		}
    }

    if (total_ret < 0)
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test Fail!");
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test OK!");

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End!\n");

	return	total_ret;
}

int _prop_test_func_ro(void)
{
    int     ret_val;
    int     total_ret = 0;
    char    key_buffer[PROPERTY_KEY_MAX];

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin!");

    sprintf(key_buffer, "%s.%d", "ro.key", __prop_test_get_time());
    ret_val = __prop_test_func_def(key_buffer, "12345678");
    if (ret_val < 0)
	{
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...1...key:%s ", key_buffer);
		total_ret = -1;
	}

    ret_val = __prop_test_func_basic(key_buffer, "87654321");
    if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...2...key:%s ", key_buffer);
		total_ret = -1;
	}

    ret_val = property_set(key_buffer, "0123456789");
    if (ret_val >= 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Fail! ...3...key:%s ", key_buffer);
		total_ret = -1;
	}

    if (total_ret < 0)
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test Fail!");
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test OK!");

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End!\n");

	return	total_ret;
}

int _prop_test_func_persist(void)
{
    int     ret_val;
    int     total_ret = 0;
    char    key_buffer[PROPERTY_KEY_MAX];
    char    value_buffer[PROPERTY_VALUE_MAX];
    char    file_buffer[128];
    FILE    *read_file;
	char    file_line[256];
    int     file_line_num;
	int		usec_num;

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin!");

	usec_num = __prop_test_get_time();

    sprintf(key_buffer, "%s%d", PROP_TEST_FILE_PREFIX, usec_num);
    sprintf(value_buffer, "%s%d", "persist.value.", usec_num);
    ret_val = __prop_test_func_def(key_buffer, "12345678");
    if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL! ...1...");
		total_ret = -1;
	}
 	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Key:%s  value:%s ", key_buffer, value_buffer);

    ret_val = __prop_test_func_basic(key_buffer, value_buffer);
    if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL! ...2...");
		total_ret = -1;
	}

    sprintf(file_buffer, "%s/%s", PROP_TEST_PERSISTENT_PROPERTY_DIR, key_buffer);
	read_file = fopen(file_buffer , "r" );
    if (!read_file) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...3... ");
        total_ret = -1;
    } else {
        file_line_num = 0;
		while (fgets(file_line, sizeof(file_line), read_file ) != NULL) {
			__prop_test_trim_line(file_line);

            file_line_num++;
            if (strcmp(value_buffer, file_line)) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...4...file_line:%s set_value:%s ", file_line, value_buffer);
                total_ret = -1;
            }
		}

        if (file_line_num != 1) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...5...num:%d ", file_line_num);
            total_ret = -1;
        }
		fclose (read_file );
    }

    if (total_ret < 0)
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test Fail!");
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test OK!");

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End!\n");

    return  total_ret;
}

int _prop_test_func_performance(void)
{
    int     ret_val = 0;
    int     total_ret = 0;
    char    key_buffer[PROPERTY_KEY_MAX];
    char    value_buffer[PROPERTY_VALUE_MAX];
 	struct  timeval start, end; //To record start time and end time
	int		elapsed_time;

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin!");

	gettimeofday(&start, NULL);

    sprintf(key_buffer, "%s.%d", "performance.test.1", __prop_test_get_time());
    ret_val = property_set(key_buffer, "12345");
	if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...2...key:%s ", key_buffer);
		total_ret = -1;
	}

	gettimeofday(&end, NULL);

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "property_set time! start=%ld, %ld usecond", start.tv_sec, start.tv_usec);
	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "property_set time!   end=%ld, %ld usecond", end.tv_sec, end.tv_usec );
	elapsed_time = (end.tv_sec - start.tv_sec)*1000000 + (end.tv_usec - start.tv_usec );
	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "property_set time!   Elapsed_time:%d ", elapsed_time);
	if (elapsed_time >= 2000)
		total_ret = -1;

	gettimeofday(&start, NULL);

    ret_val = property_get(key_buffer, value_buffer, "123");
    if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...3...key:%s ", key_buffer);
        total_ret = -1;
    }

	gettimeofday(&end, NULL);

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "property_get time! start=%ld, %ld usecond", start.tv_sec, start.tv_usec);
	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "property_get time!   end=%ld, %ld usecond", end.tv_sec, end.tv_usec );
	elapsed_time = (end.tv_sec - start.tv_sec)*1000000 + (end.tv_usec - start.tv_usec );
	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "property_get time!   Elapsed_time:%d ", elapsed_time);
	if (elapsed_time >= 2000)
		total_ret = -1;

    if (total_ret < 0)
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test Fail!");
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test OK!");

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End!\n");

    return  total_ret;
}

int _prop_test_func_stress(void)
{
    int     ret_val = 0;
    int     total_ret = 0;
    char    key_buffer[PROPERTY_KEY_MAX];
    char    value_buffer[PROPERTY_VALUE_MAX];
    char    read_buffer[PROPERTY_VALUE_MAX];
	int		index;

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin!");

	for (index = 0; index<1000; index++) {
		sprintf(key_buffer, "%s.%d", "stress.test", __prop_test_get_time()+index);
		sprintf(value_buffer, "%s.%d", "stress.test.value", __prop_test_get_time()+index);

	    ret_val = property_set(key_buffer, value_buffer);
		if (ret_val < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...1...key:%s value:%s ", key_buffer, value_buffer);
			total_ret = -1;
		}

		memset(read_buffer, 0x0, PROPERTY_VALUE_MAX);
		ret_val = property_get(key_buffer, read_buffer, "123");
		if (ret_val < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...2...key:%s value:%s ", key_buffer, value_buffer);
			total_ret = -1;
		}

		if (strcmp(read_buffer, value_buffer)) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...3...key:%s set_value:%s get_value:%s ",
							key_buffer, read_buffer, value_buffer);
			total_ret = -1;
		}
	}

    if (total_ret < 0)
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test Fail!");
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test OK!");

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End!\n");

    return  total_ret;

}

int _prop_test_unit_set(void)
{
    int     ret_val;
    int     total_val = 0;
    char    *failed_key[4] = {
                                NULL,
                                "property_set.test.key_1_01234567",
                                "property_set.test.key_1_012345678",
                                "property_set.test.key_1_0123456789012345678",
                            };

    char    *ok_key[2] = {
                            "property_set.test.key_2_01",
                            "property_set.test.key_2_0123",
                        };

    char    *ok_value[4] = {
                            "",
                            "v",
                            "property_set.test.value_1_01234567",
                            "property_set.test.value_1_0123456789_123456789_123456789_123456789_123456789_123456789_1234",
                        };

    char    *failed_value[2] = {
                            "property_set.test.value_2_0123456789_123456789_123456789_123456789_123456789_123456789_12345",
                            "property_set.test.value_2_0123456789_123456789_123456789_123456789_123456789_123456789_123456",
                        };
    char    *test_ptr;
    char    key_buffer[128];
    int     index;

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin!");
	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin Fail-Key");

    /*test the key*/
    for (index=0; index<4; index++) {
    	test_ptr = failed_key[index];
        ret_val = property_set(test_ptr, "123");
        if (ret_val >= 0) {
            if (test_ptr) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...1...key:%s length:%d", test_ptr, strlen(test_ptr));
            } else {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...1...key=NULL");
            }
            total_val = -1;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin OK-Key");

    for (index=0; index<2; index++) {
    	test_ptr = ok_key[index];
        ret_val = property_set(test_ptr, "123");
        if (ret_val < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...2...key:%s length:%d", test_ptr, strlen(test_ptr));
            total_val = -1;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin Fail-Value");

    /*test the value*/
    for (index=0; index<2; index++) {
    	test_ptr = failed_value[index];

		sprintf(key_buffer, "%s.%d", "key.1.property_set", __prop_test_get_time());
        ret_val = property_set(key_buffer, test_ptr);
        if (ret_val >= 0) {
            if (test_ptr) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...3...key:%s length:%d", test_ptr, strlen(test_ptr));
            } else {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...3...key=NULL");
            }
            total_val = -1;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin OK-Value");

    for (index=0; index<4; index++) {
    	test_ptr = ok_value[index];

		sprintf(key_buffer, "%s.%d", "key.2.property_set", __prop_test_get_time());
        ret_val = property_set(key_buffer, test_ptr);
        if (ret_val < 0) {
            if (test_ptr)
            {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...4...key:%s length:%d", test_ptr, strlen(test_ptr));
            } else {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...4...key=NULL");
            }

            total_val = -1;
        }
    }

    if (total_val < 0)
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test Fail!");
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test OK!");

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End!\n");

    return  total_val;

}

int _prop_test_unit_get(void)
{
    int     ret_val;
    int     total_val = 0;
    char    *failed_key[] = {
                                NULL,
                                "property_get.test.key_1_01234567",
                                "property_get.test.key_1_012345678",
                                "property_get.test.key_1_0123456789012345678",
                            };

    char    *ok_key[] = {
                            "property_get.test.key_2_0123456",
                            "property_get.test.key_2_0123",
                        };

    char    *failed_value[] = {
                            NULL,
                        };
    char    *test_ptr;
    char    value_buffer[PROPERTY_VALUE_MAX];
    char    key_buffer[PROPERTY_KEY_MAX];
    int     index;

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin!");
	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin Fail-Key");

    /*test the key*/
    for (index=0; index<4; index++) {
		test_ptr = failed_key[index];
        ret_val = property_get(test_ptr, value_buffer, "123");
        if (ret_val >= 0) {
            if (test_ptr) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...1...key:%s length:%d", test_ptr, strlen(test_ptr));
            } else {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...1...key=NULL");
            }

            total_val = -1;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin OK-Key");

    for (index=0; index<2; index++) {
		test_ptr = ok_key[index];
        ret_val = property_get(test_ptr, value_buffer, "123");
        if (ret_val < 0) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...2...key:%s length:%d", test_ptr, strlen(test_ptr));
            total_val = -1;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "Begin Fail-Value");

    /*test the value*/
    for (index=0; index<1; index++) {
		test_ptr = failed_value[index];
		sprintf(key_buffer, "%s.%d", "key.property_get", __prop_test_get_time());
        ret_val = property_get(key_buffer, test_ptr, "123");
        if (ret_val >= 0) {
            if (test_ptr) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...3...key:%s length:%d", test_ptr, strlen(test_ptr));
            } else {
				PROP_DEBUG_PRINT(PROP_DEBUG_TEST, "FAIL!...3...key=NULL");
            }
            total_val = -1;
        }
    }

    if (total_val < 0)
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test Fail!");
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Test OK!");

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End!\n");

    return  total_val;
}

void _prop_test_ut(const char *value)
{
    int     ret_val;
    int     total_ret = 0;
	int		test_all = 0;

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Begin...");

	if (!strncmp("all", value, strlen("all"))) {
		prop_debug_set(PROP_DEBUG_OFF);
		test_all = 1;
	}

	if (!strncmp("all_trace", value, strlen("all_trace"))) {
		prop_debug_set(PROP_DEBUG_TEST);
		test_all = 1;
	}

    if (!strncmp("set", value, strlen("set")) ||
		(test_all != 0)) {
	    ret_val = _prop_test_unit_set();
	    if (ret_val < 0)
	        total_ret = -1;
	}

    if (!strncmp("get", value, strlen("get")) ||
		(test_all != 0)) {
	    ret_val = _prop_test_unit_get();
	    if (ret_val < 0)
	        total_ret = -1;
	}

    if (!strncmp("perform", value, strlen("perform")) ||
		(test_all != 0)) {
	    ret_val = _prop_test_func_performance();
	    if (ret_val < 0)
	        total_ret = -1;
	}

    if (!strncmp("normal", value, strlen("normal")) ||
		(test_all != 0)) {
	    ret_val = _prop_test_func_normal();
	    if (ret_val < 0)
	        total_ret = -1;
	}

    if (!strncmp("ro", value, strlen("ro")) ||
		(test_all != 0)) {
	    ret_val = _prop_test_func_ro();
	    if (ret_val < 0)
	        total_ret = -1;
	}

    if (!strncmp("persist", value, strlen("persist")) ||
		(test_all != 0)) {
	    ret_val = _prop_test_func_persist();
	    if (ret_val < 0)
	        total_ret = -1;
	}

    if (!strncmp("stress", value, strlen("stress")) ||
		(test_all != 0)) {
	    ret_val = _prop_test_func_stress();
	    if (ret_val < 0)
	        total_ret = -1;
	}

    if (total_ret < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "Fail! ");
    } else {
		PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "OK!");
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_OFF, "End. \n");
}

int prop_test_func(const char *name, const char *value)
{
    char        command_buffer[256];
	char		*command_1 = "mkdir -p /usr/sc/property/test/";
	char		*command_2 = "mkdir -p /usr/sc/property/test/";
	pid_t 		pid;

	if (0 == strncmp("UT", name, strlen("UT")))
	{
		_prop_test_ut(value);
	} else if (0 == strncmp("DUMP", name, strlen("DUMP"))) {
		prop_core_dump();
	} else if (0 == strncmp("system_func", name, strlen("system_func"))) {
		sprintf(command_buffer, "%s%d;%s%d",
				command_1, __prop_test_get_time(),
				command_2, __prop_test_get_time()+1);

		system(command_buffer);
	} else if (0 == strncmp("exec_func", name, strlen("exec_func"))) {
		sprintf(command_buffer, "%s%d;%s%d",
				command_1, __prop_test_get_time(),
				command_2, __prop_test_get_time()+1);

		pid = fork();
		if (pid < 0) {
			return	-1;
		} else if (pid == 0) {
			execl("/bin/sh", "sh", "-c", command_buffer, (char *)0);
			exit(127);
		} 
	} else if (0 == strncmp("svc", name, strlen("svc"))) {
	#if 0
		char	action_buf[1024];

		if (prop_svc_get_actions_by_name(value, NULL, action_buf) >= 0) {
			printf("\n Get the ACTIONS. name:%s action:%s  ", name, action_buf);
		} else {
			printf("\n Can not get the ACTIONS. name:%s ", value);
			prop_svc_dump();
		}
    #endif
	} else if (0 == strncmp("NULL", name, strlen("NULL"))) {
		int		ret_val;
		char	value[32];

		ret_val = property_set("test.null.key.1", "");
		printf("\n Test.null.key.1.   ret:%d ", ret_val);

		ret_val = property_set("test.null.key.2", NULL);
		printf("\n Test.null.key.2.   ret:%d ", ret_val);

		value[0] = '\0';
		ret_val = property_get("test.null.key.1", value, "abc");
		printf("\n Test.null.key.1.   ret:%d value:%s!", ret_val, value);

		value[0] = '\0';
		ret_val = property_get("test.null.key.2", value, "abcd");
		printf("\n Test.null.key.2.   ret:%d value:%s!", ret_val, value);
	}
    return  0;
}
