/* SPDX-License-Identifier: MediaTekProprietary */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <signal.h>
#include <cutils/properties.h>
#include "prop_db.h"
#include "prop_core.h"
#include "prop_debug.h"
#include "prop_file.h"
#include "prop_svc.h"

static int prop_core_inited = 0;

static int _prop_core_check(const char* name)
{
    int     i;
    int     name_len;

    if (!name)
        return  -1;

    name_len = strlen(name);

    if (name_len >= PROPERTY_KEY_MAX) {
    	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "PropSet Error:[%s] namelen >= %d", name, PROPERTY_VALUE_MAX);
    	return -1;
    }
    if (name_len < 1) {
    	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "PropSet Error:[%s] namelen < 1", name);
    	return -1;
    }
    if (name[0] == '.') {
    	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "PropSet Error:[%s] name shouldn't begin with .", name);
    	return -1;
    }
    if (name[name_len - 1] == '.') {
    	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "PropSet Error:[%s] name shouldn't end with .", name);
    	return -1;
    }


    /* Only allow alphanumeric, plus '.', '-', or '_' */
    /* Don't allow ".." to appear in a property name */
    for (i = 0; i < name_len; i++) {
        if (name[i] == '.') {
            if (name[i-1] == '.') {
            	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "PropSet Error:[%s] don't allow .. in property name", name);
            	return -1;
            }
            continue;
        }
        if (name[i] == '_' || name[i] == '-') continue;
        if (name[i] >= 'a' && name[i] <= 'z') continue;
        if (name[i] >= 'A' && name[i] <= 'Z') continue;
        if (name[i] >= '0' && name[i] <= '9') {
            continue;
        } else {
        	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "PropSet Error:[%s] Only allow alphanumeric, plus '.', '-', or '_' ", name);
        	return -1;
        }
    }

    return 0;
}

void _prop_core_set_ctrl_state(prop_svc_t	*item_svc, const char *state)
{
    char 	pname[PROPERTY_VALUE_MAX];
    int 	len = strlen(item_svc->svc_key);

    if ((len + 10) > PROPERTY_VALUE_MAX) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Wrong Value length. name:%s length:%d ", item_svc->svc_key, len);
		return;
	}

	if (!strcmp(state, "stopping")) {
		item_svc->flags &= ~(PROP_SVC_RUNNING| PROP_SVC_STOPPED);
		item_svc->flags |= PROP_SVC_STOPPING;
	} else if (!strcmp(state, "stopped")) {
		item_svc->flags &= ~(PROP_SVC_RUNNING| PROP_SVC_STOPPING);
		item_svc->flags |= PROP_SVC_STOPPED;

	} else if (!strcmp(state, "running")) {
		item_svc->flags &= ~(PROP_SVC_STOPPING | PROP_SVC_STOPPED);
		item_svc->flags |= PROP_SVC_RUNNING;
	} else {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Wrong state :%s ", state);
		return;
	}

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "key:%s sate:%s flags:%d ", item_svc->svc_key, state, item_svc->flags);

    sprintf(pname, "init.svc.%s", item_svc->svc_key);
    prop_db_set(pname, state);
}

int _prop_core_set_ctrlstart(const char *prop_key)
{
#define	ACTION_BUF_LEN	256
	char		action_buf[ACTION_BUF_LEN];
    prop_svc_t  *out_svc;
	pid_t		pid;
	int			len;
    char        svc_name[PROP_SVC_TYPE_MAX_SIZE];

    /*TODO*/
	len = snprintf(svc_name, PROP_SVC_TYPE_MAX_SIZE-1, "init.%s.rc", prop_key);
	if (len >= (PROP_SVC_TYPE_MAX_SIZE-1)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Key length is error: %s", prop_key);
		return	-1;
	}

    out_svc = prop_svc_find_by_name(svc_name, prop_key, NULL);
    if (!out_svc) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to find svc. %s ", prop_key);
		return	-1;
	}

	if (out_svc->flags & PROP_SVC_RUNNING) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "svc is running! %s ", prop_key);
		return	0;
	}

#ifdef PROPERTY_SYSTEMCTL
	/*use the systemctl start command to bring up the daemon*/
	len = strlen("systemctl start ");
	strncpy(action_buf, "systemctl start ", len);
	if (prop_svc_get_actions(out_svc, action_buf+len) < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to get the svc's action: %s", out_svc->svc_key);
		return	-1;
	}
#else
#ifdef PROPERTY_ANDROID_DEF
	/*use the service-launcher to bring up the daemon*/
	len = snprintf(action_buf, ACTION_BUF_LEN-1, "service-launcher init.%s.rc", prop_key);
#else
	char	daemon_name[PROP_SVC_ACTION_MAX_SIZE] = {0};

	if (prop_svc_get_daemon_ext(out_svc, daemon_name) < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to get the svc's action: %s", out_svc->svc_key);
		return	-1;
	}
	len = snprintf(action_buf, ACTION_BUF_LEN-1, "%s", daemon_name);
#endif
	if (len >= (ACTION_BUF_LEN-1)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Length is error: %s", prop_key);
		return	-1;
	}
#endif

	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Boot-up the service! key: %s action:%s ", out_svc->svc_key, action_buf);

	if ((pid = fork()) < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to start fork() ");
		return  -1;
	} else if (pid == 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "In child! name: %s action:%s ", out_svc->svc_key, action_buf);
		execl("/bin/sh", "sh", "-c", action_buf, (char *)0);
		_exit(127);
	}

	/*Need to fix here*/
	_prop_core_set_ctrl_state(out_svc, "running");

	return  0;
}

int _prop_core_set_ctrlstop(const char *prop_key)
{
#define	ACTION_BUF_LEN	256
	char		action_buf[ACTION_BUF_LEN];
	prop_svc_t	*out_svc;
	pid_t		pid;
	int 		len;
    char        svc_name[PROP_SVC_TYPE_MAX_SIZE];


    /*TODO*/
	len = snprintf(svc_name, PROP_SVC_TYPE_MAX_SIZE-1, "init.%s.rc", prop_key);
	if (len >= (PROP_SVC_TYPE_MAX_SIZE-1)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Key length is error: %s", prop_key);
		return	-1;
	}

    out_svc = prop_svc_find_by_name(svc_name, prop_key, NULL);
	if (!out_svc) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to find svc. %s ", prop_key);
		return	-1;
	}

	if ((out_svc->flags & PROP_SVC_STOPPING) ||
		(out_svc->flags & PROP_SVC_STOPPED)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "svc is stopped %s ", prop_key);
		return	0;
	}

#ifdef PROPERTY_SYSTEMCTL
	/*use the systemctl stop command to stop the daemon*/
	len = strlen("systemctl stop ");
	strncpy(action_buf, "systemctl stop ", len);
	if (prop_svc_get_actions(out_svc, action_buf+len) < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to get the svc's action: %s", out_svc->svc_key);
		return	-1;
	}

#else
	if ((pid = prop_svc_get_pid(out_svc)) < 0) {
        char    temp_key[128];
        char    temp_value[128];
        int     length;

        sprintf(temp_key, "init.svc.%s", out_svc->svc_key);
        length = prop_db_get(temp_key, temp_value);
        if (length >= 0) {
            PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "key: %s value:%s", temp_key, temp_value);
            if (!strcmp(temp_value, "running")) {
                _prop_core_set_ctrl_state(out_svc, "stopped");
            }
        }

		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to get the svc's action: %s", out_svc->svc_key);
		return	-1;
	}
	len = snprintf(action_buf, ACTION_BUF_LEN-1, "kill -9 %d", pid); // SIGKILL
	if (len >= (ACTION_BUF_LEN-1)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Length is error: %s", prop_key);
		return	-1;
	}
#endif

	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Stop the service! key:%s action:%s ", out_svc->svc_key, action_buf);

	if ((pid = fork()) < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to start fork() ");
		return	-1;
	} else if (pid == 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "In child! name: %s action:%s ", out_svc->svc_key, action_buf);
		system(action_buf);
		exit(0);
	}

	/*Need to fix here*/
	_prop_core_set_ctrl_state(out_svc, "stopped");

	return	0;
}

int _prop_core_set_ctrlrestart(const char *value)
{
    return  0;
}

int _prop_core_set_ctrl(const char *name, const char *value)
{
    int ret_val = -1;

    if (!strcmp(name+4, "start")) {
        ret_val = _prop_core_set_ctrlstart(value);
    } else if (!strcmp(name+4, "stop")) {
        ret_val = _prop_core_set_ctrlstop(value);
    } else if (!strcmp(name+4, "restart")) {
        ret_val= _prop_core_set_ctrlrestart(value);
    } else {
        PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "unknown control msg '%s'", name);
    }

    return  ret_val;
}

static void _prop_core_set_selinux(void)
{
    return;
}

int _prop_core_exec_onecommand(const prop_svc_act_t *in_act)
{
	int				retval;

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, " command:%s args[0]:%s args_num:%d ", in_act->act_name, in_act->args[0], in_act->args_num);

	if (!strncmp(in_act->args[0], "start", strlen("start"))) {
		if (in_act->args_num < 2) {
			PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Start wrong num:%d ", in_act->args_num);
			return	-1;
		}
		retval = _prop_core_set_ctrlstart(in_act->args[1]);
	} else if (!strncmp(in_act->args[0], "stop", strlen("stop"))) {
		if (in_act->args_num < 2) {
			PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Stop wrong num:%d ", in_act->args_num);
			return	-1;
		}

		retval = _prop_core_set_ctrlstop(in_act->args[1]);
	} else if (!strncmp(in_act->args[0], "restart", strlen("restart"))) {
		if (in_act->args_num < 2){
			PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Restart wrong num:%d ", in_act->args_num);
			return	-1;
		}

		retval = _prop_core_set_ctrlrestart(in_act->args[1]);
	} else if (!strncmp(in_act->args[0], "setprop", strlen("setprop"))) {
		if (in_act->args_num < 3) {
			PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Setprop wrong num:%d ", in_act->args_num);
			return	-1;
		}

		retval = prop_core_set(in_act->args[1], in_act->args[2]);
	} else {
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, " Unknown command...command:%s ", in_act->act_name);
		retval = system(in_act->act_name);
	}

	return	retval;
}

static void _prop_core_exec_command(const prop_svc_t  *svc_item)
{
	struct list 		*act_node;
	prop_svc_act_t		*svc_act;

	if (!svc_item)
		return;

	LIST_FOR_EACH(act_node, &(svc_item->act_list)) {
		LIST_CHECK(act_node);
		svc_act = LIST_ENTRY(act_node, prop_svc_act_t, alist);
		if (svc_act) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "svc command:%s ", svc_act->act_name);
			if (svc_act->args_num) {
				if (_prop_core_exec_onecommand(svc_act) < 0) {
					PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Exec command error");

				}
			} else {
				PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " find the action without args???");
			}
		}
	}
	return;
}

static void _prop_core_trigger_action(const char *name, const char *value)
{
	if (!name || !value) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Wrong name or value");
		return;
	}

    void ( *callaback)(const prop_svc_t  *);

    callaback = _prop_core_exec_command;
    prop_svc_exe_by_name(NULL, name, value, callaback);
}

int _prop_core_set(const char *name, const char *value)
{
    int     ret_val;
    char    item_value[PROPERTY_VALUE_MAX];

    ret_val = prop_db_get(name, item_value);
    if (ret_val >= 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Find the item...value:%s get_value:%s ", value, item_value);
        if (!strcmp(item_value, value)) {
            return  0;
        }

        if (!strncmp(name, "ro.", 3)) {
            return  -1;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Can not find the item...key:%s value:%s ", name, value);

    ret_val = prop_db_set(name, value);
    if (ret_val < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Add item error.");
        return  -1;
    }

    /* If name starts with "net." treat as a DNS property. */
    if (strncmp("net.", name, strlen("net.")) == 0) {
		/*
		 * The 'net.change' property is a special property used track when any
		 * 'net.*' property name is updated. It is _ONLY_ updated here. Its value
		 * contains the last updated 'net.*' property.
		 */
        if (strcmp("net.change", name) == 0)
            return 0;
        prop_db_set("net.change", name);
    } else if (strncmp("persist.", name, strlen("persist.")) == 0) {
        prop_file_write_persistent(name, value);
    } else if (strcmp("selinux.reload_policy", name) == 0 &&
               strcmp("1", value) == 0) {
        _prop_core_set_selinux();
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "trigger the action. key:%s value:%s", name, value);
    _prop_core_trigger_action(name, value);

    return  0;
}

void prop_core_dump(void)
{
	prop_db_dump();
	prop_svc_dump();
}

int prop_core_notify_ctrl(pid_t pid)
{
	prop_svc_t	*out_svc;

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Beginning trace %d", (int)pid);

    if (!prop_core_inited) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Not Inited!");
		return	-1;
	}

	out_svc = prop_svc_find_by_pid(pid);
	if (!out_svc) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Cat not find svc. pid:%d ", (int)pid);
		return	-1;
	}

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "find the svc. svc: %s pid:%d flags:%d ", out_svc->svc_key, (int)(out_svc->pid), out_svc->flags);

	if (out_svc->flags & PROP_SVC_STOPPED) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "svc :%s is already stopped. pid:%d ",
						out_svc->svc_key,
						(int)(out_svc->pid));
		return	-1;
	}

	out_svc->pid = 0;
	_prop_core_set_ctrl_state(out_svc, "stopped");

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "OK. service: %s stopped. pid:%d ", out_svc->svc_key, (int)(out_svc->pid));

	return	0;
}

int prop_core_get(const char *name, char *value)
{
    if (!prop_core_inited){
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Not Inited!");
		return	-1;
	}

    return  prop_db_get(name, value);
}

int prop_core_set(const char *name, const char *value)
{
    int     ret_val;
    int     value_len;

    if (!prop_core_inited) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Not Inited!");
		return	-1;
	}

    ret_val = _prop_core_check(name);
    if (ret_val < 0) {
    	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "... Error:[%s:%s] property name is illegal", name, value);
    	return -1;
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, " name:%s  value:%s ", name, value);

    value_len = strlen(value);
    if (value_len >= PROPERTY_VALUE_MAX) {
    	PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "...Error:[%s:%s] valuelen %d >= %d", name, value, value_len, PROPERTY_VALUE_MAX);
    	return -1;
    }

    if (memcmp(name,"ctl.",4) == 0) {
        return  _prop_core_set_ctrl(name, value);
    } else {
        return  _prop_core_set(name, value);
    }
}

int prop_core_init()
{
    prop_svc_init();
    prop_db_init();

    prop_core_inited = 1;
    /* init comflict, trigger data abort */
   /* prop_file_load_svc(); */
    prop_file_load_props();

    return  0;
}
