/* SPDX-License-Identifier: MediaTekProprietary */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include "prop_db.h"
#include "prop_debug.h"
#include "list.h"

#define PROP_DB_HASH_SIZE		32

static struct   list prop_db_hash[PROP_DB_HASH_SIZE];
static int      prop_db_inited=0;

static int _prop_db_hash(const char *prop_key)
{
	int     i, v = 0;
	char    *c;

	for (i = 31, c = (char *)prop_key; i != 0 && *c != '\0'; i--, c++) {
		v += *c;
	}

	return (v % PROP_DB_HASH_SIZE);
}

int _prop_db_add(const char *in_key, const char *in_value)
{
    prop_db_info_t      *prop_item;
    int                 hash_value;
    int                 key_length;
    int                 value_length;
    char                *value_ptr;
    char                *key_ptr;


    /*terminate the length*/
    key_length = strlen(in_key);
    if (key_length >= PROP_DB_KEY_SIZE)
        key_length = PROP_DB_KEY_SIZE - 1;

    value_length = strlen(in_value);
    if (value_length >= PROP_DB_VALUE_SIZE)
        value_length = PROP_DB_VALUE_SIZE - 1;

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "...key:%s value:%s key-len:%d val-len:%d",
					in_key, in_value,
					key_length, value_length);

    /*Allocate the buffer*/
    prop_item = malloc(sizeof(prop_db_info_t));
	if (!prop_item) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Can't malloc buffer!!");
		return -1;
	}
    value_ptr = malloc(value_length+1);
	if (!value_ptr) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Can't malloc value buffer!!");
        free(prop_item);
		return -1;
	}

    key_ptr = malloc(key_length+1);
	if (!key_ptr) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Can't malloc key buffer!!");
        free(prop_item);
        free(value_ptr);
		return -1;
	}

    /*value the item*/
    memset(prop_item, 0x0, sizeof(prop_db_info_t));
	prop_item->prop_value = value_ptr;
	prop_item->prop_key = key_ptr;

    strncpy(prop_item->prop_value, in_value, value_length);
    prop_item->prop_value[value_length] = '\0';

    strncpy(prop_item->prop_key, in_key, key_length);
    prop_item->prop_key[key_length] = '\0';

    /*Link to the hash list*/
	INIT_LIST(&(prop_item->sibling));
    hash_value = _prop_db_hash(prop_item->prop_key);
	list_add(&(prop_item->sibling), &(prop_db_hash[hash_value]));

	return 0;
}

prop_db_info_t *_prop_db_get(const char *in_key)
{
    int             hash_value;
	struct list     *list_head;
	struct list     *list_item;
    prop_db_info_t  *prop_item;

    hash_value = _prop_db_hash(in_key);

    list_head = &(prop_db_hash[hash_value]);
	LIST_CHECK(list_head);

	LIST_FOR_EACH(list_item, list_head) {
		LIST_CHECK(list_item);

        prop_item = (prop_db_info_t *)list_item;
		if (!strcmp(in_key, prop_item->prop_key)) {
			return  prop_item;
		}
	}

	return  NULL;
}

int prop_db_get(const char *in_key, char * value)
{
    prop_db_info_t  *prop_item = NULL;
    int             length;

    if (!prop_db_inited) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Not Inited!");
		return	-1;
	}

    if (!in_key || !value) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Wrong key or value!");
		return	-1;
	}

    prop_item = _prop_db_get(in_key);
    if (!prop_item) {
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Can not find the value! key:%s ", in_key);
        return  -1;
    }

    length = strlen(prop_item->prop_value);
    if (length >= (PROP_DB_VALUE_SIZE-1))
        length = PROP_DB_VALUE_SIZE-1;

    strncpy(value, prop_item->prop_value, length);
    value[length] = '\0';

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, " key:%s..value:%s length:%d ", in_key, value, length);

    return  0;
}

int prop_db_set(const char *in_key, const char *in_value)
{
    prop_db_info_t      *prop_item;
    int                 value_length;
    char                *value_ptr;

    if (!prop_db_inited) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Not Inited!");
		return	-1;
	}

    if (!in_key || !in_value) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Wrong key or value!");
		return	-1;
	}

    /*Add the item*/
    prop_item = _prop_db_get(in_key);
    if (!prop_item) {
        return  _prop_db_add(in_key, in_value);
    }

    /*terminate the length*/
    value_length = strlen(in_value);
    if (value_length >= PROP_DB_VALUE_SIZE)
        value_length = PROP_DB_VALUE_SIZE - 1;

    /*check if old buffer can be re-used*/
    if (prop_item->prop_value) {
        if (value_length <= strlen(prop_item->prop_value)) {
            strncpy(prop_item->prop_value, in_value, value_length);
            prop_item->prop_value[value_length] = '\0';

			PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, " Update the item ");
            return  0;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, " Update item with big wrongs. key:%s value:%s ", in_key, in_value);

    /*shoule allocate first, then delete old one*/
    value_ptr = malloc(value_length+1);
	if (!value_ptr) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Can't malloc for set prop_value!!");
		return -1;
	}
    strncpy(value_ptr, in_value, value_length);
    value_ptr[value_length] = '\0';

    /*need to free old value buffer*/
    if (prop_item->prop_value)
        free(prop_item->prop_value);

    prop_item->prop_value = value_ptr;

    return  0;
}

void prop_db_dump(void)
{
    int             hash_value;
	struct list     *list_head;
	struct list     *list_item;
    prop_db_info_t  *prop_item;

    if (!prop_db_inited)
        return;

	syslog(LOG_DEBUG, "\n\n\r Dumping properties...");

	for (hash_value=0; hash_value<PROP_DB_HASH_SIZE; hash_value++) {
		list_head = &(prop_db_hash[hash_value]);
		if (list_empty(list_head))
			continue;

		LIST_FOR_EACH(list_item, list_head) {
			LIST_CHECK(list_item);

			prop_item = (prop_db_info_t *)list_item;
			syslog(LOG_DEBUG, "\n\r    %s=%s", prop_item->prop_key, prop_item->prop_value);
		}
	}

	printf("\n");

    return;
}

void prop_db_init(void)
{
    int i;

	for (i=0; i<PROP_DB_HASH_SIZE; i++) {
		INIT_LIST(&prop_db_hash[i]);
	}

    prop_db_inited = 1;
}
