/* SPDX-License-Identifier: MediaTekProprietary */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <syslog.h>
#include "prop_svc.h"
#include "prop_debug.h"
#include "list.h"

static struct list prop_svc_list;

/*
*  parse out the daemon-name.
*  Format : service service-name /path/daemon-name
*/
int prop_svc_get_daemon(const prop_svc_t *act_item, char *in_buf)
{
	struct list 		*act_node;
	prop_svc_act_t		*svc_act;
	char				*buf_ptr;
	int					len, index;

	if (!act_item || !in_buf)
		return	-1;

	LIST_FOR_EACH(act_node, &(act_item->act_list)) {
		LIST_CHECK(act_node);
		svc_act = LIST_ENTRY(act_node, prop_svc_act_t, alist);
		if (svc_act) {
			/*act_name: service service-name /path/daemon-name*/
			if (svc_act->act_name) {
				/*filter out the space*/
				len = strlen(svc_act->act_name);
				buf_ptr = svc_act->act_name;
				for (index=len-1; index>=0; index--) {
					if (buf_ptr[index] == ' ')
						continue;

					break;
				}

				/*find the reverse ' ' or '/' */
				for (; index>=0; index--) {
					if ((buf_ptr[index] == ' ') ||
						(buf_ptr[index] == '/')) {
						strcpy(in_buf, &(buf_ptr[index+1]));
						PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Get the actions...act_name:%s daemon-name:%s",
															svc_act->act_name, in_buf);

						return	0;
					}
				}
			}
		}
	}

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Can not Get the actions...name:%s", act_item->svc_key);
	return	-1;

}

/*
*  parse out the daemon-name.
*  Format : service service-name /path/daemon-name
*/
int prop_svc_get_daemon_ext(const prop_svc_t *act_item, char *in_buf)
{
	struct list 		*act_node = NULL;
	prop_svc_act_t		*svc_act = NULL;
	char				*buf_ptr = NULL;

	if (!act_item || !in_buf)
		return	-1;

	LIST_FOR_EACH(act_node, &(act_item->act_list)) {
		LIST_CHECK(act_node);
		svc_act = LIST_ENTRY(act_node, prop_svc_act_t, alist);
		/*act_name: service service-name /path/daemon-name*/
		if (svc_act && svc_act->act_name) {
			if (NULL != (buf_ptr = strchr(svc_act->act_name, '/'))) {
				strcpy(in_buf, buf_ptr);
				PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Get the actions...act_name:%s daemon-name:%s",
														svc_act->act_name, in_buf);
				return	0;
			}
		}
	}

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Can not Get the actions...name:%s", act_item->svc_key);
	return	-1;
}

pid_t prop_svc_get_pid(const prop_svc_t *act_item)
{
#define	ACTION_BUF_LEN	256
	char	daemon_name[PROP_SVC_ACTION_MAX_SIZE] = {0};
	char	action_buf[ACTION_BUF_LEN] = {0};
	char	buffer[10] = {0};
	FILE	*fp = NULL;
	int		len = 0;
	pid_t	pid = -1;

	if (prop_svc_get_daemon_ext(act_item, daemon_name) < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to get the svc's action: %s", act_item->svc_key);
		return pid;
	}

	len = snprintf(action_buf, ACTION_BUF_LEN-1, "pgrep -f \'%s\'", daemon_name);
	if (len >= (ACTION_BUF_LEN-1)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Length is error: %s", daemon_name);
		return pid;
	}

	fp = popen(action_buf, "r");
	if (NULL != fgets(buffer, sizeof(buffer)-1, fp)) {
		pid = strtoul(buffer, 0, 10);
		PROP_DEBUG_PRINT(PROP_DEBUG_INFO, "success to fgets: %s (%d)", action_buf, pid);
	} else {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "failed to fgets: %s", action_buf);
	}

	pclose(fp);
	return pid;
}

int prop_svc_get_actions(const prop_svc_t *act_item, char *in_buf)
{
	int					cur_act_len;
	struct list 		*act_node;
	prop_svc_act_t		*svc_act;
	char				*buf_ptr;

	if (!act_item || !in_buf)
		return	-1;

	buf_ptr = in_buf;
	cur_act_len = 0;

	LIST_FOR_EACH(act_node, &(act_item->act_list)) {
		LIST_CHECK(act_node);
		svc_act = LIST_ENTRY(act_node, prop_svc_act_t, alist);
		if (svc_act) {
			if (svc_act->act_name) {
				if ((cur_act_len + strlen(svc_act->act_name)) >= PROP_SVC_ACTION_BUF_SIZE) {
					PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " action max length error!...name:%s length:%d cur_len:%d ",
							svc_act->act_name, strlen(svc_act->act_name), cur_act_len);
				} else {
					if (cur_act_len) {
						buf_ptr[cur_act_len] = ';';
						cur_act_len += 1;	/*for ';'*/
					}

					strncpy(&(buf_ptr[cur_act_len]), svc_act->act_name, strlen(svc_act->act_name));
					cur_act_len += strlen(svc_act->act_name);
					buf_ptr[cur_act_len] = '\0';

					PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Find! name:%s actions:%s buffer:%s length:%d ",
										act_item->svc_key, svc_act->act_name,
										buf_ptr, cur_act_len);

				}
			} else {
				PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " find the action without name???");
			}
		}
	}

	in_buf[cur_act_len] = '\0';

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Get the actions...name:%s action:%s cur_act_len:%d ",
				act_item->svc_key, in_buf, cur_act_len);

	return	0;
}

int prop_svc_get_actions_by_name(const char *svc_name, const char *svc_key, const char *svc_value, char *in_buf)
{
	prop_svc_t 		*svc_item;

	if (!svc_key || !in_buf)
		return	-1;

	svc_item =  prop_svc_find_by_name(svc_name, svc_key, svc_value);
	if (!svc_item) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "svc not exist...name:%s ", svc_key);
		return	-1;
	}

	if (prop_svc_get_actions(svc_item, in_buf) < 0) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "can not find the actions...name:%s ", svc_key);
		return	-1;
	}

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "End. name:%s buf:%s ", svc_key, in_buf);

	return	0;
}

prop_svc_t * prop_svc_find_by_name(const char *svc_name, const char *prop_key, const char *prop_value)
{
    struct list                 *node;
    prop_svc_t                  *svc;

    if (!prop_key) {
        return  NULL;
    }

    LIST_FOR_EACH(node, &prop_svc_list) {
		LIST_CHECK(node);
        svc = LIST_ENTRY(node, prop_svc_t, slist);
        if (!strcmp(svc->svc_name, svc_name)) {
    		if (prop_value) {
    	        if (!strcmp(svc->svc_key, prop_key) && !strcmp(svc->svc_value, prop_value)) {
    				PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Find the svc! db_key:%s len:%d db_value:%s len:%d ",
    													svc->svc_key, strlen(svc->svc_key),
    													svc->svc_value, strlen(svc->svc_value));
    	            return svc;
    	        }
    		} else {
    	        if (!strcmp(svc->svc_key, prop_key)) {
    				PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Find the svc! db_key:%s len:%d ",
    													svc->svc_key, strlen(svc->svc_key));
    	            return svc;
    	        }
    		}
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Can not find! key:%s len:%d ",
										prop_key, strlen(prop_key));

    return NULL;
}

prop_svc_t * prop_svc_find_by_pid(const pid_t pid)
{
    struct list                 *node;
    prop_svc_t                  *svc;

    LIST_FOR_EACH(node, &prop_svc_list) {
		LIST_CHECK(node);
        svc = LIST_ENTRY(node, prop_svc_t, slist);
        if (svc->pid == pid) {
			PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, " Find the svc %s", svc->svc_key);
            return svc;
        }
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Can not find. pid:%d", pid);

    return NULL;

}

int prop_svc_add(const prop_svc_t *svc)
{
    prop_svc_t  *db_item;

    if (!svc)
        return  -1;

	if (strlen(svc->svc_key) >= PROPERTY_KEY_MAX) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Fail name. key:%s length:%d ", svc->svc_key, strlen(svc->svc_key));
		return	-1;
	}
	if (strlen(svc->svc_value) >= PROPERTY_VALUE_MAX) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Fail name. value:%s length:%d ", svc->svc_value, strlen(svc->svc_value));
		return	-1;
	}
	if (strlen(svc->svc_name) >= PROP_SVC_TYPE_MAX_SIZE) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, " Fail name. name:%s length:%d ", svc->svc_name, strlen(svc->svc_name));
		return	-1;
	}

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "...1...type:%s len:%d key:%s len:%d value:%s len:%d",
					 svc->svc_name, strlen(svc->svc_name),
					 svc->svc_key, strlen(svc->svc_key),
					 svc->svc_value, strlen(svc->svc_value));

    db_item = prop_svc_find_by_name(svc->svc_name, svc->svc_key, svc->svc_value);
	if (db_item) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Item already existed...name:%s ", svc->svc_key);
        return  -1;
    }

    db_item = malloc(sizeof(prop_svc_t));
    if (!db_item) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Fail to malloc..name:%s ", svc->svc_key);
		return	-1;
	}

    memset(db_item, 0x0, sizeof(prop_svc_t));

	strcpy(db_item->svc_name, svc->svc_name);
	strcpy(db_item->svc_key, svc->svc_key);
	strcpy(db_item->svc_value, svc->svc_value);
    db_item->svc_type = svc->svc_type;

	db_item->flags = svc->flags;
	INIT_LIST(&(db_item->slist));
	INIT_LIST(&(db_item->act_list));
	list_add_tail((struct list *)db_item, &prop_svc_list);

    return  0;
}

int prop_svc_add_act(const char *svc_name, const char *svc_key, const char *svc_value, const char *actions)
{
    prop_svc_t  	*db_item;
    prop_svc_act_t	*act_item;
    int         	action_len;
	int				index;
	char			*args_temp;

    if (!svc_key || !actions)
        return  -1;

	if (svc_value)
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "name:%s svc_key:%s svc_value:%s actions:%s ", svc_name, svc_key, svc_value, actions);
	else
		PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "name:%s svc_key:%s svc_value:NULL actions:%s ", svc_name, svc_key, actions);


    /*check the length ???*/
    action_len = strlen(actions);
    if ((PROP_SVC_ACTION_MAX_SIZE <= action_len) ||
		(!action_len)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "action length:%d ", action_len);
		return	-1;
	}

    db_item = prop_svc_find_by_name(svc_name, svc_key, svc_value);
    if (!db_item) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "can not find the svc");
		return	-1;
	}

    act_item = malloc(sizeof(prop_svc_act_t));
    if (!act_item) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "can not malloc");
		return	-1;
	}

	memset(act_item, 0x0, sizeof(prop_svc_act_t));

	/*parse the args and args_num*/
    strcpy(act_item->line_buf, actions);
	args_temp = act_item->line_buf;
	index = 0;
	while (args_temp[index] == ' ')index++;
	if ((args_temp[index] == '\0') ||
		(index >= action_len)) {
		PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Wrong line! action_len:%d index:%d ", action_len, index);
        free(act_item);
		return	-1;
	}

	act_item->args_num = 0;
	while (args_temp[index] != '\0') {
		act_item->args[act_item->args_num] = &(args_temp[index]);
		act_item->args_num++;
		while ((args_temp[index] != ' ') &&
				(args_temp[index] != '\0')) {
			index++;
		}
		if (index >= action_len)
			break;

		args_temp[index] = '\0';
		index++;

		while (args_temp[index] == ' ') index++;
		if (index >= action_len)
			break;
	}

    strcpy(act_item->act_name, actions);
	INIT_LIST(&(act_item->alist));
	list_add_tail(&(act_item->alist), &(db_item->act_list));

    return  0;
}

prop_svc_t * prop_svc_exe_by_name(const char *svc_name, const char *prop_key, const char *prop_value, void (*callabck)())
{
    struct list                 *node;
    prop_svc_t                  *svc;
    int                         find_times = 0;

    if (!prop_key) {
        PROP_DEBUG_PRINT(PROP_DEBUG_ERROR, "Wrong key!");

        return  NULL;
    }

    LIST_FOR_EACH(node, &prop_svc_list) {
		LIST_CHECK(node);
        svc = LIST_ENTRY(node, prop_svc_t, slist);

        if (svc_name && (strcmp(svc->svc_name, svc_name))) {
            continue;
        }

		if (prop_value) {
	        if (!strcmp(svc->svc_key, prop_key) && !strcmp(svc->svc_value, prop_value)) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Find the svc!  db_key:%s len:%d db_value:%s len:%d ",
													svc->svc_key, strlen(svc->svc_key),
													svc->svc_value, strlen(svc->svc_value));
                find_times++;
                (*callabck)(svc);
	        }
		} else {
	        if (!strcmp(svc->svc_key, prop_key)) {
				PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Find the svc! name:%s db_key:%s len:%d ",
                                                    svc->svc_value, svc->svc_key, strlen(svc->svc_key));
                find_times++;
                (*callabck)(svc);
	        }
		}
    }

	PROP_DEBUG_PRINT(PROP_DEBUG_TRACE, "Exe the svc! key:%s len:%d value:%s len:%d find:%d ",
										prop_key, strlen(prop_key),
										prop_value, strlen(prop_value), find_times);

    return NULL;
}



void prop_svc_dump(void)
{
	struct list 				*node;
	struct list 				*act_node;
	prop_svc_t					*svc;
	prop_svc_act_t				*svc_act;
	int							index;
	int							is_service;

	syslog(LOG_DEBUG, "\n\n\rDumping services...");

	LIST_FOR_EACH(node, &prop_svc_list) {
		LIST_CHECK(node);

		svc = LIST_ENTRY(node, prop_svc_t, slist);
		if (svc) {
			if (svc->svc_type == PROP_SVC_TYPE_SERVICE)
				is_service = 1;
			else
				is_service = 0;

			if (!is_service)
				syslog(LOG_DEBUG, "\n\n\r%s    on property:%s=%s", svc->svc_name, svc->svc_key, svc->svc_value);
			else
				syslog(LOG_DEBUG, "\n\n\r%s    service:%s %s", svc->svc_name, svc->svc_key, svc->svc_value);
			LIST_FOR_EACH(act_node, &(svc->act_list)) {
				LIST_CHECK(act_node);
				svc_act = LIST_ENTRY(act_node, prop_svc_act_t, alist);
				if (svc_act) {
                    syslog(LOG_DEBUG, "\n\r     %s", svc_act->act_name);

					for (index=0; index<svc_act->args_num; index++) {
						syslog(LOG_DEBUG, "\n\r         (args[%d]:%s len:%d)", index, svc_act->args[index], strlen(svc_act->args[index]));
					}
				}
			}
		}
	}
	printf("\n");
}

void prop_svc_init(void)
{
    INIT_LIST(&prop_svc_list);
}
